/**
 *  Copyright © 2021-2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.client.administer;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import javax.swing.JPanel;
import javax.swing.JSeparator;
import javax.swing.JTextField;
import net.javacomm.client.resource.Resource;
import net.javacomm.multilingual.Babelfish;
import net.javacomm.multilingual.MultilingualButton;
import net.javacomm.multilingual.MultilingualLabel;
import net.javacomm.multilingual.schema.ISO639;
import net.javacomm.multilingual.schema.KEY;
import net.javacomm.window.manager.Control;



@SuppressWarnings("serial")
public class JAdministerForum extends JPanel implements Babelfish {

  private GridBagLayout gridbag = new GridBagLayout();
  private GridBagConstraints con = new GridBagConstraints();
  private MultilingualLabel labelMinutetext = new MultilingualLabel(KEY.LABEL_DER_SCHWELLENWERT);
  private MultilingualLabel labelMinute = new MultilingualLabel(KEY.LABEL_MINUTEN);
  private JSeparator separator = new JSeparator();
  private MultilingualButton buttonSchliessen = new MultilingualButton(KEY.BUTTON_SCHLIESSEN);
  private MultilingualButton buttonSpeichern = new MultilingualButton(KEY.BUTTON_SPEICHERN);
  private GridLayout grid = new GridLayout(1, 2, 12, 12);
  private JPanel buttonpane = new JPanel(grid);
  private PropertyChangeSupport changes = new PropertyChangeSupport(this);
  private FocusAdapter focusAdapter = new FocusAdapter() {
    @Override
    public void focusGained(FocusEvent event) {
      textfieldMinute.setBackground(Color.WHITE);
      textfieldMinute.setForeground(Color.BLACK);
    }
  };

  private JTextField textfieldMinute = new JTextField(3);
  private final int margin_top = 12;
  private final int margin_left = 12;
  private final int margin_right = 12;

  public JAdministerForum() {
    setLayout(gridbag);

    con.gridx = 0;
    con.gridy = 0;
    con.gridwidth = 1;
    con.gridheight = 1;
    con.fill = GridBagConstraints.HORIZONTAL;
    con.weightx = 0.0;
    con.weighty = 0.0;
    con.insets = new Insets(margin_top, margin_left, 0, 0);
    con.anchor = GridBagConstraints.CENTER;
    gridbag.setConstraints(labelMinutetext, con);
    add(labelMinutetext);

    con.gridx = 1;
    con.gridy = 0;
    con.gridwidth = 1;
    con.gridheight = 1;
    con.fill = GridBagConstraints.NONE;
    con.weightx = 0.0;
    con.weighty = 0.0;
    con.insets = new Insets(margin_top, 48, 0, 6);
    con.anchor = GridBagConstraints.CENTER;
    gridbag.setConstraints(textfieldMinute, con);
    add(textfieldMinute);
    textfieldMinute.addFocusListener(focusAdapter);

    con.gridx = 2;
    con.gridy = 0;
    con.gridwidth = 1;
    con.gridheight = 1;
    con.fill = GridBagConstraints.NONE;
    con.weightx = 0.0;
    con.weighty = 0.0;
    con.insets = new Insets(margin_top, 0, 0, margin_right);
    con.anchor = GridBagConstraints.WEST;
    gridbag.setConstraints(labelMinute, con);
    add(labelMinute);

    con.gridx = 0;
    con.gridy = 1;
    con.gridwidth = 3;
    con.gridheight = 1;
    con.fill = GridBagConstraints.HORIZONTAL;
    con.weightx = 0.0;
    con.weighty = 0.0;
    con.insets = new Insets(24, margin_left, 0, margin_right);
    con.anchor = GridBagConstraints.CENTER;
    gridbag.setConstraints(separator, con);
    add(separator);

    buttonpane.add(buttonSpeichern);
    buttonpane.add(buttonSchliessen);
    con.gridx = 1;
    con.gridy = 2;
    con.gridwidth = 2;
    con.gridheight = 1;
    con.fill = GridBagConstraints.NONE;
    con.weightx = 0.0;
    con.weighty = 0.0;
    con.insets = new Insets(12, margin_left, 0, margin_right);
    con.anchor = GridBagConstraints.EAST;
    gridbag.setConstraints(buttonpane, con);
    add(buttonpane);

    buttonSchliessen.addActionListener((event) -> {
      changes.firePropertyChange(JAdministerForum.class.getName(), Control.NULL, Control.CLOSE);
    });

    buttonSpeichern.addActionListener((event) -> {
      changes.firePropertyChange(JAdministerForum.class.getName(), Control.NULL, Control.SAVE);
    });

  }



  public synchronized void addForumListener(PropertyChangeListener listener) {
    changes.addPropertyChangeListener(listener);
  }



  public synchronized void removeForumListener(PropertyChangeListener listener) {
    changes.removePropertyChangeListener(listener);
  }



  public synchronized void removeAllListener() {
    for (PropertyChangeListener listener : changes.getPropertyChangeListeners()) {
      removeForumListener(listener);
    }
    textfieldMinute.removeFocusListener(focusAdapter);
  }



  /**
   * Der Schwellenwert, ab wann ein öffentlicher Raum in ein Forum umgewandelt
   * wird.
   *
   *
   * @param value
   *              ein Minutenwert
   */
  public void setZeitschwelleForum(int value) {
    textfieldMinute.setText(String.valueOf(value));
  }



  /**
   * Der Schwellenwert in Minuten wird zurückgegeben.
   *
   *
   * @return ein Minutenwert
   *
   * @throws ZeitschwelleException
   *                               der Rückgabewert ist kein Zahlenwert
   */
  public int getZeitschwelleForum() throws ZeitschwelleException {
    String minute = textfieldMinute.getText();
    Integer value;
    try {
      value = Integer.valueOf(minute);
      if ((value < 1) || (value > 1440)) throw new ZeitschwelleException(minute);
    }
    catch (NumberFormatException e) {
      textfieldMinute.setBackground(Resource.JQUERY_RED);
      textfieldMinute.setForeground(Color.WHITE);
      throw new ZeitschwelleException(minute);
    }
    return value;
  }



  @Override
  public void setLanguage(ISO639 code) {
    buttonSchliessen.setLanguage(code);
    buttonSpeichern.setLanguage(code);
    labelMinutetext.setLanguage(code);
    labelMinute.setLanguage(code);
  }

}
