/**
 *  Copyright © 2021-2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.client.administer;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.Enumeration;
import java.util.List;
import javax.swing.DefaultListModel;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSeparator;
import javax.swing.ListSelectionModel;
import net.javacomm.client.gui.MultilingualServiceButton;
import net.javacomm.multilingual.Babelfish;
import net.javacomm.multilingual.MultilingualButton;
import net.javacomm.multilingual.MultilingualLabel;
import net.javacomm.multilingual.MultilingualString;
import net.javacomm.multilingual.schema.ISO639;
import net.javacomm.multilingual.schema.KEY;
import net.javacomm.window.manager.Control;



public class JProgrammversionen extends JPanel implements Babelfish {

  private static final long serialVersionUID = 7392078262880453399L;
  private GridBagLayout gridbag = new GridBagLayout();
  private GridBagConstraints con = new GridBagConstraints();
  private MultilingualLabel labelVersionen = new MultilingualLabel(KEY.LABEL_VERALTETE_VERSIONEN);
  private JScrollPane scrollpane = new JScrollPane();
  private DefaultListModel<String> model = new DefaultListModel<>();
  private JList<String> listFilter = new JList<>(model);
  private JSeparator separator = new JSeparator();

  private JPanel buttonVerwalten = new JPanel();
  private GridLayout gridVerwalten = new GridLayout(2, 1, 0, 12);
  private MultilingualButton buttonAdd = new MultilingualButton(KEY.BUTTON_VERSION_HINZUFUEGEN);
  private MultilingualButton buttonRemove = new MultilingualButton(KEY.BUTTON_VERSION_ENTFERNEN);
  private ActionListener addActionListener;
  private ActionListener removeActionListener;
  private PropertyChangeSupport changes = new PropertyChangeSupport(this);

  private MultilingualServiceButton serviceButton = new MultilingualServiceButton(
      KEY.BUTTON_SPEICHERN, KEY.BUTTON_SCHLIESSEN
  );
  private MultilingualString outdatedVersion = new MultilingualString(KEY.STRING_VERALTETE_VERSION);
  private MultilingualString version = new MultilingualString(KEY.STRING_VERSION);

  private final int margin_top = 12;
  private final int margin_left = 12;
  private final int margin_right = 12;

  public JProgrammversionen() {
    setLayout(gridbag);

    con.gridx = 0;
    con.gridy = 0;
    con.gridwidth = 1;
    con.gridheight = 1;
    con.fill = GridBagConstraints.NONE;
    con.weightx = 0.0;
    con.weighty = 0.0;
    con.insets = new Insets(margin_top, margin_left, 0, margin_right);
    con.anchor = GridBagConstraints.CENTER;
    gridbag.setConstraints(labelVersionen, con);
    add(labelVersionen);

    con.gridx = 0;
    con.gridy++;
    con.gridwidth = 1;
    con.gridheight = 1;
    con.fill = GridBagConstraints.BOTH;
    con.weightx = 1.0;
    con.weighty = 1.0;
    con.insets = new Insets(0, margin_left, 0, 0);
    con.anchor = GridBagConstraints.CENTER;
    gridbag.setConstraints(scrollpane, con);
    add(scrollpane);
    scrollpane.setViewportView(listFilter);
    scrollpane.getHorizontalScrollBar().setUnitIncrement(12);
    scrollpane.getVerticalScrollBar().setUnitIncrement(12);
    listFilter.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);

    con.gridx = 1;
    con.gridy = 1;
    con.gridwidth = 1;
    con.gridheight = 1;
    con.fill = GridBagConstraints.NONE;
    con.weightx = 0.0;
    con.weighty = 0.0;
    con.insets = new Insets(0, 0, 0, 0);
    con.anchor = GridBagConstraints.CENTER;
    gridbag.setConstraints(buttonVerwalten, con);
    add(buttonVerwalten);
    buttonVerwalten.setLayout(gridVerwalten);
    buttonVerwalten.add(buttonAdd);
    buttonVerwalten.add(buttonRemove);
    addActionListener = new ActionListener() {

      @Override
      public void actionPerformed(ActionEvent e) {
        String input = JOptionPane
            .showInputDialog(
                JProgrammversionen.this, outdatedVersion.toString(), version.toString(),
                JOptionPane.QUESTION_MESSAGE
            );
        if (input == null) return;
        if (model.contains(input)) return;
        model.addElement(input);
      }
    };
    buttonAdd.addActionListener(addActionListener);

    removeActionListener = new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        List<String> selectedValues = listFilter.getSelectedValuesList();
        selectedValues.forEach((value) -> {
          model.removeElement(value);
        });
      }
    };
    buttonRemove.addActionListener(removeActionListener);

    con.gridx = 0;
    con.gridy++;
    con.gridwidth = 2;
    con.gridheight = 1;
    con.fill = GridBagConstraints.HORIZONTAL;
    con.weightx = 0.0;
    con.weighty = 0.0;
    con.insets = new Insets(24, margin_left, 0, margin_right);
    con.anchor = GridBagConstraints.CENTER;
    gridbag.setConstraints(separator, con);
    add(separator);

    con.gridx = 1;
    con.gridy++;
    con.gridwidth = 1;
    con.gridheight = 1;
    con.fill = GridBagConstraints.NONE;
    con.weightx = 0.0;
    con.weighty = 0.0;
    con.insets = new Insets(12, margin_left, 12, margin_right);
    con.anchor = GridBagConstraints.EAST;
    gridbag.setConstraints(serviceButton, con);
    add(serviceButton);
    serviceButton.setMnemonic(KeyEvent.VK_E, KeyEvent.VK_S);
    serviceButton.addButtonListener(event -> {
      Control value = (Control) event.getNewValue();
      switch(value) {
        case BUTTON_1:
          changes.firePropertyChange(JProgrammversionen.class.getName(), 0, Control.SAVE);
          break;
        case BUTTON_2:
          changes.firePropertyChange(JProgrammversionen.class.getName(), 0, Control.CLOSE);
          break;
        default:
          break;
      }

    });
    serviceButton.setEnabledButton1();

  }



  public synchronized void addProgrammversionenListener(PropertyChangeListener l) {
    changes.addPropertyChangeListener(l);
  }



  public synchronized void removeProgrammversionenListener(PropertyChangeListener l) {
    changes.removePropertyChangeListener(l);
  }



  public synchronized void removeAllListener() {
    serviceButton.removeAllListener();
    for (PropertyChangeListener listener : changes.getPropertyChangeListeners()) {
      removeProgrammversionenListener(listener);
    }
  }



  public Enumeration<String> getVersionen() {
    return model.elements();
  }



  /**
   * Die übergebenen Versionen sind veraltet.
   *
   * @param versionen
   *                  veraltete Versionen
   */
  public void setVersionen(List<String> versionen) {
    model.clear();
    model.addAll(versionen);
  }



  @Override
  public void setLanguage(ISO639 code) {
    serviceButton.setLanguage(code);
    buttonAdd.setLanguage(code);
    buttonRemove.setLanguage(code);
    labelVersionen.setLanguage(code);
    outdatedVersion.setLanguage(code);
    version.setLanguage(code);
  }

}
