/**
 *  Copyright © 2020-2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.client.administer;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.FlowLayout;
import java.awt.event.WindowEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.regex.Matcher;
import javax.swing.JFrame;
import javax.swing.JPanel;
import org.nexuswob.util.Util;
import net.javacomm.client.resource.Resource;
import net.javacomm.multilingual.Babelfish;
import net.javacomm.multilingual.MultilingualDialog;
import net.javacomm.multilingual.MultilingualString;
import net.javacomm.multilingual.schema.ISO639;
import net.javacomm.multilingual.schema.KEY;
import net.javacomm.window.manager.Control;
import net.javacomm.window.manager.WMResource;



/**
 * Der RegistryDialog wird für eine Registrierung und eine Passwortanforderung
 * benutzt.
 *
 *
 *
 * @author llange
 *
 */
public class JRegistryFrame extends MultilingualDialog implements Babelfish {

//  private final static Logger log = LogManager.getLogger(JRegistryFrame.class);

  private static final long serialVersionUID = -6074436143307561438L;
  public final static int width = 501;
  public final static int height = 330;

  private JPanel mainPane = new JPanel(new BorderLayout(), true);
  private JRegistryPanel registryPanel = new JRegistryPanel(true);
  private JPanel pansouth = new JPanel(true);
  private PropertyChangeSupport changes = new PropertyChangeSupport(this);
  private String email;
  private boolean confirm = false;
  private RegistryButton knopfleiste;
  private MultilingualString invalidEmail = new MultilingualString(KEY.SERVER_EMAIL_INVALID);
  private MultilingualString esGehtGleichWeiter = new MultilingualString(KEY.STRING_ES_GEHT_GLEICH_WEITER);
  private ISO639 code;

  /**
   * Dieser Konstruktor wird über den JavaComm Anmeldedialog aufgerufen.
   *
   * @param frame
   *                    der Dialog ist an diesen Frame gebunden
   * @param knopfleiste
   *                    die Leiste enthält alle Knöpfe, die der Anwender klicken
   *                    kann
   */
  public JRegistryFrame(JFrame frame, RegistryButton knopfleiste) {
    this(frame, knopfleiste, "Registrierung");
  }



  /**
   * Dieser Konstruktor wird direkt von JavaComm/Neues Passwort anfordern
   * aufgerufen.
   *
   * @param frame
   *                    der Dialog ist an diesen Frame gebunden
   * @param knopfleiste
   *                    die Leiste enthält alle Knöpfe, die der Anwender klicken
   *                    kann
   * @param title
   *                    der Fenstertitel
   */
  public JRegistryFrame(JFrame frame, RegistryButton knopfleiste, String title) {
    super(frame, title, true);
    this.knopfleiste = knopfleiste;
    setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
    setContentPane(mainPane);
    knopfleiste.addKnopfListener(event -> {
      Control rc = (Control) event.getNewValue();
      switch(rc) {
        case OHNE_REGISTRIEREN:
          processWindowEvent(new WindowEvent(this, WindowEvent.WINDOW_CLOSING));
          changes.firePropertyChange(JRegistryFrame.class.getName(), Control.NULL, Control.OHNE_REGISTRIEREN);
          break;
        case WEITER:
          processWindowEvent(new WindowEvent(this, WindowEvent.WINDOW_CLOSING));
          if (confirm) {
            // ein neuer Anwender wurde angelegt
            changes.firePropertyChange(JRegistryFrame.class.getName(), Control.NULL, Control.NEUES_PASSWORD);
          }
          else {
            changes.firePropertyChange(JRegistryFrame.class.getName(), Control.NULL, Control.CONTINUE);
          }
          break;
        case REGISTRIEREN:
          Matcher matcher = Util.PATTERN_EMAIL.matcher(registryPanel.getEmail());
          if (!matcher.matches()) {
            registryPanel.setMessage(invalidEmail.toString(), Color.WHITE, Resource.JQUERY_RED);
            return;
          }
          email = registryPanel.getEmail();
          /* absenden */
          registryPanel
              .setMessage(esGehtGleichWeiter.toString(), Resource.JQUERY_YELLOW, Resource.JQUERY_BLAU);
          changes.firePropertyChange(JRegistryFrame.class.getName(), Control.NULL, Control.REGISTER);
          break;
        default:
          break;
      }
    });
    registryPanel.addSignListener((event) -> {
      Control rc = (Control) event.getNewValue();
      switch(rc) {
        case REGISTER:
          knopfleiste.hitSign();
          break;
        default:
          break;
      }
    });

    pansouth.setLayout(new FlowLayout(FlowLayout.RIGHT, 12, 12));
    pansouth.add(knopfleiste);

    mainPane.add(BorderLayout.CENTER, registryPanel);
    mainPane.add(BorderLayout.SOUTH, pansouth);
    setSize(width, height);
    setBounds(
        ((frame.getBounds().width - width) >> 1) + (int) frame.getBounds().getX(),
        ((frame.getBounds().height - height) >> 1) + (int) frame.getBounds().getY(), width, height
    );
  }



  public void addRegisterListener(PropertyChangeListener l) {
    changes.addPropertyChangeListener(l);
  }



  /**
   * Der Anwender wurde als neuer User mit CONFIRM registriert.
   */
  public void confirmUser() {
    confirm = true;
  }



  /**
   *
   * Nach einer erfolgreichen Anmeldung werden keine erneuten Eingaben
   * entgegengenommen.
   *
   */
  public void disableInput() {
    registryPanel.removeAllSignListener();
    knopfleiste.buttonRegistrieren.removeActionListener(knopfleiste.registrierenListener);
    knopfleiste.buttonRegistrieren.setForeground(WMResource.DISABLED_BUTTON_FOREGROUND);
    knopfleiste.buttonRegistrieren.setBackground(WMResource.DISABLED_BUTTON_BACKGROUND);
  }



  public String getEmail() {
    return (email);
  }



  public synchronized void removeAllRegisterListener() {
    for (PropertyChangeListener listener : changes.getPropertyChangeListeners()) {
      removeRegisterListener(listener);
    }
    knopfleiste.removeAllListener();
  }



  public synchronized void removeRegisterListener(PropertyChangeListener l) {
    changes.removePropertyChangeListener(l);
  }



  public void setTextArea(String data, Color c) {
    registryPanel.setMessage(data, c);
  }



  public void setTextArea(String data, Color foreground, Color background) {
    registryPanel.setMessage(data, foreground, background);
  }



  /**
   * Die Spracheinstellung im Fenster.
   * 
   * @return Deutsch, Englisch oder Spanisch
   */
  public ISO639 getLanguage() {
    return code;
  }



  @Override
  public void setLanguage(ISO639 code) {
    this.code = code;
    registryPanel.setLanguage(code);
    super.setLanguage(code);
    invalidEmail.setLanguage(code);
    esGehtGleichWeiter.setLanguage(code);
  }

}
