/**
 *  Copyright © 2022-2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.client.base;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;



/**
 * AudioContainer nimmt Arrays von {@code byte[]} entgegen und speichert sie
 * intern in einem {@code ByteArrayOutputStream} ab. Die überegebenen Felder
 * werden zusammengefügt und als Ganzes über {@link #get()} zurückgegeben.
 * 
 * @author llange
 *
 */
public final class AudioContainer {

  private final static Logger log = LogManager.getLogger(AudioContainer.class);
  private ByteArrayOutputStream buffer;
  private String session;




  /**
   * 
   * 
   * @param session
   *                die übergebenen Audiopakete gehören zu dieser Session
   */
  public AudioContainer(String session) {
    this.session = session;
    buffer = new ByteArrayOutputStream();
  }



  /**
   * Ein Teilaudiopaket aus einem Stream.
   * 
   * 
   * @param value
   *              ein Audiopaket
   * @throws IOException
   */
  public void add(byte[] value) throws IOException {
      buffer.write(value);
//      log.info(session + "," + buffer.size());
  }



  /**
   * Ein ByteArray mit Audiodaten wird zurückgegeben.
   * 
   * @return {@code byte[]} enthält Audiodaten
   */
  public byte[] get() {
    return buffer.toByteArray();
  }



  /**
   * Das ByteArray wird auf 0 zurückgesetzt. {@link #reset()} sollte nach
   * {@link #get()} aufgerufen werden, damit das Gesprochene nicht erneut
   * abgespielt wird.
   * 
   */
  public void reset() {
    buffer.reset();
  }



  /**
   * Wie groß ist das ByteArray mit den Audiodaten?
   * 
   * @return Anzahl Bytes
   */
  public int size() {
    return buffer.size();
  }


  /**
   * Zu welcher Session gehört der Container?
   * 
   * 
   * @return die zugehörige Session
   */
  public String getSession() {
    return session;
  }

}
