/**
 *  Copyright © 2020-2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.client.chat;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.GridLayout;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.List;
import javax.swing.DefaultListModel;
import javax.swing.ImageIcon;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.ListSelectionModel;
import javax.swing.SwingConstants;
import org.nexuswob.gui.swing.NimbusCellRenderer;
import net.javacomm.client.resource.Resource;
import net.javacomm.multilingual.Babelfish;
import net.javacomm.multilingual.MultilingualLabel;
import net.javacomm.multilingual.schema.ISO639;
import net.javacomm.multilingual.schema.KEY;
import net.javacomm.protocol.UserOnline;
import net.javacomm.window.manager.Control;



public class JOnlineliste extends JPanel implements Babelfish {

  private static final long serialVersionUID = -8434948265792542624L;

  private JScrollPane scrollpane = new JScrollPane();
  private JList<String> list = new JList<String>();
  private JPanel panNorth = new JPanel(true);
  private DefaultListModel<String> model = new DefaultListModel<String>();
  private PropertyChangeSupport changes = new PropertyChangeSupport(this);
  private MultilingualLabel online = new MultilingualLabel(KEY.LABEL_ONLINE);
  private String benutzer;
  private JLabel lefticon;
  private JLabel righticon;

  public JOnlineliste() {
    setLayout(new BorderLayout());
    add(BorderLayout.NORTH, panNorth);
    add(BorderLayout.CENTER, scrollpane);
    init();
  }



  public void addOnlinelistListener(PropertyChangeListener l) {
    changes.addPropertyChangeListener(l);
  }



  public void clearSelection() {
    list.clearSelection();
  }



  /**
   * Die Onlineliste erhält den Focus.
   * 
   * @return ein Benutzer ist markiert
   */
  public boolean focusList() {
    if (list.getSelectedIndex() == -1) {
      // nichts markiert
      list.setSelectedIndex(0);
    }
    benutzer = list.getSelectedValue();
    list.requestFocusInWindow(); // sonst funktioniert nicht die Tastatursteuerung in der Liste
    return list.getSelectedIndex() >= 0;
  }



  /**
   * Der gewählte Benutzer aus der Onlineliste. Der Rückgabewert kann {@code null}
   * sein.
   * 
   * 
   * @return ein Benutzer
   */
  public String getBenutzer() {
    return benutzer;
  }



  public int getSelectedIndex() {
    return list.getSelectedIndex();
  }

  private ImageIcon onlineIcon = new ImageIcon(getClass().getResource(Resource.ONLINE));

  private void init() {
    lefticon = new JLabel(onlineIcon, SwingConstants.LEFT);
    righticon = new JLabel(onlineIcon, SwingConstants.RIGHT);
    panNorth.setLayout(new GridLayout(1, 3));
    panNorth.add(lefticon);
    panNorth.add(online);
    panNorth.add(righticon);

    online.setHorizontalAlignment(SwingConstants.CENTER);

    scrollpane.setViewportView(list);
    list.setModel(model);
    list.setCellRenderer(new NimbusCellRenderer<>());
    list.setBackground(Color.WHITE);
    list.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
    list.setPrototypeCellValue("WWWWWWWWWWW");
    list.addKeyListener(new KeyAdapter() {
      @Override
      public void keyPressed(KeyEvent event) {

        int selectedIndex = list.getSelectedIndex();
        if (event.getKeyCode() == KeyEvent.VK_ENTER) {
          changes.firePropertyChange(JOnlineliste.class.getName(), Control.NULL, Control.PRIVATECHAT);
        }
        if (event.getKeyCode() == KeyEvent.VK_UP) {
          if (selectedIndex <= 0) return;
          benutzer = model.get(selectedIndex - 1);
          changes.firePropertyChange(JOnlineliste.class.getName(), Control.NULL, Control.SELECTED);
        }
        else if (event.getKeyCode() == KeyEvent.VK_DOWN) {
          if (selectedIndex == model.size() - 1) return;
          benutzer = model.get(selectedIndex + 1);
          changes.firePropertyChange(JOnlineliste.class.getName(), Control.NULL, Control.SELECTED);
        }
      }
    });
    list.addMouseListener(new MouseAdapter() {

      @Override
      public void mouseClicked(MouseEvent event) {
        list.requestFocusInWindow();
      }



      @Override
      public void mouseReleased(MouseEvent event) {

        if (event.getClickCount() == 2) {
          changes.firePropertyChange(JOnlineliste.class.getName(), Control.NULL, Control.PRIVATECHAT);
          return;
        }

        benutzer = list.getSelectedValue();
        if (benutzer != null) {
          changes.firePropertyChange(JOnlineliste.class.getName(), Control.NULL, Control.SELECTED);
        }
      }

    });

  }



  public void removeOnlinelistListener(PropertyChangeListener l) {
    changes.removePropertyChangeListener(l);
  }



  public void setUserList(List<UserOnline> list) {
    model.removeAllElements();
    for (UserOnline useronline : list) {
      model.addElement(useronline.getNickname());
    }

//    model.addElement("1234567890");
//    model.addElement("22222222");
//    model.addElement("3333333333");
//    model.addElement("4444444444");
//    model.addElement("55555555555");
//    model.addElement("6666666666666");
//    model.addElement("7777777777777");
//    model.addElement("788888");
//    model.addElement("99999");
//    model.addElement("101010101");

    revalidate();

  }



  @Override
  public void setLanguage(ISO639 code) {
    online.setLanguage(code);
  }

}
