/**
 *  Copyright © 2020-2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.client.chat;

import java.awt.Color;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.time.format.FormatStyle;
import java.util.Date;
import java.util.Locale;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import net.javacomm.multilingual.schema.ISO639;
import net.javacomm.protocol.Attachment;
import net.javacomm.protocol.ChatUser;
import net.javacomm.protocol.RecordInterface;



public class MessageImpl implements Message {

  final static Logger log = LogManager.getLogger(MessageImpl.class);

  private int countParagraphs;
  private String[] paragraphs;
  private Color bgColor;
  private Color fgColor;
  private String nickname;
  private long datetime;
  private String message;
  private String filename;
  private Long filesize;
  private Long number;

  public MessageImpl() {}



//TODO alle Nachrichten herausfiltern, die nur EOTs enthalten
  public MessageImpl(RecordInterface chatmessage) throws MessageException {
    init(chatmessage);
    if (getMessage().length() == 0) throw new MessageException("message is empty");
  }



  public MessageImpl(Attachment chatmessage) throws MessageException {
    init(chatmessage);
    filename = chatmessage.getFilename();
    filesize = chatmessage.getFilesize();
    number = chatmessage.getAttachment();

    if (number == null && getMessage().length() == 0)
      throw new MessageException("message is empty und kein Anhang");

  }



  private void init(RecordInterface record) {
    message = record.getMessage();
    ChatUser chatuser = record.getChatUser();
    bgColor = new Color(chatuser.getBackgroundColor());
    fgColor = new Color(chatuser.getForegroundColor());
    nickname = chatuser.getNickname();
    datetime = record.getDatetime();

  }



  @Override
  /**
   *
   * Ein Paragraph oder Absatz wird durch ein <ENTER> beschränkt.
   *
   */
  public int countParagraphs() {
    return countParagraphs;
  }



  @Override
  public String[] getParagraphs() {
    return paragraphs;
  }



  @Override
  public int getTextformatMask() {
    // TODO Auto-generated method stub
    return 0;
  }



  @Override
  public String getNickname() {
    return nickname;
  }



  @Override
  public Color getBackgroundColor() {
    return bgColor;
  }



  @Override
  public Color getForegroundColor() {
    return fgColor;
  }



  @Override
  public long getXtime() {
    return datetime;
  }



  @Override
  public long getXdate() {
    return datetime;
  }



  @Override
  public String getMddsTime(long xtime) {
    final SimpleDateFormat simpleformat = new SimpleDateFormat("HH:mm:ss");
    Date date = new Date(xtime);
    String time = simpleformat.format(date);

    return (time);
  }



  @Override
  public String getMddsDate(long value) {
    SimpleDateFormat simpleformat;
    if (System.getProperty("java.version").startsWith("1.7")) {
      simpleformat = new SimpleDateFormat("ddMMMyyyy");
    }
    else {
      simpleformat = new SimpleDateFormat("ddLLLyyyy");
    }
    Date date = new Date(value);
    String datum = simpleformat.format(date);
    return datum;
  }



  @Override
  public String getMddsDateTime(long datetime) {
    return getMddsDateTime(datetime, null);
  }



  @Override
  public String getMddsDateTime(long datetime, ISO639 iso) {
    Locale locale;
    if (iso == null) {
      locale = Locale.getDefault();
    }
    else {
      locale = new Locale(iso.value());
    }
    Date date = new Date(datetime);
    LocalDateTime localdatetime = date.toInstant().atZone(ZoneId.systemDefault()).toLocalDateTime();
    DateTimeFormatter formatter = DateTimeFormatter.ofLocalizedDateTime(FormatStyle.MEDIUM)
        .withLocale(locale);
    String datum = localdatetime.format(formatter);
    return datum;
  }



  @Override
  public String getMessage() {
    return message;
  }



  @Override
  public String getFilename() {
    return filename;
  }



  @Override
  public Long getFilesize() {
    return filesize;
  }



  @Override
  public Long getFilenumber() {
    return number;
  }

}
