/**
 *  Copyright © 2020-2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.client.gui;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.util.Enumeration;
import java.util.List;
import javax.swing.DefaultListModel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.ListSelectionModel;
import javax.swing.SwingConstants;
import org.nexuswob.gui.swing.NimbusCellRenderer;
import net.javacomm.multilingual.Babelfish;
import net.javacomm.multilingual.MultilingualLabel;
import net.javacomm.multilingual.schema.ISO639;
import net.javacomm.multilingual.schema.KEY;



/**
 * 
 * Eine Liste mit Überschrift. Eine Mehrfachauswahl in der Liste ist möglich.
 * 
 * @author llange
 *
 */
public class JUserlist extends JPanel implements Babelfish {

  private static final long serialVersionUID = -3976170409370620524L;
  private JScrollPane scrollpane = new JScrollPane();
  private JList<String> list = new JList<String>();
  private DefaultListModel<String> model = new DefaultListModel<String>();
  private MultilingualLabel label = new MultilingualLabel(KEY.LABEL_BENUTZER);

  public JUserlist() {
    setLayout(new BorderLayout());
    add(BorderLayout.NORTH, label);
    add(BorderLayout.CENTER, scrollpane);
    init();
  }



  public void clear() {
    model.removeAllElements();
  }



  /**
   * Im Model befindliche Members
   * 
   * @return
   */
  public Enumeration<String> getMembers() {
    return model.elements();
  }



  /**
   * in der GUI markierte Members
   * 
   * @return
   */
  public List<String> getSelectedMembers() {
    return list.getSelectedValuesList();
  }



  private void init() {
    scrollpane.setViewportView(list);
    scrollpane.setMinimumSize(new Dimension(162, 150));
    scrollpane.setPreferredSize(new Dimension(162, 180));
    scrollpane.setMaximumSize(new Dimension(162, 201));
    scrollpane.setBorder(null);
    list.setModel(model);
    list.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
    list.setCellRenderer(new NimbusCellRenderer<String>());
    list.setBackground(Color.WHITE);
  }



  public void removeMember(List<String> member) {
    int count = member.size();
    for (int index = 0; index < count; index++) {
      model.removeElement(member.get(index));
    }
  }



  /**
   * Die Listenüberschrift.
   * 
   * @param value
   *              die Überschrift
   */
  public void setHeader(String value) {
    label.setHorizontalAlignment(SwingConstants.CENTER);
    label.setText(value);
  }



  /**
   * Die Listenüberschrift.
   * 
   * @param key
   *             der Übersetzungsschlüssel
   * @param code
   *             der Sprachcode
   */
  public void setHeader(KEY key, ISO639 code) {
    label.setHorizontalAlignment(SwingConstants.CENTER);
    label.setLanguage(key, code);
  }



  public void setMember(List<String> member) {
    setMember(member.toArray(new String[member.size()]));
  }



  public void setMember(String[] member) {
    if (member == null) return;
    for (int index = 0; index < member.length; index++) {
      model.addElement(member[index]);
    }
  }



  @Override
  public void setLanguage(ISO639 code) {
    label.setLanguage(code);
  }

}
