/**
 *  Copyright © 2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  Previously released under Apache License 2.0; now licensed under MPL 2.0.
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.database;

import jakarta.xml.bind.JAXBContext;
import jakarta.xml.bind.JAXBElement;
import jakarta.xml.bind.JAXBException;
import jakarta.xml.bind.Unmarshaller;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import javax.xml.XMLConstants;
import javax.xml.transform.Source;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.xml.sax.SAXException;
import net.javacomm.database.sqlmapper.SQLMAPPER;



public class SQLMapper {
  private final Logger log = LogManager.getLogger(SQLMapper.class);

  private static SQLMapper instance;
  public final static String SCHEMAFILE = "/net/javacomm/database/sqlmapper/sqlmapper.xsd";
  public final static String MAPPERFILE = "/net/javacomm/database/sqlmapper/sqlmapper.xml";
  private final String SCHEMA_PATH = "net.javacomm.database.sqlmapper";
  private Schema schema;

  private SQLMapper() {
    log.info("SCHEMAFILE -----> " + SCHEMAFILE);
    log.info("MAPPERFILE -----> " + MAPPERFILE);
    Class<? extends SQLMapper> clazz = getClass();

    SchemaFactory schemafactory = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
    URL url = clazz.getResource(SCHEMAFILE);
    try {
      schema = schemafactory.newSchema(url);
    }
    catch (SAXException e) {
      log.error(e.getMessage(), e.getCause());
    }
  }

  static {
    instance = new SQLMapper();
  }

  public static synchronized SQLMapper getInstance() {
    return instance;
  }



  /**
   * 
   * Dieses Schema wurde zum Lesen und Schreiben verwendet.
   * 
   * @return das Schema
   */
  public Schema getSchema() {
    return schema;
  }



  /**
   * Eine Konfigurationsdatei wird gelesen.
   * 
   * 
   * @return der Wurzelknoten
   * @throws JAXBException
   */
  public SQLMAPPER read() throws JAXBException {
    SQLMAPPER root = null;
    try(InputStream sqlmapper_xml = getClass().getResourceAsStream(MAPPERFILE)) {
      JAXBContext jc = JAXBContext.newInstance(SCHEMA_PATH);
      Unmarshaller unmarshaller = jc.createUnmarshaller();
      unmarshaller.setSchema(getSchema());
      Source source = new StreamSource(sqlmapper_xml);
      JAXBElement<SQLMAPPER> jaxbCollection = unmarshaller.unmarshal(source, SQLMAPPER.class);
      root = jaxbCollection.getValue();
    }
    catch (IOException e) {
      log.warn(e.getMessage(), e);
    }
    return root;
  }

}
