/**
 *  Copyright © 2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  Previously released under Apache License 2.0; now licensed under MPL 2.0.
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.restserver;

import jakarta.ws.rs.GET;
import jakarta.ws.rs.POST;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.WebApplicationException;
import java.io.IOException;
import java.io.InputStream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.glassfish.jersey.media.multipart.FormDataBodyPart;
import org.glassfish.jersey.media.multipart.FormDataMultiPart;
import net.javacomm.database.DatabaseException;
import net.javacomm.database.WebdatabaseImpl;



@Path("/chat")
public class ChatFileUpload {

  private final Logger log = LogManager.getLogger(ChatFileUpload.class);

  private WebdatabaseImpl database;

  public ChatFileUpload() {
    database = (WebdatabaseImpl) WebdatabaseImpl.getInstance();
  }



  /**
   * An den Server wird ein Datenblock/Chunk gesendet.
   * 
   * @param multiPart
   *                  enthält die Anlagennummer, Chunknummer und Content
   * 
   * @return die Anzahl der gespeicherten Bytes aus dem Datenblock
   */
  @POST
  @Path("/upload/private")
  public int uploadPrivateChatfileChunks(FormDataMultiPart multiPart) {
    int count = 0;
    FormDataBodyPart anlage = multiPart.getField("anlagennummer");
    FormDataBodyPart chunknummer = multiPart.getField("chunknummer");
    FormDataBodyPart content = multiPart.getField("content");

    try(InputStream is = content.getContent()) {
      count = database
          .insertPrivateChunk(Long.valueOf(anlage.getValue()), Integer.valueOf(chunknummer.getValue()), is);
    }
    catch (NumberFormatException | IOException | DatabaseException e) {
      log.error(e.getMessage());
      throw new WebApplicationException();
    }
    return count;
  }



  @POST
  @Path("/upload")
  public int upload(FormDataMultiPart multiPart) {
    int count = 0;
    FormDataBodyPart anlage = multiPart.getField("anlagennummer");
    FormDataBodyPart chunknummer = multiPart.getField("chunknummer");
    FormDataBodyPart content = multiPart.getField("content");

    try(InputStream is = content.getContent()) {
      count = database
          .insertChunk(Long.valueOf(anlage.getValue()), Integer.valueOf(chunknummer.getValue()), is);
    }
    catch (NumberFormatException | IOException | DatabaseException e) {
      log.error(e.getMessage());
      throw new WebApplicationException();
    }
    return count;
  }



  /**
   * Erzeuge einen Anlagenidentifier.
   * 
   * @return dieser Identifier wurde erstellt
   */
  @GET
  @Path("/identifier")
  public long createIdentifier() {
    long identifier;
    try {
      identifier = database.createIdentifier();
    }
    catch (DatabaseException e) {
      log.error(e.getMessage());
      throw new WebApplicationException();
    }
    return identifier;
  }

}
