/**
 * The MIT License
 * Copyright © 2020 Thorsten Schulz <http://javacomm.net>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package net.javacomm.schulz;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.geom.Arc2D;
import javax.swing.JComponent;

/***********************************************************************/
/*                                                                     */
/*                        Class JProcentBar                            */
/*                                                                     */
/***********************************************************************/

public class JProcentBar extends JComponent {

  public enum Chart {
    BAR,
    PIE;
  }

  public enum Orientation {
    VERTICAL,
    HORIZONTAL;
  }

  public enum ColorMode {
    BLUEYELLOW(yellow, Color.BLUE, Color.RED),
    GREENWHITE(Color.WHITE, Color.GREEN, Color.RED),
    WHITEBLACK(Color.BLACK, Color.WHITE, Color.RED),
    ORANGEPURPUR(purpur, orange, Color.WHITE),
    REDBLACK(Color.BLACK, Color.RED, Color.WHITE),
    BLUEWHITE(Color.GRAY, Color.WHITE, Color.BLUE),
    PURPURGRAY(Color.GRAY, purpur, yellow);

    private Color background;
    private Color barColor;
    private Color textColor;

    private ColorMode(Color background, Color barColor, Color textColor) {
      this.background = background;
      this.barColor = barColor;
      this.textColor = textColor;
    }



    public Color getBackground() {
      return background;
    }



    public Color getBarColor() {
      return barColor;
    }



    public Color getTextColor() {
      return textColor;
    }

  }

  private static final long serialVersionUID = 4124315724272841642L;


  private Orientation orientationValue;
  private Chart chart = Chart.BAR; // Kein Kreisdiagramm zeichnen
  private double angle        = 0.0d;

  private static Color orange = new Color(250, 140, 30);
  private static Color yellow = new Color(240, 230, 10);
  private static Color purpur = new Color(90, 0, 90);// lila

  private int dimensionX;
  private int dimensionY;
  private int width;
  private int height;
  private int intProcent;
  private int textX;
  private int textY;
  private Color background;
  private Color barColor;
  private Color textColor;

  public JProcentBar(){
    orientationValue = Orientation.VERTICAL;
    dimensionX  = 40;
    dimensionY  = 120;
    width       = dimensionX;
    height      = dimensionY;
    intProcent  = 100;
    textX       = (dimensionX>>1)-10;
    textY       = (dimensionY>>1)+5;
    background = ColorMode.GREENWHITE.getBackground();
    barColor = ColorMode.GREENWHITE.getBarColor();
    textColor = ColorMode.GREENWHITE.getTextColor();
  }



  /**
   * Erzeuge eine Fortschrittsanzeige.
   * 
   * @param value
   *              ein Balken oder ein Donut
   */
  public JProcentBar(Chart value) {
    this();
    chart = value;
  }



  public JProcentBar(Orientation orientation) {
    this();
    orientationValue = orientation;
    if (orientation == Orientation.VERTICAL) return;
    dimensionX = 120;
    dimensionY = 40;
    width = dimensionX;
    height = dimensionY;
    intProcent = 100;
    textX = (dimensionX >> 1) - 10;
    textY = (dimensionY >> 1) + 5;
  }



  /**
   * 
   * 
   * @param colorMode
   *                  der Farbmodus
   */
  public JProcentBar(ColorMode colorMode) {
    this();
    background = colorMode.getBackground();
    barColor = colorMode.getBarColor();
    textColor = colorMode.getTextColor();
  }



  /**
   * Erzeuge eine Fortschrittsanzeige.
   * 
   * @param colorMode
   *                  der Farbmodus
   * @param value
   *                  ein Balken oder ein Donut
   * 
   */
  public JProcentBar(ColorMode colorMode, Chart value) {
    this(colorMode);
    chart = value;
  }



  public JProcentBar(Orientation orientation, ColorMode colorMode) {
    this(orientation);
    background = colorMode.getBackground();
    barColor = colorMode.getBarColor();
    textColor = colorMode.getTextColor();
  }



  public JProcentBar(Orientation orientation, ColorMode colorMode, Dimension componentSize) {
    this(orientation, colorMode);
    dimensionX  = componentSize.width;
    dimensionY  = componentSize.height;
    if(dimensionX<40){
      dimensionX = 40;
    }
    if(dimensionY<20){
      dimensionY =20;
    }

    if (orientation == Orientation.VERTICAL) {
      width = dimensionX;
      height = dimensionY;
      intProcent = 100;
      textX = (dimensionX >> 1) - 10;
      textY = (dimensionY >> 1) + 5;
    }
    else{
      width       = dimensionX;
      height      = dimensionY;
      intProcent  = 100;
      textX       = (dimensionX>>1)-10;
      textY       = (dimensionY>>1)+5;
    }
  }



  /**
   * Erzeuge eine Fortschrittsanzeige.
   * 
   * 
   * @param orientation
   *                      soll das Chart horizontal oder vertikal ausgerichtet
   *                      sein
   * @param colorMode
   *                      der Farbmodus
   * @param componentSize
   *                      die Gr��e der JProzentBar
   * @param value
   *                      ein Balken oder ein Donut
   */
  public JProcentBar(Orientation orientation, ColorMode colorMode, Dimension componentSize, Chart value) {
    this(orientation, colorMode, componentSize);
    chart = value;
  }



  /**
   * Zeige den Teilfortschritt im Verh�ltnis zum Ganzen an.
   * 
   * @param part
   *              ein Teil von Etwas
   * @param total
   *              das Ganze
   */
  public void showBar(int part, int total){
    if ((0 > part) || (part > total))
      throw new RuntimeException(part + " - part liegt nicht im Intervall [0..total]");
    intProcent = 100 * part / total;

    switch(orientationValue){
      case VERTICAL:
        height = (part * dimensionY) / total;
        break;
      case HORIZONTAL:
        width = (part * dimensionX) / total;
        break;
    }
    repaint();
    return;
  }



  /**
   * Zeige den Fortschrittsbalken bis n-Prozent an. Der Fortschritt liegt im
   * Bereich [0..100].
   * 
   * @param procent
   *                der aktuelle Fortschrit
   */
  public void showBar(int procent) {
    if ((0 > procent) || (procent > 100))
      throw new RuntimeException(procent + " - liegt nicht im Intervall [0..100]");
    intProcent = procent;
    switch(orientationValue) {
      case VERTICAL:
        height = (intProcent * dimensionY) / 100;
        break;
      case HORIZONTAL:
        width = (intProcent * dimensionX) / 100;
        break;
    }
    repaint();
    return;
  }



  @Override
  public void paintComponent(Graphics g){

    super.paintComponent(g);

    Graphics2D g2 = (Graphics2D) g;

    if (chart == Chart.PIE) {

      angle = intProcent / 100.0d * 360.0d;

      g2.setColor(barColor);
      g2.fill(new Arc2D.Double(0,0,
      dimensionX, dimensionY,
      0.0d, angle, Arc2D.PIE));
      g2.setColor(background);
      g2.fill(new Arc2D.Double(0,0,
      dimensionX, dimensionY,
      angle, 360.0d - angle, Arc2D.PIE));
      g2.setColor(textColor);
      g2.drawString(String.valueOf(intProcent)+"%", textX, textY);
    }
    else{
      if (orientationValue == Orientation.VERTICAL) {
        g2.setColor(background);
        g2.fill(new Rectangle(0, 0, dimensionX, dimensionY));
        g2.setColor(barColor);
        g2.fill(new Rectangle(0, (dimensionY - height), width, height));
        g2.setColor(textColor);
        g2.drawString(String.valueOf(intProcent)+"%", textX, textY);
      }
      else{
        g2.setColor(background);
        g2.fill(new Rectangle(0, 0, dimensionX, dimensionY));
        g2.setColor(barColor);
        g2.fill(new Rectangle(0, 0, width, height));
        g2.setColor(textColor);
        g2.drawString(String.valueOf(intProcent)+"%", textX, textY);
      }
    }
    return;
  }

  @Override
  public Dimension getPreferredSize(){
    return(new Dimension(dimensionX, dimensionY));
  }





}