/*
 * The MIT License
 * Copyright © 2023 Thorsten Schulz <https://javacomm.net>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package org.nexuswob.gui;

import java.awt.Cursor;
import java.awt.event.InputEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import javax.swing.JButton;



/*****************************************************************************/
/*                                                                           */
/* Class       : JMouseButton                                                */
/* Owner       : Thorsten Schulz                                             */
/* License     : All rights reserved                                         */
/* Createtd    : 20.11.2014 by Thorsten Schulz, TSchulz@javacomm.net         */
/* Last Update : 11.07.2023 by Thorsten Schulz                               */
/*                                                                           */
/*****************************************************************************/

/**
 * Wenn die Maus über den Knopf fährt, kann sie sich in eine Hand verwandeln.
 * Ebenfalls können Rechtsklicks mit der Maus abgefragt werden. Solange sich
 * niemand über {@link #addRightClickListener(PropertyChangeListener)}
 * registiert, verhält sich {@code JMouseButton} wie ein
 * {@link javax.swing.JButton}.
 * 
 * @author Thorsten Schulz
 */
public class JMouseButton extends JButton {

  private static final long serialVersionUID = -8648067532944925921L;
  private MouseAction actionMouse = new MouseAction();
  private boolean pressed = false;
  private boolean rightButtonPressed = false;
  private PropertyChangeSupport changes = new PropertyChangeSupport(this);
  private boolean hand = true;

  /**
   * Ein Mausknopf wird erzeugt.
   */
  public JMouseButton() {
    addMouseListener(actionMouse);
  }



  /**
   * Ein Mausknopf wird erzeugt.
   * 
   * @param buttonName
   *                   Schriftzug auf dem Knopf
   */
  public JMouseButton(String buttonName) {
    this();
    setText(buttonName);
  }



  /**
   * Wurde ein Knopf gedrückt, egal welcher?
   * 
   * @return {@code true}, ein Knopf wurde gedrückt.
   */
  public boolean isPressed() {
    return pressed;
  }



  /**
   * Wenn die Maus über den Knopf fährt, verwandelt sie sich in eine Hand.
   * 
   * @param value
   *              für {@code true} wird eine Hand angezeigt, sonst ein Pfeil
   */
  public void activateHandCursor(boolean value) {
    hand = value;
  }



  /**
   * Der Listener hört die rechte Maustaste ab.
   * 
   * @param listener
   *                 überwacht die rechte Maustaste
   */
  public void addRightClickListener(PropertyChangeListener listener) {
    changes.addPropertyChangeListener(listener);
  }



  /**
   * Der Listener hört nicht mehr die rechte Maustaste ab.
   * 
   * @param listener
   *                 überwacht die rechte Maustaste
   */
  public void removeRightClickListener(PropertyChangeListener listener) {
    changes.removePropertyChangeListener(listener);
  }



  /**
   * Alle Listener werden entfernt und der Speicher wird freigegeben. Nach dem
   * Aufruf erkennt die Komponente keine rechten Mausklicks mehr und der Cursor
   * verwandelt sich nicht mehr in eine Hand.
   * 
   */
  public void removeAllListener() {
    for (PropertyChangeListener pl : changes.getPropertyChangeListeners()) {
      removeRightClickListener(pl);
    }
    removeMouseListener(actionMouse);
  }



  /**
   * Wurde der rechte Mausknopf gedrückt?
   * 
   * @return {@code true}, der rechte Knopf wurde gedrückt
   */
  public boolean rightButtonPressed() {
    return rightButtonPressed;
  }

  /*****************************************************************************/
  /*                                                                           */
  /* InnerClass : MouseAction */
  /* Owner : Thorsten Schulz */
  /* License : All rights reserved */
  /* Createtd : 30.03.2004 by Thorsten Schulz, TSchulz@javacomm.net */
  /* Last Update : 31.08.2023 by Thorsten Schulz */
  /*                                                                           */
  /*****************************************************************************/

  class MouseAction extends MouseAdapter {

    @Override
    public void mouseEntered(MouseEvent event) {
      if (hand) setCursor(Cursor.getPredefinedCursor(Cursor.HAND_CURSOR));
    }



    @Override
    public void mouseExited(MouseEvent event) {
      setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
    }



    @Override
    public void mousePressed(MouseEvent event) {
      pressed = true;
      if (event.getModifiersEx() == InputEvent.BUTTON3_DOWN_MASK) {
        rightButtonPressed = true;
        changes.firePropertyChange(JMouseButton.MouseAction.class.getName(), false, true);
      }
      else {
        rightButtonPressed = false;
      }
    }



    @Override
    public void mouseReleased(MouseEvent event) {
      pressed = false;
    }
  }
}