/**
 *  Copyright © 2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  Previously released under Apache License 2.0; now licensed under MPL 2.0.
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.protocol;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;



/**
 * Aktiviere oder deaktiviere die Bildschirmteilung.
 */
public class CONFERENCEVIDEO extends MESSAGE {

  public final static String USERID = "USERID";
  public final static String ORGANISATOR = "ORGANISATOR";
  public final static String RECEIVER_SESSIONS = "RECEIVER_SESSIONS";
  public final static String SHARING = "SHARING";
  public static final String KONFERENZRAUM = "KONFERENZRAUM";
  public static final String NICKNAME = "NICKNAME";

  private String userid; // der Absender
  private String nickname; // Nickname Absender
  private String organisator; // der Konferenzorganisator
  private String[] receiverSessions; // session alle Empfänger
  private Boolean isShared; // true, wenn eine Verbindung aufgebaut wird
  private String konferenzraum;

  public CONFERENCEVIDEO() {
    super();
  }



  public String getKonferenzraum() {
    return konferenzraum;
  }



  /**
   * Pflichtfeld in REQUEST. Dieser Anwender teilt seinen Bildschirm.
   * 
   * @return dieser Nickname
   */
  public String getNickname() {
    return nickname;
  }



  public String getOrganisator() {
    return organisator;
  }



  /**
   * Die Videosession wird an diese Empfänger übertragen.
   * 
   * @return alle Empfängersessions
   */
  public String[] getReceiverSessions() {
    return receiverSessions;
  }



  /**
   * Wer hat Nachricht gesendet?
   * 
   * @return dieser Anwender
   */
  public String getUserid() {
    return userid;
  }



  /**
   * Die Bildschirmteilung wurde eingeschaltet oder ausgeschaltet.
   * 
   * @return {@code true} wenn der Bildschirm mit anderen Teilnehmern geteilt wird
   */
  public Boolean isShared() {
    return isShared;
  }



  public void setKonferenzraum(String konferenzraum) {
    this.konferenzraum = konferenzraum;
  }



  /**
   * Pflichtattribut in REQUEST.
   * 
   * @param nickname
   *                 dieser Nickname vom Absender
   */
  public void setNickname(String nickname) {
    this.nickname = nickname;
  }



  /**
   * Der Organisator richtet die Konferenz ein.
   * 
   * @param origin
   *               dieser Organisator
   */
  public void setOrganisator(String origin) {
    organisator = origin;
  }



  /**
   * An diese Empfänger wird die Videosession übertragen.
   * 
   * @param receiverSessions
   *                         diese Empfängersessions
   * 
   */
  public void setReceiverSessions(String[] receiverSessions) {
    this.receiverSessions = receiverSessions;
  }



  /**
   * Wird der Bildschirm geteilt?
   * 
   * 
   * @param isShared
   *                 {@code true} wenn der Bildschirem geteilt wird
   */
  public void setShared(Boolean isShared) {
    this.isShared = isShared;
  }



  /**
   * Der Absender.
   * 
   * @param userid
   *               dieser Anwender
   */
  public void setUserid(String userid) {
    this.userid = userid;
  }



  @Override
  public String toString() {
    GsonBuilder builder = new GsonBuilder();
    builder.registerTypeHierarchyAdapter(CONFERENCEVIDEO.class, new ConferencevideoSerializer());
    Gson gson = builder.create();
    return gson.toJson(this);
  }

}
