/**
 *  Copyright © 2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  Previously released under Apache License 2.0; now licensed under MPL 2.0.
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.restserver;

import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;
import java.lang.reflect.Type;
import java.util.List;
import net.javacomm.transfer.TransferOutdated;
import net.javacomm.transfer.TransferRoomfilter;
import net.javacomm.transfer.TransferUser;



/**
 * Der Wrapper ist eine Hilfsklasse für das Übertragen von Listen jeglichen
 * Typs.
 * 
 * @author llange
 *
 */
public class Wrapper {

  private String liste;

  private Wrapper(String value) {
    liste = value;
  }



  public static Wrapper fromString(String value) {
    return new Wrapper(value);
  }


  
  /**
   * Ein Transfer-REQUEST-Parameter für den Consumer. Der Paramter wird auf der
   * Serverseite ausgewertet.
   * 
   * @return eine Liste von Transferobjekte
   */
  public List<TransferUser> getTransferUser() {
    Gson gson = new Gson();
    Type typeOfT = new TypeToken<List<TransferUser>>() {}.getType();
    return gson.fromJson(liste, typeOfT);
  }


  
  /**
   * Ein Transfer-REQUEST-Parameter für den Consumer. Der Paramter wird auf der
   * Serverseite ausgewertet.
   * 
   * @return eine Liste von Transferobjekte
   */
  public List<TransferRoomfilter> getTransferRoomfilter() {
    Gson gson = new Gson();
    Type typeOfT = new TypeToken<List<TransferRoomfilter>>() {}.getType();
    return gson.fromJson(liste, typeOfT);
  }



  /**
   * Ein Transfer-REQUEST-Parameter für den Consumer. Der Parameter wird auf der
   * Serverseite ausgewertet.
   * 
   * @return eine Liste von Transferobjekte
   */
  public List<TransferOutdated> getTransferOutdated() {
    Gson gson = new Gson();
    Type typeOfT = new TypeToken<List<TransferOutdated>>() {}.getType();
    return gson.fromJson(liste, typeOfT);
  }



  /**
   * Ein Transfer-REQUEST-Parameter für den Consumer. Der Parameter wird auf der
   * Serverseite ausgewertet.
   * 
   * @return eine Liste von Transferobjekte
   */
  public List<String> getStringList() {
    Gson gson = new Gson();
    Type typeOfT = new TypeToken<List<String>>() {}.getType();
    return gson.fromJson(liste, typeOfT);
  }



  /**
   * Für das Einkapseln von Listen in einen JSON-String.
   * 
   * @param <T>
   *              der Listentyp
   * @param liste
   *              diese Liste wird geschickt
   * @return die gewandelte Liste als JSON-String
   */
  public static <T> String toString(List<T> liste) {
    Gson gson = new Gson();
    return gson.toJson(liste);
  }

}
