/**
 *  Copyright © 2022-2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 */
package net.javacomm.window.manager;

import java.awt.Color;
import java.util.List;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.nexuswob.gui.swing.JavacommInternalFrame;
import net.javacomm.multilingual.Multilingual;
import net.javacomm.multilingual.MultilingualString;
import net.javacomm.multilingual.schema.COMMAND;
import net.javacomm.multilingual.schema.ISO639;
import net.javacomm.multilingual.schema.KEY;
import net.javacomm.multilingual.schema.LANGUAGE;
import net.javacomm.multilingual.schema.LANGUAGES;



public abstract class WindowManagerInternalFrame extends JavacommInternalFrame implements WM {

  private final static Logger log = LogManager.getLogger(WindowManagerInternalFrame.class);
  private static final long serialVersionUID = -1567862235105377482L;
  public final static int SPLITWIDTH = 231;
  public final static int MALUS = 120;

  private ISO639 language;
  private KEY key;
  protected String extText;
  protected String reportFilename;// ändert seinen Namen nicht mehr, wenn er zugewiesen wurde
  protected String schemaFilename;// ändert seinen Namen nicht mehr, wenn er zugewiesen wurde

  protected WindowManager windowManager = new WindowManager();
  protected MultilingualString multiLauncher = new MultilingualString(KEY.FRAME_LAUNCHER);
  protected MultilingualString multiFTS = new MultilingualString(KEY.FRAME_FILESHARING);
  protected MultilingualString multiVideoservice = new MultilingualString(KEY.FRAME_BILDSCHIRMEMPFANG);
  protected MultilingualString multiTeilnehmer = new MultilingualString(KEY.FRAME_EINWILLIGUNG);
  protected MultilingualString multiTelefonEinrichten = new MultilingualString(KEY.FRAME_TELEFON_EINRICHTEN);
  protected MultilingualString multiSchwarzeliste = new MultilingualString(KEY.FRAME_SCHWARZE_LISTE);
  protected MultilingualString multiProjektor = new MultilingualString(KEY.FRAME_PROJEKTOR);
  protected MultilingualString multiVideokonferenz = new MultilingualString(KEY.FRAME_VIDEOKONFERNEZ);
  protected MultilingualString multiLizenz = new MultilingualString(KEY.FRAME_INFO);
  protected MultilingualString multiTelefonkonferenzBeitreten = new MultilingualString(
      KEY.STRING_TELEFONKONFERENZ
  );
  protected MultilingualString multiAdministrator = new MultilingualString(KEY.ADMINISTRATOR);
  protected MultilingualString multiMastermind = new MultilingualString(KEY.FRAME_MASTERMIND);
  private MultilingualString multiAnrufen = new MultilingualString(KEY.TELEFON_ANRUFEN);
  private MultilingualString multiAbnehmen = new MultilingualString(KEY.FRAME_ABNEHMEN);
  private MultilingualString multiOdxReport = new MultilingualString(KEY.FRAME_REPORT);
  private MultilingualString multiSchemapruefung = new MultilingualString(KEY.FRAME_SCHEMAPRUEFUNG);

  protected String oldnameLauncher;
  protected String oldnameFTS;
  protected String oldnameVideoservice;
  protected String oldnameTeilnehmer;
  protected String oldnameTelefonEinrichten;
  protected String oldnameSchwarzeListe;
  protected String oldnameProjektor;
  protected String oldnameVideokonferenz;
  protected String oldnameLizenz;
  protected String oldnameAnrufen;
  protected String oldnameAbnehmenFrame;
  protected String oldnameBeitreten;
  protected String oldnameAdministrator;
  protected String oldnameReportFrame;
  protected String neuReportFrame;
  protected String neuSchemaframe;
  protected String oldnameMastermind;

  private final long id;

  public WindowManagerInternalFrame() {
    super();
    language = ISO639.DE;
    try {
      Thread.sleep(1);
    }
    catch (InterruptedException e) {}
    id = System.currentTimeMillis();
  }



  /**
   * Ein Menupunkt wird in deutscher Sprache erzeugt.
   *
   * @param key
   *            der Zugriffsschlüssel auf ein Menu
   */
  public WindowManagerInternalFrame(KEY key) {
    this(key, ISO639.DE);
  }



  /**
   * Ein Menupunkt wird in einer bestimmten Sprache erzeugt.
   *
   * @param key
   *                 der Zugriffsschlüssel auf ein Menu
   * @param language
   *                 die angezeigte Sprache
   */
  public WindowManagerInternalFrame(KEY key, ISO639 language) {
    this();
    setLanguage(key, language);
  }



  public WindowManagerInternalFrame(String title) {
    this();
    setTitle(title);
  }



  @Override
  public void activated() {
    throw new UnsupportedOperationException();
  }



  /**
   * Ist im Fenstermanager der AnrufenFrame eingetragen?
   *
   * @param tmp
   *            enthält den Empfänger wie Call - anaconda oder Llamar - anaconda
   * @return {@code true}, der AnrufenFrame ist eingetragen
   */
  public boolean foundAnrufenFrame(String tmp) {
    COMMAND command = Multilingual.getInstance().getCommand(KEY.TELEFON_ANRUFEN);
    LANGUAGES languages = command.getLANGUAGES();
    List<LANGUAGE> alleSprachen = languages.getLANGUAGE();
    for (LANGUAGE language : alleSprachen) {
      if (tmp.startsWith(language.getTEXT())) {
        return true;
      }
    }
    return false;
  }



  @Override
  public String getBaumansicht() {
    return getFrameTitleId();
  }



  @Override
  public String getFrameTitleId() {
    return title;
  }



  @Override
  public long getId() {
    return id;
  }



  public MultilingualString getMultiAbnehmen() {
    return multiAbnehmen;
  }



  public MultilingualString getMultiAnrufen() {
    return multiAnrufen;
  }



  public MultilingualString getMultiOdxReport() {
    return multiOdxReport;
  }



  public MultilingualString getMultiSchemapruefung() {
    return multiSchemapruefung;
  }



  public String getTEXT(KEY key, ISO639 language) {
    if (key == null) throw new IllegalArgumentException("key ist null");
    if (language == null) throw new IllegalArgumentException("language ist null");
    this.language = language;
    this.key = key;
    COMMAND command = Multilingual.getInstance().getCommand(this.key);
    LANGUAGES languages = command.getLANGUAGES();
    if (languages == null) throw new IllegalArgumentException("languages ist null");
    for (LANGUAGE tmp : languages.getLANGUAGE()) {
      if (tmp.getISO639() != language) continue;
      if (tmp.getTEXT() != null) {
        return tmp.getTEXT();
      }
      break;
    }
    throw new IllegalArgumentException("Für den Schlüssel gibt es keinen Eintrag - " + key.toString());
  }



  @Override
  public Frames getType() {
    return null;
  }



  @Override
  public void insertWM(WM frame) {
    throw new UnsupportedOperationException();
  }



  @Override
  public void removeWM(WM frame) {
    throw new UnsupportedOperationException();

  }



  /**
   * Zeichne den Hintergrundfarbverlauf von oben nach unten in einem
   * {@code JTree}.
   * 
   * @param top
   *               von dieser Farbe
   * @param bottom
   *               nach dieser Farbe
   */
  public abstract void setBackgroundGradient(Color top, Color bottom);



  @Override
  public void setFrameTitleId(String id) {
    setTitle(id);
  }



  /**
   * Der Menuschlüssel für die Textanzeige wird gesetzt.
   *
   * @param key
   *            der Zugriffschlüssel für JMenu, JMenuItem und JCheckBoxMenuItem in
   *            der Oberfläche
   *
   */
  public void setKey(KEY key) {
    setLanguage(key, language);
  }



  /**
   * Die Sprache wird gesetzt.
   *
   * @param language
   *                 die Sprache
   */
  @Override
  public void setLanguage(ISO639 language) {
    setLanguage(key, language);
  }



  /**
   * Die Sprache darf jederzeit geändert werden.
   *
   * @param key
   *                 der Schlüssel für das ausgewählte Kommando
   * @param language
   *                 der Text wird in dieser Sprache angezeigt
   */
  public void setLanguage(KEY key, ISO639 language) {
    if (key == null) throw new IllegalArgumentException("key ist null");
    if (language == null) throw new IllegalArgumentException("language ist null");
    this.language = language;
    this.key = key;
    COMMAND command = Multilingual.getInstance().getCommand(this.key);
    LANGUAGES languages = command.getLANGUAGES();
    if (languages == null) return;
    for (LANGUAGE tmp : languages.getLANGUAGE()) {
      if (tmp.getISO639() != language) continue;
      if (tmp.getTEXT() != null) {
        setFrameTitleId(tmp.getTEXT());
      }
      break;
    }
  }



  /**
   * Im Schlüssel gibt es einen Platzhalter XXX, der durch einen beliebigen Text
   * ersetzt werden kann. Wenn der Schlüssel keinen Platzhalter hat, passiert
   * nichts. Der Aufruf ist dann wie {@link #setLanguage(KEY, ISO639)}.
   *
   *
   * @param key
   *                 der Schlüssel für das ausgewählte Kommando
   * @param language
   *                 der Text wird in dieser Sprache angezeigt
   * @param extText
   *                 ein bestehender Text wird mit diesem Text erweitert
   */
  public void setLanguage(KEY key, ISO639 language, String extText) {
    if (key == null) throw new IllegalArgumentException("key ist null");
    if (language == null) throw new IllegalArgumentException("language ist null");
    this.language = language;
    this.key = key;
    this.extText = extText;
    COMMAND command = Multilingual.getInstance().getCommand(key);
    LANGUAGES languages = command.getLANGUAGES();
    if (languages == null) return;
    for (LANGUAGE tmp : languages.getLANGUAGE()) {
      if (tmp.getISO639() != language) continue;
      String titleText = tmp.getTEXT();
      String neu = titleText.replace("XXX", extText == null ? "" : " - " + extText);
      setFrameTitleId(neu); // hier wird nur der Frame gesetzt!!!!!!!!!!!!!!!!!!!
      break;
    }
  }



  public void setMultiAbnehmen(MultilingualString multiAbnehmen) {
    this.multiAbnehmen = multiAbnehmen;
  }



  public void setMultiAnrufen(MultilingualString multiAnrufen) {
    this.multiAnrufen = multiAnrufen;
  }



  public void setMultiOdxReport(MultilingualString multiOdxReport) {
    this.multiOdxReport = multiOdxReport;
  }



  public void setMultiSchemapruefung(MultilingualString multiSchemapruefung) {
    this.multiSchemapruefung = multiSchemapruefung;
  }



  @Override
  public void setType(Frames value) {
    throw new UnsupportedOperationException();

  }



  /**
   * Im Fenstermanager wird die Baumansicht mehrsprachig angezeigt.
   * Fensteransichten mit Dateinamen werden nicht mehrsprachig angezeigt.
   *
   * @param code
   *             in dieser Sprache erscheint die Anzeige
   */
  protected void treeView(ISO639 code) {

    oldnameBeitreten = multiTelefonkonferenzBeitreten.toString();
    multiTelefonkonferenzBeitreten.setLanguage(code);
    windowManager.replaceFramename(oldnameBeitreten, multiTelefonkonferenzBeitreten.toString(), getId());

    oldnameLauncher = multiLauncher.toString();
    multiLauncher.setLanguage(code);
    windowManager.replaceFramename(oldnameLauncher, multiLauncher.toString(), getId());

    oldnameSchwarzeListe = multiSchwarzeliste.toString();
    multiSchwarzeliste.setLanguage(code);
    windowManager.replaceFramename(oldnameSchwarzeListe, multiSchwarzeliste.toString(), getId());

    oldnameVideoservice = multiVideoservice.toString();
    multiVideoservice.setLanguage(code);
    windowManager.replaceFramename(oldnameVideoservice, multiVideoservice.toString(), getId());

    oldnameFTS = multiFTS.toString();
    multiFTS.setLanguage(code);
    windowManager.replaceFramename(oldnameFTS, multiFTS.toString(), getId());

    oldnameTeilnehmer = multiTeilnehmer.toString();
    multiTeilnehmer.setLanguage(code);
    windowManager.replaceFramename(oldnameTeilnehmer, multiTeilnehmer.toString(), getId());

    oldnameTelefonEinrichten = multiTelefonEinrichten.toString();
    multiTelefonEinrichten.setLanguage(code);
    windowManager.replaceFramename(oldnameTelefonEinrichten, multiTelefonEinrichten.toString(), getId());

    oldnameProjektor = multiProjektor.toString();
    multiProjektor.setLanguage(code);
    windowManager.replaceFramename(oldnameProjektor, multiProjektor.toString(), getId());

    oldnameLizenz = multiLizenz.toString();
    multiLizenz.setLanguage(code);
    windowManager.replaceFramename(oldnameLizenz, multiLizenz.toString(), getId());

    oldnameAdministrator = multiAdministrator.toString();
    multiAdministrator.setLanguage(code);
    windowManager.replaceFramename(oldnameAdministrator, multiAdministrator.toString(), getId());

    oldnameAnrufen = multiAnrufen.toString();
    multiAnrufen.setLanguage(code);
    /// triple 3 durch ext tauschen gilt für oldname und multi
    oldnameAnrufen = oldnameAnrufen.replace("XXX", extText == null ? "" : extText);
    multiAnrufen.setLanguage(KEY.TELEFON_ANRUFEN, code, extText);
    windowManager.replaceFramename(oldnameAnrufen, multiAnrufen.toString(), getId());

    oldnameAbnehmenFrame = multiAbnehmen.toString();
    multiAbnehmen.setLanguage(code);
    oldnameAbnehmenFrame = oldnameAbnehmenFrame.replace("XXX", extText == null ? "" : extText);
    multiAbnehmen.setLanguage(KEY.FRAME_ABNEHMEN, code, extText);
    windowManager.replaceFramename(oldnameAbnehmenFrame, multiAbnehmen.toString(), getId());

    oldnameAbnehmenFrame = multiAbnehmen.toString();
    multiAbnehmen.setLanguage(code);
    oldnameAbnehmenFrame = oldnameAbnehmenFrame.replace("XXX", extText == null ? "" : extText);
    multiAbnehmen.setLanguage(KEY.FRAME_ABNEHMEN, code, extText);
    windowManager.replaceFramename(oldnameAbnehmenFrame, multiAbnehmen.toString(), getId());

    oldnameVideokonferenz = multiVideokonferenz.toString();
    multiVideokonferenz.setLanguage(code);
    oldnameVideokonferenz = oldnameVideokonferenz.replace("XXX", extText == null ? "" : extText);
    multiVideokonferenz.setLanguage(KEY.FRAME_VIDEOKONFERNEZ, code, extText);
    windowManager.replaceFramename(oldnameVideokonferenz, multiVideokonferenz.toString(), getId());

    oldnameMastermind = multiMastermind.toString();
    multiMastermind.setLanguage(code);
    windowManager.replaceFramename(oldnameMastermind, multiMastermind.toString(), getId());

    // Sonderfall es kann viele Berichte geben. XX ist der Dateiname
    // in der Baumansicht kann es viele Berichte geben
    // Wechsle alle Berichte aus
    // hole die Baumansicht und alle Berichte

//    log.info(this.getFrameTitleId());
    for (String tmp : windowManager.getFrameList()) {
      // hier werden nur Bericht/Report/Informe geändert
      String[] result = tmp.split(" - ");
      if (result.length != 2) continue;
//      if (!tmp.startsWith(result[0])) continue;
//      log.info(tmp);

      if (tmp.startsWith("Informe") || tmp.startsWith("Report") || tmp.startsWith("Bericht")) {
        if (tmp.endsWith(".odx") || tmp.endsWith("index.xml")) {

          if (!tmp.endsWith(result[1])) continue;
          // hole Frame_Report mit aktuellem DE
          //
          multiOdxReport.setLanguage(code); // neue Darstellung
          neuReportFrame = multiOdxReport.toString().replace("XXX", result[1]);
          windowManager.replaceFramename(tmp, neuReportFrame, getId());

        }
      }
    }

    for (String tmp : windowManager.getFrameList()) {
      // hier werden nur Schemaprüfung/Scheme check/Comprobación del sistema geändert
      String[] result = tmp.split(" - ");
      if (result.length != 2) continue;
      if (
        tmp.startsWith("Schemaprüfung")
            || tmp.startsWith("Scheme check")
            || tmp.startsWith("Comprobación del sistema")
      ) {
        if (!tmp.endsWith(result[1])) continue;
        multiSchemapruefung.setLanguage(code); // neue Darstellung
        neuSchemaframe = multiSchemapruefung.toString().replace("XXX", result[1]);
        windowManager.replaceFramename(tmp, neuSchemaframe, getId());
      }
    }

  }

}
