/**
 *  Copyright © 2020-2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.client.chat;

import java.awt.Color;
import java.awt.GridLayout;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JToolTip;
import javax.swing.ToolTipManager;
import org.nexuswob.gui.ArrowTooltip;
import org.nexuswob.gui.JToolTipBorder;
import net.javacomm.client.environment.GUI;
import net.javacomm.client.resource.Resource;
import net.javacomm.multilingual.Babelfish;
import net.javacomm.multilingual.MultilingualButton;
import net.javacomm.multilingual.MultilingualString;
import net.javacomm.multilingual.schema.ISO639;
import net.javacomm.multilingual.schema.KEY;
import net.javacomm.window.manager.Control;



/**
 * 
 * Ein 2-Listen-Austauschpanel.
 * 
 * @author llange
 *
 */
public class ArrowButtonPanel extends JPanel implements Babelfish {

  private static final long serialVersionUID = 779758453353282377L;
  private PropertyChangeSupport changes = new PropertyChangeSupport(this);

  @SuppressWarnings("serial")
  private JButton buttonLeft = new JButton() {
    @Override
    public JToolTip createToolTip() {
      ToolTipManager.sharedInstance().setInitialDelay(200);
      ToolTipManager.sharedInstance().setDismissDelay(4000);
      ArrowTooltip arrow = new ArrowTooltip(Resource.JQUERY_ORANGE);
      arrow.setComponent(buttonLeft);
      arrow.setTextAttributes(GUI.regularFont13, Color.BLACK, Resource.JQUERY_ORANGE);
      arrow.setBorder(new JToolTipBorder(7, Resource.JQUERY_BRAUN, Resource.JQUERY_ORANGE));
      return arrow;
    }
  };
  private MultilingualButton buttonCommit = new MultilingualButton(KEY.BUTTON_CONFIRM);
  private Class<? extends ArrowButtonPanel> clazz;
  private MultilingualString benutzerEinladen = new MultilingualString(KEY.STRING_BENUTZER_EINLADEN);
  
  
  
  
  public ArrowButtonPanel() {
    super(true);
    setLayout(new GridLayout(1, 1));
    init();
  }

  
  public synchronized void addArrowButtonListener(PropertyChangeListener l) {
    changes.addPropertyChangeListener(l);
  }
  

  private void init() {
    clazz = getClass();
    buttonLeft.setIcon(new ImageIcon(clazz.getResource(Resource.LEFTARROW_25x26)));
    buttonLeft.setToolTipText(benutzerEinladen.toString());
    buttonLeft.addActionListener((event) -> {
      changes.firePropertyChange(ArrowButtonPanel.class.getName(), Control.NULL, Control.SELECTED);
    });
    buttonCommit.addActionListener((event) -> {
      changes.firePropertyChange(ArrowButtonPanel.class.getName(), Control.NULL, Control.CONFIRM);
    });
    add(buttonLeft);
  }

  
  public synchronized void removeAllListener() {
    for (PropertyChangeListener tmp: changes.getPropertyChangeListeners()) {
      removeArrowButtonListener(tmp);
    }
  }
  
    
  
  public synchronized void removeArrowButtonListener(PropertyChangeListener l) {
    changes.removePropertyChangeListener(l);
  }



  @Override
  public void setLanguage(ISO639 code) {
    benutzerEinladen.setLanguage(code);
    buttonLeft.setToolTipText(benutzerEinladen.toString());
    buttonCommit.setLanguage(code);
  }
  

  
  
  
  
}