/**
 *  Copyright © 2020-2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.client.filetransfer;

import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.SwingConstants;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableColumnModel;
import org.nexuswob.gui.TableModel3;
import org.nexuswob.util.Cell;
import org.nexuswob.util.Util.Record;
import net.javacomm.multilingual.Babelfish;
import net.javacomm.multilingual.MultilingualString;
import net.javacomm.multilingual.schema.ISO639;
import net.javacomm.multilingual.schema.KEY;
import net.javacomm.protocol.BlacklistTypes;



@SuppressWarnings("serial")
public class JBlacklist extends JScrollPane implements Babelfish {

  private JTable table = new JTable();
  private MultilingualString beschreibung = new MultilingualString(KEY.STRING_BESCHREIBUNG);
  private MultilingualString suffix = new MultilingualString(KEY.STRING_SUFFIX);
  private MultilingualString gesperrt = new MultilingualString(KEY.STRING_GESPERRT);
  private DefaultTableCellRenderer headerRenderer;

  private TableModel3 model3 = new TableModel3() {

    @Override
    public boolean isCellEditable(int rowIndex, int columnIndex) {
      return columnIndex == Column.GESPERRT.columnNumber;
    }

  };
  private JTableHeader header;
  private TableColumnModel columnModel;

  public JBlacklist() {
    table.setModel(model3);
    table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
    model3.setHeader(Column.BESCHREIBUNG.toString(), Column.SUFFIX.toString(), Column.GESPERRT.toString());
    header = table.getTableHeader();
    centerHeader();

    table.setFillsViewportHeight(true);
//    table.setAutoResizeMode(JTable.AUTO_RESIZE_OFF); // wieder aktiviert, HorizontalScrollbar verwenden, falls
//                                                     // benötigt
    table.setRowHeight(20);

    columnModel = table.getColumnModel();
    columnModel.getColumn(Column.BESCHREIBUNG.columnNumber).setPreferredWidth(480);
    columnModel.getColumn(Column.SUFFIX.columnNumber).setPreferredWidth(65);
    columnModel.getColumn(Column.GESPERRT.columnNumber).setPreferredWidth(85);


    this.getViewport().setOpaque(true);
    setViewportView(table);
    this.setOpaque(true);
    setBorder(null);

  }



  void centerHeader() {
    headerRenderer = (DefaultTableCellRenderer) header.getDefaultRenderer();
    headerRenderer.setHorizontalAlignment(SwingConstants.CENTER);
  }


  public void setSperrliste(BlacklistTypes[] sperrliste) {
    // Liste leeren
    model3.removeAll();

    for (BlacklistTypes tmp : sperrliste) {

      Record record = new Record();

      Cell<String> description = new Cell<>();
      description.setValue(tmp.getDescription());
      record.add(description);

      Cell<String> suffix = new Cell<>();
      suffix.setValue(tmp.getSuffix());
      record.add(suffix);

      Cell<Boolean> check = new Cell<>();
      check.setValue(tmp.getChecked());
      record.add(check);

      model3.addRow(record);
    }

  }



  /**
   * Das TableModel wird zurückgegeben.
   * 
   * 
   * @return
   */
  synchronized BlacklistTypes[] toBlacklistTypes() {
    BlacklistTypes[] blacklist = new BlacklistTypes[model3.getRecords().size()];
    for (int index = 0; index < blacklist.length; index++) {
      Record record = model3.getRecordAt(index);
      blacklist[index] = new BlacklistTypes();
      blacklist[index].setDescription((String) record.getCellAt(Column.BESCHREIBUNG.columnNumber).getValue());
      blacklist[index].setSuffix((String) record.getCellAt(Column.SUFFIX.columnNumber).getValue());
      blacklist[index].setChecked((Boolean) record.getCellAt(Column.GESPERRT.columnNumber).getValue());
    }
    return blacklist;
  }

  enum Column {
    BESCHREIBUNG("Beschreibung", 0),
    SUFFIX("Suffix", 1),
    GESPERRT("Gesperrt", 2);

    private String columnName;
    private int columnNumber;

    private Column(String value, int columnNumber) {
      columnName = value;
      this.columnNumber = columnNumber;
    }



    @Override
    public String toString() {
      return columnName;
    }

  }

  @Override
  public void setLanguage(ISO639 code) {
    gesperrt.setLanguage(code);
    beschreibung.setLanguage(code);
    suffix.setLanguage(code);
    columnModel.getColumn(Column.BESCHREIBUNG.columnNumber).setHeaderValue(beschreibung.toString());
    columnModel.getColumn(Column.SUFFIX.columnNumber).setHeaderValue(suffix.toString());
    columnModel.getColumn(Column.GESPERRT.columnNumber).setHeaderValue(gesperrt.toString());
  }

}
