/**
 *  Copyright © 2020-2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.client.mediaplayer;

import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JPanel;
import net.javacomm.client.resource.Resource;
import net.javacomm.window.manager.Control;
import net.javacomm.window.manager.WMResource;



/**
 * 
 * Die Knopfleiste Start, Pause, Rückspulen.
 * 
 * @author llange
 *
 */
public class Knopfleiste extends JPanel {


  private static final long serialVersionUID = 5533666702024751454L;
  private GridLayout grid = new GridLayout(1, 3);
  private JButton buttonStart = new JButton();
  private JButton buttonPause = new JButton();
  private JButton buttonRewind = new JButton();
  private final PropertyChangeSupport changesMediabutton = new PropertyChangeSupport(Knopfleiste.class);
  private ActionListener startListener;
  private ActionListener pauseListener;
  private ActionListener rewindListener;
  private ImageIcon iconStart;
  private ImageIcon iconPause;
  private ImageIcon iconRewind;
  private Class<? extends Knopfleiste> resource;

  Knopfleiste() {
    super(true);
    init();
  }



  synchronized void addButtonListener(PropertyChangeListener l) {
    changesMediabutton.addPropertyChangeListener(l);
  }



  void buttonEnabled(JButton button) {
    button.setForeground(WMResource.ENABLED_BUTTON_FOREGROUND);
    button.setBackground(WMResource.ENABLED_BUTTON_BACKGROUND);
  }



  void init() {
    resource = getClass();
    iconStart = new ImageIcon(resource.getResource(Resource.MEDIASTART));
    iconPause = new ImageIcon(resource.getResource(Resource.MEDIAPAUSE));
    iconRewind = new ImageIcon(resource.getResource(Resource.MEDIA_SKIP_BACKWARD));

    startListener = new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        pressStart();
        changesMediabutton.firePropertyChange("START", Control.MEDIAPLAYER_PAUSE, Control.MEDIAPLAYER_START);
      }
    };
    pauseListener = new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        pressPause();
        changesMediabutton.firePropertyChange("PAUSE", Control.MEDIAPLAYER_START, Control.MEDIAPLAYER_PAUSE);
      }
    };
    rewindListener = new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        pressRewind();
        changesMediabutton
            .firePropertyChange("REWIND", Control.MEDIAPLAYER_START, Control.MEDIAPLAYER_REWIND);
      }

    };
    setLayout(grid);
    super.add(buttonStart);
    buttonStart.setOpaque(false);
    buttonStart.setIcon(iconStart);
    buttonStart.addActionListener(startListener);
    super.add(buttonPause);

    buttonPause.setIcon(iconPause);

    super.add(buttonRewind);
    buttonRewind.setIcon(iconRewind);
  }



  /**
   * Ist der Pauseknopf aktiviert oder nicht aktiviert. Kann der Pauseknopf Klicks
   * entgegennehmen?
   * 
   * @return
   */
  public boolean isPressedPaused() {
    return buttonPause.getActionListeners().length != 0;
  }



  /**
   * Das ActionEvent klicke auf Pause-Button wird ausgelöst.
   * 
   */
  public void pressedPausebutton() {
    buttonPause.doClick();
  }




  /**
   * Das ActionEvent klicke auf Start-Button wird ausgelöst.
   * 
   */
  public void pressedStartbutton() {
    buttonStart.doClick();
  }



  /**
   * Der PAUSE-Button wurde programmatisch geklickt.
   * 
   */
  void pressPause() {
    buttonStart.addActionListener(startListener);
    buttonPause.removeActionListener(pauseListener);
    buttonRewind.addActionListener(rewindListener);

    buttonPause.setBackground(Resource.JQUERY_RED);
    buttonEnabled(buttonStart);
    buttonEnabled(buttonRewind);
  }



  /**
   * Der REWIND-Button wurde programmatisch geklickt.
   * 
   */
  private void pressRewind() {
    buttonStart.addActionListener(startListener);
    buttonPause.removeActionListener(pauseListener);
    buttonRewind.removeActionListener(rewindListener);

    buttonRewind.setBackground(Resource.JQUERY_RED);
    buttonPause.setBackground(Resource.JQUERY_RED);

    buttonEnabled(buttonStart);

  }



  /**
   * Der START-Button wurde programmatisch geklickt.
   * 
   */
  void pressStart() {
    buttonStart.removeActionListener(startListener);
    buttonPause.addActionListener(pauseListener);
    buttonRewind.removeActionListener(rewindListener);

    buttonStart.setBackground(Resource.JQUERY_RED);
    buttonRewind.setBackground(Resource.JQUERY_RED);
    buttonEnabled(buttonPause);

  }



  synchronized void removeButtonListener(PropertyChangeListener l) {
    changesMediabutton.removePropertyChangeListener(l);
  }
}