/**
 *  Copyright © 2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.client.phone;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.List;
import java.util.concurrent.Callable;
import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.LineEvent;
import javax.sound.sampled.LineListener;
import javax.sound.sampled.LineUnavailableException;
import javax.sound.sampled.Mixer.Info;
import javax.sound.sampled.TargetDataLine;
import net.javacomm.multilingual.Babelfish;
import net.javacomm.multilingual.MultilingualString;
import net.javacomm.multilingual.schema.ISO639;
import net.javacomm.multilingual.schema.KEY;
import io.nayuki.flac.app.EncodeWavToFlac;



public class MicLightProducer implements Callable<byte[]>, Babelfish {

  private Telefonformat telefonformat;
  private int audioBufferSize;
  private TargetDataLine targetDataline;
  private MultilingualString mikrofonnameIstFalsch = new MultilingualString(
      KEY.STRING_DER_MIKROFONNAME_IST_UNBEKANNT
  );
  private LineListener lineListener = new LineListener() {

    @Override
    public void update(LineEvent event) {
      // wird vom JavaSoundEventDispatcher Thread ausgeführt
      if (event.getType().equals(LineEvent.Type.OPEN)) {}
      else if (event.getType().equals(LineEvent.Type.START)) {
        targetDataline.flush();
      }
      else if (event.getType().equals(LineEvent.Type.STOP)) {}
      else if (event.getType().equals(LineEvent.Type.CLOSE)) {
        targetDataline.removeLineListener(lineListener);
      }
    }
  };

  public MicLightProducer(String mikroname, Telefonformat telefonformat)
      throws MikrofonException, MicronameException {
    this(mikroname, telefonformat, 2304);
  }



  public MicLightProducer(String mikroname, Telefonformat telefonformat, int audioBufferSize)
      throws MikrofonException, MicronameException {
    this.telefonformat = telefonformat;
    this.audioBufferSize = audioBufferSize;
    setTelefonformat(mikroname, telefonformat);
  }



  private void setTelefonformat(String mikroname, Telefonformat value)
      throws MikrofonException, MicronameException {
    telefonformat = value;

    List<Info> mixer = TelefonUtil.microMixer(telefonformat);
    for (Info tmp : mixer) {
      if (tmp.getName().equals(mikroname)) {
        try {
          targetDataline = TelefonUtil.getTargetDataLine(tmp, telefonformat);
        }
        catch (LineUnavailableException e) {
          throw new MikrofonException("Für dieses Mikrofon ist die Leitung belegt. - " + mikroname);
        }
        targetDataline.addLineListener(lineListener);
        open(TelefonUtil.getAudioFormat(telefonformat));
        return;
      }
    }
    throw new MicronameException("<b>'" + mikroname + "'</b> - " + mikrofonnameIstFalsch.toString());

  }



  /**
   * Öffnet eine Mikrofonleitung. Die Mikrofonleitung darf nicht belegt sein.
   * 
   * @param audioformat
   *                    dieses Audioformat
   * @throws MikrofonException
   *                           die Mikrofonleitung wird benutzt; schließe die
   *                           Mikrofonleitung
   */
  private void open(AudioFormat audioformat) throws MikrofonException {
    try {
      if (targetDataline == null) return;
      targetDataline.open(audioformat);
    }
    catch (LineUnavailableException e) {
      throw new MikrofonException("Diese Mikrofonleitung wird bereits benutzt.");
    }
  }



  @Override
  public byte[] call() throws Exception {
    byte[] rawAudiobuffer = new byte[audioBufferSize]; // 3*564+576];
    int read = targetDataline.read(rawAudiobuffer, 0, rawAudiobuffer.length);
    byte[] waveByteArray = toWave(ByteBuffer.wrap(rawAudiobuffer, 0, read), telefonformat);
    byte[] flacArray = EncodeWavToFlac.entry(waveByteArray);
    return flacArray;
  }



  /**
   * 
   * Die gerade empfangenen Audiodaten werden übergeben. Die Audiodaten stammen
   * von einem Mikrofon.
   * 
   * 
   * @param micro
   *               aufgezeichnete Mikrofonbytes
   * @param format
   *               das Telefonformat der Mikrofonbytes
   * 
   * @return WAVE-Audiodaten als ByteArray
   * 
   * @throws AudioFormatException
   *                              das Audioformat wird nicht unterstützt
   * @throws IOException
   *                              encodieren ist fehlgeschlagen
   */
  protected byte[] toWave(ByteBuffer micro, Telefonformat format) throws AudioFormatException, IOException {
    Raw2Wave raw2wave = new Raw2Wave(format);
    return raw2wave.encode(micro);
  }



  @Override
  public void setLanguage(ISO639 code) {
    mikrofonnameIstFalsch.setLanguage(code);
  }

}
