/**
 *  Copyright © 2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  Previously released under Apache License 2.0; now licensed under MPL 2.0.
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.restserver;

import jakarta.ws.rs.FormParam;
import jakarta.ws.rs.POST;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.core.MediaType;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import net.javacomm.database.WebdatabaseImpl;
import net.javacomm.database.entity.FiletransferConfig;
import net.javacomm.share.Constants;
import net.javacomm.transfer.Starttype;
import net.javacomm.transfer.TransferFiletransferConfig;



@Produces(MediaType.APPLICATION_JSON)
@Path("/filetransfer")
public class Filetransfer {

  final Logger log = LogManager.getLogger(Filetransfer.class);
  private WebdatabaseImpl database;

  public Filetransfer() {
    database = (WebdatabaseImpl) WebdatabaseImpl.getInstance();
  }



  /**
   * Die Benutzerdaten aus der TB_FILETRANSFER_CONFIG werden gelesen.
   * 
   * 
   * @param userid
   *               von diesem Benutzer werden die Daten gelesen.
   * @return die Benutzerdaten
   */
  @POST
  @Path("/read")
  public TransferFiletransferConfig readFiletransfer(@FormParam(value = "userid") String userid) {
    TransferFiletransferConfig config = new TransferFiletransferConfig();
    FiletransferConfig result = database.fetchFiletransferConfig(userid);
    config.setPort(result.getPort());
    config.setStarttype(Starttype.toTransferStarttype(result.getStarttype()));
    config.setUid(result.getUid());
    config.setDownloadDir(result.getDownloadDir());
    config.setUploadDir(result.getUploadDir());
    return config;
  }



  @POST
  @Path("/write")
  @Produces(MediaType.TEXT_PLAIN)
  public boolean writeFiletransfer(@FormParam(value = "config") TransferFiletransferConfig config) {

    if (Constants.MIN_PORT_RANGE > config.getPort() || config.getPort() > Constants.MAX_PORT_RANGE)
      return false;
    FiletransferConfig transfer = new FiletransferConfig();
    transfer.setPort(config.getPort());
    transfer.setStarttype(config.getStarttype().getModus());
    transfer.setUid(config.getUid());
    transfer.setDownloadDir(config.getDownloadDir());
    transfer.setUploadDir(config.getUploadDir());
    return database.updateFiletransferConfig(transfer);

  }



  /**
   * Aktualisiere die Verezeichniseinträge für den Dateitransfer.
   * 
   * 
   * @param config
   *               die Benutzereinstellungen
   * @return {@code true}, die Verzeichnisse wurden gespeichert
   */
  @POST
  @Path("/write/directories")
  @Produces(MediaType.TEXT_PLAIN)
  public boolean writeDirectories(@FormParam(value = "config") TransferFiletransferConfig config) {

    FiletransferConfig transfer = new FiletransferConfig();
    transfer.setUid(config.getUid());
    transfer.setDownloadDir(config.getDownloadDir());
    transfer.setUploadDir(config.getUploadDir());
    return database.updateFiletransferDirectories(transfer);
  }

}
