/**
 *  Copyright © 2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  Previously released under Apache License 2.0; now licensed under MPL 2.0.
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.protocol;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import java.util.Objects;



public class KonferenzraumUser extends ChatUser {

  public final static String ANWESEND = "ANWESEND";
  public final static String SESSION = "SESSION";
  public final static String CHATUSER = "CHATUSER";
  public final static String MUTE = "MUTE";

  private boolean anwesend;
  private String session; // nur vorhanden, wenn der User anwesend ist
  private boolean mute; // standardmäßig ist das Mikrofon immer eingeschaltet.

  @Override
  public boolean equals(Object obj) {
    if (this == obj) return true;
    if (!super.equals(obj)) return false;
    if (getClass() != obj.getClass()) return false;
    KonferenzraumUser other = (KonferenzraumUser) obj;
    return anwesend == other.anwesend && Objects.equals(session, other.session);
  }



  /**
   * Der Server soll an diese Session senden. Wer eine Session hat, ist online.
   * 
   * @return diese Session gehört zu einem Anwender oder {@code null}, wenn der
   *         User nicht online ist
   */
  public String getSession() {
    return session;
  }



  @Override
  public int hashCode() {
    final int prime = 31;
    int result = super.hashCode();
    result = prime * result + Objects.hash(anwesend, session);
    return result;
  }



  /**
   * Ist der Anwender in Telefonkonferenz anwesend oder abwesend?
   * 
   * @return {@code true}, der Anwender ist anwesend
   */
  public boolean isAnwesend() {
    return anwesend;
  }



  /**
   * Ist das Mikrofon stumm?
   * 
   * @return {@code true}, ist stumm und nimmt nichts auf
   */
  public boolean isMute() {
    return mute;
  }



  /**
   * 
   * 
   * @param online
   *               {@code true}, wenn der User in der Telefonkonferenz anwesend
   *               ist; {@code false} abwesend
   */
  public void setAnwesend(boolean online) {
    anwesend = online;
  }



  public void setMute(boolean mute) {
    this.mute = mute;
  }



  /**
   * Der Server soll später an diese Session senden. Wer eine Session hat, ist
   * online.
   * 
   * @param session
   *                diese Session gehört zu einem Anwender
   */
  public void setSession(String session) {
    this.session = session;
  }



  @Override
  public String toString() {
    GsonBuilder builder = new GsonBuilder();
    builder.registerTypeAdapter(KonferenzraumUser.class, new KonferenzraumUserSerializer());
    Gson gson = builder.create();
    return gson.toJson(this);

  }

}
