/**
 *  Copyright © 2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  Previously released under Apache License 2.0; now licensed under MPL 2.0.
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.protocol;

import static net.javacomm.protocol.HEADER.CONFIRM;
import static net.javacomm.protocol.HEADER.ERROR;
import static net.javacomm.protocol.HEADER.REQUEST;
import static net.javacomm.protocol.HEADER.RESPONSE;
import static net.javacomm.protocol.Protocol.DATASET;
import static net.javacomm.protocol.Protocol.HEADER;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.stream.JsonWriter;
import jakarta.websocket.EncodeException;
import jakarta.websocket.Encoder;
import jakarta.websocket.EndpointConfig;
import java.io.IOException;
import java.io.StringWriter;
import java.util.Base64;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;



/**
 * 
 * Jede ausgehende {@link MESSAGE} wird über diese Klasse in einen <i>JSON</i>
 * String formatiert.
 * 
 * 
 * @author llange
 *
 */
public class MessageEncoder implements Encoder.Text<MESSAGE> {

  private final static Logger log = LogManager.getLogger(MessageEncoder.class);
  public final static String EMPTY = "";
  private GsonBuilder gsonbuilder;

  public MessageEncoder() {
    gsonbuilder = new GsonBuilder().disableHtmlEscaping();
  }



  @Override
  public void destroy() {}



  @Override
  public String encode(MESSAGE message) throws EncodeException {
    if (message == null) throw new IllegalArgumentException("message is null");
    String encoded = new String();
    try {

      encoded = encodeCONFERENCEAUDIO(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeNEWIMAGE(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeNEWMOUSEPOINTER(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeSTOPSENDINGVIDEO(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeCONFERENCEVIDEO(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeIMAGE(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeBEAMEROFF(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodePROYECTORCLOSING(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeSCREENCAST(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeBREAKROOMS(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeREADROOMS(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeCHATMESSAGE(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeCONFERENCE(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodePRIVATECHATFILE(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodePRIVATEMESSAGE(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeKEEPALIVE(message);
      if (!encoded.equals(EMPTY)) {
        return encoded;
      }

      encoded = encodeMICROERROR(message);
      if (!encoded.equals(EMPTY)) {
        return encoded;
      }

      encoded = encodeUPDATEFILETYPES(message);
      if (!encoded.equals(EMPTY)) {
        return encoded;
      }

      encoded = encodeFILETYPES(message);
      if (!encoded.equals(EMPTY)) {
        return encoded;
      }

      encoded = encodeONHOOK(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeINCOMINGCALL(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeUPDATEPHONE(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeDIAL(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeONCALL(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeSEARCHFILES(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeDOWNLOAD(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeUPLOADFILES(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeDELETEUPLOADFILES(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeCHATUSERLIST(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeCHATONLINELIST(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeHISTORYMESSAGE(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeROOMLIST(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeENTERROOM(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeLEAVEROOM(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeLEAVEPRIVATECHAT(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeCALLPRIVATECHAT(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeCALLREMOTEUSER(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeUSERONLINELIST(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeCREATETEMPROOM(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeUSRLOGIN(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeTRANSFER(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeFTSDOWNLOAD(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeUPDATEUSER(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeSIGNIN(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeDELETEUSER(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeUPDATEPORT(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeREADTOPICROOMOWNER(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeTOPICMEMBER(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeCHANGETOPICMEMBER(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeCANDIDATETOPICMEMBER(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeDELETEROOM(message);
      if (!encoded.equals(EMPTY)) return encoded;

      encoded = encodeCONFERENCEMUTE(message);
      if (!encoded.equals(EMPTY)) return encoded;

    }
    catch (IOException e) {
      throw new EncodeException(message, e.getMessage());
    }
    log.fatal(message.toString() + " - diese Botschaft konnte nicht encoded werden");
    throw new EncodeException(message, "diese Botschaft konnte nicht encoded werden");
  }



  String encodeNEWIMAGE(MESSAGE message) throws IOException, EncodeException {
    if (!(message instanceof NEWIMAGE)) return EMPTY;
    NEWIMAGE newimage = (NEWIMAGE) message;
    if (!Command.NEWIMAGE.equals(message.getCommand())) return EMPTY;
    StringWriter stringwriter = new StringWriter();
    Gson gson = gsonbuilder.create();
    JsonWriter generator = gson.newJsonWriter(stringwriter);
    generator.beginObject();
    generator.name(HEADER).value(message.getHeader().toString());
    generator.name(Command.getCommandKey()).value(message.getCommand().name());

    if (newimage.getDataset() == null)
      throw new EncodeException(newimage, "Protokollfehler: das Attribut DATASET wurde nicht definiert.");
    generator.name(DATASET);
    generator.beginObject();

    if (message.getHeader() == REQUEST) {
      if (newimage.getUserid() == null)
        throw new EncodeException(newimage, "Protokollfehler: NEWIMAGE | REQUEST, das Attribut USERID fehlt");
      generator.name(NEWIMAGE.USERID).value(newimage.getUserid());
    }
    generator.endObject();
    generator.endObject();
    return stringwriter.toString();

  }



  String encodeNEWMOUSEPOINTER(MESSAGE message) throws IOException, EncodeException {
    if (!(message instanceof NEWMOUSEPOINTER)) return EMPTY;
    NEWMOUSEPOINTER newMousepointer = (NEWMOUSEPOINTER) message;
    if (!Command.NEWMOUSEPOINTER.equals(message.getCommand())) return EMPTY;
    StringWriter stringwriter = new StringWriter();
    Gson gson = gsonbuilder.create();
    JsonWriter generator = gson.newJsonWriter(stringwriter);
    generator.beginObject();
    generator.name(HEADER).value(message.getHeader().toString());
    generator.name(Command.getCommandKey()).value(message.getCommand().name());

    if (
      newMousepointer.getDataset() == null
    ) throw new EncodeException(
        newMousepointer, "Protokollfehler: das Attribut DATASET wurde nicht definiert."
    );
    generator.name(DATASET);
    generator.beginObject();
    if (newMousepointer.getX() == null) {
      throw new EncodeException(
          newMousepointer, "Protokollfehler: NEWMOUSEPOINTER | REQUEST, das Attribut X fehlt"
      );
    }
    generator.name(NEWMOUSEPOINTER.X).value(newMousepointer.getX());

    if (newMousepointer.getY() == null) {
      throw new EncodeException(
          newMousepointer, "Protokollfehler: NEWMOUSEPOINTER | REQUEST, das Attribut Y fehlt"
      );
    }
    generator.name(NEWMOUSEPOINTER.Y).value(newMousepointer.getY());

    generator.endObject();
    generator.endObject();
    return stringwriter.toString();

  }



  private String encodeSTOPSENDINGVIDEO(MESSAGE message) throws IOException, EncodeException {
    if (!(message instanceof STOPSENDINGVIDEO)) return EMPTY;
    STOPSENDINGVIDEO stopSendingVideo = (STOPSENDINGVIDEO) message;
    if (!Command.STOPSENDINGVIDEO.equals(message.getCommand())) return EMPTY;
    StringWriter stringwriter = new StringWriter();
    Gson gson = gsonbuilder.create();
    JsonWriter generator = gson.newJsonWriter(stringwriter);
    generator.beginObject();
    generator.name(Protocol.HEADER).value(message.getHeader().toString());
    generator.name(Command.getCommandKey()).value(message.getCommand().name());
    if (stopSendingVideo.getDataset() == null)
      throw new EncodeException(stopSendingVideo, "DATASET wurde nicht definiert.");
    generator.name(stopSendingVideo.getDataset());
    generator.beginObject();
    switch(message.getHeader()) {
      case CONFIRM:
        break;
      case ERROR:
        break;
      case REQUEST:
        if (stopSendingVideo.getUserid() == null)
          throw new EncodeException(stopSendingVideo, "das Attribut USERID fehlt");
        generator.name(STOPSENDINGVIDEO.USERID).value(stopSendingVideo.getUserid());
        break;
      case RESPONSE:
        break;
      default:
        break;
    }
    generator.endObject();
    generator.endObject();
    return stringwriter.toString();
  }



  private String encodeCONFERENCEMUTE(MESSAGE message) throws IOException, EncodeException {
    if (!(message instanceof CONFERENCEMUTE)) return EMPTY;
    CONFERENCEMUTE conferencemute = (CONFERENCEMUTE) message;
    if (!Command.CONFERENCEMUTE.equals(message.getCommand())) return EMPTY;
    StringWriter stringwriter = new StringWriter();
    Gson gson = gsonbuilder.create();
    JsonWriter generator = gson.newJsonWriter(stringwriter);
    generator.beginObject();
    generator.name(Protocol.HEADER).value(message.getHeader().toString());
    generator.name(Command.getCommandKey()).value(message.getCommand().name());
    if (conferencemute.getDataset() == null)
      throw new EncodeException(conferencemute, "DATASET wurde nicht definiert.");
    generator.name(conferencemute.getDataset());
    generator.beginObject();

    switch(message.getHeader()) {
      case CONFIRM:
        break;
      case ERROR:
        break;
      case REQUEST:
        if (conferencemute.getKonferenzraum() == null)
          throw new EncodeException(conferencemute, "das Attribut KONFERENZRAUM fehlt");
        if (conferencemute.getOrganisator() == null)
          throw new EncodeException(conferencemute, "das Attribut ORGANISATOR fehlt");
        if (conferencemute.getUserid() == null)
          throw new EncodeException(conferencemute, "das Attribut USERID fehlt");
        if (conferencemute.getReceiverSessions() == null)
          throw new EncodeException(conferencemute, "das Attribut RECEIVER_SESSIONS fehlt");
        if (conferencemute.isMute() == null)
          throw new EncodeException(conferencemute, "das Attribut MUTE fehlt");

        generator.name(CONFERENCEMUTE.KONFERENZRAUM).value(conferencemute.getKonferenzraum());
        generator.name(CONFERENCEMUTE.ORGANISATOR).value(conferencemute.getOrganisator());
        generator.name(CONFERENCEMUTE.USERID).value(conferencemute.getUserid());
        generator.name(CONFERENCEMUTE.RECEIVER_SESSIONS);
        generator.beginArray();
        for (String session : conferencemute.getReceiverSessions()) {
          generator.value(session);
        }
        generator.endArray();
        generator.name(CONFERENCEMUTE.MUTE).value(conferencemute.isMute());
        break;
      case RESPONSE:
        break;
      default:
        break;
    }

    generator.endObject();
    generator.endObject();
    return stringwriter.toString();
  }



  private String encodeCONFERENCEVIDEO(MESSAGE message) throws EncodeException, IOException {
    if (!(message instanceof CONFERENCEVIDEO)) return EMPTY;
    CONFERENCEVIDEO conferencevideo = (CONFERENCEVIDEO) message;
    if (!Command.CONFERENCEVIDEO.equals(message.getCommand())) return EMPTY;
    StringWriter stringwriter = new StringWriter();
    Gson gson = gsonbuilder.create();
    JsonWriter generator = gson.newJsonWriter(stringwriter);
    generator.beginObject();
    generator.name(Protocol.HEADER).value(message.getHeader().toString());
    generator.name(Command.getCommandKey()).value(message.getCommand().name());
    if (conferencevideo.getDataset() == null)
      throw new EncodeException(conferencevideo, "DATASET wurde nicht definiert.");
    generator.name(conferencevideo.getDataset());
    switch(message.getHeader()) {
      case CONFIRM:
        break;
      case ERROR:
        generator.beginObject();
        generator.name(CONFERENCEVIDEO.MULTILINGUALKEY).value(conferencevideo.getMultilingualkey().value());
        generator.name(CONFERENCEVIDEO.SHARING).value(conferencevideo.isShared());
        generator.endObject();
        break;
      case REQUEST:
        generator.beginObject();
        if (conferencevideo.getUserid() == null)
          throw new EncodeException(conferencevideo, "das Attribut USERID fehlt");
        generator.name(CONFERENCEVIDEO.USERID).value(conferencevideo.getUserid());
        if (conferencevideo.getNickname() == null)
          throw new EncodeException(conferencevideo, "das Attribut NICKNAME fehlt");
        generator.name(CONFERENCEVIDEO.NICKNAME).value(conferencevideo.getNickname());

        if (conferencevideo.getOrganisator() == null)
          throw new EncodeException(conferencevideo, "das Attribut ORGANISATOR fehlt");
        generator.name(CONFERENCEVIDEO.ORGANISATOR).value(conferencevideo.getOrganisator());

        generator.name(CONFERENCEVIDEO.RECEIVER_SESSIONS);
        generator.beginArray();
        if (conferencevideo.getReceiverSessions() == null)
          throw new EncodeException(conferencevideo, "Das Attribut RECEIVER_SESSIONS wurde nicht definiert.");
        for (String session : conferencevideo.getReceiverSessions()) {
          generator.value(session);
        }
        generator.endArray();

        if (conferencevideo.isShared() == null)
          throw new EncodeException(conferencevideo, "das Attribut SHARING fehlt");
        generator.name(CONFERENCEVIDEO.SHARING).value(conferencevideo.isShared());

        if (conferencevideo.getKonferenzraum() == null)
          throw new EncodeException(conferencevideo, "das Attribut KONFERENZRAUM fehlt");
        generator.name(CONFERENCEVIDEO.KONFERENZRAUM).value(conferencevideo.getKonferenzraum());

        generator.endObject();
        break;
      case RESPONSE:
        break;
      default:
        break;
    }

    generator.endObject();
    return stringwriter.toString();
  }



  String encodeBEAMEROFF(MESSAGE message) throws IOException, EncodeException {
    if (!(message instanceof BEAMEROFF)) return EMPTY;
    BEAMEROFF beamer = (BEAMEROFF) message;

    if (!Command.BEAMEROFF.equals(message.getCommand())) return EMPTY;
    StringWriter stringwriter = new StringWriter();
    Gson gson = gsonbuilder.create();
    JsonWriter generator = gson.newJsonWriter(stringwriter);
    generator.beginObject();
    generator.name(Protocol.HEADER).value(message.getHeader().toString());
    generator.name(Command.getCommandKey()).value(message.getCommand().name());
    if (beamer.getDataset() == null) throw new EncodeException(beamer, "DATASET wurde nicht definiert.");

    generator.name(beamer.getDataset());
    switch(message.getHeader()) {
      case REQUEST:
        if (beamer.getUserid() == null)
          throw new EncodeException(beamer, "Das Attribut USERID wurde nicht definiert");
        generator.beginObject();
        generator.name(BEAMEROFF.USERID).value(beamer.getUserid());
        generator.endObject();
        break;
      case RESPONSE:
        if (beamer.getUserid() == null)
          throw new EncodeException(beamer, "Das Attribut USERID wurde nicht definiert");
        generator.beginObject();
        generator.name(BEAMEROFF.USERID).value(beamer.getUserid());
        generator.endObject();
        break;
    }

    generator.endObject();
    return stringwriter.toString();
  }



  String encodeBREAKROOMS(MESSAGE message) throws EncodeException, IOException {
    if (!(message instanceof BREAKROOMS)) return EMPTY;
    BREAKROOMS breakrooms = (BREAKROOMS) message;
    if (!Command.BREAKROOMS.equals(message.getCommand())) return EMPTY;
    StringWriter stringwriter = new StringWriter();
    Gson gson = gsonbuilder.create();
    JsonWriter generator = gson.newJsonWriter(stringwriter);
    generator.beginObject();
    generator.name(Protocol.HEADER).value(message.getHeader().toString());
    generator.name(Command.getCommandKey()).value(message.getCommand().name());
    if (REQUEST == message.getHeader()) {
      if (breakrooms.getDataset() == null)
        throw new EncodeException(breakrooms, "DATASET wurde nicht definiert.");
      generator.name(breakrooms.getDataset());
      generator.beginObject();
      if (breakrooms.getUserid() == null)
        throw new EncodeException(breakrooms, "Das Attribut USERID wurde nicht definiert.");
      generator.name(BREAKROOMS.USERID).value(breakrooms.getUserid());
      generator.endObject();
    }
    else if (RESPONSE.equals(message.getHeader())) {
      if (breakrooms.getDataset() == null)
        throw new EncodeException(breakrooms, "DATASET wurde nicht definiert.");
      generator.name(breakrooms.getDataset());
      generator.beginObject();
      if (breakrooms.getUserid() == null)
        throw new EncodeException(breakrooms, "Das Attribut USERID wurde nicht definiert.");
      generator.name(BREAKROOMS.USERID).value(breakrooms.getUserid());
      if (breakrooms.getRooms() == null)
        throw new EncodeException(breakrooms, "Das Attribut ROOMS wurde nicht definiert.");
      generator.name(BREAKROOMS.ROOMS);
      generator.beginArray();
      for (String tmp : breakrooms.getRooms()) {
        generator.value(tmp);
      }
      generator.endArray();
      generator.endObject();
    }
    generator.endObject();
    generator.flush();
    generator.close();
    return stringwriter.toString();
  }



  public String encodeCALLPRIVATECHAT(MESSAGE message) throws EncodeException, IOException {
    if (!(message instanceof CALLPRIVATECHAT)) return EMPTY;
    CALLPRIVATECHAT call = (CALLPRIVATECHAT) message;
    if (Command.CALLPRIVATECHAT.equals(call.getCommand())) {
      StringWriter stringwriter = new StringWriter();
      Gson gson = gsonbuilder.create();
      JsonWriter generator = gson.newJsonWriter(stringwriter);
      generator.beginObject();
      generator.name(Protocol.HEADER).value(message.getHeader().toString());
      generator.name(Command.getCommandKey()).value(message.getCommand().name());
      if (REQUEST == message.getHeader()) {
        generator.name(message.getDataset()).beginObject();
        try {
          generator.name(CALLPRIVATECHAT.SENDER_UID).value(call.getSenderUID());
          generator.name(CALLPRIVATECHAT.LOCAL_NICKNAME).value(call.getLocalNickname());
          generator.name(CALLPRIVATECHAT.REMOTE_NICKNAME).value(call.getRemoteNickname());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde ein Attribut nicht definiert.");
        }
        generator.endObject();
      }
      else if (CONFIRM == message.getHeader()) {
        try {
          generator.name(message.getDataset()).beginObject();
          generator.name(CALLPRIVATECHAT.LOCAL_NICKNAME).value(call.getLocalNickname());
          generator.name(CALLPRIVATECHAT.LOCAL_SESSIONID).value(call.getLocalSessionid());
          generator.name(CALLPRIVATECHAT.REMOTE_NICKNAME).value(call.getRemoteNickname());
          generator.name(CALLPRIVATECHAT.REMOTE_SESSIONID).value(call.getRemoteSessionid());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde ein Attribut nicht definiert.");
        }
        generator.endObject();
      }
      else if (RESPONSE.equals(message.getHeader())) {
        generator.name(message.getDataset()).beginObject();
        try {
          generator.name(CALLPRIVATECHAT.LOCAL_NICKNAME).value(call.getLocalNickname());
          generator.name(CALLPRIVATECHAT.LOCAL_SESSIONID).value(call.getLocalSessionid());
          generator.name(CALLPRIVATECHAT.REMOTE_NICKNAME).value(call.getRemoteNickname());
          generator.name(CALLPRIVATECHAT.REMOTE_SESSIONID).value(call.getRemoteSessionid());
          generator.name(MESSAGE.TEXT).value(call.getText());
          if (message.getMultilingualkey() != null)
            generator.name(MESSAGE.MULTILINGUALKEY).value(message.getMultilingualkey().value());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde ein Attribut nicht definiert.");
        }
        generator.endObject();
      }
      else if (ERROR == message.getHeader()) {
        generator.name(message.getDataset()).beginObject();
        if (call.getLocalNickname() != null)
          generator.name(CALLPRIVATECHAT.LOCAL_NICKNAME).value(call.getLocalNickname());
        if (call.getLocalSessionid() != null)
          generator.name(CALLPRIVATECHAT.LOCAL_SESSIONID).value(call.getLocalSessionid());
        if (call.getRemoteNickname() != null)
          generator.name(CALLPRIVATECHAT.REMOTE_NICKNAME).value(call.getRemoteNickname());
        if (call.getRemoteSessionid() != null)
          generator.name(CALLPRIVATECHAT.REMOTE_SESSIONID).value(call.getRemoteSessionid());
        if (call.getErrorMessage() != null)
          generator.name(MESSAGE.ERRORMESSAGE).value(call.getErrorMessage());
        if (message.getText() != null) {
          generator.name(MESSAGE.TEXT).value(call.getText());
        }
        if (message.getMultilingualkey() != null)
          generator.name(MESSAGE.MULTILINGUALKEY).value(message.getMultilingualkey().value());
        generator.endObject();
      }
      generator.endObject();
      generator.flush();
      generator.close();
      return stringwriter.toString();
    }
    return EMPTY;
  }



  public String encodeCALLREMOTEUSER(MESSAGE message) throws EncodeException, IOException {
    if (!(message instanceof CALLREMOTEUSER)) return EMPTY;
    CALLREMOTEUSER call = (CALLREMOTEUSER) message;
    if (Command.CALLREMOTEUSER.equals(call.getCommand())) {
      StringWriter stringwriter = new StringWriter();
      Gson gson = gsonbuilder.create();
      JsonWriter generator = gson.newJsonWriter(stringwriter);
      generator.beginObject();
      generator.name(Protocol.HEADER).value(message.getHeader().toString());
      generator.name(Command.getCommandKey()).value(message.getCommand().name());
      if (REQUEST == message.getHeader()) {
        generator.name(message.getDataset()).beginObject();
        try {
          generator.name(CALLREMOTEUSER.LOCAL_NICKNAME).value(call.getLocalNickname());
          generator.name(CALLREMOTEUSER.LOCAL_SESSIONID).value(call.getLocalSessionid());
          generator.name(CALLREMOTEUSER.REMOTE_NICKNAME).value(call.getRemoteNickname());
          generator.name(CALLREMOTEUSER.REMOTE_SESSIONID).value(call.getRemoteSessionid());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde ein Attribut nicht definiert.");
        }
        generator.endObject();
      }
      else if (CONFIRM == message.getHeader()) {
        generator.name(message.getDataset()).beginObject();
        try {
          generator.name(CALLREMOTEUSER.LOCAL_NICKNAME).value(call.getLocalNickname());
          generator.name(CALLREMOTEUSER.LOCAL_SESSIONID).value(call.getLocalSessionid());
          generator.name(CALLREMOTEUSER.REMOTE_NICKNAME).value(call.getRemoteNickname());
          generator.name(CALLREMOTEUSER.REMOTE_SESSIONID).value(call.getRemoteSessionid());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde ein Attribut nicht definiert.");
        }
        generator.endObject();
      }
      else if (RESPONSE.equals(message.getHeader())) {
        generator.name(message.getDataset()).beginObject();
        try {
          generator.name(CALLREMOTEUSER.LOCAL_NICKNAME).value(call.getLocalNickname());
          generator.name(CALLREMOTEUSER.LOCAL_SESSIONID).value(call.getLocalSessionid());
          generator.name(CALLREMOTEUSER.REMOTE_NICKNAME).value(call.getRemoteNickname());
          generator.name(CALLREMOTEUSER.REMOTE_SESSIONID).value(call.getRemoteSessionid());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde ein Attribut nicht definiert.");
        }
        generator.endObject();
      }
      else if (ERROR == message.getHeader()) {
        generator.name(message.getDataset()).beginObject();
        try {
          generator.name(MESSAGE.ERRORMESSAGE).value(call.getErrorMessage());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde ein Attribut nicht definiert.");
        }
        generator.endObject();
      }
      generator.endObject();
      generator.flush();
      generator.close();
      return stringwriter.toString();
    }
    return EMPTY;
  }



  String encodeCANDIDATETOPICMEMBER(MESSAGE message) throws EncodeException, IOException {
    if (!(message instanceof CANDIDATETOPICMEMBER)) return EMPTY;
    CANDIDATETOPICMEMBER member = (CANDIDATETOPICMEMBER) message;
    if (Command.CANDIDATETOPICMEMBER.equals(member.getCommand())) {
      StringWriter stringwriter = new StringWriter();
      Gson gson = gsonbuilder.create();
      JsonWriter generator = gson.newJsonWriter(stringwriter);
      generator.beginObject();
      generator.name(Protocol.HEADER).value(message.getHeader().toString());
      generator.name(Command.getCommandKey()).value(message.getCommand().name());
      if (REQUEST == message.getHeader()) {
        generator.name(message.getDataset()).beginObject();
        try {
          generator.name(MEMBER.USERID).value(member.getUserid());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde das Attribut USERID nicht definiert.");
        }
        try {
          generator.name(MEMBER.ROOM).value(member.getRoom());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde das Attribut ROOM nicht definiert.");
        }
        generator.endObject();
      }
      else if (RESPONSE.equals(message.getHeader())) {
        generator.name(message.getDataset()).beginObject();
        try {
          generator.name(MEMBER.USERID).value(member.getUserid());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde das Attribut USERID nicht definiert.");
        }
        try {
          generator.name(MEMBER.ROOM).value(member.getRoom());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde das Attribut ROOM nicht definiert.");
        }
        try {
          generator.name(MEMBER.ROOMTYPE).value(member.getRoomtype().toString());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde das Attribut ROOMTYPE nicht definiert.");
        }
        // es kommen die User
        generator.name(MEMBER.TOPICMEMBER).beginArray();
        ChatUser[] rooms = member.getChatUser();
        if (rooms != null) {
          for (ChatUser tmp : rooms) {
            generator.beginObject();
            if (
              tmp.getUserid() == null
            ) throw new EncodeException(
                message, "Im Datensatz wurde das Attribut TOPICMEMBER/USERID nicht definiert."
            );
            generator.name(ChatUser.USERID).value(tmp.getUserid());
            if (
              tmp.getNickname() == null
            ) throw new EncodeException(
                message, "Im Datensatz wurde das Attribut TOPICMEMBER/NICKNAME nicht definiert."
            );
            generator.name(ChatUser.NICKNAME).value(tmp.getNickname());
            generator.endObject();
          }
        }
        generator.endArray(); // Array
        generator.endObject();
      }
      generator.endObject();
      generator.flush();
      generator.close();
      return stringwriter.toString();
    }
    return EMPTY;
  }



  String encodeCHANGETOPICMEMBER(MESSAGE message) throws EncodeException, IOException {
    if (!(message instanceof CHANGETOPICMEMBER)) return EMPTY;
    CHANGETOPICMEMBER member = (CHANGETOPICMEMBER) message;
    if (Command.CHANGETOPICMEMBER.equals(member.getCommand())) {
      StringWriter stringwriter = new StringWriter();

//      Gson gson = new GsonBuilder().
      Gson gson = new GsonBuilder().disableHtmlEscaping().create();

      JsonWriter generator = gson.newJsonWriter(stringwriter);
      generator.beginObject();
      generator.name(Protocol.HEADER).value(message.getHeader().toString());
      generator.name(Command.getCommandKey()).value(message.getCommand().name());
      if (REQUEST == message.getHeader()) {
        generator.name(member.getDataset());
        generator.beginObject();
        try {
          generator.name(MEMBER.USERID).value(member.getUserid());
        }
        catch (NullPointerException e) {
          throw new EncodeException(member, "Im Datensatz wurde das Attribut USERID nicht definiert.");
        }
        try {
          generator.name(MEMBER.ROOM).value(member.getRoom());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde das Attribut ROOM nicht definiert.");
        }
        try {
          generator.name(MEMBER.ROOMTYPE).value(member.getRoomtype().toString());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde das Attribut ROOMTYPE nicht definiert.");
        }
        generator.name(MEMBER.TOPICMEMBER);
        generator.beginArray();
        ChatUser[] chatuser = member.getChatUser();
        if (chatuser == null)
          throw new EncodeException(message, "Im Datensatz wurden keine TOPICMEMBER definiert.");
        int count = chatuser.length;
        for (int index = 0; index < count; index++) {
          generator.beginObject();
          if (chatuser[index].getUserid() != null)
            generator.name(ChatUser.USERID).value(chatuser[index].getUserid());
          if (
            chatuser[index].getNickname() == null
          ) throw new EncodeException(
              message, "Im Datensatz wurde das Attribut TOPICMEMBER.NICKNAME nicht definiert."
          );
          generator.name(ChatUser.NICKNAME).value(chatuser[index].getNickname());
          generator.endObject();
        }
        generator.endArray();
        generator.endObject();
      }
      else if (RESPONSE.equals(message.getHeader())) {
        generator.name(message.getDataset());
        generator.beginObject();
        try {
          generator.name(MEMBER.USERID).value(member.getUserid());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde das Attribut USERID nicht definiert.");
        }
        try {
          generator.name(MEMBER.ROOM).value(member.getRoom());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde das Attribut ROOM nicht definiert.");
        }
        try {
          generator.name(MEMBER.ROOMTYPE).value(member.getRoomtype().toString());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde das Attribut ROOMTYPE nicht definiert.");
        }
        // es kommen die User
        generator.name(MEMBER.TOPICMEMBER);
        generator.beginArray();
        ChatUser[] rooms = member.getChatUser();
        if (rooms != null) {
          for (ChatUser tmp : rooms) {
            generator.beginObject();
            if (
              tmp.getUserid() == null
            ) throw new EncodeException(
                message, "Im Datensatz wurde das Attribut TOPICMEMBER/USERID nicht definiert."
            );
            generator.name(ChatUser.USERID).value(tmp.getUserid());
            if (
              tmp.getNickname() == null
            ) throw new EncodeException(
                message, "Im Datensatz wurde das Attribut TOPICMEMBER/NICKNAME nicht definiert."
            );
            generator.name(ChatUser.NICKNAME).value(tmp.getNickname());
            generator.endObject();
          }
        }
        generator.endArray(); // Array
        generator.endObject();

      }
      else if (ERROR == message.getHeader()) {
        generator.name(message.getDataset()).beginObject();
        try {
          generator.name(MEMBER.USERID).value(member.getUserid());
          generator.name(MEMBER.ROOM).value(member.getRoom());
          generator.name(MESSAGE.ERRORMESSAGE).value(member.getErrorMessage());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde ein Attribut nicht definiert.");
        }
        generator.endObject();
      }
      generator.endObject();
      generator.flush();
      generator.close();
      return stringwriter.toString();
    }
    return EMPTY;
  }



  public String encodeCHATMESSAGE(MESSAGE message) throws EncodeException, IOException {
    if (!(message instanceof CHATMESSAGE)) return EMPTY;
    CHATMESSAGE chatmessage = (CHATMESSAGE) message;
    if (Command.CHATMESSAGE.equals(chatmessage.getCommand())) {
      StringWriter stringwriter = new StringWriter();
      Gson gson = gsonbuilder.create();
      JsonWriter generator = gson.newJsonWriter(stringwriter);
      generator.beginObject();
      generator.name(Protocol.HEADER).value(message.getHeader().toString());
      generator.name(Command.getCommandKey()).value(message.getCommand().name());
      if (REQUEST == message.getHeader()) {
        generator.name(message.getDataset()).beginObject();
        try {
          generator.name(CHATMESSAGE.ROOM).value(chatmessage.getRoom());
          if (chatmessage.getMessage() == null)
            throw new EncodeException(message, "Im Datensatz ist keine MESSAGE enthalten.");
          generator.name(CHATMESSAGE.MESSAGE).value(chatmessage.getMessage());

          if (chatmessage.getAttachment() != null) {
            generator.name(Attachment.ATTACHMENT).value(chatmessage.getAttachment());
            if (chatmessage.getFilename() == null)
              throw new EncodeException(message, "Das Attribut FILENAME wurde nicht definiert.");
            generator.name(Attachment.FILENAME).value(chatmessage.getFilename());
            if (chatmessage.getFilesize() == null)
              throw new EncodeException(message, "Das Attribut FILESIZE wurde nicht definiert.");
            generator.name(Attachment.FILESIZE).value(chatmessage.getFilesize());
          }
          generator.name(CHATMESSAGE.CHATUSER).beginObject();
          ChatUser chatuser = chatmessage.getChatUser();
          if (chatuser == null)
            throw new EncodeException(message, "Im Datensatz wurde die Userbeschreibung nicht definiert.");
          generator.name(ChatUser.BACKGROUND_COLOR).value(chatuser.getBackgroundColor());
          generator.name(ChatUser.FOREGROUND_COLOR).value(chatuser.getForegroundColor());
          generator.name(ChatUser.USERID).value(chatuser.getUserid());
          generator.name(ChatUser.NICKNAME).value(chatuser.getNickname());
          generator.endObject();
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde ein Attribut nicht definiert.");
        }
        generator.endObject();
      }
      else if (RESPONSE.equals(chatmessage.getHeader())) {
        generator.name(message.getDataset()).beginObject();
        try {
          generator.name(CHATMESSAGE.ROOM).value(chatmessage.getRoom());
          generator.name(CHATMESSAGE.DATETIME).value(chatmessage.getDatetime());
          if (chatmessage.getMessage() == null)
            throw new EncodeException(message, "Im Datensatz ist keine MESSAGE enthalten.");
          generator.name(CHATMESSAGE.MESSAGE).value(chatmessage.getMessage());
          if (chatmessage.getAttachment() != null) {
            generator.name(Attachment.ATTACHMENT).value(chatmessage.getAttachment());
            if (chatmessage.getFilename() == null)
              throw new EncodeException(message, "Das Attribut FILENAME wurde nicht definiert.");
            generator.name(Attachment.FILENAME).value(chatmessage.getFilename());
            if (chatmessage.getFilesize() == null)
              throw new EncodeException(message, "Das Attribut FILESIZE wurde nicht definiert.");
            generator.name(Attachment.FILESIZE).value(chatmessage.getFilesize());
          }
          generator.name(CHATMESSAGE.CHATUSER).beginObject();
          ChatUser chatuser = chatmessage.getChatUser();
          if (chatuser == null)
            throw new EncodeException(message, "Im Datensatz wurde die Userbeschreibung nicht definiert.");
          generator.name(ChatUser.BACKGROUND_COLOR).value(chatuser.getBackgroundColor());
          generator.name(ChatUser.FOREGROUND_COLOR).value(chatuser.getForegroundColor());
          generator.name(ChatUser.USERID).value(chatuser.getUserid());
          generator.name(ChatUser.NICKNAME).value(chatuser.getNickname());
          generator.endObject();
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde ein Attribut nicht definiert.");
        }
        generator.endObject();
      }
      generator.endObject();
      generator.flush();
      generator.close();
      return stringwriter.toString();
    }
    return EMPTY;
  }



  public String encodeCHATONLINELIST(MESSAGE message) throws IOException {
    if (!(message instanceof CHATONLINELIST)) return EMPTY;
    CHATONLINELIST chatonlinelist = (CHATONLINELIST) message;
    if (Command.CHATONLINELIST.equals(chatonlinelist.getCommand())) {
      StringWriter stringwriter = new StringWriter();
      Gson gson = gsonbuilder.create();
      JsonWriter generator = gson.newJsonWriter(stringwriter);
      generator.beginObject();
      generator.name(Protocol.HEADER).value(message.getHeader().toString());
      generator.name(Command.getCommandKey()).value(message.getCommand().name());
      if (REQUEST == message.getHeader()) {
        generator.name(message.getDataset()).beginObject();
        generator.name(CHATONLINELIST.CHATONLINELIST).beginArray();
        ChatOnline[] online = chatonlinelist.getChatOnline();
        if (online != null) {
          for (ChatOnline element : online) {
            generator.beginObject();
            generator.name(ChatOnline.ROOM).value(element.getRoom());
            generator.name(ChatOnline.ONLINE).value(element.getOnline());
            generator.endObject();
          }
        }
        generator.endArray();
        generator.endObject();
      }
      generator.endObject();
      generator.flush();
      generator.close();
      return stringwriter.toString();
    }
    return EMPTY;
  }



  public String encodeCHATUSERLIST(MESSAGE message) throws EncodeException, IOException {
    if (!(message instanceof CHATUSERLIST)) return EMPTY;
    CHATUSERLIST userlist = (CHATUSERLIST) message;
    if (Command.CHATUSERLIST.equals(userlist.getCommand())) {
      StringWriter stringwriter = new StringWriter();
      Gson gson = gsonbuilder.create();
      JsonWriter generator = gson.newJsonWriter(stringwriter);
      generator.beginObject();
      generator.name(Protocol.HEADER).value(message.getHeader().toString());
      generator.name(Command.getCommandKey()).value(message.getCommand().name());
      if (RESPONSE.equals(message.getHeader())) {
        generator.name(message.getDataset()).beginObject();
        generator.name(CHATUSERLIST.ROOM).value(userlist.getRoom());
        generator.name(CHATUSERLIST.CHATUSERLIST).beginArray();
        ChatUser[] chatuser = userlist.getChatUser();
        if (chatuser != null) {
          for (ChatUser element : chatuser) {
            try {
              generator.beginObject();
              if (element.getUserid() != null) generator.name(ChatUser.USERID).value(element.getUserid()); // optinal
              generator.name(ChatUser.FOREGROUND_COLOR).value(element.getForegroundColor());
              generator.name(ChatUser.BACKGROUND_COLOR).value(element.getBackgroundColor());
              generator.name(ChatUser.NICKNAME).value(element.getNickname());
              generator.endObject();
            }
            catch (NullPointerException e) {
              throw new EncodeException(message, "Im Datensatz wurde ein Attribut nicht definiert.");
            }
          }
        }
        generator.endArray();
        generator.endObject();
      }
      else if (REQUEST == message.getHeader()) {
        try {
          generator.name(message.getDataset()).beginObject();
          generator.name(CHATUSERLIST.ROOM).value(userlist.getRoom());
          generator.endObject();
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde ein Attribut nicht definiert.");
        }
      }
      else if (ERROR == message.getHeader()) {
        if (message.getDataset() != null) {
          generator.name(message.getDataset()).beginObject();
          if (message.getErrorCode() != 0) generator.name(MESSAGE.ERRORCODE).value(message.getErrorCode());
          if (message.getErrorMessage() != null)
            generator.name(MESSAGE.ERRORMESSAGE).value(message.getErrorMessage());
          generator.endObject();
        }
      }
      generator.endObject();
      generator.flush();
      generator.close();
      return stringwriter.toString();
    }
    return EMPTY;
  }



  String encodeCONFERENCE(MESSAGE message) throws IOException, EncodeException {
    if (!(message instanceof CONFERENCE)) return EMPTY;
    CONFERENCE conference = (CONFERENCE) message;
    if (!Command.CONFERENCE.equals(message.getCommand())) return EMPTY;
    StringWriter stringwriter = new StringWriter();
    Gson gson = gsonbuilder.create();
    JsonWriter generator = gson.newJsonWriter(stringwriter);
    generator.beginObject();
    generator.name(Protocol.HEADER).value(message.getHeader().toString());
    generator.name(Command.getCommandKey()).value(message.getCommand().name());
    if (conference.getDataset() == null)
      throw new EncodeException(conference, "DATASET wurde nicht definiert.");
    generator.name(conference.getDataset());
    switch(message.getHeader()) {
      case CONFIRM:
        break;
      case ERROR:
        break;
      case REQUEST:
        generator.beginObject();
        if (conference.getKonferenzname() == null)
          throw new EncodeException(conference, "Das Attribut KONFERENZNAME wurde nicht definiert.");
        generator.name(CONFERENCE.KONFERENZNAME).value(conference.getKonferenzname());

        if (conference.getOrganisatorUid() == null)
          throw new EncodeException(conference, "Das Attribut ORGANISATOR_UID wurde nicht definiert.");
        generator.name(CONFERENCE.ORGANISATOR_UID).value(conference.getOrganisatorUid());

        if (conference.getUserid() == null)
          throw new EncodeException(conference, "Das Attribut USERID wurde nicht definiert.");
        generator.name(CONFERENCE.USERID).value(conference.getUserid());
        if (conference.getSession() == null)
          throw new EncodeException(conference, "Das Attribut SESSION wurde nicht definiert.");
        generator.name(CONFERENCE.SESSION).value(conference.getSession());

        generator.name(CONFERENCE.OFFHOOK).value(conference.isOffhook());

        generator.endObject();
        break;
      case RESPONSE:
        generator.beginObject();
        if (conference.getKonferenzname() == null)
          throw new EncodeException(conference, "Das Attribut KONFERENZNAME wurde nicht definiert.");
        generator.name(CONFERENCE.KONFERENZNAME).value(conference.getKonferenzname());
        if (conference.getOrganisatorUid() == null)
          throw new EncodeException(conference, "Das Attribut ORGANISATOR_UID wurde nicht definiert.");
        generator.name(CONFERENCE.ORGANISATOR_UID).value(conference.getOrganisatorUid());
        if (conference.getUserid() == null)
          throw new EncodeException(conference, "Das Attribut USERID wurde nicht definiert.");
        generator.name(CONFERENCE.USERID).value(conference.getUserid());
        if (conference.isOffhook() == null)
          throw new EncodeException(conference, "Das Attribut OFFHOOK wurde nicht definiert.");
        generator.name(CONFERENCE.OFFHOOK).value(conference.isOffhook());
        generator.name(CONFERENCE.KONFERENZRAUMUSER).beginArray();
        for (KonferenzraumUser konferenzraumuser : conference.getKonferenzraumUser()) {
          generator.beginObject();
          generator.name(KonferenzraumUser.ANWESEND).value(konferenzraumuser.isAnwesend());
          generator.name(KonferenzraumUser.SESSION).value(konferenzraumuser.getSession());
          generator.name(KonferenzraumUser.MUTE).value(konferenzraumuser.isMute());
          generator.name(KonferenzraumUser.CHATUSER).beginObject();
          generator.name(ChatUser.USERID).value(konferenzraumuser.getUserid());
          generator.name(ChatUser.NICKNAME).value(konferenzraumuser.getNickname());
          generator.name(ChatUser.FOREGROUND_COLOR).value(konferenzraumuser.getForegroundColor());
          generator.name(ChatUser.BACKGROUND_COLOR).value(konferenzraumuser.getBackgroundColor());
          generator.endObject();
          generator.endObject();
        }
        generator.endArray();
        generator.endObject();
        break;
      default:
        break;
    }
    generator.endObject();
    return stringwriter.toString();
  }



  String encodeCONFERENCEAUDIO(MESSAGE message) throws IOException, EncodeException {
    if (!(message instanceof CONFERENCEAUDIO)) return EMPTY;
    CONFERENCEAUDIO conferenceaudio = (CONFERENCEAUDIO) message;
    if (!Command.CONFERENCEAUDIO.equals(message.getCommand())) return EMPTY;

    StringWriter stringwriter = new StringWriter();
    Gson gson = gsonbuilder.create();
    JsonWriter generator = gson.newJsonWriter(stringwriter);

    generator.beginObject();
    generator.name(Protocol.HEADER).value(message.getHeader().toString());
    generator.name(Command.getCommandKey()).value(message.getCommand().name());

    if (conferenceaudio.getDataset() == null)
      throw new EncodeException(conferenceaudio, "DATASET wurde nicht definiert.");
    generator.name(conferenceaudio.getDataset());

    switch(message.getHeader()) {
      case CONFIRM:
        break;
      case ERROR:
        break;
      case REQUEST:
        generator.beginObject();

        generator.name(CONFERENCEAUDIO.USERID).value(conferenceaudio.getUserid());

        generator.name(CONFERENCEAUDIO.TARGET);
        generator.beginArray();
        if (conferenceaudio.getReceiverSessions() == null)
          throw new EncodeException(conferenceaudio, "Das Attribut TARGET wurde nicht definiert.");
        for (String session : conferenceaudio.getReceiverSessions()) {
          generator.value(session);
        }
        generator.endArray();

//        if (conferenceaudio.getContent() == null)
//          throw new EncodeException(conferenceaudio, "Das Attribut CONTENT wurde nicht definiert.");

        // WICHTIG: FLAC-Bytes als Base64 schreiben
        if (conferenceaudio.getContent() != null) {
          generator.name(CONFERENCEAUDIO.CONTENT);
          String base64Content = Base64.getEncoder().encodeToString(conferenceaudio.getContent());
          generator.value(base64Content);
        }

        generator.name(CONFERENCEAUDIO.STREAM_ID);
        generator.value(conferenceaudio.getStreamId());

        generator.endObject();
        break;
      case RESPONSE:
        break;
      default:
        break;
    }

    generator.endObject();
    generator.close(); // schließt JsonWriter
    return stringwriter.toString();
  }



  public String encodeCREATETEMPROOM(MESSAGE message) throws EncodeException, IOException {
    if (!(message instanceof CREATETEMPROOM)) return EMPTY;
    CREATETEMPROOM temproom = (CREATETEMPROOM) message;
    if (Command.CREATETEMPROOM.equals(temproom.getCommand())) {
      StringWriter stringwriter = new StringWriter();
      Gson gson = gsonbuilder.create();
      JsonWriter generator = gson.newJsonWriter(stringwriter);
      generator.beginObject();
      generator.name(Protocol.HEADER).value(message.getHeader().toString());
      generator.name(Command.getCommandKey()).value(message.getCommand().name());
      if (REQUEST == message.getHeader()) {
        generator.name(message.getDataset()).beginObject();
        if (temproom.getUser() == null)
          throw new EncodeException(message, "Im Datensatz wurde das Attribut USER nicht definiert.");
        generator.name(CREATETEMPROOM.USER).value(temproom.getUser());
        if (temproom.getNickname() == null)
          throw new EncodeException(message, "Im Datensatz wurde das Attribut NICKNAME nicht definiert.");
        generator.name(CREATETEMPROOM.NICKNAME).value(temproom.getNickname());
        if (temproom.getRoom() == null)
          throw new EncodeException(message, "Im Datensatz wurde das Attribut ROOM nicht definiert.");
        generator.name(CREATETEMPROOM.ROOM).value(temproom.getRoom());
        if (temproom.getRoomtype() == null)
          throw new EncodeException(message, "Im Datensatz wurde das Attribut ROOMTYPE nicht definiert.");
        generator.name(CREATETEMPROOM.ROOMTYPE).value(temproom.getRoomtype().name());
        generator.name(CREATETEMPROOM.ENTRY).value(temproom.getEntry().name());
        generator.name(CREATETEMPROOM.LIFETIME).value(temproom.getLifetime().name());
        if (temproom.getPassword() != null) {
          generator.name(CREATETEMPROOM.PASSWORD).value(temproom.getPassword());
        }
        generator.endObject();
      }
      else if (RESPONSE.equals(message.getHeader())) {
        generator.name(message.getDataset()).beginObject();
        if (temproom.getUser() == null)
          throw new EncodeException(message, "Im Datensatz wurde das Attribut USER nicht definiert.");
        generator.name(CREATETEMPROOM.USER).value(temproom.getUser());
        if (temproom.getRoom() == null)
          throw new EncodeException(message, "Im Datensatz wurde das Attribut ROOM nicht definiert.");
        generator.name(CREATETEMPROOM.ROOM).value(temproom.getRoom());
        if (temproom.getRoomtype() == null)
          throw new EncodeException(message, "Im Datensatz wurde das Attribut ROOMTYPE nicht definiert.");
        generator.name(CREATETEMPROOM.ROOMTYPE).value(temproom.getRoomtype().name());
        generator.name(CREATETEMPROOM.ENTRY).value(temproom.getEntry().name());
        generator.name(CREATETEMPROOM.LIFETIME).value(temproom.getLifetime().name());
        if (temproom.getPassword() != null) {
          generator.name(CREATETEMPROOM.PASSWORD).value(temproom.getPassword());
        }
        if (temproom.getText() != null) {
          // optional
          generator.name(MESSAGE.TEXT).value(temproom.getText());
        }
        if (temproom.getMultilingualkey() != null) {
          // optional
          generator.name(MESSAGE.MULTILINGUALKEY).value(temproom.getMultilingualkey().value());
        }
        generator.endObject();
      }
      else if (ERROR == message.getHeader()) {
        generator.name(message.getDataset()).beginObject();
        if (temproom.getUser() == null)
          throw new EncodeException(message, "Im Datensatz wurde das Attribut USER nicht definiert.");
        generator.name(CREATETEMPROOM.USER).value(temproom.getUser());
        if (temproom.getRoom() == null)
          throw new EncodeException(message, "Im Datensatz wurde das Attribut ROOM nicht definiert.");
        generator.name(CREATETEMPROOM.ROOM).value(temproom.getRoom());
        if (temproom.getRoomtype() != null)
          generator.name(CREATETEMPROOM.ROOMTYPE).value(temproom.getRoomtype().name());
        if (temproom.getEntry() != null)
          generator.name(CREATETEMPROOM.ENTRY).value(temproom.getEntry().name());
        if (temproom.getLifetime() != null)
          generator.name(CREATETEMPROOM.LIFETIME).value(temproom.getLifetime().name());
        if (temproom.getErrorCode() != null) generator.name(MESSAGE.ERRORCODE).value(temproom.getErrorCode());
        if (temproom.getErrorMessage() != null)
          generator.name(MESSAGE.ERRORMESSAGE).value(temproom.getErrorMessage());
        if (message.getText() != null) generator.name(MESSAGE.TEXT).value(message.getText());
        if (temproom.getMultilingualkey() != null) {
          // optional
          generator.name(MESSAGE.MULTILINGUALKEY).value(temproom.getMultilingualkey().value());
        }

        generator.endObject();
      }
      generator.endObject();
      generator.flush();
      generator.close();
      return stringwriter.toString();
    }
    return EMPTY;
  }



  String encodeDELETEROOM(MESSAGE message) throws EncodeException, IOException {
    if (!(message instanceof DELETEROOM)) return EMPTY;
    DELETEROOM delete = (DELETEROOM) message;
    if (Command.DELETEROOM.equals(message.getCommand())) {
      StringWriter stringwriter = new StringWriter();
      Gson gson = gsonbuilder.create();
      JsonWriter generator = gson.newJsonWriter(stringwriter);
      generator.beginObject();
      generator.name(Protocol.HEADER).value(message.getHeader().toString());
      generator.name(Command.getCommandKey()).value(message.getCommand().name());
      if (REQUEST == message.getHeader()) {
        generator.name(delete.getDataset());
        generator.beginObject();
        if (delete.getUserid() == null)
          throw new EncodeException(delete, "Im Datensatz wurde das Attribut USERID nicht definiert.");
        generator.name(DELETEROOM.USER).value(delete.getUserid());
        if (delete.getRoom() == null)
          throw new EncodeException(delete, "Im Datensatz wurde das Attribut ROOM nicht definiert.");
        generator.name(DELETEROOM.ROOM).value(delete.getRoom());
        generator.endObject();
      }
      else if (RESPONSE.equals(message.getHeader())) {
        generator.name(delete.getDataset());
        generator.beginObject();
        if (delete.getUserid() == null)
          throw new EncodeException(delete, "Im Datensatz wurde das Attribut USERID nicht definiert.");
        generator.name(DELETEROOM.USER).value(delete.getUserid());
        if (delete.getRoom() == null)
          throw new EncodeException(delete, "Im Datensatz wurde das Attribut ROOM nicht definiert.");
        generator.name(DELETEROOM.ROOM).value(delete.getRoom());
        if (delete.getText() != null) generator.name(MESSAGE.TEXT).value(delete.getText());
        generator.endObject();
      }
      else if (ERROR == message.getHeader()) {
        generator.name(delete.getDataset());
        generator.beginObject();
        if (delete.getUserid() == null)
          throw new EncodeException(delete, "Im Datensatz wurde das Attribut USERID nicht definiert.");
        generator.name(DELETEROOM.USER).value(delete.getUserid());
        if (delete.getRoom() == null)
          throw new EncodeException(delete, "Im Datensatz wurde das Attribut ROOM nicht definiert.");
        generator.name(DELETEROOM.ROOM).value(delete.getRoom());
        if (delete.getText() == null)
          throw new EncodeException(delete, "Im Datensatz wurde das Attribut TEXT nicht definiert.");
        generator.name(MESSAGE.TEXT).value(delete.getText());
        generator.endObject();
      }
      generator.endObject();
      generator.flush();
      generator.close();
      return stringwriter.toString();
    }
    return EMPTY;
  }



  public String encodeDELETEUPLOADFILES(MESSAGE message) throws IOException {
    if (!(message instanceof DELETEUPLOADFILES)) return EMPTY;
    DELETEUPLOADFILES deleteUploadfiles = (DELETEUPLOADFILES) message;
    if (Command.DELETEUPLOADFILES.equals(deleteUploadfiles.getCommand())) {
      StringWriter stringwriter = new StringWriter();
      Gson gson = gsonbuilder.create();
      JsonWriter generator = gson.newJsonWriter(stringwriter);
      generator.beginObject();
      generator.name(Protocol.HEADER).value(message.getHeader().toString());
      generator.name(Command.getCommandKey()).value(message.getCommand().name());
      if (REQUEST == message.getHeader()) {
        generator.name(message.getDataset()).beginObject();
        generator.name(DELETEUPLOADFILES.USERID).value(deleteUploadfiles.getUserid());
        generator.name(DELETEUPLOADFILES.IP).value(deleteUploadfiles.getIp());
        generator.endObject();
      }
      else if (CONFIRM == message.getHeader()) {

      }
      else if (ERROR == message.getHeader()) {
        generator.name(message.getDataset()).beginObject();
        generator.name(MESSAGE.TEXT).value(deleteUploadfiles.getText());
        generator.name(MESSAGE.ERRORCODE).value(deleteUploadfiles.getErrorCode());
        generator.name(MESSAGE.ERRORMESSAGE).value(deleteUploadfiles.getErrorMessage());
        generator.endObject();
      }
      generator.endObject();
      generator.flush();
      generator.close();
      return stringwriter.toString();
    }
    return EMPTY;
  }



  public String encodeDELETEUSER(MESSAGE message) throws IOException {
    if (!(message instanceof DELETEUSER)) return EMPTY;
    if (message.getCommand().equals(Command.DELETE_USER)) {
      if (message.getHeader().equals(REQUEST)) {
        StringWriter stringwriter = new StringWriter();
        Gson gson = gsonbuilder.create();
        JsonWriter generator = gson.newJsonWriter(stringwriter);
        generator.beginObject();
        generator.name(Protocol.HEADER).value(message.getHeader().toString());
        generator.name(Command.getCommandKey()).value(message.getCommand().name());
        DELETEUSER deleteuser = (DELETEUSER) message;
        generator.name(message.getDataset()).beginObject();
        generator.name(USER.USERID).value(deleteuser.getUserid());
        generator.endObject();
        generator.endObject();
        generator.flush();
        generator.close();
        return stringwriter.toString();
      }
      else if (message.getHeader() == ERROR) {
        StringWriter stringwriter = new StringWriter();
        Gson gson = gsonbuilder.create();
        JsonWriter generator = gson.newJsonWriter(stringwriter);
        generator.beginObject();
        generator.name(Protocol.HEADER).value(message.getHeader().toString());
        generator.name(Command.getCommandKey()).value(message.getCommand().name());
        generator.endObject();
        generator.flush();
        generator.close();
        return stringwriter.toString();
      }
      else if (message.getHeader() == CONFIRM) {
        StringWriter stringwriter = new StringWriter();
        Gson gson = gsonbuilder.create();
        JsonWriter generator = gson.newJsonWriter(stringwriter);
        generator.beginObject();
        generator.name(Protocol.HEADER).value(message.getHeader().toString());
        generator.name(Command.getCommandKey()).value(message.getCommand().name());
        generator.endObject();
        generator.flush();
        generator.close();
        return stringwriter.toString();
      }
    }
    return EMPTY;
  }



  public String encodeDIAL(MESSAGE message) throws EncodeException, IOException {
    if (!(message instanceof DIAL)) return EMPTY;
    DIAL dial = (DIAL) message;
    if (message.getCommand().equals(Command.DIAL)) {
      StringWriter stringwriter = new StringWriter();
      Gson gson = gsonbuilder.create();
      JsonWriter generator = gson.newJsonWriter(stringwriter);
      generator.beginObject();
      generator.name(Protocol.HEADER).value(message.getHeader().toString());
      generator.name(Command.getCommandKey()).value(message.getCommand().name());
      if (message.getHeader().equals(REQUEST)) {
        if (dial.getCallerUserid() == null)
          throw new EncodeException(message, "Das Attribut  CALLER_USERID wurde nicht definiert");
        if (dial.getCallerVoice() == null)
          throw new EncodeException(message, "Das Attribut CALLER_VOICE wurde nicht definiert");
        if (dial.getCallerNickname() == null)
          throw new EncodeException(message, "Das Attribut CALLER_NICKNAME wurde nicht definiert");
        if (dial.getReceiverNickname() == null)
          throw new EncodeException(message, "Das Attribut RECEIVER_NICKNAME wurde nicht definiert");

        generator.name(dial.getDataset()).beginObject();
        generator.name(DIAL.CALLER_USERID).value(dial.getCallerUserid());
        generator.name(DIAL.CALLER_NICKNAME).value(dial.getCallerNickname());
        generator.name(DIAL.CALLER_VOICE).value(dial.getCallerVoice());
        generator.name(DIAL.RECEIVER_IP).value(dial.getReceiverIp());
        generator.name(DIAL.RECEIVER_PORT).value(dial.getReceiverPort());
        generator.name(DIAL.RECEIVER_NICKNAME).value(dial.getReceiverNickname());
        generator.endObject(); // DATASET End
      }
      else if (message.getHeader() == RESPONSE) {
        if (dial.getCallerNickname() == null)
          throw new EncodeException(message, "Das Attribut CALLER_NICKNAME wurde nicht definiert");
        if (dial.getCallerVoice() == null)
          throw new EncodeException(message, "Das Attribut CALLER_VOICE wurde nicht definiert");
        if (dial.getReceiverUserid() == null)
          throw new EncodeException(message, "Das Attribut RECEIVER_USERID wurde nicht definiert");
        if (dial.getReceiverNickname() == null)
          throw new EncodeException(message, "Das Attribut RECEIVER_NICKNAME wurde nicht definiert");
        if (dial.getCallerSession() == null)
          throw new EncodeException(message, "Das Attribut CALLER_SESSION wurde nicht definiert");
        if (dial.getReceiverSession() == null)
          throw new EncodeException(message, "Das Attribut RECEIVER_SESSION wurde nicht definiert");

        generator.name(dial.getDataset()).beginObject();
        generator.name(DIAL.CALLER_USERID).value(dial.getCallerUserid());
        generator.name(DIAL.CALLER_NICKNAME).value(dial.getCallerNickname());
        generator.name(DIAL.RECEIVER_IP).value(dial.getReceiverIp());
        generator.name(DIAL.RECEIVER_PORT).value(dial.getReceiverPort());
        generator.name(DIAL.CALLER_VOICE).value(dial.getCallerVoice());
        generator.name(DIAL.RECEIVER_NICKNAME).value(dial.getReceiverNickname());
        generator.name(DIAL.RECEIVER_USERID).value(dial.getReceiverUserid());
        generator.name(DIAL.CALLER_SESSION).value(dial.getCallerSession());
        generator.name(DIAL.RECEIVER_SESSION).value(dial.getReceiverSession());
        generator.endObject(); // DATASET End
      }
      else if (message.getHeader() == ERROR) {
        generator.name(dial.getDataset()).beginObject();
        generator.name(DIAL.CALLER_USERID).value(dial.getCallerUserid());
        generator.name(DIAL.CALLER_NICKNAME).value(dial.getCallerNickname());
        generator.name(DIAL.CALLER_VOICE).value(dial.getCallerVoice());
        generator.name(DIAL.RECEIVER_IP).value(dial.getReceiverIp());
        generator.name(DIAL.RECEIVER_PORT).value(dial.getReceiverPort());
        if (dial.getReceiverNickname() == null)
          throw new EncodeException(message, "Das Attribut RECEIVER_NICKNAME ist null");
        generator.name(DIAL.RECEIVER_NICKNAME).value(dial.getReceiverNickname());
        if (dial.getErrorMessage() != null)
          generator.name(MESSAGE.ERRORMESSAGE).value(dial.getErrorMessage());
        if (dial.getMultilingualkey() != null)
          generator.name(MESSAGE.MULTILINGUALKEY).value(dial.getMultilingualkey().value());
        generator.endObject(); // DATASET End
      }
      generator.endObject();
      generator.flush();
      generator.close();
      return stringwriter.toString();
    }

    return EMPTY;
  }



  public String encodeDOWNLOAD(MESSAGE message) throws EncodeException, IOException {
    if (!(message instanceof DOWNLOAD)) return EMPTY;
    DOWNLOAD download = (DOWNLOAD) message;
    if (Command.DOWNLOAD.equals(download.getCommand())) {
      StringWriter stringwriter = new StringWriter();
      Gson gson = gsonbuilder.create();
      JsonWriter generator = gson.newJsonWriter(stringwriter);
      generator.beginObject();
      generator.name(Protocol.HEADER).value(message.getHeader().toString());
      generator.name(Command.getCommandKey()).value(message.getCommand().name());
      if (REQUEST == message.getHeader()) {
        generator.name(message.getDataset()).beginObject();
        try {
          generator.name(DOWNLOAD.USERID).value(download.getUserid());
          generator.name(DOWNLOAD.SLOT).value(download.getSlot());
          generator.name(DOWNLOAD.FILENAME).value(download.getFilename());
          generator.name(DOWNLOAD.INDEX).value(download.getIndex());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde ein Attribut nicht definiert.");
        }
        generator.endObject();
      }
      else if (RESPONSE.equals(message.getHeader())) {
        generator.name(message.getDataset()).beginObject();
        try {
          generator.name(DOWNLOAD.USERID).value(download.getUserid());
          generator.name(DOWNLOAD.SLOT).value(download.getSlot());
          generator.name(DOWNLOAD.FILENAME).value(download.getFilename());
          generator.name(DOWNLOAD.IP).value(download.getIp());
          generator.name(DOWNLOAD.PATHFILE).value(download.getPathfile());
          generator.name(DOWNLOAD.INDEX).value(download.getIndex());
          if (download.getPort() <= 0)
            throw new EncodeException(message, "ung�ltiger Port " + String.valueOf(download.getPort()));
          if (download.getPort() > 65535)
            throw new EncodeException(message, "ung�ltiger Port " + String.valueOf(download.getPort()));
          generator.name(DOWNLOAD.PORT).value(download.getPort());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde ein Attribut nicht definiert.");
        }
        generator.endObject();
      }
      else if (ERROR == message.getHeader()) {
        generator.name(message.getDataset()).beginObject();
        try {
          generator.name(DOWNLOAD.USERID).value(download.getUserid());
          generator.name(DOWNLOAD.SLOT).value(download.getSlot());
          generator.name(DOWNLOAD.FILENAME).value(download.getFilename());
          generator.name(MESSAGE.TEXT).value(download.getText());
          generator.name(DOWNLOAD.INDEX).value(download.getIndex());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde ein Attribut nicht definiert.");
        }
        generator.endObject();
      }
      generator.endObject();
      generator.flush();
      generator.close();
      return stringwriter.toString();
    }
    return EMPTY;
  }



  public String encodeENTERROOM(MESSAGE message) throws EncodeException, IOException {
    if (!(message instanceof ENTERROOM)) return EMPTY;
    ENTERROOM enterroom = (ENTERROOM) message;
    if (enterroom.getCommand().equals(Command.ENTERROOM)) {
      StringWriter stringwriter = new StringWriter();
      Gson gson = gsonbuilder.create();
      JsonWriter generator = gson.newJsonWriter(stringwriter);
      generator.beginObject();
      generator.name(Protocol.HEADER).value(message.getHeader().toString());
      generator.name(Command.getCommandKey()).value(message.getCommand().name());
      if (REQUEST == message.getHeader()) {
        generator.name(enterroom.getDataset()).beginObject();
        if (enterroom.getUserid() == null)
          throw new EncodeException(message, "Das Attribut USERID wurde nicht definiert.");
        if (enterroom.getNickname() == null)
          throw new EncodeException(message, "Das Attribut NICKNAME wurde nicht definiert.");
        if (enterroom.getRoom() == null)
          throw new EncodeException(message, "Das Attribut ROOM wurde nicht definiert.");
        generator.name(ENTERROOM.USERID).value(enterroom.getUserid());
        generator.name(ENTERROOM.NICKNAME).value(enterroom.getNickname());
        generator.name(ENTERROOM.ROOM).value(enterroom.getRoom());
        generator.endObject();
      }
      else if (RESPONSE.equals(message.getHeader())) {
        generator.name(enterroom.getDataset()).beginObject();
        if (enterroom.getUserid() == null)
          throw new EncodeException(message, "Das Attribut USERID wurde nicht definiert.");
        if (enterroom.getNickname() == null)
          throw new EncodeException(message, "Das Attribut NICKNAME wurde nicht definiert.");
        if (enterroom.getRoom() == null)
          throw new EncodeException(message, "Das Attribut ROOM wurde nicht definiert.");
        if (enterroom.getRoomtype() == null)
          throw new EncodeException(message, "Das Attribut ROOMTYPE wurde nicht definiert.");
        generator.name(ENTERROOM.USERID).value(enterroom.getUserid());
        generator.name(ENTERROOM.NICKNAME).value(enterroom.getNickname());
        generator.name(ENTERROOM.ROOM).value(enterroom.getRoom());
        generator.name(ENTERROOM.ROOMTYPE).value(enterroom.getRoomtype().name());
        generator.endObject();
      }
      else if (ERROR == message.getHeader()) {
        generator.name(enterroom.getDataset()).beginObject();
        generator.name(MESSAGE.MULTILINGUALKEY).value(enterroom.getMultilingualkey().value());
        generator.name(MESSAGE.TEXT).value(enterroom.getText());
        generator.endObject();
      }
      generator.endObject();
      generator.flush();
      generator.close();
      return stringwriter.toString();
    }
    return EMPTY;
  }



  String encodeFILETYPES(MESSAGE message) throws IOException, EncodeException {
    if (!(message instanceof FILETYPES)) return EMPTY;
    FILETYPES filetypes = (FILETYPES) message;
    if (!Command.FILETYPES.equals(message.getCommand())) return EMPTY;
    StringWriter stringwriter = new StringWriter();
    Gson gson = gsonbuilder.create();
    JsonWriter generator = gson.newJsonWriter(stringwriter);
    generator.beginObject();
    generator.name(Protocol.HEADER).value(message.getHeader().toString());
    generator.name(Command.getCommandKey()).value(message.getCommand().name());
    if (REQUEST == message.getHeader()) {
      if (filetypes.getDataset() == null)
        throw new EncodeException(filetypes, "DATASET wurde nicht definiert.");
      generator.name(filetypes.getDataset());
      generator.beginObject();
      if (filetypes.getUserid() == null)
        throw new EncodeException(filetypes, "Das Attribut USERID wurde nicht definiert.");
      generator.name(FILETYPES.USERID).value(filetypes.getUserid());
      generator.endObject();
    }
    else if (RESPONSE.equals(message.getHeader())) {
      if (filetypes.getDataset() == null)
        throw new EncodeException(filetypes, "DATASET wurde nicht definiert.");
      generator.name(filetypes.getDataset());
      generator.beginObject();
      if (filetypes.getUserid() == null)
        throw new EncodeException(filetypes, "Das Attribut USERID wurde nicht definiert.");
      generator.name(FILETYPES.USERID).value(filetypes.getUserid());
      generator.name(FILETYPES.FILETYPES);
      generator.beginArray();
      BlacklistTypes[] blacklist = filetypes.getFiletypes();
      if (blacklist == null)
        throw new EncodeException(message, "Im Datensatz wurden keine FILETYPES definiert.");
      int count = blacklist.length;
      for (int index = 0; index < count; index++) {
        generator.beginObject();
        if (blacklist[index].getSuffix() == null)
          throw new EncodeException(message, "Im Datensatz wurde das Attribut SUFFIX nicht definiert.");
        generator.name(BlacklistTypes.SUFFIX).value(blacklist[index].getSuffix());
        if (blacklist[index].getDescription() == null)
          throw new EncodeException(message, "Im Datensatz wurde das Attribut DESCRIPTION nicht definiert.");
        generator.name(BlacklistTypes.DESCRIPTION).value(blacklist[index].getDescription());
        if (blacklist[index].getChecked() == null)
          throw new EncodeException(message, "Im Datensatz wurde das Attribut CHECKED nicht definiert.");
        generator.name(BlacklistTypes.CHECKED).value(blacklist[index].getChecked());
        generator.endObject();
      }
      generator.endArray();
      generator.endObject();
    }
    generator.endObject();
    generator.flush();
    generator.close();
    return stringwriter.toString();
  }



  private String encodeFTSDOWNLOAD(MESSAGE message) throws EncodeException, IOException {
    if (!(message instanceof FTSDOWNLOAD)) return EMPTY;
    FTSDOWNLOAD ftsdownload = (FTSDOWNLOAD) message;
    if (Command.FTSDOWNLOAD.equals(ftsdownload.getCommand())) {
      StringWriter stringwriter = new StringWriter();
      Gson gson = gsonbuilder.create();
      JsonWriter generator = gson.newJsonWriter(stringwriter);
      generator.beginObject();
      generator.name(Protocol.HEADER).value(message.getHeader().toString());
      generator.name(Command.getCommandKey()).value(message.getCommand().name());
      if (REQUEST == message.getHeader()) {
        generator.name(message.getDataset()).beginObject();
        try {
          generator.name(FTSDOWNLOAD.SLOT).value(ftsdownload.getSlot());
          generator.name(FTSDOWNLOAD.FILENAME).value(ftsdownload.getFilename());
          generator.name(FTSDOWNLOAD.PATH).value(ftsdownload.getPath());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde ein Attribut nicht definiert.");
        }
        generator.endObject();
      }
      else if (RESPONSE.equals(message.getHeader())) {
        generator.name(message.getDataset()).beginObject();
        try {
          generator.name(FTSDOWNLOAD.SLOT).value(ftsdownload.getSlot());
          generator.name(FTSDOWNLOAD.FILENAME).value(ftsdownload.getFilename());
          generator.name(FTSDOWNLOAD.PATH).value(ftsdownload.getPath());
          generator.name(FILETRANSFER.MAXDATABLOCKS).value(ftsdownload.getMaxDatablocks());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde ein Attribut nicht definiert.");
        }
        generator.endObject();
      }
      else if (ERROR == message.getHeader()) {
        generator.name(message.getDataset()).beginObject();
        try {
          generator.name(FTSDOWNLOAD.SLOT).value(ftsdownload.getSlot());
          generator.name(FTSDOWNLOAD.FILENAME).value(ftsdownload.getFilename());
          generator.name(FTSDOWNLOAD.PATH).value(ftsdownload.getPath());
          generator.name(MESSAGE.ERRORMESSAGE).value(ftsdownload.getErrorMessage());
          generator.name(MESSAGE.TEXT).value(ftsdownload.getText());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde ein Attribut nicht definiert.");
        }
        generator.endObject();
      }

      generator.endObject();
      generator.flush();
      generator.close();
      return stringwriter.toString();
    }
    return EMPTY;
  }



  public String encodeHISTORYMESSAGE(MESSAGE message) throws EncodeException, IOException {
    if (!(message instanceof HISTORYMESSAGE)) return EMPTY;
    HISTORYMESSAGE historymessage = (HISTORYMESSAGE) message;
    if (Command.HISTORYMESSAGE.equals(historymessage.getCommand())) {
      StringWriter stringwriter = new StringWriter();
      Gson gson = gsonbuilder.create();
      JsonWriter generator = gson.newJsonWriter(stringwriter);
      generator.beginObject();
      generator.name(Protocol.HEADER).value(message.getHeader().toString());
      generator.name(Command.getCommandKey()).value(message.getCommand().name());
      if (REQUEST == message.getHeader()) {
        generator.name(message.getDataset()).beginObject();
        generator.name(HISTORYMESSAGE.ROOM).value(historymessage.getRoom());
        generator.endObject();
      }
      else if (RESPONSE.equals(message.getHeader())) {
        try {
          generator.name(message.getDataset()).beginObject();
          generator.name(HISTORYMESSAGE.ROOM).value(historymessage.getRoom());
          generator.name(HISTORYMESSAGE.RECORD).beginArray();
          Attachment[] records = historymessage.getAttachment();
          if (records != null) {
            for (Attachment record : historymessage.getAttachment()) {
              generator.beginObject();
              generator.name(RecordInterface.ROOM).value(record.getRoom());
              generator.name(RecordInterface.DATETIME).value(record.getDatetime());
              generator.name(RecordInterface.MESSAGE).value(record.getMessage());

              if (record.getAttachment() != null) {
                generator.name(Attachment.ATTACHMENT).value(record.getAttachment());
                if (record.getFilename() == null)
                  throw new EncodeException(message, "Das Attribut FILENAME wurde nicht definiert.");
                generator.name(Attachment.FILENAME).value(record.getFilename());
                if (record.getFilesize() == null)
                  throw new EncodeException(message, "Das Attribut FILESIZE wurde nicht definiert.");
                generator.name(Attachment.FILESIZE).value(record.getFilesize());
              }

              ChatUser chatuser = record.getChatUser();
              generator.name(RecordInterface.CHATUSER).beginObject();
              generator.name(ChatUser.BACKGROUND_COLOR).value(chatuser.getBackgroundColor());
              generator.name(ChatUser.FOREGROUND_COLOR).value(chatuser.getForegroundColor());
              generator.name(ChatUser.NICKNAME).value(chatuser.getNickname());
              generator.name(ChatUser.USERID).value(chatuser.getUserid());
              generator.endObject();
              generator.endObject();
            }
          }
          generator.endArray(); // Array
          generator.endObject(); // Dataset
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde ein Attribut nicht definiert.");
        }
      }
      generator.endObject();
      generator.flush();
      generator.close();
      return stringwriter.toString();
    }
    return EMPTY;
  }



  String encodeIMAGE(MESSAGE message) throws IOException, EncodeException {
    if (!(message instanceof IMAGE)) return EMPTY;
    IMAGE image = (IMAGE) message;
    if (!Command.IMAGE.equals(message.getCommand())) return EMPTY;
    StringWriter stringwriter = new StringWriter();
    Gson gson = gsonbuilder.create();
    JsonWriter generator = gson.newJsonWriter(stringwriter);
    generator.beginObject();
    generator.name(Protocol.HEADER).value(message.getHeader().toString());
    generator.name(Command.getCommandKey()).value(message.getCommand().name());
    if (image.getDataset() == null) throw new EncodeException(image, "DATASET wurde nicht definiert.");
    generator.name(image.getDataset());
    switch(message.getHeader()) {
      case REQUEST:
        generator.beginObject();
        if (image.getSenderUid() == null) {
          log.fatal("Protokollfehler: IMAGE | REQUEST, das Attribut SENDERUID wurde nicht gesetzt.");
          throw new EncodeException(
              image, "Protokollfehler: IMAGE | REQUEST, das Attribut SENDERUID wurde nicht gesetzt."
          );
        }

        if (image.getSenderNickname() == null) {
          log.fatal("Protokollfehler: IMAGE | REQUEST, das Attribut SENDERNICKNAME wurde nicht gesetzt.");
          throw new EncodeException(
              image, "Protokollfehler: IMAGE | REQUEST, das Attribut SENDERNICKNAME wurde nicht gesetzt."
          );
        }

        if (image.getReceiverNickname() == null) {
          log.fatal("Protokollfehler: IMAGE | REQUEST, das Attribut RECEIVERNICKNAME wurde nicht gesetzt.");
          throw new EncodeException(
              image, "Protokollfehler: IMAGE | REQUEST, das Attribut RECEIVERNICKNAME wurde nicht gesetzt."
          );
        }

        generator.name(IMAGE.SENDERUID).value(image.getSenderUid());
        generator.name(IMAGE.SENDERNICKNAME).value(image.getSenderNickname());
        generator.name(IMAGE.RECEIVERNICKNAME).value(image.getReceiverNickname());

        generator.endObject();
        break;
      case RESPONSE:
        generator.beginObject();

        if (image.getSenderUid() == null) {
          log.fatal("Protokollfehler: IMAGE | RESPONSE, das Attribut SENDERUID wurde nicht gesetzt.");
          throw new EncodeException(
              image, "Protokollfehler: IMAGE | RESPONSE, das Attribut SENDERUID wurde nicht gesetzt."
          );
        }

        if (image.getSenderNickname() == null) {
          log.fatal("Protokollfehler: IMAGE | RESPONSE, das Attribut SENDERNICKNAME wurde nicht gesetzt.");
          throw new EncodeException(
              image, "Protokollfehler: IMAGE | RESPONSE, das Attribut SENDERNICKNAME wurde nicht gesetzt."
          );
        }

        if (image.getReceiverNickname() == null) {
          log.fatal("Protokollfehler: IMAGE | RESPONSE, das Attribut RECEIVERNICKNAME wurde nicht gesetzt.");
          throw new EncodeException(
              image, "Protokollfehler: IMAGE | RESPONSE, das Attribut RECEIVERNICKNAME wurde nicht gesetzt."
          );
        }

        if (image.getReceiverUid() == null) {
          log.fatal("Protokollfehler: IMAGE | RESPONSE, das Attribut RECEIVERUID wurde nicht gesetzt.");
          throw new EncodeException(
              image, "Protokollfehler: IMAGE | RESPONSE, das Attribut RECEIVERUID wurde nicht gesetzt."
          );
        }

        if (image.getReceiverSession() == null) {
          log.fatal("Protokollfehler: IMAGE | RESPONSE, das Attribut RECEIVER_SESSION wurde nicht gesetzt.");
          throw new EncodeException(
              image, "Protokollfehler: IMAGE | RESPONSE, das Attribut RECEIVER_SESSION wurde nicht gesetzt."
          );
        }

        generator.name(IMAGE.SENDERUID).value(image.getSenderUid());
        generator.name(IMAGE.SENDERNICKNAME).value(image.getSenderNickname());
        generator.name(IMAGE.RECEIVERNICKNAME).value(image.getReceiverNickname());
        generator.name(IMAGE.RECEIVERUID).value(image.getReceiverUid());
        generator.name(IMAGE.RECEIVER_SESSION).value(image.getReceiverSession());

        generator.endObject();
        break;
      case CONFIRM:
        generator.beginObject();

        generator.endObject();
        break;
      case ERROR:
        generator.beginObject();
        if (image.getSenderUid() == null)
          throw new EncodeException(image, "Das Attribut SENDERUSERID wurde nicht definiert");
        if (image.getSenderNickname() == null)
          throw new EncodeException(image, "Das Attribut SENDERNICKNAME wurde nicht definiert");
        if (image.getReceiverNickname() == null)
          throw new EncodeException(image, "Das Attribut RECEIVERNICKNAME wurde nicht definiert");
        if (image.getMultilingualkey() == null)
          throw new EncodeException(image, "Das Attribut MULTILINGUALKEY wurde nicht definiert");

        generator.name(IMAGE.SENDERUID).value(image.getSenderUid());
        generator.name(IMAGE.SENDERNICKNAME).value(image.getSenderNickname());
        generator.name(IMAGE.RECEIVERNICKNAME).value(image.getReceiverNickname());
        generator.name(MESSAGE.MULTILINGUALKEY).value(image.getMultilingualkey().value());

        generator.endObject();
        break;
    }

    generator.endObject();
    return stringwriter.toString();
  }



  String encodeINCOMINGCALL(MESSAGE message) throws IOException, EncodeException {
    if (!(message instanceof INCOMINGCALL)) return EMPTY;
    INCOMINGCALL incoming = (INCOMINGCALL) message;
    if (Command.INCOMINGCALL.equals(message.getCommand())) {
      StringWriter stringwriter = new StringWriter();
      Gson gson = gsonbuilder.create();
      JsonWriter generator = gson.newJsonWriter(stringwriter);
      generator.beginObject();
      generator.name(Protocol.HEADER).value(message.getHeader().toString());
      generator.name(Command.getCommandKey()).value(message.getCommand().toString());
      if (RESPONSE.equals(message.getHeader())) {
        generator.name(incoming.getDataset());
        generator.beginObject();

        if (incoming.getCallerNickname() == null)
          throw new EncodeException(incoming, "Das Attribut CALLER_NICKNAME wurde nicht definiert.");
        if (incoming.getCallerUserid() == null)
          throw new EncodeException(incoming, "Das Attribut CALLER_USERID wurde nicht definiert.");
        if (incoming.getReceiverUserid() == null)
          throw new EncodeException(incoming, "Das Attribut RECEIVER_USERID wurde nicht definiert.");
        if (incoming.getReceiverNickname() == null)
          throw new EncodeException(incoming, "Das Attribut RECEIVER_NICKNAME wurde nicht definiert.");
        if (incoming.getCallerVoice() == null)
          throw new EncodeException(incoming, "Das Attribut CALLER_VOICE wurde nicht definiert.");
        if (incoming.getReceiverVoice() == null)
          throw new EncodeException(incoming, "Das Attribut RECEIVER_VOICE wurde nicht definiert.");
        if (incoming.getCallerSession() == null)
          throw new EncodeException(incoming, "Das Attribut CALLER_SESSION wurde nicht definiert.");
        if (incoming.getReceiverSession() == null)
          throw new EncodeException(incoming, "Das Attribut RECEIVER_SESSION wurde nicht definiert.");

        generator.name(INCOMINGCALL.CALLER_NICKNAME).value(incoming.getCallerNickname());
        generator.name(INCOMINGCALL.CALLER_USERID).value(incoming.getCallerUserid());
        generator.name(INCOMINGCALL.CALLER_VOICE).value(incoming.getCallerVoice());
        generator.name(INCOMINGCALL.CALLER_SESSION).value(incoming.getCallerSession());
        generator.name(INCOMINGCALL.RECEIVER_USERID).value(incoming.getReceiverUserid());
        generator.name(INCOMINGCALL.RECEIVER_NICKNAME).value(incoming.getReceiverNickname());
        generator.name(INCOMINGCALL.RECEIVER_SESSION).value(incoming.getReceiverSession());
        generator.name(INCOMINGCALL.RECEIVER_VOICE).value(incoming.getReceiverVoice());

        generator.endObject();
      }
      else if (REQUEST == message.getHeader()) {
        generator.name(incoming.getDataset());
        generator.beginObject();
        if (incoming.getCallerVoice() == null)
          throw new EncodeException(incoming, "Das Attribut CALLER_VOICE wurde nicht definiert.");
        if (incoming.getCallerNickname() == null)
          throw new EncodeException(incoming, "Das Attribut CALLER_NICKNAME wurde nicht definiert.");
        if (incoming.getCallerUserid() == null)
          throw new EncodeException(incoming, "Das Attribut CALLER_USERID wurde nicht definiert.");
        if (incoming.getCallerSession() == null)
          throw new EncodeException(incoming, "Das Attribut CALLER_SESSION wurde nicht definiert.");
        if (incoming.getReceiverUserid() == null)
          throw new EncodeException(incoming, "Das Attribut RECEIVER_USERID wurde nicht definiert.");
        if (incoming.getReceiverNickname() == null)
          throw new EncodeException(incoming, "Das Attribut RECEIVER_NICKNAME wurde nicht definiert.");
        if (incoming.getReceiverSession() == null)
          throw new EncodeException(incoming, "Das Attribut RECEIVER_SESSION wurde nicht definiert.");
        generator.name(INCOMINGCALL.CALLER_NICKNAME).value(incoming.getCallerNickname());
        generator.name(INCOMINGCALL.CALLER_USERID).value(incoming.getCallerUserid());
        generator.name(INCOMINGCALL.CALLER_VOICE).value(incoming.getCallerVoice());
        generator.name(INCOMINGCALL.CALLER_SESSION).value(incoming.getCallerSession());
        generator.name(INCOMINGCALL.RECEIVER_USERID).value(incoming.getReceiverUserid());
        generator.name(INCOMINGCALL.RECEIVER_NICKNAME).value(incoming.getReceiverNickname());
        generator.name(INCOMINGCALL.RECEIVER_SESSION).value(incoming.getReceiverSession());
        generator.endObject();
      }
      else if (ERROR == message.getHeader()) {
        generator.name(incoming.getDataset());
        generator.beginObject();
        generator.name(INCOMINGCALL.RECEIVER_NICKNAME).value(incoming.getReceiverNickname());
        generator.name(MESSAGE.ERRORMESSAGE).value(incoming.getErrorMessage());
        if (incoming.getMultilingualkey() == null)
          throw new EncodeException(incoming, "Das Attribut MULTILINGUALKEY wurde nicht definiert.");
        generator.name(MESSAGE.MULTILINGUALKEY).value(incoming.getMultilingualkey().value());
        generator.endObject();
      }
      else if (CONFIRM == message.getHeader()) {
        generator.name(incoming.getDataset());
        generator.beginObject();

        if (incoming.getCallerVoice() == null)
          throw new EncodeException(incoming, "Das Attribut CALLER_VOICE wurde nicht definiert.");
        if (incoming.getCallerNickname() == null)
          throw new EncodeException(incoming, "Das Attribut CALLER_NICKNAME wurde nicht definiert.");
        if (incoming.getCallerUserid() == null)
          throw new EncodeException(incoming, "Das Attribut CALLER_USERID wurde nicht definiert.");
        if (incoming.getReceiverUserid() == null)
          throw new EncodeException(incoming, "Das Attribut RECEIVER_USERID wurde nicht definiert.");
        if (incoming.getReceiverNickname() == null)
          throw new EncodeException(incoming, "Das Attribut RECEIVER_NICKNAME wurde nicht definiert.");
        if (incoming.getReceiverSession() == null)
          throw new EncodeException(incoming, "Das Attribut RECEIVER_SESSION wurde nicht definiert.");

        generator.name(INCOMINGCALL.CALLER_NICKNAME).value(incoming.getCallerNickname());
        generator.name(INCOMINGCALL.CALLER_USERID).value(incoming.getCallerUserid());
        generator.name(INCOMINGCALL.CALLER_VOICE).value(incoming.getCallerVoice());
        generator.name(INCOMINGCALL.CALLER_SESSION).value(incoming.getCallerSession());
        generator.name(INCOMINGCALL.RECEIVER_USERID).value(incoming.getReceiverUserid());
        generator.name(INCOMINGCALL.RECEIVER_NICKNAME).value(incoming.getReceiverNickname());
        generator.name(INCOMINGCALL.RECEIVER_SESSION).value(incoming.getReceiverSession());
        generator.endObject();
      }
      generator.endObject();
      generator.flush();
      generator.close();
      return stringwriter.toString();
    }
    return EMPTY;
  }



  public String encodeKEEPALIVE(MESSAGE message) throws IOException {
    if (!(message instanceof KEEPALIVE)) return EMPTY;
    KEEPALIVE keepalive = (KEEPALIVE) message;
    if (Command.KEEPALIVE.equals(keepalive.getCommand())) {
      StringWriter stringwriter = new StringWriter();
      Gson gson = gsonbuilder.create();
      JsonWriter generator = gson.newJsonWriter(stringwriter);

      generator.beginObject();

      generator.name(Protocol.HEADER).value(message.getHeader().toString());
      generator.name(Command.getCommandKey()).value(message.getCommand().name());

      if (message.getDataset() != null) {
        generator.name(message.getDataset()).beginObject();
        generator.name(KEEPALIVE.SESSIONID).value(keepalive.getSessionId());
        generator.endObject();
      }
      generator.endObject();
      generator.flush();
      generator.close();
      return stringwriter.toString();
    }
    return EMPTY;

  }



  public String encodeLEAVEPRIVATECHAT(MESSAGE message) throws EncodeException, IOException {
    if (!(message instanceof LEAVEPRIVATECHAT)) return EMPTY;
    LEAVEPRIVATECHAT chat = (LEAVEPRIVATECHAT) message;
    if (Command.LEAVEPRIVATECHAT.equals(chat.getCommand())) {
      StringWriter stringwriter = new StringWriter();
      Gson gson = gsonbuilder.create();
      JsonWriter generator = gson.newJsonWriter(stringwriter);
      generator.beginObject();
      generator.name(Protocol.HEADER).value(message.getHeader().toString());
      generator.name(Command.getCommandKey()).value(message.getCommand().name());
      if (REQUEST == message.getHeader()) {
        generator.name(message.getDataset()).beginObject();
        try {
          generator.name(LEAVEPRIVATECHAT.GONE_SESSIONID).value(chat.getGoneSessionid());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde ein Attribut nicht definiert.");
        }
        generator.endObject();
      }
      else if (RESPONSE.equals(message.getHeader())) {
        generator.name(message.getDataset()).beginObject();
        try {
          generator.name(LEAVEPRIVATECHAT.REMOTE_SESSIONID).value(chat.getRemotesession());
          generator.name(LEAVEPRIVATECHAT.GONE_SESSIONID).value(chat.getGoneSessionid());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde ein Attribut nicht definiert.");
        }
        generator.endObject();
      }
      generator.endObject();
      generator.flush();
      generator.close();
      return stringwriter.toString();
    }
    return EMPTY;
  }



  public String encodeLEAVEROOM(MESSAGE message) throws IOException {
    if (!(message instanceof LEAVEROOM)) return EMPTY;
    LEAVEROOM leaveroom = (LEAVEROOM) message;
    if (Command.LEAVEROOM.equals(leaveroom.getCommand())) {
      StringWriter stringwriter = new StringWriter();
      Gson gson = gsonbuilder.create();
      JsonWriter generator = gson.newJsonWriter(stringwriter);
      generator.beginObject();
      generator.name(Protocol.HEADER).value(message.getHeader().toString());
      generator.name(Command.getCommandKey()).value(message.getCommand().name());
      if (REQUEST == message.getHeader()) {
        generator.name(message.getDataset()).beginObject();
        generator.name(LEAVEROOM.USERID).value(leaveroom.getUserid());
        generator.name(LEAVEROOM.ROOM).value(leaveroom.getRoom());
        generator.endObject();
      }
      else if (RESPONSE.equals(message.getHeader())) {
        generator.name(message.getDataset()).beginObject();
        generator.name(LEAVEROOM.USERID).value(leaveroom.getUserid());
        generator.name(LEAVEROOM.ROOM).value(leaveroom.getRoom());
        generator.endObject();
      }
      else if (ERROR == message.getHeader()) {
        generator.name(message.getDataset()).beginObject();
        generator.name(LEAVEROOM.USERID).value(leaveroom.getUserid());
        generator.name(LEAVEROOM.ROOM).value(leaveroom.getRoom());
        generator.endObject();
      }
      generator.endObject();
      generator.flush();
      generator.close();
      return stringwriter.toString();
    }

    return EMPTY;
  }



  String encodeMICROERROR(MESSAGE message) throws IOException, EncodeException {
    if (!(message instanceof MICROERROR)) return EMPTY;
    MICROERROR microerror = (MICROERROR) message;
    if (!Command.MICROERROR.equals(message.getCommand())) return EMPTY;
    StringWriter stringwriter = new StringWriter();
    Gson gson = gsonbuilder.create();
    JsonWriter generator = gson.newJsonWriter(stringwriter);
    generator.beginObject();
    generator.name(Protocol.HEADER).value(message.getHeader().toString());
    generator.name(Command.getCommandKey()).value(message.getCommand().name());
    if (REQUEST == message.getHeader()) {
      if (microerror.getDataset() == null)
        throw new EncodeException(microerror, "DATASET wurde nicht definiert.");
      generator.name(microerror.getDataset());
      generator.beginObject();
      if (microerror.getFromNickname() == null)
        throw new EncodeException(microerror, "Das Attribut FROM_NICKNAME wurde nicht definiert.");
      generator.name(MICROERROR.FROM_NICKNAME).value(microerror.getFromNickname());
      if (microerror.getFromUserid() == null)
        throw new EncodeException(microerror, "Das Attribut FROM_USERID wurde nicht definiert.");
      generator.name(MICROERROR.FROM_USERID).value(microerror.getFromUserid());
      if (microerror.getText() == null)
        throw new EncodeException(microerror, "Das Attribut TEXT wurde nicht definiert.");
      generator.name(MESSAGE.TEXT).value(microerror.getText());
      if (microerror.getToUserid() == null)
        throw new EncodeException(microerror, "Das Attribut TO_USERID wurde nicht definiert.");
      generator.name(MICROERROR.TO_USERID).value(microerror.getToUserid());
      generator.endObject();
    }
    generator.endObject();
    generator.flush();
    generator.close();
    return stringwriter.toString();
  }



  String encodeONCALL(MESSAGE message) throws IOException, EncodeException {
    if (!(message instanceof ONCALL)) return EMPTY;
    ONCALL oncall = (ONCALL) message;
    if (Command.ONCALL.equals(message.getCommand())) {
      StringWriter stringwriter = new StringWriter();
      Gson gson = gsonbuilder.create();
      JsonWriter generator = gson.newJsonWriter(stringwriter);
      generator.beginObject();
      generator.name(Protocol.HEADER).value(message.getHeader().toString());
      generator.name(Command.getCommandKey()).value(message.getCommand().name());
      if (REQUEST == message.getHeader()) {
        generator.name(oncall.getDataset());
        generator.beginObject();
        if (oncall.getUserid() == null)
          throw new EncodeException(oncall, "Im Datensatz wurde das Attribut USERID nicht definiert.");
        generator.name(ONCALL.USERID).value(oncall.getUserid());
        generator.name(ONCALL.ONCALL).value(oncall.isOnCall());
        generator.endObject();
      }
      else if (RESPONSE.equals(message.getHeader())) {
        generator.name(oncall.getDataset());
        generator.beginObject();
        if (oncall.getUserid() == null)
          throw new EncodeException(oncall, "Im Datensatz wurde das Attribut USERID nicht definiert.");
        generator.name(ONCALL.USERID).value(oncall.getUserid());
        generator.name(ONCALL.ONCALL).value(oncall.isOnCall());
        generator.endObject();
      }
      generator.endObject();
      generator.flush();
      generator.close();
      return stringwriter.toString();
    }
    return EMPTY;
  }



  String encodeONHOOK(MESSAGE message) throws IOException, EncodeException {
    if (!(message instanceof ONHOOK)) return EMPTY;
    ONHOOK onhook = (ONHOOK) message;
    if (Command.ONHOOK.equals(message.getCommand())) {
      StringWriter stringwriter = new StringWriter();
      Gson gson = gsonbuilder.create();
      JsonWriter generator = gson.newJsonWriter(stringwriter);
      generator.beginObject();
      generator.name(Protocol.HEADER).value(message.getHeader().toString());
      generator.name(Command.getCommandKey()).value(message.getCommand().name());
      if (onhook.getDataset() == null) throw new EncodeException(onhook, "DATASET wurde nicht definiert.");
      generator.name(onhook.getDataset());

      switch(message.getHeader()) {
        case REQUEST:
          if (onhook.getReceiverUserid() == null)
            throw new EncodeException(onhook, "Das Attribut RECEIVER_USERID wurde nicht definiert.");
          if (onhook.getCallerUserid() == null)
            throw new EncodeException(onhook, "Das Attribut CALLER_USERID wurde nicht definiert.");
          if (onhook.isTo() == null)
            throw new EncodeException(onhook, "Das Attribut TO wurde nicht definiert.");
          generator.beginObject();
          generator.name(ONHOOK.RECEIVER_USERID).value(onhook.getReceiverUserid());
          generator.name(ONHOOK.CALLER_USERID).value(onhook.getCallerUserid());
          generator.name(ONHOOK.TO).value(onhook.isTo());
          generator.name(ONHOOK.SIGNAL).value(onhook.isSignal());
          generator.endObject();
          break;
        case RESPONSE:
          if (onhook.getReceiverUserid() == null)
            throw new EncodeException(onhook, "Das Attribut RECEIVER_USERID wurde nicht definiert.");
          if (onhook.getCallerUserid() == null)
            throw new EncodeException(onhook, "Das Attribut CALLER_USERID wurde nicht definiert.");
          if (onhook.isTo() == null)
            throw new EncodeException(onhook, "Das Attribut TO wurde nicht definiert.");
          generator.beginObject();
          generator.name(ONHOOK.RECEIVER_USERID).value(onhook.getReceiverUserid());
          generator.name(ONHOOK.CALLER_USERID).value(onhook.getCallerUserid());
          generator.name(ONHOOK.TO).value(onhook.isTo());
          generator.endObject();
          break;
      }

      generator.endObject();
      generator.flush();
      generator.close();
      return stringwriter.toString();
    }
    return EMPTY;
  }



  String encodePRIVATECHATFILE(MESSAGE message) throws IOException, EncodeException {
    if (!(message instanceof PRIVATECHATFILE)) return EMPTY;
    PRIVATECHATFILE privatechatfile = (PRIVATECHATFILE) message;
    if (!Command.PRIVATECHATFILE.equals(message.getCommand())) return EMPTY;
    StringWriter stringwriter = new StringWriter();
    Gson gson = gsonbuilder.create();
    JsonWriter generator = gson.newJsonWriter(stringwriter);
    generator.beginObject();
    generator.name(Protocol.HEADER).value(message.getHeader().toString());
    generator.name(Command.getCommandKey()).value(message.getCommand().name());
    if (privatechatfile.getDataset() == null)
      throw new EncodeException(privatechatfile, "DATASET wurde nicht definiert.");
    generator.name(privatechatfile.getDataset());
    switch(message.getHeader()) {
      case REQUEST:
        generator.beginObject();
        if (privatechatfile.getUserid() == null)
          throw new EncodeException(privatechatfile, "Das Attribut USERID wurde nicht definiert");
        if (privatechatfile.getFilename() == null)
          throw new EncodeException(privatechatfile, "Das Attribut FILENAME wurde nicht definiert");
        if (privatechatfile.getRemoteSessionId() == null)
          throw new EncodeException(privatechatfile, "Das Attribut REMOTE_SESSION_ID wurde nicht definiert");
        if (privatechatfile.getAbsoluteFilename() == null)
          throw new EncodeException(privatechatfile, "Das Attribut ABSOLUTE_FILENAME wurde nicht definiert");
        if (privatechatfile.getMessage() == null)
          throw new EncodeException(privatechatfile, "Das Attribut MESSAGE wurde nicht definiert");
        if (privatechatfile.getRemoteNickname() == null)
          throw new EncodeException(privatechatfile, "Das Attribut REMOTE_NICKNAME wurde nicht definiert");

        generator.name(PRIVATECHATFILE.USERID).value(privatechatfile.getUserid());
        generator.name(PRIVATECHATFILE.FILENAME).value(privatechatfile.getFilename());
        generator.name(PRIVATECHATFILE.REMOTE_SESSION_ID).value(privatechatfile.getRemoteSessionId());
        generator.name(PRIVATECHATFILE.REMOTE_NICKNAME).value(privatechatfile.getRemoteNickname());
        generator.name(PRIVATECHATFILE.ABSOLUTE_FILENAME).value(privatechatfile.getAbsoluteFilename());
        generator.name(PRIVATECHATFILE.FILESIZE).value(privatechatfile.getFilesize());
        generator.name(PRIVATECHATFILE.MESSAGE).value(privatechatfile.getMessage());

        generator.endObject();
        break;
      case RESPONSE:
        generator.beginObject();
        if (privatechatfile.getNumber() == null)
          throw new EncodeException(privatechatfile, "Das Attribut NUMBER wurde nicht definiert");
        generator.name(PRIVATECHATFILE.NUMBER).value(privatechatfile.getNumber());
        if (privatechatfile.getRemoteNickname() == null)
          throw new EncodeException(privatechatfile, "Das Attribut REMOTE_NICKNAME wurde nicht definiert");
        generator.name(PRIVATECHATFILE.REMOTE_NICKNAME).value(privatechatfile.getRemoteNickname());
        if (privatechatfile.getAbsoluteFilename() == null)
          throw new EncodeException(privatechatfile, "Das Attribut ABSOLUTE_FILENAME wurde nicht definiert");
        generator.name(PRIVATECHATFILE.ABSOLUTE_FILENAME).value(privatechatfile.getAbsoluteFilename());

        // optional
        if (privatechatfile.getUserid() != null)
          generator.name(PRIVATECHATFILE.USERID).value(privatechatfile.getUserid());
        if (privatechatfile.getFilename() != null)
          generator.name(PRIVATECHATFILE.FILENAME).value(privatechatfile.getFilename());
        if (privatechatfile.getRemoteSessionId() != null)
          generator.name(PRIVATECHATFILE.REMOTE_SESSION_ID).value(privatechatfile.getRemoteSessionId());
        generator.name(PRIVATECHATFILE.FILESIZE).value(privatechatfile.getFilesize());
        if (privatechatfile.getMessage() != null)
          generator.name(PRIVATECHATFILE.MESSAGE).value(privatechatfile.getMessage());

        generator.endObject();
        break;
      case CONFIRM:
        generator.beginObject();

        generator.endObject();
        break;
      case ERROR:
        generator.beginObject();

        generator.endObject();
        break;
    }

    generator.endObject();
    return stringwriter.toString();
  }



  public String encodePRIVATEMESSAGE(MESSAGE message) throws EncodeException, IOException {
    if (!(message instanceof PRIVATEMESSAGE)) return EMPTY;
    PRIVATEMESSAGE privatemessage = (PRIVATEMESSAGE) message;
    if (Command.PRIVATEMESSAGE.equals(privatemessage.getCommand())) {
      StringWriter stringwriter = new StringWriter();
      Gson gson = gsonbuilder.create();
      JsonWriter generator = gson.newJsonWriter(stringwriter);
      generator.beginObject();
      generator.name(Protocol.HEADER).value(message.getHeader().toString());
      generator.name(Command.getCommandKey()).value(message.getCommand().name());
      if (REQUEST == message.getHeader()) {
        generator.name(message.getDataset()).beginObject();

        if (privatemessage.getRemoteSessionid() == null)
          throw new EncodeException(message, "Das Attribut REMOTE_SESSIONID wurde nicht gesetzt");
        generator.name(PRIVATEMESSAGE.REMOTE_SESSIONID).value(privatemessage.getRemoteSessionid());

        if (privatemessage.getSenderUID() != null)
          generator.name(PRIVATEMESSAGE.SENDER_UID).value(privatemessage.getSenderUID());
        if (privatemessage.getLocalSessionid() != null)
          generator.name(PRIVATEMESSAGE.LOCAL_SESSIONID).value(privatemessage.getLocalSessionid());
        if (privatemessage.getDatetime() != null)
          generator.name(PRIVATEMESSAGE.DATETIME).value(privatemessage.getDatetime());
        generator.name(PRIVATEMESSAGE.MESSAGE).value(privatemessage.getMessage());
        ChatUser chatuser = privatemessage.getChatUser();
        if (chatuser != null) {
          generator.name(PRIVATEMESSAGE.CHATUSER).beginObject();
          generator.name(ChatUser.USERID).value(chatuser.getUserid());
          generator.name(ChatUser.NICKNAME).value(chatuser.getNickname());
          generator.name(ChatUser.FOREGROUND_COLOR).value(chatuser.getForegroundColor());
          generator.name(ChatUser.BACKGROUND_COLOR).value(chatuser.getBackgroundColor());
          generator.endObject();
        }

        generator.name(Attachment.ATTACHMENT).value(privatemessage.getAttachment());
        generator.name(Attachment.FILENAME).value(privatemessage.getFilename());
        generator.name(Attachment.FILESIZE).value(privatemessage.getFilesize());
        generator.endObject();
      }
      else if (RESPONSE.equals(message.getHeader())) {
        generator.name(message.getDataset()).beginObject();
        if (privatemessage.getSenderUID() != null)
          generator.name(PRIVATEMESSAGE.SENDER_UID).value(privatemessage.getSenderUID());
        if (privatemessage.getLocalSessionid() != null)
          generator.name(PRIVATEMESSAGE.LOCAL_SESSIONID).value(privatemessage.getLocalSessionid());
        try {
          generator.name(PRIVATEMESSAGE.REMOTE_SESSIONID).value(privatemessage.getRemoteSessionid());
          if (privatemessage.getDatetime() != null)
            generator.name(PRIVATEMESSAGE.DATETIME).value(privatemessage.getDatetime());
          generator.name(PRIVATEMESSAGE.MESSAGE).value(privatemessage.getMessage());
          ChatUser chatuser = privatemessage.getChatUser();
          if (chatuser != null) {
            generator.name(PRIVATEMESSAGE.CHATUSER).beginObject();
            generator.name(ChatUser.USERID).value(chatuser.getUserid());
            generator.name(ChatUser.NICKNAME).value(chatuser.getNickname());
            generator.name(ChatUser.FOREGROUND_COLOR).value(chatuser.getForegroundColor());
            generator.name(ChatUser.BACKGROUND_COLOR).value(chatuser.getBackgroundColor());
            generator.endObject();
          }
          generator.name(Attachment.ATTACHMENT).value(privatemessage.getAttachment());
          generator.name(Attachment.FILENAME).value(privatemessage.getFilename());
          generator.name(Attachment.FILESIZE).value(privatemessage.getFilesize());

        }
        catch (NullPointerException e) {
          throw new EncodeException(
              message, "PRIVATEMESSAGE-RESPONSE, im Datensatz wurde ein Attribut nicht definiert."
          );
        }
        generator.endObject();
      }

      generator.endObject();
      generator.flush();
      generator.close();
      return stringwriter.toString();
    }
    return EMPTY;
  }



  String encodePROYECTORCLOSING(MESSAGE message) throws IOException, EncodeException {
    if (!(message instanceof PROYECTORCLOSING)) return EMPTY;
    PROYECTORCLOSING proyector = (PROYECTORCLOSING) message;
    if (!Command.PROYECTORCLOSING.equals(message.getCommand())) return EMPTY;
    StringWriter stringwriter = new StringWriter();
    Gson gson = gsonbuilder.create();
    JsonWriter generator = gson.newJsonWriter(stringwriter);
    generator.beginObject();
    generator.name(Protocol.HEADER).value(message.getHeader().toString());
    generator.name(Command.getCommandKey()).value(message.getCommand().name());
    if (proyector.getDataset() == null)
      throw new EncodeException(proyector, "DATASET wurde nicht definiert.");
    generator.name(proyector.getDataset());
    switch(message.getHeader()) {
      case RESPONSE:
        generator.beginObject();
        if (proyector.getUserid() == null)
          throw new EncodeException(proyector, "Das Attribut USERID wurde nicht definiert");
        generator.name(PROYECTORCLOSING.USERID).value(proyector.getUserid());
        generator.endObject();
        break;
      case REQUEST:
        generator.beginObject();
        if (proyector.getUserid() == null)
          throw new EncodeException(proyector, "Das Attribut USERID wurde nicht definiert");
        generator.name(PROYECTORCLOSING.USERID).value(proyector.getUserid());

        generator.endObject();
        break;
    }
    generator.endObject();
    return stringwriter.toString();
  }



  String encodeREADROOMS(MESSAGE message) throws IOException, EncodeException {
    if (!(message instanceof READROOMS)) return EMPTY;
    READROOMS readrooms = (READROOMS) message;
    if (!Command.READROOMS.equals(message.getCommand())) return EMPTY;
    StringWriter stringwriter = new StringWriter();
    Gson gson = gsonbuilder.create();
    JsonWriter generator = gson.newJsonWriter(stringwriter);
    generator.beginObject();
    generator.name(Protocol.HEADER).value(message.getHeader().toString());
    generator.name(Command.getCommandKey()).value(message.getCommand().name());
    if (readrooms.getDataset() == null)
      throw new EncodeException(readrooms, "DATASET wurde nicht definiert.");
    generator.name(readrooms.getDataset());
    switch(message.getHeader()) {
      case REQUEST:
        generator.beginObject();
        if (readrooms.getUserid() == null)
          throw new EncodeException(readrooms, "Das Attribut USERID wurde nicht definiert");
        generator.name(READROOMS.USERID).value(readrooms.getUserid());
        if (readrooms.getRoomtype() == null)
          throw new EncodeException(readrooms, "Das Attribut ROOMTYPE wurde nicht definiert");
        generator.name(READROOMS.ROOMTYPE).value(readrooms.getRoomtype().getRoomtype());
        generator.endObject();
        break;
      case RESPONSE:
        generator.beginObject();
        if (readrooms.getUserid() == null)
          throw new EncodeException(readrooms, "Das Attribut USERID wurde nicht definiert");
        generator.name(READROOMS.USERID).value(readrooms.getUserid());
        if (readrooms.getRoomtype() == null)
          throw new EncodeException(readrooms, "Das Attribut ROOMTYPE wurde nicht definiert");
        generator.name(READROOMS.ROOMTYPE).value(readrooms.getRoomtype().getRoomtype());
        if (readrooms.getRoomtype() == null)
          throw new EncodeException(readrooms, "Das Attribut ROOMS wurde nicht definiert");
        generator.name(READROOMS.ROOMS);
        generator.beginArray();
        for (String room : readrooms.getRooms()) {
          generator.value(room);
        }
        generator.endArray();
        generator.endObject();
        break;
    }
    generator.endObject();
    return stringwriter.toString();
  }



  String encodeREADTOPICROOMOWNER(MESSAGE message) throws EncodeException, IOException {
    if (!(message instanceof READTOPICROOMOWNER)) return EMPTY;
    READTOPICROOMOWNER topicroom = (READTOPICROOMOWNER) message;
    if (Command.READTOPICROOMOWNER.equals(topicroom.getCommand())) {
      StringWriter stringwriter = new StringWriter();
      Gson gson = gsonbuilder.create();
      JsonWriter generator = gson.newJsonWriter(stringwriter);
      generator.beginObject();
      generator.name(Protocol.HEADER).value(message.getHeader().toString());
      generator.name(Command.getCommandKey()).value(message.getCommand().name());
      if (REQUEST == message.getHeader()) {
        generator.name(message.getDataset()).beginObject();
        try {
          generator.name(READTOPICROOMOWNER.USERID).value(topicroom.getUserid());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde das Attribut USERID nicht definiert.");
        }
        try {
          generator.name(READTOPICROOMOWNER.OWNER).value(topicroom.getOwner());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde das Attribut OWNER nicht definiert.");
        }
        generator.endObject();
      }
      else if (RESPONSE.equals(message.getHeader())) {
        generator.name(message.getDataset()).beginObject();
        try {
          generator.name(READTOPICROOMOWNER.USERID).value(topicroom.getUserid());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde das Attribut USERID nicht definiert.");
        }
        try {
          generator.name(READTOPICROOMOWNER.OWNER).value(topicroom.getOwner());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde das Attribut OWNER nicht definiert.");
        }
        // kommen die Rooms
        generator.name(ROOMLIST.ROOMLIST).beginArray();
        Room[] rooms = topicroom.getRoom();
        if (rooms != null) {
          for (Room tmp : rooms) {
            generator.beginObject();
            generator.name(Room.ROOM).value(tmp.getChatid());
            generator.name(Room.ENTRY).value(tmp.getEntry().name());
            generator.name(Room.LIFETIME).value(tmp.getLifetime().name());
            if (tmp.getOwner() != null) generator.name(Room.OWNER).value(tmp.getOwner());
            generator.endObject();
          }
        }
        generator.endArray(); // Array
        generator.endObject();
      }
      generator.endObject();
      generator.flush();
      generator.close();
      return stringwriter.toString();
    }
    return EMPTY;
  }



  public String encodeROOMLIST(MESSAGE message) throws IOException {
    if (!(message instanceof ROOMLIST)) return EMPTY;
    ROOMLIST roomlist = (ROOMLIST) message;
    if (roomlist.getCommand().equals(Command.ROOMLIST)) {
      StringWriter stringwriter = new StringWriter();
      Gson gson = gsonbuilder.create();
      JsonWriter generator = gson.newJsonWriter(stringwriter);
      generator.beginObject();
      generator.name(Protocol.HEADER).value(message.getHeader().toString());
      generator.name(Command.getCommandKey()).value(message.getCommand().name());
      if (message.getHeader().equals(RESPONSE)) {
        generator.name(roomlist.getDataset()).beginObject();
        // Array of Rooms festlegen ist ein Array of Objects
        generator.name(ROOMLIST.ROOMLIST).beginArray();
        Room[] rooms = roomlist.getRoom();
        if (rooms != null) {
          for (Room tmp : rooms) {
            generator.beginObject();
            generator.name(Room.ENTRY).value(tmp.getEntry().name());
            generator.name(Room.LIFETIME).value(tmp.getLifetime().name());
            if (tmp.getOwner() != null) generator.name(Room.OWNER).value(tmp.getOwner());
            generator.name(Room.ROOM).value(tmp.getChatid());
            generator.name(Room.ROOMTYPE).value(tmp.getRoomtype().name());
            generator.endObject();
          }
        }
        generator.endArray(); // Array
        generator.endObject();
      }
      else if (message.getHeader().equals(REQUEST)) {

      }
      generator.endObject();
      generator.flush();
      generator.close();
      return stringwriter.toString();
    }
    return EMPTY;
  }



  String encodeSCREENCAST(MESSAGE message) throws IOException, EncodeException {
    if (!(message instanceof SCREENCAST)) return EMPTY;
    SCREENCAST screencast = (SCREENCAST) message;

    if (!Command.SCREENCAST.equals(message.getCommand())) return EMPTY;
    StringWriter stringwriter = new StringWriter();
    Gson gson = gsonbuilder.create();
    JsonWriter generator = gson.newJsonWriter(stringwriter);
    generator.beginObject();
    generator.name(Protocol.HEADER).value(message.getHeader().toString());
    generator.name(Command.getCommandKey()).value(message.getCommand().name());
    if (screencast.getDataset() == null)
      throw new EncodeException(screencast, "DATASET wurde nicht definiert.");

    generator.name(screencast.getDataset());

    switch(message.getHeader()) {
      case REQUEST:
        generator.beginObject();
        if (screencast.getCallerUserid() == null)
          throw new EncodeException(screencast, "Das Attribut CALLER_USERID wurde nicht definiert");
        if (screencast.getCallerNickname() == null)
          throw new EncodeException(screencast, "Das Attribut CALLER_NICKNAME wurde nicht definiert");
        if (screencast.getReceiverNickname() == null)
          throw new EncodeException(screencast, "Das Attribut RECEIVER_NICKNAME wurde nicht definiert");
        if (screencast.getCallerIp() == null)
          throw new EncodeException(screencast, "Das Attribut CALLER_IP wurde nicht definiert");
        if (screencast.getCallerPort() == null)
          throw new EncodeException(screencast, "Das Attribut CALLER_PORT wurde nicht definiert");

        generator.name(SCREENCAST.CALLER_USERID).value(screencast.getCallerUserid());
        generator.name(SCREENCAST.CALLER_NICKNAME).value(screencast.getCallerNickname());
        generator.name(SCREENCAST.CALLER_IP).value(screencast.getCallerIp());
        generator.name(SCREENCAST.CALLER_PORT).value(screencast.getCallerPort());
        generator.name(SCREENCAST.RECEIVER_NICKNAME).value(screencast.getReceiverNickname());
        generator.endObject();
        break;
      case RESPONSE:
        generator.beginObject();
        if (screencast.getCallerUserid() == null)
          throw new EncodeException(screencast, "Das Attribut CALLER_USERID wurde nicht definiert");
        if (screencast.getReceiverUserid() == null)
          throw new EncodeException(screencast, "Das Attribut RECEIVER_USERID wurde nicht definiert");
        if (screencast.getReceiverNickname() == null)
          throw new EncodeException(screencast, "Das Attribut RECEIVER_NICKNAME wurde nicht definiert");
        generator.name(SCREENCAST.CALLER_USERID).value(screencast.getCallerUserid());
        generator.name(SCREENCAST.RECEIVER_USERID).value(screencast.getReceiverUserid());
        generator.name(SCREENCAST.RECEIVER_NICKNAME).value(screencast.getReceiverNickname());
        generator.endObject();
        break;
      case CONFIRM:
        generator.beginObject();
        generator.endObject();
        break;
      case ERROR:
        generator.beginObject();
        if (screencast.getReceiverNickname() == null)
          throw new EncodeException(screencast, "Das Attribut RECEIVER_NICKNAME wurde nicht definiert");
        generator.name(SCREENCAST.RECEIVER_NICKNAME).value(screencast.getReceiverNickname());
        generator.name(SCREENCAST.RECEIVER_USERID).value(screencast.getReceiverUserid());
        generator.name(MESSAGE.ERRORMESSAGE).value(screencast.getErrorMessage());
        generator.endObject();
        break;
    }
    generator.endObject();

    return stringwriter.toString();

  }



  public String encodeSEARCHFILES(MESSAGE message) throws IOException {
    if (!(message instanceof SEARCHFILES)) return EMPTY;
    SEARCHFILES searchfiles = (SEARCHFILES) message;
    if (Command.SEARCHFILES.equals(searchfiles.getCommand())) {
      StringWriter stringwriter = new StringWriter();
      Gson gson = gsonbuilder.create();
      JsonWriter generator = gson.newJsonWriter(stringwriter);
      generator.beginObject();
      generator.name(Protocol.HEADER).value(message.getHeader().toString());
      generator.name(Command.getCommandKey()).value(message.getCommand().name());
      if (REQUEST == message.getHeader()) {
        generator.name(message.getDataset()).beginObject();
        generator.name(SEARCHFILES.USERID).value(searchfiles.getUserid());
        generator.name(SEARCHFILES.SEARCHFILES).beginArray();
        String[] files = searchfiles.getPatterns();
        if (files != null) {
          for (String file : files) {
            generator.beginObject();
            generator.name(SEARCHFILES.PATTERN).value(file);
            generator.endObject();
          }
        }
        generator.endArray();
        generator.endObject();
      }
      else if (RESPONSE.equals(message.getHeader())) {
        generator.name(message.getDataset()).beginObject();
        generator.name(SEARCHFILES.USERID).value(searchfiles.getUserid());
        generator.name(SEARCHFILES.START).value(searchfiles.hasStarted());
        // Key f�r ein Array
        generator.name(SEARCHFILES.SEARCHFILE).beginArray();

        Searchfile[] files = searchfiles.getSearchfiles();
        if (files != null) {
          for (Searchfile tmp : files) {
            generator.beginObject();
            generator.name(Searchfile.INDEX).value(tmp.getIndex());
            generator.name(Searchfile.FILENAME).value(tmp.getFilename());
            generator.name(Searchfile.FILESIZE).value(tmp.getFilesize());
            generator.endObject();
          }
        }
        generator.endArray(); // SEARCHFILE
        generator.endObject(); // DATASET
      }
      generator.endObject();
      generator.flush();
      generator.close();
      return stringwriter.toString();
    }
    return EMPTY;
  }



  public String encodeSIGNIN(MESSAGE message) throws EncodeException, IOException {
    if (!(message instanceof SIGNIN)) return EMPTY;
    SIGNIN signin = (SIGNIN) message;
    if (message.getCommand().equals(Command.SIGNIN)) {
      StringWriter stringwriter = new StringWriter();
      Gson gson = gsonbuilder.create();
      JsonWriter generator = gson.newJsonWriter(stringwriter);
      generator.beginObject();
      generator.name(Protocol.HEADER).value(message.getHeader().toString());
      generator.name(Command.getCommandKey()).value(message.getCommand().name());
      if (message.getHeader().equals(REQUEST)) {
        // SIGNIN REQUEST
        generator.name(signin.getDataset()).beginObject();
        generator.name(SIGNIN.EMAIL).value(signin.getEmail());
        generator.endObject(); // DATASET End
      }
      else if (message.getHeader() == CONFIRM) {
        generator.name(signin.getDataset()).beginObject();
        if (signin.getEmail() == null) {
          log.fatal("Protokollfehler: das Attribute EMAIL wurde nicht gesetzt.");
          throw new EncodeException(message, "Protokollfehler: das Attribute EMAIL wurde nicht gesetzt.");

        }
        generator.name(SIGNIN.EMAIL).value(signin.getEmail());
        if (signin.getUserid() == null) {
          log.fatal("Protokollfehler: das Attribute USERID wurde nicht gesetzt.");
          throw new EncodeException(message, "Protokollfehler: das Attribute USERID wurde nicht gesetzt.");
        }
        generator.name(SIGNIN.USERID).value(signin.getUserid());

        if (signin.getPassword() == null) {
          log.fatal("Protokollfehler: das Attribute PASSWORD wurde nicht gesetzt.");
          throw new EncodeException(message, "Protokollfehler: das Attribute PASSWORD wurde nicht gesetzt.");
        }
        generator.name(SIGNIN.PASSWORD).value(signin.getPassword());

        if (signin.getForegroundColor() == null) {
          log.fatal("Protokollfehler: das Attribute FOREGROUND_COLOR wurde nicht gesetzt.");
          throw new EncodeException(
              message, "Protokollfehler: das Attribute FOREGROUND_COLOR wurde nicht gesetzt."
          );

        }
        generator.name(SIGNIN.FOREGROUND_COLOR).value(signin.getForegroundColor());

        if (signin.getBackgroundColor() == null) {
          log.fatal("Protokollfehler: das Attribute BACKGROUND_COLOR wurde nicht gesetzt.");
          throw new EncodeException(
              message, "Protokollfehler: das Attribute BACKGROUND_COLOR wurde nicht gesetzt."
          );
        }
        generator.name(SIGNIN.BACKGROUND_COLOR).value(signin.getBackgroundColor());

        if (signin.getNickname() == null) {
          log.fatal("Protokollfehler: das Attribute NICKNAME wurde nicht gesetzt.");
          throw new EncodeException(message, "Protokollfehler: das Attribute NICKNAME wurde nicht gesetzt.");
        }
        generator.name(SIGNIN.NICKNAME).value(signin.getNickname());

        if (signin.getMultilingualkey() == null) {
          log.fatal("Protokollfehler: das Attribute MULTILINGUALKEY wurde nicht gesetzt.");
          throw new EncodeException(
              message, "Protokollfehler: das Attribute MULTILINGUALKEY wurde nicht gesetzt."
          );
        }
        generator.name(MESSAGE.MULTILINGUALKEY).value(signin.getMultilingualkey().value());

        if (signin.getText() == null) {
          log.fatal("Protokollfehler: das Attribute TEXT wurde nicht gesetzt.");
          throw new EncodeException(message, "Protokollfehler: das Attribute TEXT wurde nicht gesetzt.");
        }
        generator.name(MESSAGE.TEXT).value(signin.getText());

        generator.endObject(); // DATASET End
      }
      else if (message.getHeader() == ERROR) {
        generator.name(Protocol.DATASET).beginObject();

        if (signin.getEmail() == null) {
          log.fatal("Protokollfehler: das Attribute EMAIL wurde nicht gesetzt.");
          throw new EncodeException(message, "Protokollfehler: das Attribute EMAIL wurde nicht gesetzt.");
        }
        generator.name(USER.EMAIL).value(signin.getEmail());

        if (signin.getErrorCode() == null) {
          log.fatal("Protokollfehler: das Attribute ERRORCODE wurde nicht gesetzt.");
          throw new EncodeException(message, "Protokollfehler: das Attribute ERRORCODE wurde nicht gesetzt.");
        }
        generator.name(MESSAGE.ERRORCODE).value(signin.getErrorCode());

        generator.name(MESSAGE.ERRORMESSAGE).value(signin.getErrorMessage());

        if (signin.getText() == null) {
          log.fatal("Protokollfehler: das Attribute TEXT wurde nicht gesetzt.");
          throw new EncodeException(message, "Protokollfehler: das Attribute TEXT wurde nicht gesetzt.");
        }
        generator.name(MESSAGE.TEXT).value(signin.getText());

        if (signin.getMultilingualkey() == null) {
          log.fatal("Protokollfehler: das Attribute MULTILINGUALKEY wurde nicht gesetzt.");
          throw new EncodeException(
              message, "Protokollfehler: das Attribute MULTILINGUALKEY wurde nicht gesetzt."
          );
        }
        generator.name(MESSAGE.MULTILINGUALKEY).value(signin.getMultilingualkey().value());

        generator.endObject();
      }
      else if (message.getHeader() == RESPONSE) {
        generator.name(Protocol.DATASET).beginObject();
        if (signin.getUserid() == null) {
          log.fatal("SIGNIN | RESPONSE, das Attribute USERID wurde nicht gesetzt.");
          throw new EncodeException(message, "SIGNIN | RESPONSE, das Attribut USERID wurde nicht gesetzt.");

        }
        generator.name(SIGNIN.USERID).value(signin.getUserid());

        if (signin.getNeuesPassword() == null) {
          log.fatal("SIGNIN | RESPONSE, das Attribut NEUES_PASSWORD wurde nicht gesetzt.");
          throw new EncodeException(
              message, "SIGNIN | RESPONSE, das Attribut NEUES_PASSWORD wurde nicht gesetzt."
          );
        }
        generator.name(SIGNIN.NEUES_PASSWORD).value(signin.getNeuesPassword());

        if (signin.getConfirmationKey() == null) {
          log.fatal("SIGNIN | RESPONSE, das Attribut CONFIRMATION_KEY wurde nicht gesetzt.");
          throw new EncodeException(
              message, "SIGNIN | RESPONSE, das Attribut CONFIRMATION_KEY wurde nicht gesetzt."
          );
        }
        generator.name(SIGNIN.CONFIRMATION_KEY).value(signin.getConfirmationKey());

        if (signin.getNickname() == null) {
          log.fatal("SIGNIN | RESPONSE, das Attribut NICKNAME wurde nicht gesetzt.");
          throw new EncodeException(message, "SIGNIN | RESPONSE, das Attribut NICKNAME wurde nicht gesetzt.");
        }
        generator.name(SIGNIN.NICKNAME).value(signin.getNickname());

        if (signin.getBackgroundColor() == null) {
          log.fatal("SIGNIN | RESPONSE, das Attribut BACKGROUND_COLOR wurde nicht gesetzt.");
          throw new EncodeException(
              message, "SIGNIN | RESPONSE, das Attribut BACKGROUND_COLOR wurde nicht gesetzt."
          );
        }
        generator.name(SIGNIN.BACKGROUND_COLOR).value(signin.getBackgroundColor());

        if (signin.getForegroundColor() == null) {
          log.fatal("SIGNIN | RESPONSE, das Attribut FOREGROUND_COLOR wurde nicht gesetzt.");
          throw new EncodeException(
              message, "SIGNIN | RESPONSE, das Attribut FOREGROUND_COLOR wurde nicht gesetzt."
          );
        }
        generator.name(SIGNIN.FOREGROUND_COLOR).value(signin.getForegroundColor());

        if (signin.getMultilingualkey() == null) {
          log.fatal("SIGNIN | RESPONSE, das Attribute MULTILINGUALKEY wurde nicht gesetzt.");
          throw new EncodeException(
              message, "SIGNIN | RESPONSE, das Attribut MULTILINGUALKEY wurde nicht gesetzt"
          );
        }
        generator.name(SIGNIN.MULTILINGUALKEY).value(signin.getMultilingualkey().value());

        if (signin.getText() == null) {
          log.fatal("SIGNIN | RESPONSE, das Attribut TEXT wurde nicht gesetzt.");
          throw new EncodeException(message, "SIGNIN | RESPONSE, das Attribut TEXT wurde nicht gesetzt.");
        }
        generator.name(SIGNIN.TEXT).value(signin.getText());

        generator.endObject();
      }

      generator.endObject();
      generator.flush();
      generator.close();
      return stringwriter.toString();
    }
    return EMPTY;
  }



  String encodeTOPICMEMBER(MESSAGE message) throws EncodeException, IOException {
    if (!(message instanceof TOPICMEMBER)) return EMPTY;
    TOPICMEMBER member = (TOPICMEMBER) message;
    if (Command.TOPICMEMBER.equals(member.getCommand())) {
      StringWriter stringwriter = new StringWriter();
      Gson gson = gsonbuilder.create();
      JsonWriter generator = gson.newJsonWriter(stringwriter);
      generator.beginObject();
      generator.name(Protocol.HEADER).value(message.getHeader().toString());
      generator.name(Command.getCommandKey()).value(message.getCommand().name());
      if (REQUEST == message.getHeader()) {
        generator.name(message.getDataset()).beginObject();
        try {
          generator.name(MEMBER.USERID).value(member.getUserid());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde das Attribut USERID nicht definiert.");
        }
        try {
          generator.name(MEMBER.ROOM).value(member.getRoom());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde das Attribut ROOM nicht definiert.");
        }
        generator.endObject();
      }
      else if (RESPONSE.equals(message.getHeader())) {
        generator.name(message.getDataset()).beginObject();
        try {
          generator.name(MEMBER.USERID).value(member.getUserid());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde das Attribut USERID nicht definiert.");
        }
        try {
          generator.name(MEMBER.ROOM).value(member.getRoom());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde das Attribut ROOM nicht definiert.");
        }
        try {
          generator.name(MEMBER.ROOMTYPE).value(member.getRoomtype().toString());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde das Attribut ROOMTYPE nicht definiert.");
        }
        // es kommen die User
        generator.name(MEMBER.TOPICMEMBER).beginArray();
        ChatUser[] rooms = member.getChatUser();
        if (rooms != null) {
          for (ChatUser tmp : rooms) {
            generator.beginObject();
            if (
              tmp.getUserid() == null
            ) throw new EncodeException(
                message, "Im Datensatz wurde das Attribut TOPICMEMBER/USERID nicht definiert."
            );
            generator.name(ChatUser.USERID).value(tmp.getUserid());
            if (
              tmp.getNickname() == null
            ) throw new EncodeException(
                message, "Im Datensatz wurde das Attribut TOPICMEMBER/NICKNAME nicht definiert."
            );
            generator.name(ChatUser.NICKNAME).value(tmp.getNickname());
            generator.endObject();
          }
        }
        generator.endArray(); // Array
        generator.endObject();

      }
      generator.endObject();
      generator.flush();
      generator.close();
      return stringwriter.toString();
    }
    return EMPTY;
  }



  String encodeTRANSFER(MESSAGE message) throws EncodeException, IOException {
    if (!(message instanceof TRANSFER)) return EMPTY;
    TRANSFER transfer = (TRANSFER) message;
    if (Command.TRANSFER.equals(transfer.getCommand())) {
      StringWriter stringwriter = new StringWriter();
      Gson gson = gsonbuilder.create();
      JsonWriter generator = gson.newJsonWriter(stringwriter);
      generator.beginObject();
      generator.name(Protocol.HEADER).value(message.getHeader().toString());
      generator.name(Command.getCommandKey()).value(message.getCommand().name());
      if (REQUEST == message.getHeader()) {
        generator.name(message.getDataset()).beginObject();
        try {
          generator.name(TRANSFER.SLOT).value(transfer.getSlot());
          if (
            transfer.getBlockindex() < 0
          ) throw new EncodeException(
              message, "blockindex is negative, " + String.valueOf(transfer.getBlockindex())
          );
          generator.name(TRANSFER.BLOCKINDEX).value(transfer.getBlockindex());
          generator.name(TRANSFER.FILENAME).value(transfer.getFilename());
          generator.name(TRANSFER.PATHFILE).value(transfer.getPathfile());
          generator.name(FILETRANSFER.MAXDATABLOCKS).value(transfer.getMaxDatablocks());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde ein Attribut nicht definiert.");
        }
        generator.endObject();

      }
      else if (RESPONSE.equals(message.getHeader())) {
        generator.name(message.getDataset()).beginObject();
        try {
          generator.name(TRANSFER.SLOT).value(transfer.getSlot());
          if (
            transfer.getBlockindex() < 0
          ) throw new EncodeException(
              message, "blockindex is negative, " + String.valueOf(transfer.getBlockindex())
          );
          generator.name(TRANSFER.BLOCKINDEX).value(transfer.getBlockindex());
          generator.name(TRANSFER.FILENAME).value(transfer.getFilename());
          generator.name(TRANSFER.PATHFILE).value(transfer.getPathfile());
          generator.name(TRANSFER.ENDFLAG).value(transfer.getEndflag());
          generator.name(TRANSFER.PAYLOAD).value(transfer.getPayload());
          generator.name(FILETRANSFER.MAXDATABLOCKS).value(transfer.getMaxDatablocks());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde ein Attribut nicht definiert.");
        }
        generator.endObject();
      }
      else if (ERROR == message.getHeader()) {
        generator.name(message.getDataset()).beginObject();
        try {
          generator.name(TRANSFER.SLOT).value(transfer.getSlot());
          generator.name(TRANSFER.FILENAME).value(transfer.getFilename());
          generator.name(TRANSFER.PATHFILE).value(transfer.getPathfile());
          generator.name(MESSAGE.ERRORMESSAGE).value(transfer.getErrorMessage());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde ein Attribut nicht definiert.");
        }
        generator.endObject();
      }
      generator.endObject();
      generator.flush();
      generator.close();
      return stringwriter.toString();

    }
    return EMPTY;
  }



  String encodeUPDATEFILETYPES(MESSAGE message) throws IOException, EncodeException {
    if (!(message instanceof UPDATEFILETYPES)) return EMPTY;
    UPDATEFILETYPES filetypes = (UPDATEFILETYPES) message;
    if (!Command.UPDATEFILETYPES.equals(message.getCommand())) return EMPTY;
    StringWriter stringwriter = new StringWriter();
    Gson gson = gsonbuilder.create();
    JsonWriter generator = gson.newJsonWriter(stringwriter);
    generator.beginObject();
    generator.name(Protocol.HEADER).value(message.getHeader().toString());
    generator.name(Command.getCommandKey()).value(message.getCommand().name());

    if (REQUEST == message.getHeader()) {
      if (filetypes.getDataset() == null)
        throw new EncodeException(filetypes, "DATASET wurde nicht definiert.");
      generator.name(filetypes.getDataset());
      generator.beginObject();
      if (filetypes.getUserid() == null)
        throw new EncodeException(filetypes, "Das Attribut USERID wurde nicht definiert.");
      generator.name(FILETYPES.USERID).value(filetypes.getUserid());
      generator.name(FILETYPES.FILETYPES);
      generator.beginArray();
      BlacklistTypes[] blacklist = filetypes.getFiletypes();
      if (blacklist == null)
        throw new EncodeException(message, "Im Datensatz wurden keine FILETYPES definiert.");
      int count = blacklist.length;
      for (int index = 0; index < count; index++) {
        generator.beginObject();
        if (blacklist[index].getSuffix() == null)
          throw new EncodeException(message, "Im Datensatz wurde das Attribut SUFFIX nicht definiert.");
        generator.name(BlacklistTypes.SUFFIX).value(blacklist[index].getSuffix());
        if (blacklist[index].getDescription() == null)
          throw new EncodeException(message, "Im Datensatz wurde das Attribut DESCRIPTION nicht definiert.");
        generator.name(BlacklistTypes.DESCRIPTION).value(blacklist[index].getDescription());
        if (blacklist[index].getChecked() == null)
          throw new EncodeException(message, "Im Datensatz wurde das Attribut CHECKED nicht definiert.");
        generator.name(BlacklistTypes.CHECKED).value(blacklist[index].getChecked());
        generator.endObject();
      }
      generator.endArray();
      generator.endObject();
    }
    else if (CONFIRM == message.getHeader()) {
      if (filetypes.getDataset() == null)
        throw new EncodeException(filetypes, "DATASET wurde nicht definiert.");
      generator.name(filetypes.getDataset());
      generator.beginObject();
      if (filetypes.getUserid() == null)
        throw new EncodeException(filetypes, "Das Attribut USERID wurde nicht definiert.");
      generator.name(FILETYPES.USERID).value(filetypes.getUserid());
      generator.endObject();
    }
    generator.endObject();
    generator.flush();
    generator.close();
    return stringwriter.toString();
  }



  public String encodeUPDATEPHONE(MESSAGE message) throws EncodeException, IOException {
    if (!(message instanceof UPDATEPHONE)) return EMPTY;
    UPDATEPHONE updatePhone = (UPDATEPHONE) message;
    if (message.getCommand().equals(Command.UPDATEPHONE)) {
      StringWriter stringwriter = new StringWriter();
      Gson gson = gsonbuilder.create();
      JsonWriter generator = gson.newJsonWriter(stringwriter);
      generator.beginObject();
      generator.name(Protocol.HEADER).value(message.getHeader().toString());
      generator.name(Command.getCommandKey()).value(message.getCommand().name());
      if (message.getHeader().equals(REQUEST)) {
        generator.name(updatePhone.getDataset()).beginObject();
        if (updatePhone.getUserid() == null)
          throw new EncodeException(message, "Das Attribut USERID wurde nicht definiert");
        generator.name(UPDATEPORT.USERID).value(updatePhone.getUserid());
        generator.name(UPDATEPORT.IP).value(updatePhone.getIp());
        generator.name(UPDATEPORT.PORT).value(updatePhone.getPort());
        if (updatePhone.isOnBusy() == null)
          throw new EncodeException(message, "Das Attribut ONBUSY wurde nicht definiert");
        generator.name(UPDATEPHONE.ONBUSY).value(updatePhone.isOnBusy());
        generator.endObject(); // DATASET End
      }
      else if (message.getHeader() == RESPONSE) {
        generator.name(updatePhone.getDataset()).beginObject();
        if (updatePhone.getUserid() == null)
          throw new EncodeException(message, "Das Attribut USERID wurde nicht definiert");
        generator.name(UPDATEPORT.USERID).value(updatePhone.getUserid());
        generator.name(UPDATEPORT.IP).value(updatePhone.getIp());
        generator.name(UPDATEPORT.PORT).value(updatePhone.getPort());
        if (updatePhone.isOnBusy() == null)
          throw new EncodeException(message, "Das Attribut ONBUSY wurde nicht definiert");

        generator.name(UPDATEPHONE.ONBUSY).value(updatePhone.isOnBusy());
        generator.endObject(); // DATASET End
      }
      else if (message.getHeader() == ERROR) {

      }
      generator.endObject();
      generator.flush();
      generator.close();
      return stringwriter.toString();

    }

    return EMPTY;
  }



  private String encodeUPDATEPORT(MESSAGE message) throws EncodeException, IOException {
    if (!(message instanceof UPDATEPORT)) return EMPTY;
    UPDATEPORT updateport = (UPDATEPORT) message;
    if (Command.UPDATEPORT.equals(updateport.getCommand())) {
      StringWriter stringwriter = new StringWriter();
      Gson gson = gsonbuilder.create();
      JsonWriter generator = gson.newJsonWriter(stringwriter);
      generator.beginObject();
      generator.name(Protocol.HEADER).value(message.getHeader().toString());
      generator.name(Command.getCommandKey()).value(message.getCommand().name());
      if (REQUEST == message.getHeader()) {
        generator.name(message.getDataset()).beginObject();
        try {
          generator.name(UPDATEPORT.USERID).value(updateport.getUserid());
          generator.name(UPDATEPORT.IP).value(updateport.getIp());
          generator.name(UPDATEPORT.PORT).value(updateport.getPort());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde ein Attribut nicht definiert.");
        }
        generator.endObject();
      }
      else if (RESPONSE.equals(message.getHeader())) {
        generator.name(message.getDataset()).beginObject();
        try {
          generator.name(UPDATEPORT.USERID).value(updateport.getUserid());
          generator.name(UPDATEPORT.IP).value(updateport.getIp());
          generator.name(UPDATEPORT.PORT).value(updateport.getPort());
        }
        catch (NullPointerException e) {
          throw new EncodeException(message, "Im Datensatz wurde ein Attribut nicht definiert.");
        }
        generator.endObject();
      }
      generator.endObject();
      generator.flush();
      generator.close();
      return stringwriter.toString();
    }
    return EMPTY;
  }



  public String encodeUPDATEUSER(MESSAGE message) throws EncodeException, IOException {
    if (!(message instanceof UPDATEUSER)) return EMPTY;
    UPDATEUSER updateuser = (UPDATEUSER) message;
    if (message.getCommand().equals(Command.UPDATEUSER)) {
      StringWriter stringwriter = new StringWriter();
      Gson gson = gsonbuilder.create();
      JsonWriter generator = gson.newJsonWriter(stringwriter);
      generator.beginObject();
      generator.name(Protocol.HEADER).value(message.getHeader().toString());
      generator.name(Command.getCommandKey()).value(message.getCommand().name());
      if (message.getHeader().equals(REQUEST)) {

        generator.name(updateuser.getDataset()).beginObject();

        if (updateuser.getIdentity() == null) {
          log.fatal("UPDATEUSER | REQUEST, das Attribut IDENTITY wurde nicht gesetzt.");
          throw new EncodeException(
              message, "UPDATEUSER | REQUEST, das Attribut IDENTITY wurde nicht gesetzt."
          );
        }
        generator.name(UPDATEUSER.IDENTITY).value(updateuser.getIdentity());

        if (updateuser.getForegroundColor() == null) {
          log.fatal("UPDATEUSER | REQUEST, das Attribut FOREGROUND_COLOR wurde nicht gesetzt.");
          throw new EncodeException(
              message, "UPDATEUSER | REQUEST, das Attribut FOREGROUND_COLOR wurde nicht gesetzt."
          );
        }
        generator.name(USER.FOREGROUND_COLOR).value(updateuser.getForegroundColor());

        if (updateuser.getBackgroundColor() == null) {
          log.fatal("UPDATEUSER | REQUEST, das Attribut BACKGROUND_COLOR wurde nicht gesetzt.");
          throw new EncodeException(
              message, "UPDATEUSER | REQUEST, das Attribut BACKGROUND_COLOR wurde nicht gesetzt."
          );
        }
        generator.name(UPDATEUSER.BACKGROUND_COLOR).value(updateuser.getBackgroundColor());

        if (updateuser.getNickname() == null) {
          log.fatal("UPDATEUSER | REQUEST, das Attribut NICKNAME wurde nicht gesetzt.");
          throw new EncodeException(
              message, "UPDATEUSER | REQUEST, das Attribut NICKNAME wurde nicht gesetzt."
          );
        }
        generator.name(UPDATEUSER.NICKNAME).value(updateuser.getNickname());

        generator.endObject();
        generator.endObject();
        generator.flush();
        generator.close();
        return stringwriter.toString();
      }
      else if (message.getHeader() == CONFIRM) {

        generator.name(updateuser.getDataset()).beginObject();

        if (updateuser.getIdentity() == null) {
          log.fatal("UPDATEUSER | CONFIRM, das Attribut IDENTITY wurde nicht gesetzt.");
          throw new EncodeException(message, "Protokollfehler: das Attribut IDENTITY wurde nicht gesetzt.");
        }
        generator.name(UPDATEUSER.IDENTITY).value(updateuser.getIdentity());

        if (updateuser.getForegroundColor() == null) {
          log.fatal("UPDATEUSER | CONFIRM, das Attribut FOREGROUND_COLOR wurde nicht gesetzt.");
          throw new EncodeException(
              message, "UPDATEUSER | CONFIRM, das Attribut FOREGROUND_COLOR wurde nicht gesetzt."
          );
        }
        generator.name(UPDATEUSER.FOREGROUND_COLOR).value(updateuser.getForegroundColor());

        if (updateuser.getBackgroundColor() == null) {
          log.fatal("UPDATEUSER | CONFIRM, das Attribut BACKGROUND_COLOR wurde nicht gesetzt.");
          throw new EncodeException(
              message, "UPDATEUSER | CONFIRM, das Attribut BACKGROUND_COLOR wurde nicht gesetzt."
          );
        }
        generator.name(UPDATEUSER.BACKGROUND_COLOR).value(updateuser.getBackgroundColor());

        if (updateuser.getNickname() == null) {
          log.fatal("UPDATEUSER | CONFIRM, das Attribut NICKNAME wurde nicht gesetzt.");
          throw new EncodeException(
              message, "UPDATEUSER | CONFIRM, das Attribut NICKNAME wurde nicht gesetzt."
          );
        }
        generator.name(UPDATEUSER.NICKNAME).value(updateuser.getNickname());

        generator.endObject();
        generator.endObject();
        generator.flush();
        generator.close();
        return stringwriter.toString();
      }
      else if (message.getHeader() == ERROR) {
        if (updateuser.getDataset() == null) {
          log.fatal("UPDATEUSER | ERROR, das Attribut DATASET wurde nicht gesetzt.");
          throw new EncodeException(message, "UPDATEUSER | ERROR, das Attribut DATASET wurde nicht gesetzt.");
        }
        generator.name(updateuser.getDataset()).beginObject();

        if (updateuser.getText() == null) {
          log.fatal("UPDATEUSER | ERROR, das Attribut TEXT wurde nicht gesetzt.");
          throw new EncodeException(message, "UPDATEUSER | ERROR, das Attribut TEXT wurde nicht gesetzt.");
        }
        generator.name(UPDATEUSER.TEXT).value(updateuser.getText());

        generator.endObject();
        generator.endObject();
        generator.flush();
        generator.close();
        return stringwriter.toString();
      }
    }
    return EMPTY;
  }



  public String encodeUPLOADFILES(MESSAGE message) throws EncodeException, IOException {
    if (!(message instanceof UPLOADFILES)) return EMPTY;
    UPLOADFILES uploadfiles = (UPLOADFILES) message;
    if (Command.UPLOADFILES.equals(uploadfiles.getCommand())) {
      StringWriter stringwriter = new StringWriter();
      Gson gson = gsonbuilder.create();
      JsonWriter generator = gson.newJsonWriter(stringwriter);
      generator.beginObject();
      generator.name(Protocol.HEADER).value(message.getHeader().toString());
      generator.name(Command.getCommandKey()).value(message.getCommand().name());
      if (REQUEST == message.getHeader()) {
        generator.name(message.getDataset()).beginObject();
        generator.name(UPLOADFILES.START).value(uploadfiles.hasStarted());
        generator.name(UPLOADFILES.USERID).value(uploadfiles.getUserid());
        generator.name(UPLOADFILES.IP).value(uploadfiles.getIp());
        if (uploadfiles.getPort() <= 0)
          throw new EncodeException(message, "ung�ltiger Port ---> " + String.valueOf(uploadfiles.getPort()));
        if (uploadfiles.getPort() > 65535)
          throw new EncodeException(message, "ung�ltiger Port ---> " + String.valueOf(uploadfiles.getPort()));
        generator.name(UPLOADFILES.PORT).value(uploadfiles.getPort());
        generator.name(UPLOADFILES.UPLOADFILES).beginArray();
        Uploadfile[] uploadfile = uploadfiles.getUploadfile();
        if (uploadfile != null) {
          for (Uploadfile tmp : uploadfile) {
            generator.beginObject();
            generator.name(Uploadfile.FILENAME).value(tmp.getFilename());
            generator.name(Uploadfile.FILESIZE).value(tmp.getFilesize());
            generator.endObject();
          }
        }
        generator.endArray();
        generator.endObject();
      }
      else if (RESPONSE.equals(message.getHeader())) {
        generator.name(message.getDataset()).beginObject();
        generator.name(UPLOADFILES.START).value(uploadfiles.hasStarted());
        generator.name(UPLOADFILES.USERID).value(uploadfiles.getUserid());
        generator.name(UPLOADFILES.IP).value(uploadfiles.getIp());
        generator.endObject();
      }
      generator.endObject();
      generator.flush();
      generator.close();
      return stringwriter.toString();
    }
    return EMPTY;
  }



  public String encodeUSERONLINELIST(MESSAGE message) throws EncodeException, IOException {
    if (!(message instanceof USERONLINELIST)) return EMPTY;
    USERONLINELIST userlist = (USERONLINELIST) message;
    if (Command.USERONLINELIST.equals(userlist.getCommand())) {
      StringWriter stringwriter = new StringWriter();
      Gson gson = gsonbuilder.create();
      JsonWriter generator = gson.newJsonWriter(stringwriter);
      generator.beginObject();
      generator.name(Protocol.HEADER).value(message.getHeader().toString());
      generator.name(Command.getCommandKey()).value(message.getCommand().name());
      if (RESPONSE.equals(message.getHeader())) {
        generator.name(message.getDataset()).beginObject();
        generator.name(USERONLINELIST.USERID).value(userlist.getUserid());
        generator.name(USERONLINELIST.USERONLINELIST).beginArray();
        for (UserOnline online : userlist.getUserOnline()) {
          generator.beginObject();
          generator.name(UserOnline.IP).value(online.getIp());
          generator.name(UserOnline.USERID).value(online.getUserid());
          generator.name(UserOnline.NICKNAME).value(online.getNickname());
          generator.endObject();
        }
        generator.endArray();
        generator.endObject();
      }
      else if (REQUEST == message.getHeader()) {
        generator.name(message.getDataset()).beginObject();
        generator.name(USERONLINELIST.USERID).value(userlist.getUserid());
        generator.endObject();
      }
      generator.endObject();
      generator.flush();
      generator.close();
      return stringwriter.toString();
    }
    return EMPTY;
  }



  public String encodeUSRLOGIN(MESSAGE message) throws EncodeException, IOException {
    if (!(message instanceof USRLOGIN)) return EMPTY;
    USRLOGIN usrlogin = (USRLOGIN) message;
    if (message.getCommand().name().equals(Command.USRLOGIN.name())) {
      StringWriter stringwriter = new StringWriter();
      Gson gson = gsonbuilder.create();
      JsonWriter generator = gson.newJsonWriter(stringwriter);
      generator.beginObject();
      generator.name(Protocol.HEADER).value(message.getHeader().toString());
      generator.name(Command.getCommandKey()).value(message.getCommand().name());
      if (message.getHeader().equals(REQUEST)) {
        // USRLOGIN REQUEST
        generator.name(usrlogin.getDataset()).beginObject();
        if (usrlogin.getAgent() == null) {
          log.fatal("USRLOGIN | REQUEST, das Attribut AGENT wurde nicht definiert.");
          throw new EncodeException(message, "USRLOGIN | REQUEST, das Attribut AGENT wurde nicht definiert.");

        }
        generator.name(USRLOGIN.AGENT).value(usrlogin.getAgent().toString());

        if (usrlogin.getIdentity() == null) {
          log.fatal("USRLOGIN | REQUEST, das Attribut IDENTITY wurde nicht definiert.");
          throw new EncodeException(
              message, "USRLOGIN | REQUEST, das Attribut IDENTITY wurde nicht definiert."
          );
        }
        generator.name(USRLOGIN.IDENTITY).value(usrlogin.getIdentity());

        generator.endObject(); // DATASET End
      }
      else if (message.getHeader() == CONFIRM) {
        generator.name(usrlogin.getDataset()).beginObject();

        if (usrlogin.getSession() == null) {
          log.fatal("USRLOGIN | CONFIRM, das Attribut SESSION wurde nicht definiert.");
          throw new EncodeException(
              message, "USRLOGIN | CONFIRM, das Attribut SESSION wurde nicht definiert."
          );
        }
        generator.name(USRLOGIN.SESSION).value(usrlogin.getSession());

        if (usrlogin.getIdentity() == null) {
          log.fatal("USRLOGIN | CONFIRM, das Attribut IDENTITY wurde nicht gesetzt.");
          throw new EncodeException(
              message, "USRLOGIN | CONFIRM, das Attribut IDENTITY wurde nicht gesetzt."
          );
        }
        generator.name(USRLOGIN.IDENTITY).value(usrlogin.getIdentity());

        if (usrlogin.getForegroundColor() == null) {
          log.fatal("USRLOGIN | CONFIRM, das Attribut FOREGROUND_COLOR wurde nicht definiert.");
          throw new EncodeException(
              message, "USRLOGIN | CONFIRM, das Attribut FOREGROUND_COLOR wurde nicht definiert."
          );
        }
        generator.name(USRLOGIN.FOREGROUND_COLOR).value(usrlogin.getForegroundColor());

        if (usrlogin.getBackgroundColor() == null) {
          log.fatal("USRLOGIN | CONFIRM, das Attribut BACKGROUND_COLOR wurde nicht definiert.");
          throw new EncodeException(
              message, "USRLOGIN | CONFIRM, das Attribut BACKGROUND_COLOR wurde nicht definiert."
          );
        }
        generator.name(USRLOGIN.BACKGROUND_COLOR).value(usrlogin.getBackgroundColor());

        if (usrlogin.getOncall() == null) {
          log.fatal("USRLOGIN | CONFIRM, das Attribut ONCALL wurde nicht gesetzt.");
          throw new EncodeException(message, "USRLOGIN | CONFIRM, das Attribut ONCALL wurde nicht gesetzt.");
        }
        generator.name(USRLOGIN.ONCALL).value(usrlogin.getOncall());

        if (usrlogin.getVolume() == null) {
          log.fatal("USRLOGIN | CONFIRM, das Attribut VOLUME wurde nicht gesetzt.");
          throw new EncodeException(message, "USRLOGIN | CONFIRM, das Attribut VOLUME wurde nicht gesetzt.");
        }
        generator.name(USRLOGIN.VOLUME).value(usrlogin.getVolume());

        if (usrlogin.getNickname() == null) {
          log.fatal("USRLOGIN | CONFIRM, das Attribut NICKNAME wurde nicht definiert.");
          throw new EncodeException(
              message, "USRLOGIN | CONFIRM, das Attribut NICKNAME wurde nicht definiert."
          );
        }
        generator.name(USRLOGIN.NICKNAME).value(usrlogin.getNickname());

        generator.endObject(); // DATASET End
      }
      else if (message.getHeader() == ERROR) {
        generator.name(usrlogin.getDataset()).beginObject();

        if (usrlogin.getMultilingualkey() != null)
          generator.name(USRLOGIN.MULTILINGUALKEY).value(usrlogin.getMultilingualkey().value());
        if (usrlogin.getText() == null) {
          log.fatal("USRLOGIN | ERROR, das Attribut TEXT wurde nicht gesetzt.");
          throw new EncodeException(message, "USRLOGIN | ERROR, das Attribut TEXT wurde nicht gesetzt.");
        }
        generator.name(USRLOGIN.TEXT).value(usrlogin.getText());

        generator.endObject();
      }
      generator.endObject();
      generator.flush();
      generator.close();
      return stringwriter.toString();
    }
    return EMPTY;
  }



  @Override
  public void init(EndpointConfig endpointconfig) {

  }

}
