/**
 *  Copyright © 2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  Previously released under Apache License 2.0; now licensed under MPL 2.0.
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.protocol;

/*
 * Copyright © 2025 Luis Andrés Lange <https://javacomm.net>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import jakarta.websocket.DecodeException;
import jakarta.websocket.Decoder;
import jakarta.websocket.EndpointConfig;
import java.nio.ByteBuffer;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;



public class PassThroughBinaryDecoder implements Decoder.Binary<ByteBuffer> {
  private static final Logger log = LogManager.getLogger(PassThroughBinaryDecoder.class);

  @Override
  public void init(EndpointConfig config) {
    log.info("PassThroughBinaryDecoder.init called, config={}", config);
  }



  @Override
  public ByteBuffer decode(ByteBuffer bytes) throws DecodeException {
    try {
      ByteBuffer copy = ByteBuffer.allocate(bytes.remaining());
      copy.put(bytes);
      copy.flip();
      log.debug("PassThroughBinaryDecoder.decode: {} bytes", copy.remaining());
      return copy;
    }
    catch (Throwable t) {
      log.error("PassThroughBinaryDecoder.decode threw: {}", t.getMessage(), t);
      throw new DecodeException(bytes, "decode failed", t);
    }
  }



  @Override
  public boolean willDecode(ByteBuffer bytes) {
    boolean ok = bytes != null && bytes.remaining() > 0;
    log.debug("PassThroughBinaryDecoder.willDecode -> {}", ok);
    return ok;
  }



  @Override
  public void destroy() {
    log.info("PassThroughBinaryDecoder.destroy");
  }
}
