/**
 *  Copyright © 2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  Previously released under Apache License 2.0; now licensed under MPL 2.0.
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.protocol;

import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import java.lang.reflect.Type;
import net.javacomm.multilingual.schema.KEY;



public class UsrloginDeserializer implements JsonDeserializer<USRLOGIN> {

  public UsrloginDeserializer() {}



  @Override
  public USRLOGIN deserialize(JsonElement json, Type type, JsonDeserializationContext context)
      throws JsonParseException {

    USRLOGIN login = new USRLOGIN();
    JsonObject jsonobject = json.getAsJsonObject();
    HEADER header = HEADER.toHeader(jsonobject.getAsJsonPrimitive(Protocol.HEADER).getAsString());
    JsonObject dataset = jsonobject.getAsJsonObject(Protocol.DATASET);

    try {
      String session = dataset.get(USRLOGIN.SESSION).getAsString();
      login.setSession(session);
    }
    catch (NullPointerException e) {}

    try {
      Agent agent = Agent.valueOf(dataset.get(USRLOGIN.AGENT).getAsString());
      login.setAgent(agent);
    }
    catch (NullPointerException e) {}

    try {
      String identity = dataset.get(USRLOGIN.IDENTITY).getAsString();
      login.setIdentity(identity);
    }
    catch (NullPointerException e) {}

    try {
      String nickname = dataset.get(USRLOGIN.NICKNAME).getAsString();
      login.setNickname(nickname);
    }
    catch (NullPointerException e) {}

    try {
      Integer fgColor = dataset.get(USRLOGIN.FOREGROUND_COLOR).getAsInt();
      login.setForegroundColor(fgColor);
    }
    catch (NullPointerException e) {}

    try {
      Integer bgColor = dataset.get(USRLOGIN.BACKGROUND_COLOR).getAsInt();
      login.setBackgroundColor(bgColor);
    }
    catch (NullPointerException e) {}

    try {
      Integer volume = dataset.get(USRLOGIN.VOLUME).getAsInt();
      login.setVolume(volume);
    }
    catch (NullPointerException e) {}

    try {
      Boolean oncall = dataset.get(USRLOGIN.ONCALL).getAsBoolean();
      login.setOncall(oncall);
    }
    catch (NullPointerException e) {}

    try {
      KEY multilingualkey = KEY.fromValue(dataset.getAsJsonPrimitive(SIGNIN.MULTILINGUALKEY).getAsString());
      login.setMultilingualkey(multilingualkey);
    }
    catch (NullPointerException e) {}
    try {
      String text = dataset.get(USRLOGIN.TEXT).getAsString();
      login.setText(text);
    }
    catch (NullPointerException e) {}

    login.setHeader(header);
    login.setCommand(Command.USRLOGIN);
    login.setDataset(Protocol.DATASET);
    return login;
  }

}
