/**
 * This software is released as part of the Pumpernickel project.
 * 
 * All com.pump resources in the Pumpernickel project are distributed under the
 * MIT License:
 * https://github.com/mickleness/pumpernickel/raw/master/License.txt
 * 
 * More information about the Pumpernickel project is available here:
 * https://mickleness.github.io/pumpernickel/
 */
package com.pump.icon;

import java.awt.Color;
import java.awt.Component;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.geom.AffineTransform;
import java.awt.geom.Ellipse2D;
import java.awt.geom.GeneralPath;

import javax.swing.Icon;

import com.pump.blog.ResourceSample;

/**
 * A silhouette icon of a gear.
 *
 * 
 * <!-- ======== START OF AUTOGENERATED SAMPLES ======== -->
 * <p>
 * Here are some samples:
 * <table summary="Resource&#160;Samples&#160;for&#160;com.pump.swing.resources.GearIcon">
 * <tr>
 * <td></td>
 * <td><img src=
 * "https://github.com/mickleness/pumpernickel/raw/master/resources/samples/GearIcon/12x12.png"
 * alt=
 * "new&#160;com.pump.swing.resources.GearIcon(&#160;12,&#160;java.awt.Color.darkGray)"
 * ></td>
 * <td><img src=
 * "https://github.com/mickleness/pumpernickel/raw/master/resources/samples/GearIcon/24x24.png"
 * alt=
 * "new&#160;com.pump.swing.resources.GearIcon(&#160;24,&#160;java.awt.Color.darkGray)"
 * ></td>
 * </tr>
 * <tr>
 * <td></td>
 * <td>12x12</td>
 * <td>24x24</td>
 * </tr>
 * <tr>
 * </tr>
 * </table>
 * <!-- ======== END OF AUTOGENERATED SAMPLES ======== -->
 */
@ResourceSample(sample = {
		"new com.pump.swing.resources.GearIcon( 12, java.awt.Color.darkGray)",
		"new com.pump.swing.resources.GearIcon( 24, java.awt.Color.darkGray)" }, names = {
		"12x12", "24x24" })
public class GearIcon implements Icon {
	final int diameter;
	final Color color;

	GeneralPath gear = new GeneralPath();

	public GearIcon(int diameter, Color color) {
		this.diameter = diameter;
		if (color == null) {
			this.color = new Color(0x484848);
		} else {
			this.color = color;
		}

		float center = diameter / 2f;
		float r1 = diameter / 2f;
		float r2 = r1 * 2f / 3f;
		float r3 = r1 * 1f / 5f;

		gear.append(new Ellipse2D.Float(center - r3, center - r3, 2 * r3,
				2 * r3).getPathIterator(null), false);
		gear.closePath();

		boolean empty = true;
		for (int notch = 15; notch >= 0; notch--) {
			double angle = Math.PI * notch / 8.0;

			double firstR, lastR;

			if (notch % 2 == 0) {
				firstR = r1;
				lastR = r2;
			} else {
				firstR = r2;
				lastR = r1;
			}

			float x = (float) (center - firstR * Math.cos(angle));
			float y = (float) (center - firstR * Math.sin(angle));
			if (empty) {
				gear.moveTo(x, y);
				empty = false;
			} else {
				gear.lineTo(x, y);
			}
			x = (float) (center - lastR * Math.cos(angle));
			y = (float) (center - lastR * Math.sin(angle));
			gear.lineTo(x, y);
		}
		gear.transform(AffineTransform.getRotateInstance(2 * Math.PI / 32,
				center, center));
	}

	public int getIconHeight() {
		return diameter;
	}

	public int getIconWidth() {
		return diameter;
	}

	public void paintIcon(Component c, Graphics g0, int x, int y) {
		Graphics2D g = (Graphics2D) g0;
		g.setColor(color);
		g.translate(x, y);
		g.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
				RenderingHints.VALUE_ANTIALIAS_ON);
		g.fill(gear);
		g.translate(-x, -y);
	}
}