/*
 * Copyright © 2022 Luis Andrés Lange <http://javacomm.net>
 *
 * The MIT Licence
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

package net.javacomm.multilingual;

import jakarta.xml.bind.JAXBException;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import net.javacomm.multilingual.schema.COMMAND;
import net.javacomm.multilingual.schema.COMMANDS;
import net.javacomm.multilingual.schema.KEY;
import net.javacomm.multilingual.schema.MULTILINGUAL;



public class Multilingual {

  private final static Logger log = LogManager.getLogger(Multilingual.class);
  private static Multilingual instance;
  private MULTILINGUAL multilingual;

  static {
    try {
      instance = new Multilingual();
    }
    catch (JAXBException e) {
      log.error(e.getMessage(), e);
    }
  }

  private Multilingual() throws JAXBException {
    MultilingualMapper mapper = MultilingualMapper.getInstance();
    multilingual = mapper.read();
  }



  /**
   * Multilingual ist ein Singleton, weil es sich während des Programmablaufs
   * nicht verändern kann.
   *
   * @return
   */
  public synchronized static Multilingual getInstance() {
    return instance;
  }



  /**
   * Hole alle Übersetzungen für das COMMAND mit dem Schlüssel 'key'.
   *
   * @param key
   *            ein Suchschlüssel
   * @return alle Übersetzungen
   */
  public COMMAND getCommand(KEY key) {
    if (key == null) throw new MultilingualException("key ist null");
    COMMANDS commands = getCommands();
    if (commands == null)
      throw new MultilingualException("In der Konfigurtionsdatei gibt es keine COMMANDS.");
    for (COMMAND command : commands.getCOMMAND()) {
      if (command.getKEY() == key) return command;
    }
    log.fatal("Für den Schlüssel '" + key.toString() + "' gibt es kein COMMAND");
    throw new MultilingualException("Für den Schlüssel '" + key.toString() + "' gibt es kein COMMAND");
  }



  /**
   * Hole alle COMMANDS.
   * 
   * @return ab dem Knoten COMMANDS
   */
  public COMMANDS getCommands() {
    return multilingual.getCOMMANDS();
  }



  public MULTILINGUAL getRoot() {
    return multilingual;
  }

}
