/*
 * Copyright © 2022 Luis Andrés Lange <http://javacomm.net>
 *
 * The MIT Licence
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
package net.javacomm.multilingual;

import jakarta.xml.bind.JAXBContext;
import jakarta.xml.bind.JAXBElement;
import jakarta.xml.bind.JAXBException;
import jakarta.xml.bind.Unmarshaller;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import javax.xml.XMLConstants;
import javax.xml.transform.Source;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.xml.sax.SAXException;
import net.javacomm.multilingual.schema.MULTILINGUAL;



/**
 * Über den MultilingualMapper werden die Sprachübersetzungen aus einer
 * Konfigurationsdatei ausgelesen.
 *
 *
 * @author llange
 *
 */
public class MultilingualMapper {

  private final Logger log = LogManager.getLogger(MultilingualMapper.class);
  private static MultilingualMapper instance;
  public final static String SCHEMAFILE = "/net/javacomm/multilingual/schema/multilingual.xsd";
  public final static String MAPPERFILE = "/net/javacomm/multilingual/schema/multilingual.xml";
  private final String SCHEMA_PATH = "net.javacomm.multilingual.schema";
  private Schema schema;

  private MultilingualMapper() {
    log.info("SCHEMAFILE -----> " + SCHEMAFILE);
    log.info("MAPPERFILE -----> " + MAPPERFILE);
    Class<? extends MultilingualMapper> clazz = getClass();

    SchemaFactory schemafactory = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
    URL url = clazz.getResource(SCHEMAFILE);
//    url = clazz.getClassLoader().getResource(SCHEMAFILE);
    try {
      schema = schemafactory.newSchema(url);
    }
    catch (SAXException e) {
      log.error(e.getMessage(), e.getCause());
    }
  }

  static {
    instance = new MultilingualMapper();
  }

  public static synchronized MultilingualMapper getInstance() {
    return instance;
  }



  /**
   *
   * Dieses Schema wurde zum Lesen und Schreiben verwendet.
   *
   * @return das Schema
   */
  public Schema getSchema() {
    return schema;
  }



  /**
   * Eine Konfigurationsdatei wird gelesen.
   *
   *
   * @return der Wurzelknoten
   * @throws JAXBException
   */
  public MULTILINGUAL read() throws JAXBException {
    MULTILINGUAL root = null;
    URL url = getClass().getResource(MAPPERFILE);
    try(InputStream sqlmapper_xml = url.openStream()) {
      JAXBContext jc = JAXBContext.newInstance(SCHEMA_PATH);
      Unmarshaller unmarshaller = jc.createUnmarshaller();
      unmarshaller.setSchema(getSchema());
      Source source = new StreamSource(sqlmapper_xml);
      JAXBElement<MULTILINGUAL> jaxbCollection = unmarshaller.unmarshal(source, MULTILINGUAL.class);
      root = jaxbCollection.getValue();
    }
    catch (IOException e) {
      log.warn(e.getMessage(), e);
    }
    return root;
  }


}
