/**
 *  Copyright © 2023-2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.client.administer;

import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.event.ActionListener;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import javax.swing.JPanel;
import javax.swing.JSeparator;
import javax.swing.JSpinner;
import javax.swing.SpinnerNumberModel;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import net.javacomm.multilingual.Babelfish;
import net.javacomm.multilingual.MultilingualButton;
import net.javacomm.multilingual.MultilingualLabel;
import net.javacomm.multilingual.schema.ISO639;
import net.javacomm.multilingual.schema.KEY;
import net.javacomm.share.Constants;
import net.javacomm.window.manager.Control;



@SuppressWarnings("serial")
public class JChatModul extends JPanel implements Babelfish {

  private final static Logger log = LogManager.getLogger(JChatModul.class);

  private MultilingualLabel labelNachWieVielenStunden = new MultilingualLabel(KEY.LABEL_CHAT_STUNDEN);
  private JSeparator separator = new JSeparator();
  private MultilingualButton buttonSchliessen = new MultilingualButton(KEY.BUTTON_SCHLIESSEN);
  private MultilingualButton buttonSpeichern = new MultilingualButton(KEY.BUTTON_SPEICHERN);
  private GridLayout grid = new GridLayout(1, 2, 12, 12);
  private JPanel buttonpane = new JPanel(grid);
  private PropertyChangeSupport changes = new PropertyChangeSupport(this);
  private GridBagLayout gridbag = new GridBagLayout();
  private GridBagConstraints con = new GridBagConstraints();
  private JSpinner spinner = new JSpinner();
  private SpinnerNumberModel model = new SpinnerNumberModel();
  private JSpinner.NumberEditor editorSpinner;

  public JChatModul() {
    setLayout(gridbag);

    con.gridx = 0;
    con.gridy = 0;
    con.gridwidth = 1;
    con.gridheight = 1;
    con.fill = GridBagConstraints.HORIZONTAL;
    con.weightx = 0.0;
    con.weighty = 0.0;
    con.insets = new Insets(0, 0, 0, 0);
    con.anchor = GridBagConstraints.CENTER;
    gridbag.setConstraints(labelNachWieVielenStunden, con);
    add(labelNachWieVielenStunden);

    spinner.setModel(model);
    model.setMinimum(Constants.MIN_DURATION_TIME);
    model.setStepSize(24);
    model.setMaximum(720);
    model.setValue(24);

    con.gridx++;
//    con.gridy++;
    con.gridwidth = 1;
    con.gridheight = 1;
    con.fill = GridBagConstraints.NONE;
    con.weightx = 0.0;
    con.weighty = 0.0;
    con.insets = new Insets(0, 6, 0, 0);
    con.anchor = GridBagConstraints.CENTER;
    gridbag.setConstraints(spinner, con);
    add(spinner);
    editorSpinner = (JSpinner.NumberEditor) spinner.getEditor();
    editorSpinner.setPreferredSize(new Dimension(40, 32));

    con.gridx = 0;
    con.gridy++;
    con.gridwidth = 2;
    con.gridheight = 1;
    con.fill = GridBagConstraints.HORIZONTAL;
    con.weightx = 0.0;
    con.weighty = 0.0;
    con.insets = new Insets(24, 0, 0, 0);
    con.anchor = GridBagConstraints.CENTER;
    gridbag.setConstraints(separator, con);
    add(separator);

    buttonpane.add(buttonSpeichern);
    buttonpane.add(buttonSchliessen);
    con.gridx = 0;
    con.gridy++;
    con.gridwidth = 2;
    con.gridheight = 1;
    con.fill = GridBagConstraints.NONE;
    con.weightx = 0.0;
    con.weighty = 0.0;
    con.insets = new Insets(12, 0, 0, 0);
    con.anchor = GridBagConstraints.EAST;
    gridbag.setConstraints(buttonpane, con);
    add(buttonpane);

    buttonSchliessen.addActionListener((event) -> {
      changes.firePropertyChange(JChatModul.class.getName(), Control.NULL, Control.CLOSE);
    });

    buttonSpeichern.addActionListener((event) -> {
      changes.firePropertyChange(JChatModul.class.getName(), Control.NULL, Control.SAVE);
    });

  }



  public void addChatmodulListener(PropertyChangeListener listener) {
    changes.addPropertyChangeListener(listener);
  }



  public void removeChatmodulListener(PropertyChangeListener listener) {
    changes.removePropertyChangeListener(listener);
  }



  public void removeAllListener() {
    for (PropertyChangeListener pl : changes.getPropertyChangeListeners()) {
      removeChatmodulListener(pl);
    }
    for (ActionListener al : buttonSchliessen.getActionListeners()) {
      buttonSchliessen.removeActionListener(al);
    }
    for (ActionListener al : buttonSpeichern.getActionListeners()) {
      buttonSpeichern.removeActionListener(al);
    }
  }



  @Override
  public void setLanguage(ISO639 code) {
    labelNachWieVielenStunden.setLanguage(code);
    buttonSchliessen.setLanguage(code);
    buttonSpeichern.setLanguage(code);
  }



  /**
   * Nach n-Stunden wird die Nachricht gelöscht.
   * 
   * @param hours
   *              Stunden
   */
  public void setHours(int hours) {
    model.setValue(hours);
  }



  /**
   * Nach wie vielen Stunden soll eine Nachricht spätestens gelöscht werden?
   * 
   * @return Anzahl Stunden
   */
  public int getHours() {
    return model.getValue() instanceof Integer hours ? hours : Constants.MIN_DURATION_TIME;
  }

}
