/**
 *  Copyright © 2021-2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.client.administer;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.event.ActionListener;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import javax.swing.ImageIcon;
import javax.swing.JPanel;
import javax.swing.JSeparator;
import javax.swing.JTextField;
import org.nexuswob.gui.NimbusPasswordField;
import net.javacomm.client.resource.Resource;
import net.javacomm.multilingual.Babelfish;
import net.javacomm.multilingual.MultilingualButton;
import net.javacomm.multilingual.MultilingualLabel;
import net.javacomm.multilingual.MultilingualString;
import net.javacomm.multilingual.schema.ISO639;
import net.javacomm.multilingual.schema.KEY;
import net.javacomm.window.manager.Control;



@SuppressWarnings("serial")
public class JMailserver extends JPanel implements Babelfish {

  private final int textfield_len = 20;
  private GridBagLayout gridbag = new GridBagLayout();
  private GridBagConstraints con = new GridBagConstraints();
  private MultilingualLabel labelSmtpUser = new MultilingualLabel(KEY.LABEL_BENUTZERNAME_POSTAUSGANG);
  private JTextField textfieldSmtpUser = new JTextField(textfield_len);
  private MultilingualLabel labelSmtpHost = new MultilingualLabel(KEY.LABEL_POSTAUSGANGSERVER_MAILVERSAND);
  private JTextField textfieldSmtpHost = new JTextField(textfield_len);

  private MultilingualString ueberDenMailserver = new MultilingualString(KEY.STRING_UEBER_DEN_MAILSERVER);

  private MultilingualLabel labelPort = new MultilingualLabel(KEY.LABEL_PORT_AUSGANGSSERVER);
  private JTextField textfieldPort = new JTextField(4);

  private MultilingualLabel labelPasswort = new MultilingualLabel(KEY.LABEL_PASSWORT_POSTAUSGANGSERVER);
  private NimbusPasswordField textfieldPasswort;
  private MultilingualLabel labelAdministratorPostfach = new MultilingualLabel(
      KEY.LABEL_ADMINISTRATORPOSTFACH
  );
  private JTextField textfieldAdministratorPostfach = new JTextField(textfield_len);
  private JSeparator separator = new JSeparator();
  private MultilingualButton buttonSchliessen = new MultilingualButton(KEY.BUTTON_SCHLIESSEN);
  private MultilingualButton buttonTestmail = new MultilingualButton(KEY.BUTTON_TESTMAIL);
  private GridLayout grid = new GridLayout(1, 3, 12, 12);
  private JPanel buttonpane = new JPanel(grid);
  private PropertyChangeSupport changes = new PropertyChangeSupport(this);
  private final int textfield_left = 6;
  private final int margin_left = 12;
  private final int margin_right = 12;
  private final int margin_top = 12;
  private MultilingualLabel labelEinleitung = new MultilingualLabel();
  private FocusAdapter focusAdapter = new FocusAdapter() {
    @Override
    public void focusGained(FocusEvent event) {
      textfieldPort.setBackground(Color.WHITE);
      textfieldPort.setForeground(Color.BLACK);
    }
  };

  public JMailserver() {
    setLayout(gridbag);

    con.gridx = 0;
    con.gridy = 0;
    con.gridwidth = 2;
    con.gridheight = 1;
    con.fill = GridBagConstraints.NONE;
    con.weightx = 0.0;
    con.weighty = 0.0;
    con.insets = new Insets(margin_top, margin_left, 24, margin_right);
    con.anchor = GridBagConstraints.WEST;
    gridbag.setConstraints(labelEinleitung, con);
    add(labelEinleitung);

    con.gridx = 0;
    con.gridy = 1;
    con.gridwidth = 1;
    con.gridheight = 1;
    con.fill = GridBagConstraints.NONE;
    con.weightx = 0.0;
    con.weighty = 0.0;
    con.insets = new Insets(0, margin_left, 0, 0);
    con.anchor = GridBagConstraints.EAST;
    gridbag.setConstraints(labelSmtpUser, con);
    add(labelSmtpUser);

    con.gridx = 1;
    con.gridy = 1;
    con.gridwidth = 1;
    con.gridheight = 1;
    con.fill = GridBagConstraints.NONE;
    con.weightx = 0.0;
    con.weighty = 0.0;
    con.insets = new Insets(0, textfield_left, 0, margin_right);
    con.anchor = GridBagConstraints.WEST;
    gridbag.setConstraints(textfieldSmtpUser, con);
    add(textfieldSmtpUser);

    con.gridx = 0;
    con.gridy = 2;
    con.gridwidth = 1;
    con.gridheight = 1;
    con.fill = GridBagConstraints.NONE;
    con.weightx = 0.0;
    con.weighty = 0.0;
    con.insets = new Insets(12, margin_left, 0, 0);
    con.anchor = GridBagConstraints.EAST;
    gridbag.setConstraints(labelSmtpHost, con);
    add(labelSmtpHost);

    con.gridx = 1;
    con.gridy = 2;
    con.gridwidth = 1;
    con.gridheight = 1;
    con.fill = GridBagConstraints.NONE;
    con.weightx = 0.0;
    con.weighty = 0.0;
    con.insets = new Insets(12, textfield_left, 0, margin_right);
    con.anchor = GridBagConstraints.WEST;
    gridbag.setConstraints(textfieldSmtpHost, con);
    add(textfieldSmtpHost);

    con.gridx = 0;
    con.gridy = 3;
    con.gridwidth = 1;
    con.gridheight = 1;
    con.fill = GridBagConstraints.NONE;
    con.weightx = 0.0;
    con.weighty = 0.0;
    con.insets = new Insets(12, margin_left, 0, 0);
    con.anchor = GridBagConstraints.EAST;
    gridbag.setConstraints(labelPort, con);
    add(labelPort);

    con.gridx = 1;
    con.gridy = 3;
    con.gridwidth = 1;
    con.gridheight = 1;
    con.fill = GridBagConstraints.NONE;
    con.weightx = 0.0;
    con.weighty = 0.0;
    con.insets = new Insets(12, textfield_left, 0, margin_right);
    con.anchor = GridBagConstraints.WEST;
    gridbag.setConstraints(textfieldPort, con);
    add(textfieldPort);
    textfieldPort.addFocusListener(focusAdapter);

    con.gridx = 0;
    con.gridy = 4;
    con.gridwidth = 1;
    con.gridheight = 1;
    con.fill = GridBagConstraints.NONE;
    con.weightx = 0.0;
    con.weighty = 0.0;
    con.insets = new Insets(12, margin_left, 0, 0);
    con.anchor = GridBagConstraints.EAST;
    gridbag.setConstraints(labelPasswort, con);
    add(labelPasswort);

    con.gridx = 1;
    con.gridy = 4;
    con.gridwidth = 1;
    con.gridheight = 1;
    con.fill = GridBagConstraints.NONE;
    con.weightx = 0.0;
    con.weighty = 0.0;
    con.insets = new Insets(12, textfield_left, 0, margin_right);
    con.anchor = GridBagConstraints.WEST;

    ImageIcon eye = new ImageIcon(getClass().getResource(Resource.EYE));
    textfieldPasswort = new NimbusPasswordField(textfield_len, eye, 12);

    gridbag.setConstraints(textfieldPasswort, con);
    add(textfieldPasswort);

    con.gridx = 0;
    con.gridy = 5;
    con.gridwidth = 1;
    con.gridheight = 1;
    con.fill = GridBagConstraints.NONE;
    con.weightx = 0.0;
    con.weighty = 0.0;
    con.insets = new Insets(12, margin_left, 0, 0);
    con.anchor = GridBagConstraints.EAST;
    gridbag.setConstraints(labelAdministratorPostfach, con);
    add(labelAdministratorPostfach);

    con.gridx = 1;
    con.gridy = 5;
    con.gridwidth = 1;
    con.gridheight = 1;
    con.fill = GridBagConstraints.NONE;
    con.weightx = 0.0;
    con.weighty = 0.0;
    con.insets = new Insets(12, textfield_left, 0, margin_right);
    con.anchor = GridBagConstraints.WEST;
    gridbag.setConstraints(textfieldAdministratorPostfach, con);
    add(textfieldAdministratorPostfach);

    con.gridx = 0;
    con.gridy = 6;
    con.gridwidth = 2;
    con.gridheight = 1;
    con.fill = GridBagConstraints.HORIZONTAL;
    con.weightx = 0.0;
    con.weighty = 0.0;
    con.insets = new Insets(24, margin_left, 0, margin_right);
    con.anchor = GridBagConstraints.CENTER;
    gridbag.setConstraints(separator, con);
    add(separator);

    buttonpane.add(buttonTestmail);
    buttonpane.add(buttonSchliessen);
    con.gridx = 0;
    con.gridy = 7;
    con.gridwidth = 2;
    con.gridheight = 1;
    con.fill = GridBagConstraints.NONE;
    con.weightx = 0.0;
    con.weighty = 0.0;
    con.insets = new Insets(12, margin_left, 12, margin_right);
    con.anchor = GridBagConstraints.EAST;
    gridbag.setConstraints(buttonpane, con);
    add(buttonpane);

    buttonTestmail.addActionListener(event -> {
      changes.firePropertyChange(JMailserver.class.getName(), Control.NULL, Control.TESTMAIL);
    });

    buttonSchliessen.addActionListener(event -> {
      changes.firePropertyChange(JMailserver.class.getName(), Control.NULL, Control.CLOSE);
    });

  }



  public void addMailserverListener(PropertyChangeListener listener) {
    changes.addPropertyChangeListener(listener);
  }



  String getAdministratorPostfach() {
    return textfieldAdministratorPostfach.getText();
  }



  String getBenutzername() {
    return textfieldSmtpUser.getText();
  }



  String getHost() {
    return textfieldSmtpHost.getText();
  }



  char[] getPasswort() {
    return textfieldPasswort.getPassword();
  }



  /**
   * Der SMTP-Port wird zurückgegeben.
   * 
   * 
   * @return der Port
   * @throws StringException
   *                         der Rückgabewert ist ein String
   */
  int getPort() throws StringException {
    String port = textfieldPort.getText();
    Integer value;
    try {
      value = Integer.valueOf(port);
    }
    catch (NumberFormatException e) {
      textfieldPort.setBackground(Resource.JQUERY_RED);
      textfieldPort.setForeground(Color.WHITE);
      throw new StringException(port);
    }
    return value;
  }



  public void removeAllListener() {
    for (PropertyChangeListener listener : changes.getPropertyChangeListeners()) {
      removeMailserverListener(listener);
    }
    textfieldPort.removeFocusListener(focusAdapter);
    for (ActionListener al : buttonSchliessen.getActionListeners()) {
      buttonSchliessen.removeActionListener(al);
    }
    for (ActionListener al : buttonTestmail.getActionListeners()) {
      buttonTestmail.removeActionListener(al);
    }
  }



  public void removeMailserverListener(PropertyChangeListener listener) {
    changes.removePropertyChangeListener(listener);
  }



  /**
   * Legt die Postfachadresse fest.
   * 
   * @param value
   *              ein Postafach
   */
  public void setAdministratorPostfach(String value) {
    textfieldAdministratorPostfach.setText(value);
  }



  /**
   * Legt den Benutzernamen für den Mailserver fest.
   * 
   * @param value
   *              der Benutzername auf dem Postausgangsserver
   */
  public void setBenutzername(String value) {
    textfieldSmtpUser.setText(value);
  }



  /**
   * Der Hostname für den Mailserver.
   * 
   * @param value
   *              der Postausgangsserver
   */
  public void setHost(String value) {
    textfieldSmtpHost.setText(value);
  }



  @Override
  public void setLanguage(ISO639 code) {
    ueberDenMailserver.setLanguage(code);
    labelEinleitung.setText("<html>" + "<p>" + ueberDenMailserver.toString() + "</p>" + "</html>");
    labelPort.setLanguage(code);
    labelSmtpUser.setLanguage(code);
    labelSmtpHost.setLanguage(code);
    labelPasswort.setLanguage(code);
    labelAdministratorPostfach.setLanguage(code);
    buttonSchliessen.setLanguage(code);
    buttonTestmail.setLanguage(code);
  }



  void setPasswort(String passwort) {
    textfieldPasswort.setPassword(passwort);
  }



  public void setPort(int port) {
    textfieldPort.setText(String.valueOf(port));
  }

}
