/**
 *  Copyright © 2020-2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.client.base;

import java.util.ArrayList;
import java.util.List;



/**
 * Die Klasse parzelliert ein StringArray in mehrere Teillisten. Sie ist
 * nützlich für Netzwerkübertragungen, weil große Arrays die Puffergrößen von
 * Websockets übersteigen können.
 * 
 * @author llange
 *
 */
public class TextBuffer {

	public String[] text;
	private int bufferlen;
	private int gesamtanzahlChunks;
  private ArrayList<String> partial;
  private int aktuellerChunk = 0;
	
	/**
   * 
   * Ein String-Array wird in mehrere Teillisten zerlegt. Jede Teilliste hat die
   * maximale Größe {@code chunkLen}. Die Summe aller Teillisten ergibt wieder den
   * Inhalt des Arrays. GesamtanzahlChunks = nStrings.length / chunkLen. 30
   * Strings in einem Array können so auf 5 Strings in 6 Chunks gesplittet werden.
   * 
   * @param nStrings
   *                 das zu zerlegende String-Array
   * @param chunkLen
   *                 jede Teilliste wird auf diese maximale Größe beschränkt.
   *                 {@code chunkLen} ist nicht die Stringlänge, sondern die
   *                 Anzahl herausgebrochener Strings aus dem Ursprungarray.
   */
	public TextBuffer(String[] nStrings, int chunkLen) {
		if (chunkLen<1) throw new IllegalArgumentException("len=" + chunkLen);
    if (nStrings == null) throw new IllegalArgumentException("nStrings is null");
		text = nStrings;
		this.bufferlen = chunkLen;
		partial = new ArrayList<String>(chunkLen);
		if (nStrings.length == 0) {
			gesamtanzahlChunks = 0;
		}
		else if (nStrings.length <= chunkLen) {
			gesamtanzahlChunks = 1;
		}
		else if (nStrings.length > chunkLen) {
			gesamtanzahlChunks = nStrings.length / chunkLen;
			if ((nStrings.length % chunkLen)!=0) {
				gesamtanzahlChunks++;
			}
		}
		reset();
	}
	

	
	public void reset() {
		aktuellerChunk = 0;
	}
	
	
	
  /**
   * Ein Chunk wird aus dem Ursprungarray herausgebrochen und zurückgegeben.
   * Wurden alle Chunks gelesen, ist die Liste leer aber definiert. Die Anzahl der
   * Elemente in der Liste ist {@code readPartial().size()==0}. Die Methode kann
   * beliebig oft aufgerufen werden.
   * 
   * @return ein Chunk von Strings
   */
	public List<String> readPartial() {
		partial.clear();
		if (gesamtanzahlChunks==0) return partial;
		if (aktuellerChunk<gesamtanzahlChunks) {
			int start = aktuellerChunk*bufferlen;
			int end = start + bufferlen;
			for (int index=start; index<end && index<text.length; index++) {
				partial.add(text[index]);
			}
			aktuellerChunk++;
		}
		return partial;
	}
	
	
}
