/**
 *  Copyright © 2020-2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.client.chat;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyVetoException;
import javax.swing.AbstractAction;
import javax.swing.ImageIcon;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JSeparator;
import javax.swing.JTextField;
import javax.swing.KeyStroke;
import javax.swing.event.InternalFrameAdapter;
import javax.swing.event.InternalFrameEvent;
import org.nexuswob.gui.InputDocument;
import net.javacomm.client.resource.Resource;
import net.javacomm.multilingual.MultilingualButton;
import net.javacomm.multilingual.MultilingualLabel;
import net.javacomm.multilingual.schema.ISO639;
import net.javacomm.multilingual.schema.KEY;
import net.javacomm.window.manager.Control;
import net.javacomm.window.manager.WindowManagerInternalFrame;



public class JPausenraum extends WindowManagerInternalFrame {

  private static final long serialVersionUID = -8849895820719653247L;
  public final static int WIDTH = 540;
  public final static int HEIGHT = 280;
  private PropertyChangeSupport changes = new PropertyChangeSupport(this);
  private GridBagLayout gridbag = new GridBagLayout();
  private GridBagConstraints con = new GridBagConstraints();
  private MultilingualLabel labelRaum = new MultilingualLabel(KEY.LABEL_RAEUME);
  private int minutos;

  private JSeparator separator = new JSeparator();
  private MultilingualButton buttonRaumerstellen = new MultilingualButton(KEY.BUTTON_RAUM_ERSTELLEN);
  private MultilingualButton buttonSchliessen = new MultilingualButton(KEY.BUTTON_SCHLIESSEN);
  private JTextField textfieldRoom = new JTextField(Resource.FELD_LEN_ROOM);

  private MultilingualLabel labelPausenraum = new MultilingualLabel(KEY.LABEL_LICHT);
  private InternalFrameAdapter frameAdapter = new InternalFrameAdapter() {
    @Override
    public void internalFrameClosed(InternalFrameEvent event) {
      changes.firePropertyChange(JPausenraum.class.getName(), Control.NULL, Control.CLOSE);
    }
  };
  private JPanel dummy = new JPanel();
  private KeyAdapter raumAdapter = new KeyAdapter() {

    @Override
    public void keyReleased(KeyEvent event) {
      if (event.getKeyCode() == KeyEvent.VK_ENTER) return;
      if (textfieldRoom.getText().length() == 0) {
        disableButtonRaumErstellen();
        return;
      }
      buttonRaumerstellen.setBackground(Resource.JQUERY_GREEN);
      buttonRaumerstellen.setForeground(Color.WHITE);
    }
  };

  public JPausenraum() {
    init();
  }



  /**
   * Der Knopf Raum erstellen wird rot angezeigt.
   * 
   */
  public void disableButtonRaumErstellen() {
    buttonRaumerstellen.setBackground(Resource.JQUERY_RED);
    buttonRaumerstellen.setForeground(Color.WHITE);
  }



  public synchronized void addPausenroomListener(PropertyChangeListener l) {
    changes.addPropertyChangeListener(l);
  }



  @Override
  public Dimension getMinimumSize() {
    return new Dimension(WIDTH - 80, HEIGHT - 20);
  }



  public final String getRoom() {
    return textfieldRoom.getText();
  }



  @SuppressWarnings("serial")
  void init() {
    setOpaque(true);
    setLayout(gridbag);

    final int leftMargin = 12;
    final int rightMargin = 12;
    final int marginTop = 12;
    final int marginBottom = 12;

    con.gridx = 0;
    con.gridy = 0;
    con.gridwidth = 2;
    con.fill = GridBagConstraints.HORIZONTAL;
    con.weightx = 1.0;
    con.weighty = 0.0;
    con.insets = new Insets(marginTop, leftMargin, 0, rightMargin);
    con.anchor = GridBagConstraints.CENTER;
    gridbag.setConstraints(labelPausenraum, con);
    add(labelPausenraum);
    labelPausenraum.setIcon(new ImageIcon(getClass().getResource(Resource.PAUSENRAUM45)));

    con.gridx = 0;
    con.gridy = 1;
    con.gridwidth = 1;
    con.fill = GridBagConstraints.NONE;
    con.weightx = 0.0;
    con.weighty = 0.0;
    con.insets = new Insets(18, leftMargin, 0, 0);
    con.anchor = GridBagConstraints.WEST;
    gridbag.setConstraints(labelRaum, con);
    add(labelRaum);

    textfieldRoom.setDocument(new InputDocument(Resource.FELD_LEN_ROOM));
    textfieldRoom.setMinimumSize(new Dimension(192, 32));
    con.gridx = 0;
    con.gridy = 2;
    con.gridwidth = 1;
    con.fill = GridBagConstraints.NONE;
    con.weightx = 0.0;
    con.weighty = 0.0;
    con.insets = new Insets(0, leftMargin, 0, rightMargin);
    con.anchor = GridBagConstraints.WEST;
    gridbag.setConstraints(textfieldRoom, con);
    add(textfieldRoom);
    textfieldRoom.addKeyListener(raumAdapter);

    textfieldRoom.getInputMap(JComponent.WHEN_FOCUSED)
        .put(KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0), "ENTER");
    textfieldRoom.getActionMap().put("ENTER", new AbstractAction() {
      @Override
      public void actionPerformed(ActionEvent e) {
        buttonRaumerstellen.doClick();
      }
    });

    con.gridx = 1;
    con.gridwidth = 1;
    con.fill = GridBagConstraints.NONE;
    con.weightx = 1.0;
    con.weighty = 0.0;
    con.insets = new Insets(0, 0, 0, rightMargin);
    con.anchor = GridBagConstraints.WEST;
    buttonRaumerstellen.setBackground(Resource.JQUERY_RED);
    buttonRaumerstellen.setForeground(Color.WHITE);
    buttonRaumerstellen.setEnabled(true);
    buttonRaumerstellen.addActionListener((event) -> {
      if (getRoom().length() == 0) {
        buttonRaumerstellen.setBackground(Resource.JQUERY_RED);
        return;
      }
      changes.firePropertyChange(JPausenraum.class.getName(), Control.NULL, Control.CREATE);
    });

    gridbag.setConstraints(buttonRaumerstellen, con);
    add(buttonRaumerstellen);

    con.gridx = 0;
    con.gridy = 3;
    con.gridwidth = 2;
    con.fill = GridBagConstraints.HORIZONTAL;
    con.weightx = 1.0;
    con.weighty = 0.0;
    con.insets = new Insets(24, leftMargin, 0, rightMargin);
    con.anchor = GridBagConstraints.CENTER;
    gridbag.setConstraints(separator, con);
    add(separator);

    con.gridx = 1;
    con.gridy++;
    con.gridwidth = 1;
    con.fill = GridBagConstraints.NONE;
    con.weightx = 0.0;
    con.weighty = 0.0;
    con.insets = new Insets(12, 0, marginBottom, rightMargin);
    con.anchor = GridBagConstraints.EAST;
    gridbag.setConstraints(buttonSchliessen, con);
    add(buttonSchliessen);
    buttonSchliessen.addActionListener((event) -> {
      try {
        setClosed(true);
      }
      catch (PropertyVetoException e) {}
    });
    addInternalFrameListener(frameAdapter);

    con.gridx = 0;
    con.gridy++;
    con.gridwidth = 1;
    con.fill = GridBagConstraints.NONE;
    con.weightx = 0.0;
    con.weighty = 1.0;
    con.insets = new Insets(0, 0, 0, 0);
    con.anchor = GridBagConstraints.CENTER;
    gridbag.setConstraints(dummy, con);
    add(dummy);

  }



  /**
   * Der Schwellenwert in der Textausgabe
   *
   *
   * @param value
   *              die Anzahl in Minuten
   */
  public void setMinuten(int value) {
    minutos = value;
  }



  public synchronized void removeAllListener() {
    for (PropertyChangeListener listener : changes.getPropertyChangeListeners()) {
      removePausenroomListener(listener);
    }
    removeInternalFrameListener(frameAdapter);
    textfieldRoom.removeKeyListener(raumAdapter);
  }



  public synchronized void removePausenroomListener(PropertyChangeListener l) {
    changes.removePropertyChangeListener(l);
  }



  @Override
  public void setLanguage(ISO639 code) {
    buttonSchliessen.setLanguage(code);
    buttonRaumerstellen.setLanguage(code);
    labelRaum.setLanguage(code);
    labelPausenraum.setLanguage(code);
    String help = labelPausenraum.getText().replaceAll("XXX", String.valueOf(minutos));
    labelPausenraum.setText("<html>" + help + "</html>");
    setLanguage(KEY.FRAME_PAUSENRAUM, code);

  }



  @Override
  public void setBackgroundGradient(Color top, Color bottom) {
    windowManager.setBackgroundGradient(top, bottom);
  }

}
