/**
 *  Copyright © 2020-2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.client.filetransfer;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GridLayout;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyVetoException;
import javax.swing.JPanel;
import javax.swing.JSplitPane;
import javax.swing.SwingConstants;
import javax.swing.event.InternalFrameAdapter;
import javax.swing.event.InternalFrameEvent;
import net.javacomm.multilingual.MultilingualButton;
import net.javacomm.multilingual.MultilingualLabel;
import net.javacomm.multilingual.schema.ISO639;
import net.javacomm.multilingual.schema.KEY;
import net.javacomm.protocol.BlacklistTypes;
import net.javacomm.window.manager.Control;
import net.javacomm.window.manager.Frames;
import net.javacomm.window.manager.WM;
import net.javacomm.window.manager.WindowAnrufenEvent;
import net.javacomm.window.manager.WindowAnrufenListener;
import net.javacomm.window.manager.WindowManagerInternalFrame;



public class JSchwarzeListeDialog extends WindowManagerInternalFrame implements WindowAnrufenListener {

  public final static int WIDTH = 900;
  public final static int HEIGHT = 450;
  private static final long serialVersionUID = 1617463493750507213L;

  private JPanel mainpane = new JPanel();
  private MultilingualLabel labelTop = new MultilingualLabel(KEY.LABEL_SCHWARZE_LISTE);

  private JButtonPane buttonPane = new JButtonPane();
  private JPanel southPanel = new JPanel(new FlowLayout(FlowLayout.RIGHT, 12, 12));
  private PropertyChangeSupport changes = new PropertyChangeSupport(this);
  private JBlacklist blacklist = new JBlacklist();
  private JSplitPane splitpane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
  private FrameAction frameAction = new FrameAction();

  public JSchwarzeListeDialog() {
    init();
  }



  @Override
  public void activated() {
    try {
      setSelected(true);
      windowManager.setActivNode(this.getFrameTitleId());
    }
    catch (PropertyVetoException e) {}
  }



  public synchronized void addBlacklistListener(PropertyChangeListener l) {
    changes.addPropertyChangeListener(l);
  }



  public void close() {
    try {
      setClosed(true);
    }
    catch (Exception e) {}
  }



  public void centerHeader() {
    blacklist.centerHeader();
  }



  @Override
  public String getFrameTitleId() {
    return this.getTitle();
  }



  @Override
  public Dimension getMinimumSize() {
    return new Dimension(WIDTH, HEIGHT);
  }



  public synchronized BlacklistTypes[] getSperrliste() {
    return blacklist.toBlacklistTypes();
  }



  @Override
  public Frames getType() {
    return Frames.SCHWARZE_LISTE;
  }



  void init() {
    mainpane.setLayout(new BorderLayout());

    splitpane.setRightComponent(windowManager);
    splitpane.setLeftComponent(mainpane);
    splitpane.setDividerLocation(WIDTH - WindowManagerInternalFrame.SPLITWIDTH);
    splitpane.setOneTouchExpandable(true);
    splitpane.setOpaque(true);
    labelTop.setHorizontalAlignment(SwingConstants.LEFT);

    blacklist.getVerticalScrollBar().setUnitIncrement(12);
    blacklist.getHorizontalScrollBar().setUnitIncrement(12);

    southPanel.add(buttonPane);

    mainpane.add(labelTop, BorderLayout.NORTH);
    mainpane.add(blacklist, BorderLayout.CENTER);
    mainpane.add(southPanel, BorderLayout.SOUTH);

    windowManager.addWMListener((event) -> {
      changes.firePropertyChange(event);

    });
    windowManager.setMinimumSize(new Dimension(0, 0));
    setContentPane(splitpane);

    this.addInternalFrameListener(frameAction);

  }



  @Override
  public void insertWM(WM value) {
    windowManager.addWindow(value);
  }



  public synchronized void removeAllListener() {
    for (PropertyChangeListener listener : changes.getPropertyChangeListeners()) {
      changes.removePropertyChangeListener(listener);
    }
    windowManager.removeAllListener();
  }



  public synchronized void removeBlacklistListener(PropertyChangeListener l) {
    changes.removePropertyChangeListener(l);
  }



  @Override
  public void removeWM(WM frame) {
    windowManager.removeWM(frame);
  }



  @Override
  public void setFrameTitleId(String id) {
    setTitle(id);
  }



  public void setSperrliste(BlacklistTypes[] sperrliste) {
    blacklist.setSperrliste(sperrliste);
  }



  @Override
  public void setType(Frames value) {
    // TODO Auto-generated method stub

  }

  /*****************************************************************************/
  /*                                                                           */
  /* Innerclass FrameAction */
  /*                                                                           */
  /*****************************************************************************/

  class FrameAction extends InternalFrameAdapter {

    @Override
    public void internalFrameClosing(InternalFrameEvent event) {
      changes.firePropertyChange(JSchwarzeListeDialog.class.getName(), Control.NULL, Control.CLOSE);
    }

  }

  class JButtonPane extends JPanel {
    private static final long serialVersionUID = 5603407466776609565L;
    private MultilingualButton buttonSchliessen = new MultilingualButton(KEY.BUTTON_SCHLIESSEN);
    private final MultilingualButton buttonSpeichernUndSchliessen = new MultilingualButton(
        KEY.BUTTON_SPEICHERN_SCHLIESSEN
    );
    private final GridLayout grid = new GridLayout(1, 2, 18, 12);

    JButtonPane() {
      setLayout(grid);
      add(buttonSpeichernUndSchliessen);
      buttonSpeichernUndSchliessen.addActionListener((event) -> {
        changes
            .firePropertyChange(JSchwarzeListeDialog.class.getName(), Control.NULL, Control.SAVE_AND_CLOSE);
      });
      add(buttonSchliessen);
      buttonSchliessen.addActionListener((event) -> {
        close();
      });
    }

  }

  @Override
  public void setLanguage(ISO639 code) {
    buttonPane.buttonSchliessen.setLanguage(code);
    buttonPane.buttonSpeichernUndSchliessen.setLanguage(code);
    labelTop.setLanguage(code);
    blacklist.setLanguage(code);
    setLanguage(KEY.FRAME_SCHWARZE_LISTE, code); // der neue Name wird gesetzt
    treeView(code);
  }



  @Override
  public void ringRing(WindowAnrufenEvent event) {
    // TODO Auto-generated method stub
    extText = event.getNickname();
    getMultiAbnehmen().setLanguage(KEY.FRAME_ABNEHMEN, event.getCode(), extText);
    getMultiAnrufen().setLanguage(KEY.TELEFON_ANRUFEN, event.getCode(), extText);
  }



  @Override
  public void setBackgroundGradient(Color top, Color bottom) {
    windowManager.setBackgroundGradient(top, bottom);
  }

}
