/**
 *  Copyright © 2022-2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.client.filetransfer;

import java.nio.file.Path;
import java.nio.file.attribute.FileTime;
import java.util.Objects;



/**
 * UploadFile ist eine Datei, die auf andere Rechner kopiert wird.
 * 
 * @author llange
 *
 */
public class UploadFile {

  private Path path;
  private FileTime lastModified;

  public UploadFile() {}



  /**
   * Wo liegt die Datei im Verzeichnisbaum? Der Dateiname ist im Pfad enthalten.
   * 
   * @return ein vollständiger Dateipfad
   */
  public Path getPath() {
    return path;
  }



  /**
   * Das Uploadverzeichnis für den FTService.
   * 
   * @param path
   *             ein vollständiger Dateipfad
   */
  public void setPath(Path path) {
    this.path = path;
  }



  /**
   * Wann wurde die Datei das letzte Mal verändert oder neu geschrieben?
   * 
   * @return ein Zeitpunkt
   */
  public FileTime getLastModified() {
    return lastModified;
  }



  /**
   * Zu diesem Zeitpunkt wurde die Datei das letze Mal in das Verzeichnis kopiert
   * oder gespeichert.
   * 
   * @param lastModified
   *                     ein Zeitpunkt
   */
  public void setLastModified(FileTime lastModified) {
    this.lastModified = lastModified;
  }



  @Override
  public int hashCode() {
    return Objects.hash(lastModified, path);
  }



  @Override
  public boolean equals(Object obj) {
    if (this == obj) return true;
    if (obj == null) return false;
    if (getClass() != obj.getClass()) return false;
    UploadFile other = (UploadFile) obj;
    return Objects.equals(lastModified, other.lastModified) && Objects.equals(path, other.path);
  }

}
