/**
 *  Copyright © 2020-2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.client.phone;

import java.util.EventObject;



/**
 * Alle Ereignise während einer Sprachaufzeichnung.
 * 
 * @author llange
 *
 */
public class RecordMicEvent extends EventObject {

  private static final long serialVersionUID = 237076400450693212L;
  
  public enum Event {ON_START, ON_RECORD, ON_FINISH};
  
  private Event id;
  private byte[] content;
  private int len;
  
  
  /**
   * Das Mikrofon meldet ein Ereignis.
   * 
   * @param source
   *               das auslösende Objekt
   */
  public RecordMicEvent(Object source) {
    super(source);
  }


  
  /**
   * 
   * Das Mikrofon meldet ein Ereignis.
   * 
   * @param source
   *               das auslösende Objekt
   * @param event
   *               das Ereignis, der Grund
   */
  public RecordMicEvent(Object source, Event event) {
    super(source);
    this.setId(event);
  }
  
  
  
  /**
   * Das Mikrofon hält aufgezeichnete Daten bereit.
   * 
   * 
   * @param source
   *                das auslösende Objekt
   * @param content
   *                Audiodaten
   * @param len
   *                die Anzahl der Bytes
   */
  public RecordMicEvent(Object source, byte[] content, int len) {
    super(source);
    setId(Event.ON_RECORD);
    this.content = content;
    this.len = len;
  }

  

  
  /**
   * Die Anzahl der gelesenen Bytes.
   * 
   * @return Anzahl der Bytes in {@code content}
   */
  public int getLen() {
    return len;
  }
  

  
  
  /**
   * Die Daten aus einer Sprachaufnahme.
   * 
   * 
   * @return Audiodaten
   */
  public byte[] getContent() {
    return content;
  }
  
  
  /**
   * Was war passiert?
   * 
   * @return das Event
   */
  public Event getId() {
    return id;
  }
  
  
  
  /**
   * Dem auslösenden Objekt wird ein Event zugeordnet.
   * 
   * @param id
   *           Event
   */
  public void setId(Event id) {
    this.id = id;
  }
  
  
  
}
