/**
 *  Copyright © 2021-2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.client.resource;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.GradientPaint;
import java.awt.Graphics2D;
import java.awt.Insets;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import javax.swing.JMenuBar;
import javax.swing.JSlider;
import javax.swing.JSplitPane;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.Painter;
import javax.swing.UIDefaults;
import javax.swing.UIManager;
import javax.swing.plaf.nimbus.NimbusLookAndFeel;
import net.javacomm.client.environment.GUI;
import net.javacomm.multilingual.Babelfish;
import net.javacomm.multilingual.MultilingualString;
import net.javacomm.multilingual.schema.ISO639;
import net.javacomm.multilingual.schema.KEY;
import net.javacomm.multilingual.schema.VIRTUALKEY;
import net.javacomm.window.manager.WMResource;



public class Erdbeere implements Babelfish {

  private NimbusLookAndFeel nimbus;
  private MultilingualString dateiname = new MultilingualString(KEY.STRING_DATEINAME);
  private MultilingualString dateityp = new MultilingualString(KEY.STRING_DATEITYP);
  private MultilingualString cancel = new MultilingualString(KEY.BUTTON_ABBRECHEN);
  private MultilingualString open = new MultilingualString(KEY.STRING_OPEN);
  private MultilingualString searchIn = new MultilingualString(KEY.STRING_SEARCH_IN);
  private MultilingualString alleFormate = new MultilingualString(KEY.STRING_ALLE_FORMATE);
  private MultilingualString name = new MultilingualString(KEY.STRING_NAME);
  private MultilingualString size = new MultilingualString(KEY.STRING_SIZE);
  private MultilingualString type = new MultilingualString(KEY.STRING_TYPE);
  private MultilingualString modified = new MultilingualString(KEY.STRING_MODIFIED);
  private MultilingualString directoryname = new MultilingualString(KEY.STRING_VERZEICHNISNAME);
  private VIRTUALKEY vk;

  public Erdbeere() {
    init();
  }



  public NimbusLookAndFeel getLookAndFeel() {
    return nimbus;
  }



  private void init() {
    nimbus = new NimbusLookAndFeel() {

      private static final long serialVersionUID = -5430544919268533739L;

      @Override
      public UIDefaults getDefaults() {
        UIDefaults hash = super.getDefaults();
        hash.put("menubar.top", Resource.JQUERY_ERDBEERE);
        hash.put("menubar.bottom", new Color(251, 41, 67));

        hash.put("defaultFont", GUI.regularFont13);
        hash.put("control", Resource.JQUERY_TEXTBRAUN);
        hash.put("nimbusDisabledText", Resource.JQUERY_RED);
        hash.put("nimbusBase", new Color(255, 0, 90)); // HOMEPAGE
        hash.put("nimbusSelectionBackground", new Color(253, 12, 113));
        hash.put("textForeground", Resource.JQUERY_TEXTBRAUN);
        hash.put("text", Color.WHITE);
        hash.put("nimbusSelectedText", Color.BLACK);
        hash.put("TitledBorder.titleColor", Resource.JQUERY_ORANGE);
        hash.put("TitledBorder.font", GUI.regularBold);
        hash.put("TextField.textForeground", Color.BLACK);
        hash.put("TextField.background", new Color(255, 157, 179));

        hash.put(
            "Spinner:Panel:\"Spinner.formattedTextField\"[Enabled].backgroundPainter",
            new Painter<JTextField>() {

              @Override
              public void paint(Graphics2D g, JTextField object, int width, int height) {
                g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);

                g.setColor(new Color(115, 164, 209));
                g.drawPolyline(
                    new int[] {width - 1, 1, 1, width - 1}, new int[] {2, 2, height - 3, height - 3}, 4
                );

                g.setColor(Resource.JQUERY_ERDBEERE);
                g.fillRect(2, 3, width - 2, height - 6);

              }

            }
        );

        hash.put("FileChooser.acceptAllFileFilterText", alleFormate.toString());
        hash.put("FileChooser.saveInLabelText", "Save in:");
        hash.put("FileChooser.lookInLabelText", searchIn.toString());
        hash.put("FileChooser.fileNameLabelText", dateiname.toString());
        hash.put("FileChooser.filesOfTypeLabelText", dateityp.toString());
        hash.put("FileChooser.upFolderToolTipText", "Eine Verzeichnisebene nach oben");
        hash.put("FileChooser.homeFolderToolTipText", "Heimatverzeichnis");
        hash.put("FileChooser.newFolderToolTipText", "Ein neues Verzeichnis anlegen");
        hash.put("FileChooser.listViewButtonToolTipText", "Inhaltsansicht");
        hash.put("FileChooser.detailsViewButtonToolTipText", "Detailansicht");
        hash.put("FileChooser.cancelButtonText", cancel.toString());
        hash.put("FileChooser.cancelButtonToolTipText", "Dialogfeld schließen");
        hash.put("FileChooser.folderNameLabelText", directoryname.toString());
        hash.put("FileChooser.openButtonText", open.toString());
        hash.put("FileChooser.openButtonToolTipText", "Datei oder Verzeichnis klicken");
        hash.put("Button.background", WMResource.ENABLED_BUTTON_BACKGROUND);
        hash.put("Button[Enabled].textForeground", WMResource.ENABLED_BUTTON_FOREGROUND);
        hash.put("Label.textForeground", Resource.JQUERY_HELLBLAU);
        hash.put("CheckBox.textForeground", Resource.JQUERY_HELLBLAU);
        hash.put("Table.background", Color.WHITE); // treffer!
        hash.put("Table.alternateRowColor", Color.WHITE); // treffer!
        hash.put("Table.textForeground", Color.BLACK); // treffer!
        hash.put("List.background", Color.WHITE); // treffer!
        hash.put("Viewport.contentMargins", new Insets(0, 0, 0, 0)); // treffer!
        hash.put("Label.contentMargins", new Insets(0, 0, 0, 0)); // treffer!
        hash.put("RadioButton.textForeground", Resource.JQUERY_HELLBLAU);
        hash.put("TextArea.textForeground", hash.get("textForeground"));
        hash.put("TextArea[Disabled].backgroundPainter", new Painter<JTextArea>() {
          @Override
          public void paint(Graphics2D g, JTextArea object, int width, int height) {
            g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
            g.setColor(Resource.JQUERY_ORANGE);
            g.fillRect(0, 0, width, height);
          }
        });
        hash.put("List[Selected].textBackground", hash.get("nimbusSelectionBackground"));
//        hash.put("List[Selected].textForeground", Color.WHITE); // wegen FileChooser
        hash.put("Tree.background", Resource.JQUERY_ORANGE);
        hash.put("Tree.textForeground", Resource.JQUERY_BLAU);
        hash.put("Tree.selectionBackground", Resource.JQUERY_BLAU);
        hash.put("Tree.textBackground", Resource.JQUERY_BLAU);
        hash.put("MenuBar[Enabled].backgroundPainter", new Painter<JMenuBar>() {

          @Override
          public void paint(Graphics2D g, JMenuBar object, int width, int height) {
            g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
            GradientPaint gp = new GradientPaint(
                0, 0, hash.getColor("menubar.top"), width, 0, hash.getColor("menubar.bottom")
            );
            g.setPaint(gp);
            g.fillRect(0, 0, width, height);
          }

        });
        hash.put("ScrollPane.contentMargins", new Insets(0, 0, 0, 0));
        hash.put("TableHeader.font", GUI.semiFont15);
        hash.put("MenuItem.font", GUI.semiFont15); // ok
        hash.put("MenuBar.font", GUI.semiFont15);
        hash.put("Menu.font", GUI.semiFont15);
        hash.put("CheckBoxMenuItem.font", GUI.semiFont15);
        hash.put("PopupMenu.font", GUI.semiFont15);
        hash.put("InternalFrame.titleFont", GUI.regularFrametitle);
        hash.put("SplitPane.centerOneTouchButtons", Boolean.TRUE); // ok
        hash.put("SplitPane.oneTouchButtonOffset", 6); // ok
        hash.put("SplitPane.size", 12);
        hash.put("SplitPane.contentMargins", new Insets(0, 0, 0, 0)); // ok
        // background / foreground nimbusBase beeinflusst den Hintergrund /
        // Arrow //
        hash.put("SplitPane:SplitPaneDivider[Enabled].backgroundPainter", new Painter<JSplitPane>() {
          @Override
          public void paint(Graphics2D g, JSplitPane object, int width, int height) {
            g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
            g.setColor((Color) hash.get("nimbusSelectionBackground"));
            g.fillRect(0, 0, width, height);
          }
        });
        hash.put("Slider.tickColor", Resource.JQUERY_YELLOW);
        hash.put("Slider:SliderTrack[Enabled].backgroundPainter", new Painter<JSlider>() {
          @Override
          public void paint(Graphics2D g, JSlider slider, int width, int height) {
            int arc = 10;
            int trackHeight = 8;
            int trackWidth = width - 2;
            int fillTop = 4;
            int fillLeft = 1;
            g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
            g.setStroke(new BasicStroke(1.5f));
            g.setColor(Resource.JQUERY_ORANGE);
            g.fillRoundRect(fillLeft, fillTop, trackWidth, trackHeight, arc, arc);
            int fillBottom = fillTop + trackHeight;
            int fillRight = xPositionForValue(
                slider.getValue(), slider, new Rectangle(fillLeft, fillTop, trackWidth, fillBottom - fillTop)
            );
            g.setColor(Resource.JQUERY_BLAU);
            g.fillRect(fillLeft + 1, fillTop + 1, fillRight - fillLeft, fillBottom - fillTop);
            g.setColor((Color) hash.get("text"));
            g.drawRoundRect(fillLeft, fillTop, trackWidth, trackHeight, arc, arc);
          }



          protected int xPositionForValue(int value, JSlider slider, Rectangle trackRect) {
            int min = slider.getMinimum();
            int max = slider.getMaximum();
            int trackLength = trackRect.width;
            double valueRange = (double) max - (double) min;
            double pixelsPerValue = trackLength / valueRange;
            int trackLeft = trackRect.x;
            int trackRight = trackRect.x + (trackRect.width - 1);
            int xPosition;
            xPosition = trackLeft;
            xPosition += Math.round(pixelsPerValue * ((double) value - min));
            xPosition = Math.max(trackLeft, xPosition);
            xPosition = Math.min(trackRight, xPosition);
            return xPosition;
          }
        });
        hash.put("TextPane.background", hash.get("control"));
        hash.put("TextPane.foreground", Resource.JQUERY_HELLBLAU);
        hash.put("CheckBox.foreground", Resource.JQUERY_HELLBLAU);

        return hash;
      }
    };
  }



  @Override
  public void setLanguage(ISO639 code) {
    dateiname.setLanguage(code);
    UIManager.put("FileChooser.fileNameLabelText", dateiname.toString());

    cancel.setLanguage(code);
    UIManager.put("FileChooser.cancelButtonText", cancel.toString());

    open.setLanguage(code);
    UIManager.put("FileChooser.openButtonText", open.toString());

    dateityp.setLanguage(code);
    UIManager.put("FileChooser.filesOfTypeLabelText", dateityp.toString());

    searchIn.setLanguage(code);
    UIManager.put("FileChooser.lookInLabelText", searchIn.toString());

    // SynthFileChooserUIImpl
    vk = searchIn.getVirtualKey();
    if (vk != null) {
      UIManager.put("FileChooser.lookInLabelMnemonic", vk.ordinal() + 65);
    }

    alleFormate.setLanguage(code);
    UIManager.put("FileChooser.acceptAllFileFilterText", alleFormate.toString());

    name.setLanguage(code);
    UIManager.put("FileChooser.fileNameHeaderText", name.toString());

    size.setLanguage(code);
    UIManager.put("FileChooser.fileSizeHeaderText", size.toString());

    type.setLanguage(code);
    UIManager.put("FileChooser.fileTypeHeaderText", type.toString());

    modified.setLanguage(code);
    UIManager.put("FileChooser.fileDateHeaderText", modified.toString());

    directoryname.setLanguage(code);
    UIManager.put("FileChooser.folderNameLabelText", directoryname.toString());

    vk = directoryname.getVirtualKey();
    if (vk != null) {
      UIManager.put("FileChooser.folderNameLabelMnemonic", vk.ordinal() + 65);
    }

  }

}
