/**
 *  Copyright © 2021-2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.client.screencaster;

import java.nio.file.Path;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import net.javacomm.client.environment.Environment;
import uk.co.caprica.vlcj.factory.MediaPlayerFactory;
import uk.co.caprica.vlcj.player.base.MediaPlayer;



/**
 * Der Screensaver zeichnet den Bildschirm auf und sendet ihn über das Netzwerk
 * an einen Server.
 *
 * 
 * @author llange
 *
 */
public class Caster {

  private final static Logger log = LogManager.getLogger(Caster.class);
  private MediaPlayerFactory factory;
  private MediaPlayer mediaplayer;
  private boolean hasStarted = false;

  /**
   * Es wird nur eine Instanz erzeugt.
   * 
   */
  public Caster() {}



  /**
   * Der Server beginnt mit der Bildschirmaufzeichnung.
   * 
   * 
   * @param serverhost
   *                   ein Domänenname oder eine IP-Adresse
   * @param port
   *                   der Serverport
   */
  public void serverDomain(String serverhost, int port) {
    if (hasStarted) return;
    hasStarted = true;

    Path maus = Environment.getInstance().unpackMousecursor();
    factory = new MediaPlayerFactory(
        "--quiet", "--screen-fps=12", "--live-caching=560", "--screen-follow-mouse",
        "--screen-mouse-image=" + maus.toString()
    );
    mediaplayer = factory.mediaPlayers().newMediaPlayer();
    StringBuilder builder = new StringBuilder();
    builder.append(":sout=#transcode{vcodec=h264,vb=512,acodec=none,scodec=none}:http{mux=ts,dst=")
        .append(serverhost).append(":").append(port).append("}");
    log.info(builder.toString());
    mediaplayer.media().play("screen://", builder.toString());
  }

  
  
  /**
   * Läuft gerade eine Übertragung?
   * 
   * @return
   */
  public synchronized boolean isRunning() {
    return hasStarted;
  }




  /**
   * Die Bildschirmaufzeichnung wird abgebrochen. Alle Ressourcen werden
   * freigegeben und der AWT-Thread benachrichtigt. Falls keine
   * Bildschirmaufzeichnung läuft, hat der Aufruf keine Wirkung.
   * 
   */
  public Runnable stopRecording() {
    Runnable runnable = () -> {
      try {
        if (!hasStarted) return;
        hasStarted = false;
        mediaplayer.controls().stop();
        mediaplayer.release();
        factory.release();
      }
      catch (Exception e) {
        log.warn(e.getMessage(), e);
      }
    };
    return runnable;
  }




}
