/**
 *  Copyright © 2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.client.screencaster;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;



/**
 * FrameBuffer mit genau einem Slot. Producer überschreibt älteres Bild,
 * Consumer erhält die aktuell gespeicherte FrameData und der Slot wird geleert.
 */
public class FrameBuffer {

  private FrameData lastFrame = null;
  private final ReentrantLock lock = new ReentrantLock();
  private final Condition newFrameAvailable = lock.newCondition();

  /**
   * Einfaches DTO: byte[] + fortlaufende Nummer (vom Producer vergeben).
   * Zusätzlich: timestampNanos - Erzeugungszeitpunkt (nanoTime).
   */
  public static class FrameData {
    final byte[] bytes;
    final long number;
    final long timestampNanos;

    FrameData(byte[] bytes, long number, long timestampNanos) {
      this.bytes = bytes;
      this.number = number;
      this.timestampNanos = timestampNanos;
    }
  }

  /**
   * Setzt das aktuelle Frame (überschreibt vorhandenes). Producer blockiert
   * nicht.
   */
  public void setFrame(FrameData frame) {
    lock.lock();
    try {
      lastFrame = frame;
      newFrameAvailable.signal();
    }
    finally {
      lock.unlock();
    }
  }



  /**
   * Liefert das aktuelle Frame und leert den Slot. Blockiert, bis etwas vorhanden
   * ist.
   */
  public FrameData takeFrame() throws InterruptedException {
    lock.lock();
    try {
      while (lastFrame == null) {
        newFrameAvailable.await();
      }
      FrameData ret = lastFrame;
      lastFrame = null; // Slot leeren
      return ret;
    }
    finally {
      lock.unlock();
    }
  }
}
