/**
 *  Copyright © 2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.client.screencaster;

import static net.javacomm.share.Constants.DOMAIN;
import static net.javacomm.share.Constants.HTTP;
import jakarta.ws.rs.client.Client;
import jakarta.ws.rs.client.ClientBuilder;
import jakarta.ws.rs.client.Entity;
import jakarta.ws.rs.core.MediaType;
import java.awt.MouseInfo;
import java.awt.Point;
import java.io.IOException;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import org.glassfish.jersey.media.multipart.FormDataMultiPart;
import org.glassfish.jersey.media.multipart.MultiPartFeature;
import net.javacomm.restserver.Wrapper;



/**
 * Der MousePointer wird an den Server gesendet.
 */
public class Mousepointer extends Thread {

  private volatile Point mousepointer;
  private Point old = new Point(0, 0);
  private volatile CopyOnWriteArrayList<String> receiverlist;
  private Client client;

  /**
   * 
   * @param receiverIds
   *                    alle Mousepointerempfänger
   */
  public Mousepointer(List<String> receiverIds) {
    receiverlist = new CopyOnWriteArrayList<>(receiverIds);
    client = ClientBuilder.newClient();
    client.register(MultiPartFeature.class);

  }



  @Override
  public void run() {
    StringBuilder url = new StringBuilder(HTTP).append(DOMAIN)
        .append("/javacommserver/screenshot/mousepointer");
    while (!Thread.currentThread().isInterrupted()) {
      mousepointer = MouseInfo.getPointerInfo().getLocation();
      try {
        Thread.sleep(40);
        if (mousepointer.equals(old)) continue;
        old = new Point(mousepointer);
        // senden
        try(FormDataMultiPart multipart = new FormDataMultiPart()) {
          multipart.field("x", (int) mousepointer.getX(), MediaType.TEXT_PLAIN_TYPE);
          multipart.field("y", (int) mousepointer.getY(), MediaType.TEXT_PLAIN_TYPE);
          multipart.field("receivers", Wrapper.toString(receiverlist));
          client.target(url.toString()).request(MediaType.TEXT_PLAIN)
              .post(Entity.entity(multipart, multipart.getMediaType()));
        }
        catch (IOException e) {}
      }
      catch (InterruptedException e) {
        break;
      }
    }
    client.close();
  }



  /**
   * An welcher Bildschirmposition befindet sich der Mauszeiger?
   * 
   * @return diese Mausposition
   */
  public Point getMousepoint() {
    return mousepointer != null ? new Point(mousepointer) : new Point(0, 0);
  }



  /**
   * Die Empfängerliste wird aktualisiert.
   * 
   * @param newReceiverlist
   */
  public void setReceiverlist(List<String> newReceiverlist) {
    receiverlist = new CopyOnWriteArrayList<>(newReceiverlist);
  }

}
