/**
 *  Copyright © 2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.client.screencaster;

import java.awt.EventQueue;
import java.awt.Graphics2D;
import java.awt.MouseInfo;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Robot;
import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.util.concurrent.ArrayBlockingQueue;
import javax.imageio.ImageIO;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import net.javacomm.client.resource.Resource;



/**
 * Der ScreenshotProducer erstellt Screenshots im PNG-Format.
 * 
 */
public class ScreenshotProducerPNG implements Runnable {

  private final static Logger log = LogManager.getLogger(ScreenshotProducerPNG.class);

  private Robot robot;
  private Rectangle screenSize;
  private ArrayBlockingQueue<byte[]> queue;
  private BufferedImage duke = null;
  BufferedImage image;
  ByteArrayOutputStream outputStream;

  /**
   * Der Producer wird initialisiert.
   * 
   * 
   * @param queue
   *                   in diese Queue werden die Screenshots geschrieben
   * @param robot
   *                   nimmt den Screenshot vor
   * @param screenSize
   *                   diese Bildschirmgröße
   */
  public ScreenshotProducerPNG(ArrayBlockingQueue<byte[]> queue, Robot robot, Rectangle screenSize) {
    this.queue = queue;
    this.robot = robot;
    this.screenSize = screenSize;
    outputStream = new ByteArrayOutputStream();
    ImageIO.setUseCache(false);
    try {
      duke = ImageIO.read(getClass().getResource(Resource.DUKEMAUS));
    }
    catch (IOException e) {
      log.warn("Mauscursor konnte nicht gelesen werden");
    }

  }



  @Override
  public void run() {
    if (duke == null) return;
    while (!Thread.currentThread().isInterrupted()) {

      try {
        EventQueue.invokeAndWait(() -> {
          image = robot.createScreenCapture(screenSize);
        });
      }
      catch (InvocationTargetException | InterruptedException e1) {
        return;
      }

      Graphics2D g2 = image.createGraphics();
      Point mousepointer = MouseInfo.getPointerInfo().getLocation();
      g2.drawImage(duke, mousepointer.x, mousepointer.y, null);
      g2.dispose();

      try {
        outputStream.reset();
        ImageIO.write(image, "png", outputStream);
        image.flush();
        queue.put(outputStream.toByteArray());
      }
      catch (IOException e) {
        log.error("ImageIO - " + e.getMessage());
      }
      catch (InterruptedException e) {
        return;
      }
    }
  }
}
