/**
 *  Copyright © 2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  Previously released under Apache License 2.0; now licensed under MPL 2.0.
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.restserver;

import static net.javacomm.protocol.HEADER.RESPONSE;
import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.PUT;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.PathParam;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.QueryParam;
import jakarta.ws.rs.WebApplicationException;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response.Status;
import java.sql.SQLException;
import java.util.List;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import net.javacomm.database.DatabaseException;
import net.javacomm.database.WebdatabaseImpl;
import net.javacomm.protocol.CHATMESSAGE;
import net.javacomm.protocol.Command;
import net.javacomm.protocol.HISTORYMESSAGE;
import net.javacomm.protocol.Protocol;
import net.javacomm.protocol.SEARCHFILES;
import net.javacomm.protocol.Searchfile;



/**
 * 
 * Das Websocket Protocol.
 * 
 * @author llange
 *
 */
@Path("/protocol")
public class ProtocolMessage {

  private Logger log = LogManager.getLogger(ProtocolMessage.class);
  private WebdatabaseImpl database;

  public ProtocolMessage() {
    database = (WebdatabaseImpl) WebdatabaseImpl.getInstance();
  }



  /**
   * Ein Anwender sucht nach Dateien.
   * 
   * @param pattern
   *                mehrere Suchmuster
   * 
   * @return keine oder mehrere gefundene Dateien
   */
  @GET
  @Path("/searchfiles")
  @Produces(MediaType.APPLICATION_JSON)
  public SEARCHFILES searchfiles(@QueryParam(value = "pattern") String[] pattern) {

    // er kann unbegrenzt viele Treffer übertragen
    List<Searchfile> searchfiles = database.selectSearchfiles(pattern, Integer.MAX_VALUE);

    SEARCHFILES response = new SEARCHFILES();
    response.setCommand(Command.SEARCHFILES);
    response.setHeader(RESPONSE);
    response.setDataset(Protocol.DATASET);
    response.setSearchfiles(searchfiles.toArray(new Searchfile[searchfiles.size()]));
    response.setStarted(true); // die alte Liste im Client wird gelöscht

    return response;
  }



  @GET
  @Path("/historymessage/{room}")
  @Produces(MediaType.APPLICATION_JSON)
  public HISTORYMESSAGE historymessage(@PathParam(value = "room") String room) {

    HISTORYMESSAGE historymessage = database.fetchHistorymessage(room);
    return historymessage;
  }



  /**
   * Eine einzelne Nachricht wird in einen Raum geschickt.
   * 
   * 
   * @param chatmessage
   *                    die Nachricht
   * @throws SQLException
   */
  @PUT
  @Path("/chatmessage")
  @Produces(MediaType.TEXT_PLAIN)
  @Consumes(MediaType.APPLICATION_JSON)
  public void chatmessage(CHATMESSAGE chatmessage) {
    try {
      database.insertChatmessage(
          chatmessage.getRoom(), chatmessage.getChatUser().getUserid(), chatmessage.getAttachment(),
          chatmessage.getMessage()
      );
    }
    catch (DatabaseException e) {
      log.error(e);
      throw new WebApplicationException(e, Status.INTERNAL_SERVER_ERROR);
    }

  }

}
