/**
 * The MIT License
 * Copyright © 2026 Luis Andrés Lange <https://javacomm.net>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package org.nexuswob.gui;

import java.awt.BorderLayout;
import java.awt.Cursor;
import java.awt.Font;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.BorderFactory;
import javax.swing.Icon;
import javax.swing.JPanel;
import javax.swing.JPasswordField;
import javax.swing.JToggleButton;



/**
 * NimbusPasswordField ist eine Komponente für Nimbus Look & Feel. Passwortfeld
 * mit Eye-Button zum Anzeigen/Verstecken. Unterstützt: 1. PNG/SVG Icon 2.
 * Unicode 👁 (skalierbar)
 */
@SuppressWarnings("serial")
public class NimbusPasswordField extends JPanel {

  private final JPasswordField passwordField;
  private final JToggleButton eyeButton;
  private final char echoChar;

  /**
   * Konstruktor mit Icon
   * 
   * @param columns
   *                           Spalten für Passwortfeld
   * @param icon
   *                           Eye-Icon (PNG/SVG), null falls Unicode verwendet
   *                           werden soll
   * @param unicodeEyeFontSize
   *                           Schriftgröße für Unicode-Eye (wenn icon == null)
   */
  public NimbusPasswordField(int columns, Icon icon, float unicodeEyeFontSize) {
    super(new BorderLayout(5, 0));

    // Passwortfeld initialisieren
    passwordField = new JPasswordField(columns);
    echoChar = passwordField.getEchoChar();

    // Eye-Button initialisieren
    eyeButton = new JToggleButton();
    eyeButton.setFocusable(false);
    eyeButton.setBorder(BorderFactory.createEmptyBorder());
    eyeButton.setContentAreaFilled(false);
    eyeButton.setRolloverEnabled(true);
    eyeButton.setCursor(Cursor.getPredefinedCursor(Cursor.HAND_CURSOR));

    if (icon != null) {
      // PNG/SVG Icon Variante
      eyeButton.setIcon(icon);
    }
    else {
      // Unicode 👁 Variante
      eyeButton.setText("👁");
      Font currentFont = eyeButton.getFont();
      eyeButton.setFont(currentFont.deriveFont(unicodeEyeFontSize));
    }

    // ActionListener zum Umschalten
    eyeButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        if (eyeButton.isSelected()) {
          passwordField.setEchoChar((char) 0); // sichtbar
        }
        else {
          passwordField.setEchoChar(echoChar); // maskiert
        }
      }
    });

    // Layout
    this.add(passwordField, BorderLayout.CENTER);
    this.add(eyeButton, BorderLayout.EAST);
  }



  /** Zugriff auf das Passwortfeld */
  public JPasswordField getPasswordField() {
    return passwordField;
  }



  /** Passwort auslesen */
  public char[] getPassword() {
    return passwordField.getPassword();
  }



  /** Passwort programmatisch setzen */
  public void setPassword(String password) {
    passwordField.setText(password);
  }



  /** Zugriff auf den Eye-Button */
  public JToggleButton getEyeButton() {
    return eyeButton;
  }
}
