/**
 *  Copyright © 2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  Previously released under Apache License 2.0; now licensed under MPL 2.0.
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.protocol;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import java.util.Objects;



public class ChatUser {

  public final static String USERID = "USERID";
  public final static String NICKNAME = "NICKNAME";
  public final static String FOREGROUND_COLOR = "FOREGROUND_COLOR";
  public final static String BACKGROUND_COLOR = "BACKGROUND_COLOR";

  private int bgColor;
  private int fgColor;
  private String userid;
  private String nickname;

  @Override
  public boolean equals(Object obj) {
    if (this == obj) return true;
    if (obj == null) return false;
    if (getClass() != obj.getClass()) return false;
    ChatUser other = (ChatUser) obj;
    return bgColor == other.bgColor
        && fgColor == other.fgColor
        && Objects.equals(nickname, other.nickname)
        && Objects.equals(userid, other.userid);
  }



  public int getBackgroundColor() {
    return bgColor;
  }



  public int getForegroundColor() {
    return fgColor;
  }



  public String getNickname() {
    return nickname;
  }



  /**
   * Jeder User hat eine eindeutige id.
   * 
   * @return userid
   */
  public final String getUserid() {
    return userid;
  }



  @Override
  public int hashCode() {
    return Objects.hash(bgColor, fgColor, nickname, userid);
  }



  public void setBackgroundColor(int color) {
    bgColor = color;
  }



  public void setForegroundColor(int color) {
    fgColor = color;
  }



  public void setNickname(String value) {
    nickname = value;
  }



  public final void setUserid(String value) {
    userid = value;
  }



  public String toString() {
    GsonBuilder builder = new GsonBuilder();
    Gson gson = builder.registerTypeAdapter(ChatUser.class, new ChatUserSerializer()).create();
    return gson.toJson(this);

  }

}
