/**
 *  Copyright © 2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  Previously released under Apache License 2.0; now licensed under MPL 2.0.
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.protocol;

import com.google.gson.JsonArray;
import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import java.lang.reflect.Type;



public class HistoryMessageDeserializer implements JsonDeserializer<HISTORYMESSAGE> {

  @Override
  public HISTORYMESSAGE deserialize(JsonElement json, Type typeOfT, JsonDeserializationContext context)
      throws JsonParseException {

    JsonObject jsonobject = json.getAsJsonObject();
    HEADER header = HEADER.toHeader(jsonobject.getAsJsonPrimitive(Protocol.HEADER).getAsString());
    JsonObject dataset = jsonobject.getAsJsonObject(Protocol.DATASET);
    String room = dataset.get(HISTORYMESSAGE.ROOM).getAsString();

    JsonArray record = dataset.getAsJsonArray(HISTORYMESSAGE.RECORD);
    int count = record.size();
    AttachmentImpl[] attachment = new AttachmentImpl[count];
    for (int index = 0; index < count; index++) {
      attachment[index] = new AttachmentImpl();

      JsonObject recordobject = record.get(index).getAsJsonObject();
      attachment[index].setDatetime(recordobject.getAsJsonPrimitive(RecordInterface.DATETIME).getAsLong());
      attachment[index].setMessage(recordobject.getAsJsonPrimitive(RecordInterface.MESSAGE).getAsString());
      attachment[index].setRoom(recordobject.getAsJsonPrimitive(RecordInterface.ROOM).getAsString());
      JsonObject chatuserobject = recordobject.getAsJsonObject(RecordInterface.CHATUSER);
      ChatUser chatuser = new ChatUser();
      chatuser.setBackgroundColor(chatuserobject.getAsJsonPrimitive(ChatUser.BACKGROUND_COLOR).getAsInt());
      chatuser.setForegroundColor(chatuserobject.getAsJsonPrimitive(ChatUser.FOREGROUND_COLOR).getAsInt());
      chatuser.setNickname(chatuserobject.getAsJsonPrimitive(ChatUser.NICKNAME).getAsString());
      chatuser.setUserid(chatuserobject.getAsJsonPrimitive(ChatUser.USERID).getAsString());
      attachment[index].setChatUser(chatuser);
      try {
        attachment[index].setFilename(recordobject.getAsJsonPrimitive(Attachment.FILENAME).getAsString());
        attachment[index].setFilesize(recordobject.getAsJsonPrimitive(Attachment.FILESIZE).getAsLong());
        attachment[index].setAttachment(recordobject.getAsJsonPrimitive(Attachment.ATTACHMENT).getAsLong());
      }
      catch (NullPointerException e) {}

    }

    HISTORYMESSAGE historymessage = new HISTORYMESSAGE();
    historymessage.setHeader(header);
    historymessage.setCommand(Command.HISTORYMESSAGE);
    historymessage.setDataset(Protocol.DATASET);
    historymessage.setRoom(room);
    historymessage.setAttachment(attachment);

    return historymessage;
  }

}
