/**
 *  Copyright © 2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  Previously released under Apache License 2.0; now licensed under MPL 2.0.
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.protocol;

import com.google.gson.Gson;
import net.javacomm.multilingual.schema.KEY;



/**
 * 
 * Eine Botschaft hat einen HEADER, ein COMMAND und optional einen DATASET.
 * HEADER und COMMAND sind für jede Botschaft zwingend vorgeschrieben. Ein
 * DATASET ist abhängig vom verwendeten COMMAND. COMMAND kann eine
 * Fehlerbeschreibung enthalten.
 * 
 * @author llange
 *
 */
public class MESSAGE {

  public final static String ERRORCODE = "ERRORCODE";
  public final static String ERRORMESSAGE = "ERRORMESSAGE";
  public final static String TEXT = "TEXT";
  public final static String MULTILINGUALKEY = "MULTILINGUALKEY";

  private HEADER header;
  private Command command;
  private String dataset;
  private Integer errorcode;
  private String errormessage;
  private String text;
  private KEY multilingualkey;

  /**
   * Wie lautet der Befehl?
   * 
   * @return das Kommando
   */
  public Command getCommand() {
    return command;
  }



  /**
   * Das Kommando wird beim ENCODING verwendet und allen Attributen eines
   * Kommandos vorangestellt.
   * 
   * @return alle Attribute eines Kommandos
   */
  public String getDataset() {
    return dataset;
  }



  public Integer getErrorCode() {
    return errorcode;
  }



  public String getErrorMessage() {
    return errormessage;
  }



  /**
   * {@link Protocol#REQUEST}, {@link Protocol#RESPONSE}, {@link Protocol#CONFIRM}
   * oder {@link Protocol#ERROR}?
   * 
   * @return der Header
   */
  public HEADER getHeader() {
    return header;
  }



  public KEY getMultilingualkey() {
    return multilingualkey;
  }



  /**
   * Ein beliebiger Kommentar oder eine Beschreibung.
   * 
   * @return text
   */
  public String getText() {
    return text;
  }



  /**
   * 
   * Jedes definierte Kommando aus {@link Protocol}.
   * 
   * @param value
   *              Kommando als String
   */
  public void setCommand(Command value) {
    command = value;
  }



  /**
   * Dataset ist abhängig vom verwendeten Kommando.
   * 
   * @param value
   *              Datensatz als String
   * 
   * @see DIAGNOSTICS
   * @see GETINTERFACES
   * @see REFRESHINTERFACES
   * @see USRLOGIN
   */
  public void setDataset(String value) {
    dataset = value;
  }



  /**
   * Eine Fehlerbeschreibung als Zahlenwert.
   * 
   * @param value
   *              Fehlerbeschreibung
   */
  public void setErrorCode(Integer value) {
    errorcode = value;
  }



  /**
   * Eine genaue Fehlerbescrheibung
   * 
   * @param value
   *              Fehlerbeschreibung
   */
  public void setErrorMessage(String value) {
    errormessage = value;
  }



  public void setHeader(HEADER header) {
    this.header = header;
  }



  public void setMultilingualkey(KEY multilingualkey) {
    this.multilingualkey = multilingualkey;
  }



  /**
   * Freier Text.
   * 
   * @param value
   *              text
   */
  public void setText(String value) {
    text = value;
  }



  @Override
  public String toString() {
    Gson gson = new Gson();
    return gson.toJson(this);
  }

}
