/**
 *  Copyright © 2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  Previously released under Apache License 2.0; now licensed under MPL 2.0.
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 *  In short:
 *  - This file may be used, modified, and distributed under MPL 2.0 only.
 *  - It may NOT be relicensed under GPL, LGPL, AGPL, or any other Secondary License.
 *
 *  Rationale:
 *  - Ensures that the code remains MPL-2.0.
 *  - Avoids legal conflicts with GPL-licensed libraries (e.g., VideoLAN).
 *  - Maximizes usability for commercial and security-critical applications.
 *
 */
package net.javacomm.protocol;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;



/**
 * Die Portalanmeldung erfolgt über das Kommando USRLOGIN. Ein User muss kein
 * Mensch sein.
 * 
 * @author llange
 *
 */
public class USRLOGIN extends MESSAGE {

  public final static String SESSION = "SESSION";
  public final static String AGENT = "AGENT";
  public final static String IDENTITY = "IDENTITY";
  public final static String NICKNAME = USER.NICKNAME;
  public final static String FOREGROUND_COLOR = USER.FOREGROUND_COLOR;
  public final static String BACKGROUND_COLOR = USER.BACKGROUND_COLOR;
  public final static String ONCALL = USER.ONCALL;
  public final static String VOLUME = USER.VOLUME;

  private Agent agent;
  private String identity;
  private Integer backgroundColor;
  private Integer foregroundColor;
  private Integer volume;
  private Boolean oncall;
  private String nickname;
  private String sessionid;

  /**
   * Deserialisiere einen JSON String in ein USRLOGIN Objekt.
   * 
   * @param deserialize
   *                    dieser JSON String
   * 
   * @return dieser USRLOGIN
   */
  public static USRLOGIN toUSRLOGIN(String deserialize) {
    GsonBuilder builder = new GsonBuilder();
    builder.registerTypeHierarchyAdapter(USRLOGIN.class, new UsrloginDeserializer());
    Gson gson = builder.create();
    return gson.fromJson(deserialize, USRLOGIN.class);
  }



  /**
   * Über welches Gerät erfolgte die Anmeldung?
   * 
   * @return diese Gerät
   */
  public Agent getAgent() {
    return agent;
  }



  public final Integer getBackgroundColor() {
    return backgroundColor;
  }



  public final Integer getForegroundColor() {
    return foregroundColor;
  }



  /**
   * Die Anmeldeinformationen wurden im X.509 Format Base64 codiert.
   * 
   * @return diese Anmeldeinformationen sind RSA 2048 verschlüsselt
   */
  public String getIdentity() {
    return identity;
  }



  public final String getNickname() {
    return nickname;
  }



  public final Boolean getOncall() {
    return oncall;
  }



  /**
   * Der Sitzungsschlüssel für die Websocketverbindung.
   * 
   * @return dieser Sitzungsschlüssel
   */
  public String getSession() {
    return sessionid;
  }



  public final String getSessionid() {
    return sessionid;
  }



  public final Integer getVolume() {
    return volume;
  }



  /**
   * Der Agent beeinflusst die Anzahl der geöffneten Chats. Wenn der Agent nicht
   * gesetzt ist, geht der Server von einem <b>Browser</b> aus.<br>
   * <br>
   * 
   * REQUEST: <b>optional</b><br>
   * CONFIRM: nicht vorhanden<br>
   * RESPONSE: nicht vorhanden<br>
   * ERROR: nicht vorhanden<br>
   * 
   * @param agent
   *              über dieses Gerät erfolgt die Anmeldung
   */
  public void setAgent(Agent agent) {
    this.agent = agent;
  }



  public final void setBackgroundColor(Integer backgroundColor) {
    this.backgroundColor = backgroundColor;
  }



  public final void setForegroundColor(Integer foregroundColor) {
    this.foregroundColor = foregroundColor;
  }



  /**
   * Die Anmeldinformationen wurden RSA 2048 verschlüsselt. Das verwendete Format
   * ist X.509. Das übergebene Token ist Base64 codiert. Das Token selber ist ein
   * JSON String mit den Keys USERID, EMAIL, PASSWORD, AES.<br>
   * <br>
   * 
   * REQUEST: <b>zwingend</b><br>
   * CONFIRM: nicht vorhanden<br>
   * RESPONSE: nicht vorhanden<br>
   * ERROR: nicht vorhanden<br>
   * 
   * @param identity
   *                 enthält alle Anmeldeinformationen
   */
  public final void setIdentity(String identity) {
    this.identity = identity;
  }



  public final void setNickname(String nickname) {
    this.nickname = nickname;
  }



  public final void setOncall(Boolean oncall) {
    this.oncall = oncall;
  }



  /**
   * Der Sitzungsschlüssel für die Websocketverbindung.<br>
   * <br>
   * 
   * REQUEST: nicht vorhanden<br>
   * CONFIRM: <b>zwingend</b><br>
   * RESPONSE: nicht vorhanden<br>
   * ERROR: nicht vorhanden<br>
   * 
   * @param session
   *                diese Session
   */
  public void setSession(String session) {
    sessionid = session;
  }



  public final void setSessionid(String sessionid) {
    this.sessionid = sessionid;
  }



  public final void setVolume(Integer volume) {
    this.volume = volume;
  }



  @Override
  public String toString() {
    GsonBuilder builder = new GsonBuilder();
    builder.registerTypeAdapter(USRLOGIN.class, new UsrloginSerializer());
    Gson gson = builder.create();
    return gson.toJson(this);
  }

}
