package net.javacomm.protocol;

import static net.javacomm.protocol.HEADER.CONFIRM;
import static net.javacomm.protocol.HEADER.ERROR;
import static net.javacomm.protocol.HEADER.REQUEST;
import static net.javacomm.protocol.HEADER.RESPONSE;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertNull;
import static org.junit.jupiter.api.Assertions.assertTrue;
import static org.junit.jupiter.api.Assertions.fail;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonSyntaxException;
import jakarta.websocket.DecodeException;
import jakarta.websocket.EncodeException;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.Arrays;
import java.util.regex.Matcher;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;
import net.javacomm.multilingual.schema.KEY;



public class TestMessageDecoder {

  MessageDecoder messagedecoder;

  @Test
  public void decodeBEAMEROFF_REQUEST() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"BEAMEROFF\","
        + "\"DATASET\":{"
        + "\"USERID\":\"anaconda\""
        + "}}";

    BEAMEROFF beameroff = (BEAMEROFF) messagedecoder.decode(encoded);
    assertEquals(Command.BEAMEROFF, beameroff.getCommand(), "BEAMEROFF");
    assertEquals(REQUEST, beameroff.getHeader(), "REQUEST");
    assertEquals("anaconda", beameroff.getUserid(), "USERID");
  }



  @Test
  public void decodeBEAMEROFF_RESPONSE() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"BEAMEROFF\","
        + "\"DATASET\":{"
        + "\"USERID\":\"anaconda\""
        + "}}";

    BEAMEROFF beameroff = (BEAMEROFF) messagedecoder.decode(encoded);
    assertEquals(Command.BEAMEROFF, beameroff.getCommand(), "BEAMEROFF");
    assertEquals(RESPONSE, beameroff.getHeader(), "RESPONSE");
    assertEquals("anaconda", beameroff.getUserid(), "USERID");
  }



  @Test
  public void decodeBREAKROOMS_REQUEST() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"BREAKROOMS\","
        + "\"DATASET\":{"
        + "\"USERID\":\"anaconda\""
        + "}}";

    BREAKROOMS breakrooms = (BREAKROOMS) messagedecoder.decode(encoded);
    assertEquals(Command.BREAKROOMS, breakrooms.getCommand(), "BREAKROOMS");
    assertEquals(REQUEST, breakrooms.getHeader(), "REQUEST");
    assertEquals("anaconda", breakrooms.getUserid(), "USERID");
  }



  @Test
  public void decodeBREAKROOMS_RESPONSE() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"BREAKROOMS\","
        + "\"DATASET\":{"
        + "\"USERID\":\"anaconda\","
        + "\"ROOMS\":[\"schach\",\"fussball\"]"
        + "}}";

    BREAKROOMS breakrooms = (BREAKROOMS) messagedecoder.decode(encoded);
    assertEquals(Command.BREAKROOMS, breakrooms.getCommand(), "BREAKROOMS");
    assertEquals(RESPONSE, breakrooms.getHeader(), "RESPONSE");
    assertEquals("anaconda", breakrooms.getUserid(), "USERID");
    assertEquals("schach", breakrooms.getRooms()[0], "schach");
    assertEquals("fussball", breakrooms.getRooms()[1], "fussball");
  }



  @Test
  public void decodeCALLPRIVATECHAT_CONFIRM() throws DecodeException {
    String encoded = "{\"HEADER\":\"CONFIRM\"," + "\"COMMAND\":\"CALLPRIVATECHAT\","
        + "\"DATASET\":{"
        + "\"LOCAL_NICKNAME\":\"anaconda\","
        + "\"LOCAL_SESSIONID\":\"7\","
        + "\"REMOTE_NICKNAME\":\"Jester\","
        + "\"REMOTE_SESSIONID\":\"8\"}"
        + "}";

    CALLPRIVATECHAT call = (CALLPRIVATECHAT) messagedecoder.decode(encoded);

    assertEquals(Command.CALLPRIVATECHAT, call.getCommand(), "command");
    assertEquals(CONFIRM, call.getHeader(), "header");
    assertEquals(Protocol.DATASET, call.getDataset(), "dataset");
    assertEquals("anaconda", call.getLocalNickname(), "local Nickname");
    assertEquals("Jester", call.getRemoteNickname(), "remote Nickname");
    assertEquals(null, call.getSenderUID(), "sender uid");
    assertEquals("7", call.getLocalSessionid(), "remote Nickname");
    assertEquals("8", call.getRemoteSessionid(), "remote Nickname");
  }



  @Test
  public void decodeCALLPRIVATECHAT_HEADER() throws DecodeException {
    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"CALLPRIVATECHAT\","
        + "\"DATASET\":{"
        + "\"LOCAL_NICKNAME\":\"anaconda\","
        + "\"LOCAL_SESSIONID\":\"7\","
        + "\"REMOTE_NICKNAME\":\"anaconda\","
        + "\"REMOTE_SESSIONID\":\"7\","
        + "\"ERRORMESSAGE\":\"abc\"}"
        + "}";

    CALLPRIVATECHAT call = (CALLPRIVATECHAT) messagedecoder.decode(encoded);

    assertEquals(Command.CALLPRIVATECHAT, call.getCommand(), "command");
    assertEquals(ERROR, call.getHeader(), "header");
    assertEquals(Protocol.DATASET, call.getDataset(), "dataset");
    assertEquals("anaconda", call.getLocalNickname(), "local Nickname");
    assertEquals("anaconda", call.getRemoteNickname(), "remote Nickname");
    assertEquals(null, call.getSenderUID(), "sender uid");
    assertEquals("7", call.getLocalSessionid(), "remote Nickname");
    assertEquals("7", call.getRemoteSessionid(), "remote Nickname");
    assertEquals("abc", call.getErrorMessage(), "errorMessage");
  }



  @Test
  public void decodeCALLPRIVATECHAT_REQUEST() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CALLPRIVATECHAT\","
        + "\"DATASET\":{"
        + "\"SENDER_UID\":\"hdh39678\","
        + "\"LOCAL_NICKNAME\":\"anaconda\","
        + "\"REMOTE_NICKNAME\":\"jester\"}}";

    CALLPRIVATECHAT call = (CALLPRIVATECHAT) messagedecoder.decode(encoded);

    assertEquals(Command.CALLPRIVATECHAT, call.getCommand(), "command");
    assertEquals(REQUEST, call.getHeader(), "header");
    assertEquals(Protocol.DATASET, call.getDataset(), "dataset");
    assertEquals("anaconda", call.getLocalNickname(), "local Nickname");
    assertEquals("jester", call.getRemoteNickname(), "remote Nickname");
    assertEquals("hdh39678", call.getSenderUID(), "sender uid");

  }



  @Test
  public void decodeCALLPRIVATECHAT_RESPONSE() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CALLPRIVATECHAT\","
        + "\"DATASET\":{"
        + "\"LOCAL_NICKNAME\":\"anaconda\","
        + "\"LOCAL_SESSIONID\":\"7\","
        + "\"REMOTE_NICKNAME\":\"Jester\","
        + "\"REMOTE_SESSIONID\":\"8\","
        + "\"TEXT\":\"Der Anwender nimmt keine Anrufe entgegen.\"}"
        + "}";

    CALLPRIVATECHAT call = (CALLPRIVATECHAT) messagedecoder.decode(encoded);

    assertEquals(Command.CALLPRIVATECHAT, call.getCommand(), "command");
    assertEquals(RESPONSE, call.getHeader(), "header");
    assertEquals(Protocol.DATASET, call.getDataset(), "dataset");
    assertEquals("anaconda", call.getLocalNickname(), "local Nickname");
    assertEquals("Jester", call.getRemoteNickname(), "remote Nickname");
    assertEquals(null, call.getSenderUID(), "sender uid");
    assertEquals("7", call.getLocalSessionid(), "remote Nickname");
    assertEquals("8", call.getRemoteSessionid(), "remote Nickname");
    assertEquals("Der Anwender nimmt keine Anrufe entgegen.", call.getText(), "text");
  }



  @Test
  public void decodeCALLPRIVATECHAT_RESPONSE_MULTILINGUALKEY() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CALLPRIVATECHAT\","
        + "\"DATASET\":{"
        + "\"LOCAL_NICKNAME\":\"anaconda\","
        + "\"LOCAL_SESSIONID\":\"7\","
        + "\"REMOTE_NICKNAME\":\"Jester\","
        + "\"REMOTE_SESSIONID\":\"8\","
        + "\"TEXT\":\"Der Anwender nimmt keine Anrufe entgegen.\","
        + "\"MULTILINGUALKEY\":\"Server_Lehnt_Chatanfragen_Ab\""
        + "}}";

    CALLPRIVATECHAT call = (CALLPRIVATECHAT) messagedecoder.decode(encoded);

    assertEquals(Command.CALLPRIVATECHAT, call.getCommand(), "command");
    assertEquals(RESPONSE, call.getHeader(), "header");
    assertEquals(Protocol.DATASET, call.getDataset(), "dataset");
    assertEquals("anaconda", call.getLocalNickname(), "local Nickname");
    assertEquals("Jester", call.getRemoteNickname(), "remote Nickname");
    assertEquals(null, call.getSenderUID(), "sender uid");
    assertEquals("7", call.getLocalSessionid(), "remote Nickname");
    assertEquals("8", call.getRemoteSessionid(), "remote Nickname");
    assertEquals("Der Anwender nimmt keine Anrufe entgegen.", call.getText(), "text");
    assertEquals(KEY.SERVER_LEHNT_CHATANFRAGEN_AB, call.getMultilingualkey(), "MULTILINGUALKEY");
  }



  @Test
  public void decodeCALLPRIVATECHAT_ERROR_OHNE_ErrorMessage() throws DecodeException {
    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"CALLPRIVATECHAT\","
        + "\"DATASET\":{"
        + "\"LOCAL_NICKNAME\":\"anaconda\","
        + "\"LOCAL_SESSIONID\":\"7\","
        + "\"REMOTE_NICKNAME\":\"anaconda\","
        + "\"REMOTE_SESSIONID\":\"8\","
        + "\"TEXT\":\"einsame Seelen\","
        + "\"MULTILINGUALKEY\":\"String_Einsame_Seelen\""
        + "}}";

    CALLPRIVATECHAT call = (CALLPRIVATECHAT) messagedecoder.decode(encoded);

    assertEquals(Command.CALLPRIVATECHAT, call.getCommand(), "command");
    assertEquals(ERROR, call.getHeader(), "header");
    assertEquals(Protocol.DATASET, call.getDataset(), "dataset");
    assertEquals("anaconda", call.getLocalNickname(), "local Nickname");
    assertEquals("anaconda", call.getRemoteNickname(), "remote Nickname");
    assertEquals(null, call.getSenderUID(), "sender uid");
    assertEquals("7", call.getLocalSessionid(), "remote Nickname");
    assertEquals("8", call.getRemoteSessionid(), "remote Nickname");
    assertEquals("einsame Seelen", call.getText(), "text");
    assertEquals(KEY.STRING_EINSAME_SEELEN, call.getMultilingualkey(), "MULTILINGUALKEY");
  }



  @Test
  public void decodeCALLPRIVATECHAT_ERROR_ErrorMessage_MULTILINGUALKEY() throws DecodeException {
    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"CALLPRIVATECHAT\","
        + "\"DATASET\":{"
        + "\"ERRORMESSAGE\":\"anaconda kann keine weiteren Chats entgegennehmen.\","
        + "\"MULTILINGUALKEY\":\"Server_Chat_Exhausted\""
        + "}}";

    CALLPRIVATECHAT call = (CALLPRIVATECHAT) messagedecoder.decode(encoded);

    assertEquals(Command.CALLPRIVATECHAT, call.getCommand(), "command");
    assertEquals(ERROR, call.getHeader(), "header");
    assertEquals(Protocol.DATASET, call.getDataset(), "dataset");
    assertEquals(
        "anaconda kann keine weiteren Chats entgegennehmen.", call.getErrorMessage(), "ERRORMESSAGE"
    );
    assertEquals(KEY.SERVER_CHAT_EXHAUSTED, call.getMultilingualkey(), "MULTILINGUALKEY");
  }



  @Test
  public void decodeCALLPRIVATECHAT_RESPONSE_ohne_TEXT() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CALLPRIVATECHAT\","
        + "\"DATASET\":{"
        + "\"LOCAL_NICKNAME\":\"anaconda\","
        + "\"LOCAL_SESSIONID\":\"7\","
        + "\"REMOTE_NICKNAME\":\"Jester\","
        + "\"REMOTE_SESSIONID\":\"8\""
        + "}}";

    CALLPRIVATECHAT call = (CALLPRIVATECHAT) messagedecoder.decode(encoded);

    assertEquals(Command.CALLPRIVATECHAT, call.getCommand(), "command");
    assertEquals(RESPONSE, call.getHeader(), "header");
    assertEquals(Protocol.DATASET, call.getDataset(), "dataset");
    assertEquals("anaconda", call.getLocalNickname(), "local Nickname");
    assertEquals("Jester", call.getRemoteNickname(), "remote Nickname");
    assertEquals(null, call.getSenderUID(), "sender uid");
    assertEquals("7", call.getLocalSessionid(), "remote Nickname");
    assertEquals("8", call.getRemoteSessionid(), "remote Nickname");
    assertNull(call.getText(), "text");
  }



  @Test
  public void decodeCALLREMOTEUSER_CONFIRM() throws DecodeException {

    String encoded = "{\"HEADER\":\"CONFIRM\"," + "\"COMMAND\":\"CALLREMOTEUSER\","
        + "\"DATASET\":{"
        + "\"LOCAL_NICKNAME\":\"anaconda\","
        + "\"LOCAL_SESSIONID\":\"7\","
        + "\"REMOTE_NICKNAME\":\"Jester\","
        + "\"REMOTE_SESSIONID\":\"8\"}"
        + "}";

    CALLREMOTEUSER call = (CALLREMOTEUSER) messagedecoder.decode(encoded);

    assertEquals(Command.CALLREMOTEUSER, call.getCommand(), "command");
    assertEquals(CONFIRM, call.getHeader(), "header");
    assertEquals(Protocol.DATASET, call.getDataset(), "dataset");
    assertEquals("anaconda", call.getLocalNickname(), "local Nickname");
    assertEquals("Jester", call.getRemoteNickname(), "remote Nickname");
    assertEquals("8", call.getRemoteSessionid(), "remote sessionid");
    assertEquals("7", call.getLocalSessionid(), "local sessionid");

  }



  @Test
  public void decodeCALLREMOTEUSER_ERROR() throws DecodeException {

    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"CALLREMOTEUSER\","
        + "\"DATASET\":{"
        + "\"ERRORMESSAGE\":\"The user is offline\"}"
        + "}";

    CALLREMOTEUSER call = (CALLREMOTEUSER) messagedecoder.decode(encoded);

    assertEquals(Command.CALLREMOTEUSER, call.getCommand(), "command");
    assertEquals(ERROR, call.getHeader(), "header");
    assertEquals(Protocol.DATASET, call.getDataset(), "dataset");
    assertEquals("The user is offline", call.getErrorMessage(), "errorMessage");

  }



  @Test
  public void decodeCALLREMOTEUSER_REQUEST() throws DecodeException {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CALLREMOTEUSER\","
        + "\"DATASET\":{"
        + "\"LOCAL_NICKNAME\":\"anaconda\","
        + "\"LOCAL_SESSIONID\":\"7\","
        + "\"REMOTE_NICKNAME\":\"Jester\","
        + "\"REMOTE_SESSIONID\":\"8\"}"
        + "}";

    CALLREMOTEUSER call = (CALLREMOTEUSER) messagedecoder.decode(encoded);

    assertEquals(Command.CALLREMOTEUSER, call.getCommand(), "command");
    assertEquals(REQUEST, call.getHeader(), "header");
    assertEquals(Protocol.DATASET, call.getDataset(), "dataset");
    assertEquals("anaconda", call.getLocalNickname(), "local Nickname");
    assertEquals("Jester", call.getRemoteNickname(), "remote Nickname");
    assertEquals("8", call.getRemoteSessionid(), "remote sessionid");
    assertEquals("7", call.getLocalSessionid(), "local sessionid");

  }



  @Test
  public void decodeCALLREMOTEUSER_RESPONSE() throws DecodeException {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CALLREMOTEUSER\","
        + "\"DATASET\":{"
        + "\"LOCAL_NICKNAME\":\"anaconda\","
        + "\"LOCAL_SESSIONID\":\"7\","
        + "\"REMOTE_NICKNAME\":\"Jester\","
        + "\"REMOTE_SESSIONID\":\"8\"}"
        + "}";

    CALLREMOTEUSER call = (CALLREMOTEUSER) messagedecoder.decode(encoded);

    assertEquals(Command.CALLREMOTEUSER, call.getCommand(), "command");
    assertEquals(RESPONSE, call.getHeader(), "header");
    assertEquals(Protocol.DATASET, call.getDataset(), "dataset");
    assertEquals("anaconda", call.getLocalNickname(), "local Nickname");
    assertEquals("Jester", call.getRemoteNickname(), "remote Nickname");
    assertEquals("8", call.getRemoteSessionid(), "remote sessionid");
    assertEquals("7", call.getLocalSessionid(), "local sessionid");

  }



  @Test
  public void decodeCANDIDATETOPICMEMBER_REQUEST() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CANDIDATETOPICMEMBER\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"ROOM\":\"fussball\"}"
        + "}";

    CANDIDATETOPICMEMBER result = (CANDIDATETOPICMEMBER) messagedecoder.decode(encoded);
    assertEquals("louisxiv", result.getUserid(), "userid");
    assertEquals("fussball", result.getRoom(), "room");

  }



  @Test
  public void decodeCANDIDATETOPICMEMBER_RESPONSE() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CANDIDATETOPICMEMBER\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"ROOM\":\"fussball\","
        + "\"ROOMTYPE\":\"BESPRECHUNGSRAUM\","
        + "\"TOPICMEMBER\":["
        + "{\"USERID\":\"jester\",\"NICKNAME\":\"anaconda\"},"
        + "{\"USERID\":\"luis\",\"NICKNAME\":\"viper\""
        + "}]"
        + "}}";

    CANDIDATETOPICMEMBER result = (CANDIDATETOPICMEMBER) messagedecoder.decode(encoded);
    assertEquals("louisxiv", result.getUserid(), "userid");
    assertEquals("fussball", result.getRoom(), "room");
    ChatUser[] chatuser = result.getChatUser();
    assertEquals(2, chatuser.length, "2 Datensätze");
    assertEquals("jester", chatuser[0].getUserid(), "userid0");
    assertEquals("anaconda", chatuser[0].getNickname(), "nickname0");
    assertEquals("luis", chatuser[1].getUserid(), "userid1");
    assertEquals("viper", chatuser[1].getNickname(), "nickname1");
    assertEquals(Roomtype.BESPRECHUNGSRAUM, result.getRoomtype(), "ROOMTYPE");

  }



  @Test
  public void decodeCHANGETOPICMEMBER_ERROR() throws DecodeException {
    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"CHANGETOPICMEMBER\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"ROOM\":\"Nexus\","
        + "\"ERRORMESSAGE\":\"Room <Nexus> does not exist.\""
        + "}}";

    CHANGETOPICMEMBER result = (CHANGETOPICMEMBER) messagedecoder.decode(encoded);
    assertEquals("12345678", result.getUserid(), "userid");
    assertEquals("Nexus", result.getRoom(), "room");
    assertEquals("Room <Nexus> does not exist.", result.getErrorMessage(), "errorMessage");

  }



  @Test
  public void decodeCHANGETOPICMEMBER_REQUEST() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CHANGETOPICMEMBER\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"ROOM\":\"fussball\","
        + "\"ROOMTYPE\":\"BESPRECHUNGSRAUM\","
        + "\"TOPICMEMBER\":["
        + "{\"USERID\":\"jester\",\"NICKNAME\":\"anaconda\"},"
        + "{\"USERID\":\"luis\",\"NICKNAME\":\"viper\""
        + "}]"

        + "}}";

    CHANGETOPICMEMBER result = (CHANGETOPICMEMBER) messagedecoder.decode(encoded);
    ChatUser[] chatuser = result.getChatUser();
    assertEquals("louisxiv", result.getUserid(), "userid");
    assertEquals("fussball", result.getRoom(), "room");
    assertEquals(2, chatuser.length, "chatuser");
    assertEquals("jester", chatuser[0].getUserid(), "userid1");
    assertEquals("anaconda", chatuser[0].getNickname(), "nickname1");
    assertEquals("luis", chatuser[1].getUserid(), "userid2");
    assertEquals("viper", chatuser[1].getNickname(), "nickname2");
    assertEquals(Roomtype.BESPRECHUNGSRAUM, result.getRoomtype(), "");

  }



  @Test
  public void decodeCHANGETOPICMEMBER_REQUEST_DecodeException() {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CHANGETOPICMEMBER\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"ROOM\":\"fussball\""
        + "}}";

    CHANGETOPICMEMBER result;
    try {
      result = (CHANGETOPICMEMBER) messagedecoder.decode(encoded);
      fail("DecodeException nicht geworfen");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void decodeCHANGETOPICMEMBER_REQUEST_LEER() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CHANGETOPICMEMBER\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"ROOM\":\"fussball\","
        + "\"ROOMTYPE\":\"BESPRECHUNGSRAUM\","
        + "\"TOPICMEMBER\":["
        + "]"

        + "}}";

    CHANGETOPICMEMBER result = (CHANGETOPICMEMBER) messagedecoder.decode(encoded);
    ChatUser[] chatuser = result.getChatUser();
    assertEquals("louisxiv", result.getUserid(), "userid");
    assertEquals("fussball", result.getRoom(), "room");
    assertEquals(0, chatuser.length, "chatuser");
    assertEquals(Roomtype.BESPRECHUNGSRAUM, result.getRoomtype(), "");

  }



  @Test
  public void decodeCHANGETOPICMEMBER_RESPONSE() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CHANGETOPICMEMBER\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"ROOM\":\"fussball\","
        + "\"ROOMTYPE\":\"BESPRECHUNGSRAUM\","
        + "\"TOPICMEMBER\":["
        + "{\"USERID\":\"jester\",\"NICKNAME\":\"anaconda\"},"
        + "{\"USERID\":\"luis\",\"NICKNAME\":\"viper\""
        + "}]"
        + "}}";

    CHANGETOPICMEMBER result = (CHANGETOPICMEMBER) messagedecoder.decode(encoded);
    assertEquals("louisxiv", result.getUserid(), "userid");
    assertEquals("fussball", result.getRoom(), "room");
    ChatUser[] chatuser = result.getChatUser();
    assertEquals(2, chatuser.length, "2 Datensätze");
    assertEquals("jester", chatuser[0].getUserid(), "userid0");
    assertEquals("anaconda", chatuser[0].getNickname(), "nickname0");
    assertEquals("luis", chatuser[1].getUserid(), "userid1");
    assertEquals("viper", chatuser[1].getNickname(), "nickname1");
    assertEquals(Roomtype.BESPRECHUNGSRAUM, result.getRoomtype(), "");

  }



  @Test
  public void decodeCHATMESSAGE_REQUEST() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CHATMESSAGE\","
        + "\"DATASET\":{"
        + "\"ROOM\":\"room\","
        + "\"MESSAGE\":\"Hallo meine Freunde\","
        + "\"CHATUSER\":{"
        + "\"BACKGROUND_COLOR\":-1,"
        + "\"FOREGROUND_COLOR\":-16777216,"
        + "\"USERID\":\"louisxiv\","
        + "\"NICKNAME\":\"anaconda\"}}}";

    CHATMESSAGE chatmessage = (CHATMESSAGE) messagedecoder.decode(encoded);

    assertEquals(Command.CHATMESSAGE, chatmessage.getCommand(), "command");
    assertEquals(REQUEST, chatmessage.getHeader(), "header");
    assertEquals(Protocol.DATASET, chatmessage.getDataset(), "dataset");
    ChatUser chatuser = chatmessage.getChatUser();
    assertEquals("louisxiv", chatuser.getUserid(), "userid");
    assertEquals("room", chatmessage.getRoom(), "room");
    assertEquals(-16777216, chatuser.getForegroundColor(), "fgColor");
    assertEquals(-1, chatuser.getBackgroundColor(), "bgColor");
    assertEquals("anaconda", chatuser.getNickname(), "nickname");
    assertEquals("Hallo meine Freunde", chatmessage.getMessage(), "message");

  }



  @Test
  public void decodeCHATMESSAGE_REQUEST_2() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\",\"COMMAND\":\"CHATMESSAGE\",\"DATASET\":{"
        + "\"ROOM\":\"Tester\","
        + "\"MESSAGE\":\"anhang\","
        + "\"ATTACHMENT\":29098336975847574,"
        + "\"FILENAME\":\"Top 5 Highest Win Rate Openings for White (2025).mp4\","
        + "\"FILESIZE\":67934764,"
        + "\"CHATUSER\":{\"BACKGROUND_COLOR\":-1772550,"
        + "\"FOREGROUND_COLOR\":-14607386,"
        + "\"USERID\":\"louisxiv\","
        + "\"NICKNAME\":\"anaconda\"}}}";

    CHATMESSAGE chatmessage = (CHATMESSAGE) messagedecoder.decode(encoded);
    assertEquals(Command.CHATMESSAGE, chatmessage.getCommand(), "CHATMESSAGE");
    assertEquals(REQUEST, chatmessage.getHeader(), "header");
    assertEquals(Protocol.DATASET, chatmessage.getDataset(), "dataset");
    ChatUser chatuser = chatmessage.getChatUser();
    assertEquals("louisxiv", chatuser.getUserid(), "userid");
    assertEquals("Tester", chatmessage.getRoom(), "room");
    assertEquals(-14607386, chatuser.getForegroundColor(), "fgColor");
    assertEquals(-1772550, chatuser.getBackgroundColor(), "bgColor");
    assertEquals("anaconda", chatuser.getNickname(), "nickname");
    assertEquals("anhang", chatmessage.getMessage(), "message");

    assertEquals(29098336975847574l, chatmessage.getAttachment(), "attachment");
    assertEquals("Top 5 Highest Win Rate Openings for White (2025).mp4", chatmessage.getFilename());
    assertEquals(67934764l, chatmessage.getFilesize());

  }



  @Test
  public void decodeCHATMESSAGE_REQUEST_ATTACHMENT() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CHATMESSAGE\","
        + "\"DATASET\":{"
        + "\"ROOM\":\"room\","
        + "\"MESSAGE\":\"Hallo meine Freunde\","
        + "\"ATTACHMENT\":33,"
        + "\"FILENAME\":\"name.txt\","
        + "\"FILESIZE\":1024,"
        + "\"CHATUSER\":{"
        + "\"BACKGROUND_COLOR\":-1,"
        + "\"FOREGROUND_COLOR\":-16777216,"
        + "\"USERID\":\"louisxiv\","
        + "\"NICKNAME\":\"anaconda\"}}}";

    CHATMESSAGE chatmessage = (CHATMESSAGE) messagedecoder.decode(encoded);

    assertEquals(Command.CHATMESSAGE, chatmessage.getCommand(), "command");
    assertEquals(REQUEST, chatmessage.getHeader(), "header");
    assertEquals(Protocol.DATASET, chatmessage.getDataset(), "dataset");
    ChatUser chatuser = chatmessage.getChatUser();
    assertEquals("louisxiv", chatuser.getUserid(), "userid");
    assertEquals("room", chatmessage.getRoom(), "room");
    assertEquals(-16777216, chatuser.getForegroundColor(), "fgColor");
    assertEquals(-1, chatuser.getBackgroundColor(), "bgColor");
    assertEquals("anaconda", chatuser.getNickname(), "nickname");
    assertEquals("Hallo meine Freunde", chatmessage.getMessage(), "message");
    assertEquals(33, chatmessage.getAttachment(), "ATTACHMENT");
    assertEquals("name.txt", chatmessage.getFilename());
    assertEquals(1024, chatmessage.getFilesize());

  }



  @Test
  public void decodeCHATMESSAGE_REQUEST_UMLAUTE() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CHATMESSAGE\","
        + "\"DATASET\":{"
        + "\"ROOM\":\"room\","
        + "\"MESSAGE\":\"äöüßéáúéíóú\","
        + "\"ATTACHMENT\":33,"
        + "\"FILENAME\":\"name.txt\","
        + "\"FILESIZE\":1024,"
        + "\"CHATUSER\":{"
        + "\"BACKGROUND_COLOR\":-1,"
        + "\"FOREGROUND_COLOR\":-16777216,"
        + "\"USERID\":\"louisxiv\","
        + "\"NICKNAME\":\"anaconda\"}}}";

    CHATMESSAGE chatmessage = (CHATMESSAGE) messagedecoder.decode(encoded);

    assertEquals(Command.CHATMESSAGE, chatmessage.getCommand(), "command");
    assertEquals(REQUEST, chatmessage.getHeader(), "header");
    assertEquals(Protocol.DATASET, chatmessage.getDataset(), "dataset");
    ChatUser chatuser = chatmessage.getChatUser();
    assertEquals("louisxiv", chatuser.getUserid(), "userid");
    assertEquals("room", chatmessage.getRoom(), "room");
    assertEquals(-16777216, chatuser.getForegroundColor(), "fgColor");
    assertEquals(-1, chatuser.getBackgroundColor(), "bgColor");
    assertEquals("anaconda", chatuser.getNickname(), "nickname");
    assertEquals("äöüßéáúéíóú", chatmessage.getMessage(), "message");
    assertEquals(33, chatmessage.getAttachment(), "ATTACHMENT");
    assertEquals("name.txt", chatmessage.getFilename());
    assertEquals(1024, chatmessage.getFilesize());

  }



  @Test
  public void decodeCHATMESSAGE_REQUEST2() throws DecodeException {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CHATMESSAGE\","
        + "\"DATASET\":{"
        + "\"ROOM\":\"Everyone\","
        + "\"MESSAGE\":\"Hallo meine Freunde\","
        + "\"CHATUSER\":{\"BACKGROUND_COLOR\":-12975360,"
        + "\"FOREGROUND_COLOR\":-65536,"
        + "\"USERID\":\"louisxiv\","
        + "\"NICKNAME\":\"python\"}}}";

    CHATMESSAGE chatmessage = (CHATMESSAGE) messagedecoder.decode(encoded);

    assertEquals(Command.CHATMESSAGE, chatmessage.getCommand(), "command");
    assertEquals(REQUEST, chatmessage.getHeader(), "header");
    assertEquals(Protocol.DATASET, chatmessage.getDataset(), "dataset");
    ChatUser chatuser = chatmessage.getChatUser();
    assertEquals("louisxiv", chatuser.getUserid(), "userid");
    assertEquals("Everyone", chatmessage.getRoom(), "room");
    assertEquals(-65536, chatuser.getForegroundColor(), "fgColor");
    assertEquals(-12975360, chatuser.getBackgroundColor(), "bgColor");
    assertEquals("python", chatuser.getNickname(), "nickname");
    assertEquals("Hallo meine Freunde", chatmessage.getMessage(), "message");

  }



  @Test
  public void decodeCHATMESSAGE_RESPONSE() throws DecodeException {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CHATMESSAGE\","
        + "\"DATASET\":{"
        + "\"ROOM\":\"Everyone\","
        + "\"DATETIME\":9223372036854775807,"
        + "\"MESSAGE\":\"Hallo meine Freunde\","
        + "\"CHATUSER\":{\"BACKGROUND_COLOR\":-12975360,"
        + "\"FOREGROUND_COLOR\":-65536,"
        + "\"USERID\":\"louisxiv\","
        + "\"NICKNAME\":\"python\"}}}";

    CHATMESSAGE chatmessage = (CHATMESSAGE) messagedecoder.decode(encoded);

    assertEquals(Command.CHATMESSAGE, chatmessage.getCommand(), "command");
    assertEquals(RESPONSE, chatmessage.getHeader(), "header");
    assertEquals(Protocol.DATASET, chatmessage.getDataset(), "dataset");
    ChatUser chatuser = chatmessage.getChatUser();
    assertEquals("louisxiv", chatuser.getUserid(), "userid");
    assertEquals("Everyone", chatmessage.getRoom(), "room");
    assertEquals(-65536, chatuser.getForegroundColor(), "fgColor");
    assertEquals(-12975360, chatuser.getBackgroundColor(), "bgColor");
    assertEquals("python", chatuser.getNickname(), "nickname");
    assertEquals("Hallo meine Freunde", chatmessage.getMessage(), "message");
    assertEquals(Long.MAX_VALUE, chatmessage.getDatetime(), "datetime");

  }



  @Test
  public void decodeCHATMESSAGE_RESPONSE_ATTACHMENT() throws DecodeException {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CHATMESSAGE\","
        + "\"DATASET\":{"
        + "\"ROOM\":\"Everyone\","
        + "\"DATETIME\":666,"
        + "\"MESSAGE\":\"Hallo meine Freunde\","
        + "\"ATTACHMENT\":99,"
        + "\"FILENAME\":\"name.txt\","
        + "\"FILESIZE\":"
        + String.valueOf(Long.MAX_VALUE)
        + ","
        + "\"CHATUSER\":{"
        + "\"BACKGROUND_COLOR\":-1,"
        + "\"FOREGROUND_COLOR\":-65536,"
        + "\"USERID\":\"louisxiv\","
        + "\"NICKNAME\":\"python\"}}}";

    CHATMESSAGE chatmessage = (CHATMESSAGE) messagedecoder.decode(encoded);

    assertEquals(Command.CHATMESSAGE, chatmessage.getCommand(), "command");
    assertEquals(RESPONSE, chatmessage.getHeader(), "header");
    assertEquals(Protocol.DATASET, chatmessage.getDataset(), "dataset");
    ChatUser chatuser = chatmessage.getChatUser();
    assertEquals("louisxiv", chatuser.getUserid(), "userid");
    assertEquals("Everyone", chatmessage.getRoom(), "room");
    assertEquals(-65536, chatuser.getForegroundColor(), "fgColor");
    assertEquals(-1, chatuser.getBackgroundColor(), "bgColor");
    assertEquals("python", chatuser.getNickname(), "nickname");
    assertEquals("Hallo meine Freunde", chatmessage.getMessage(), "message");
    assertEquals(666, chatmessage.getDatetime(), "datetime");
    assertEquals(Long.MAX_VALUE, chatmessage.getFilesize());
    assertEquals("name.txt", chatmessage.getFilename());
    assertEquals(99, chatmessage.getAttachment());

  }



  @Test
  public void decodeCHATMESSAGE_RESPONSE_Ohne_Time() throws DecodeException {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CHATMESSAGE\","
        + "\"DATASET\":{"
        + "\"ROOM\":\"Everyone\","
        + "\"MESSAGE\":\"Hallo meine Freunde\","
        + "\"CHATUSER\":{\"BACKGROUND_COLOR\":-12975360,"
        + "\"FOREGROUND_COLOR\":-65536,"
        + "\"USERID\":\"louisxiv\","
        + "\"NICKNAME\":\"python\"}}}";

    CHATMESSAGE chatmessage = (CHATMESSAGE) messagedecoder.decode(encoded);

    assertEquals(Command.CHATMESSAGE, chatmessage.getCommand(), "command");
    assertEquals(RESPONSE, chatmessage.getHeader(), "header");
    assertEquals(Protocol.DATASET, chatmessage.getDataset(), "dataset");
    ChatUser chatuser = chatmessage.getChatUser();
    assertEquals("louisxiv", chatuser.getUserid(), "userid");
    assertEquals("Everyone", chatmessage.getRoom(), "room");
    assertEquals(-65536, chatuser.getForegroundColor(), "fgColor");
    assertEquals(-12975360, chatuser.getBackgroundColor(), "bgColor");
    assertEquals("python", chatuser.getNickname(), "nickname");
    assertEquals("Hallo meine Freunde", chatmessage.getMessage(), "message");
    assertNull(chatmessage.getDatetime(), "datetime");

  }



  @Test
  public void decodeCHATONLINELIST_REQUEST()
      throws DecodeException, JsonSyntaxException, UnsupportedEncodingException {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CHATONLINELIST\","
        + "\"DATASET\":{\"CHATONLINELIST\":["
        + "{\"ROOM\":\"Nexus\",\"ONLINE\":1},"
        + "{\"ROOM\":\"Everyone\",\"ONLINE\":2},"
        + "{\"ROOM\":\"Developer\",\"ONLINE\":2}]}}";

    CHATONLINELIST chatonline = (CHATONLINELIST) messagedecoder.decodeCHATONLINELIST(encoded);
    assertEquals(Command.CHATONLINELIST, chatonline.getCommand(), "command");
    assertEquals(REQUEST, chatonline.getHeader(), "header");
    assertEquals(Protocol.DATASET, chatonline.getDataset(), "dataset");
    assertEquals(3, chatonline.getChatOnline().length, "3 Felder");
  }



  @Test
  public void decodeCHATONLINELIST_REQUEST_2() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CHATONLINELIST\","
        + "\"DATASET\":{"
        + "\"CHATONLINELIST\":["
        + "{\"ROOM\":\"everyone\",\"ONLINE\":99},"
        + "{\"ROOM\":\"ingress\",\"ONLINE\":7}"
        + "]}}";

    CHATONLINELIST onlinelist = (CHATONLINELIST) messagedecoder.decode(encoded);
    assertEquals(Command.CHATONLINELIST, onlinelist.getCommand(), "command");
    assertEquals(REQUEST, onlinelist.getHeader(), "command");
    assertEquals(Protocol.DATASET, onlinelist.getDataset(), "command");
    ChatOnline[] chatonline = onlinelist.getChatOnline();

    assertEquals("everyone", chatonline[0].getRoom(), "room_0");
    assertEquals(99, chatonline[0].getOnline(), "online_0");
    assertEquals("ingress", chatonline[1].getRoom(), "room_1");
    assertEquals(7, chatonline[1].getOnline(), "online_1");

  }



  @Test
  public void decodeCHATUSERLIST_ERROR() throws DecodeException {

    String encoded = "{\"HEADER\":\"ERROR\",\"COMMAND\":\"CHATUSERLIST\"}";

    CHATUSERLIST chatuseronline = (CHATUSERLIST) messagedecoder.decode(encoded);
    assertEquals(Command.CHATUSERLIST, chatuseronline.getCommand(), "command");
    assertEquals(ERROR, chatuseronline.getHeader(), "header");
  }



  @Test
  public void decodeCHATUSERLIST_ERROR_mitErrorCode() throws DecodeException {

    String encoded = "{\"HEADER\":\"ERROR\",\"COMMAND\":\"CHATUSERLIST\",\"DATASET\":{\"ERRORCODE\":-1,\"ERRORMESSAGE\":\"error\"}}";
    String bla = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"CHATUSERLIST\","
        + "\"DATASET\":{\"ERRORCODE\":-1,\"ERRORMESSAGE\":\"error\"}}";

    CHATUSERLIST chatuseronline = (CHATUSERLIST) messagedecoder.decode(encoded);
    assertEquals(Command.CHATUSERLIST, chatuseronline.getCommand(), "command");
    assertEquals(ERROR, chatuseronline.getHeader(), "header");
    assertEquals(-1, chatuseronline.getErrorCode(), "errorcode");
    assertEquals("error", chatuseronline.getErrorMessage(), "errormessage");
  }



  @Test
  public void decodeCHATUSERLIST_REQEUST() throws DecodeException {

    String encoded = "{\"HEADER\":\"REQUEST\",\"COMMAND\":\"CHATUSERLIST\",\"DATASET\":{\"ROOM\":\"Developer\"}}";

    CHATUSERLIST chatuseronline = (CHATUSERLIST) messagedecoder.decode(encoded);
    assertEquals(Command.CHATUSERLIST, chatuseronline.getCommand(), "command");
    assertEquals(REQUEST, chatuseronline.getHeader(), "header");
    assertEquals(Protocol.DATASET, chatuseronline.getDataset(), "dataset");
    assertEquals("Developer", chatuseronline.getRoom(), "room");

  }



  @Test
  public void decodeCHATUSERLIST_RESPONSE() throws DecodeException {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CHATUSERLIST\","
        + "\"DATASET\":{"
        + "\"ROOM\":\"room\","
        + "\"CHATUSERLIST\":["
        + "{\"USERID\":\"12345678\","
        + "\"FOREGROUND_COLOR\":-1,"
        + "\"BACKGROUND_COLOR\":-2,"
        + "\"NICKNAME\":\"louisxiv\"}]}}";

    CHATUSERLIST chatuseronline = (CHATUSERLIST) messagedecoder.decode(encoded);
    assertEquals(Command.CHATUSERLIST, chatuseronline.getCommand(), "command");
    assertEquals(RESPONSE, chatuseronline.getHeader(), "header");
    assertEquals(Protocol.DATASET, chatuseronline.getDataset(), "dataset");
    assertEquals(1, chatuseronline.getChatUser().length, "1 Felder");
    assertEquals("louisxiv", chatuseronline.getChatUser()[0].getNickname(), "Nickname");
    assertEquals("12345678", chatuseronline.getChatUser()[0].getUserid(), "userid");
    assertEquals(-2, chatuseronline.getChatUser()[0].getBackgroundColor(), "bgColor");
    assertEquals(-1, chatuseronline.getChatUser()[0].getForegroundColor(), "fgColor");
    assertEquals("room", chatuseronline.getRoom(), "rootm");
  }



  @Test
  public void decodeCHATUSERLIST_RESPONSE_ohneUSERID() throws DecodeException {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CHATUSERLIST\","
        + "\"DATASET\":{"
        + "\"ROOM\":\"room\","
        + "\"CHATUSERLIST\":["
        + "{\"FOREGROUND_COLOR\":-1,\"BACKGROUND_COLOR\":-2,\"NICKNAME\":\"louisxiv\"}]}}";

    CHATUSERLIST chatuseronline = (CHATUSERLIST) messagedecoder.decode(encoded);
    assertEquals(Command.CHATUSERLIST, chatuseronline.getCommand(), "command");
    assertEquals(RESPONSE, chatuseronline.getHeader(), "header");
    assertEquals(Protocol.DATASET, chatuseronline.getDataset(), "dataset");
    assertEquals(1, chatuseronline.getChatUser().length, "1 Felder");
    assertEquals("louisxiv", chatuseronline.getChatUser()[0].getNickname(), "Nickname");
    assertEquals(null, chatuseronline.getChatUser()[0].getUserid(), "userid,optional");
    assertEquals(-2, chatuseronline.getChatUser()[0].getBackgroundColor(), "bgColor");
    assertEquals(-1, chatuseronline.getChatUser()[0].getForegroundColor(), "fgColor");
    assertEquals("room", chatuseronline.getRoom(), "room");
  }



  @Test
  public void decodeCONFERENCE_REQUEST() throws DecodeException {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CONFERENCE\","
        + "\"DATASET\":{"
        + "\"KONFERENZNAME\":\"Dienstag Treff\","
        + "\"ORGANISATOR_UID\":\"anaconda\","
        + "\"USERID\":\"12345678\","
        + "\"SESSION\":\"3a\","
        + "\"OFFHOOK\":true"
        + "}}";

    CONFERENCE conference = (CONFERENCE) messagedecoder.decode(encoded);
    assertEquals(Command.CONFERENCE, conference.getCommand(), "COMMAND");
    assertEquals(HEADER.REQUEST, conference.getHeader(), "header");
    assertEquals(Protocol.DATASET, conference.getDataset(), "dataset");

    assertEquals("Dienstag Treff", conference.getKonferenzname());
    assertEquals("anaconda", conference.getOrganisatorUid());
    assertEquals("12345678", conference.getUserid());
    assertEquals("3a", conference.getSession());
    assertEquals(true, conference.isOffhook());
  }



  @Test
  public void decodeCONFERENCE_REQUEST_ohne_USERID() {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CONFERENCE\","
        + "\"DATASET\":{"
        + "\"KONFERENZNAME\":\"Dienstag Treff\","
        + "\"ORGANISATOR_UID\":\"anaconda\","
        + "\"SESSION\":\"3a\","
        + "\"OFFHOOK\":true"
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("DecodeException nicht geworfen, USERID fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void decodeCONFERENCE_REQUEST_ohne_SESSION() {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CONFERENCE\","
        + "\"DATASET\":{"
        + "\"KONFERENZNAME\":\"Dienstag Treff\","
        + "\"ORGANISATOR_UID\":\"anaconda\","
        + "\"USERID\":\"12345678\","
        + "\"OFFHOOK\":true"
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("DecodeException nicht geworfen, SESSION fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void decodeCONFERENCE_REQUEST_ERROR() {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CONFERENCE\","
        + "\"DATASET\":{"
        + "\"KONFERENZNAME\":\"Dienstag Treff\""
        + "}}";

    try {
      CONFERENCE conference = (CONFERENCE) messagedecoder.decode(encoded);
      fail("Exception nicht geworfen");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void decodeCONFERENCE_RESPONSE() throws DecodeException {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CONFERENCE\","
        + "\"DATASET\":{"
        + "\"KONFERENZNAME\":\"Dienstag Treff\","
        + "\"ORGANISATOR_UID\":\"anaconda\","
        + "\"USERID\":\"admin678\","
        + "\"OFFHOOK\":true,"
        + "\"KONFERENZRAUMUSER\":[{\"ANWESEND\":true,\"SESSION\":\"ab\",\"MUTE\":true,\"CHATUSER\":{\"USERID\":\"userid1\",\"NICKNAME\":\"admin678\",\"FOREGROUND_COLOR\":-2839,\"BACKGROUND_COLOR\":238}},"
        + "{\"ANWESEND\":true,\"SESSION\":\"cd\",\"MUTE\":true,\"CHATUSER\":{\"USERID\":\"userid2\",\"NICKNAME\":\"freigeist\",\"FOREGROUND_COLOR\":-9283,\"BACKGROUND_COLOR\":-293}},"
        + "{\"ANWESEND\":false,\"MUTE\":false,\"CHATUSER\":{\"USERID\":\"userid3\",\"NICKNAME\":\"Jester\",\"FOREGROUND_COLOR\":-7263,\"BACKGROUND_COLOR\":-9283}},"
        + "{\"ANWESEND\":false,\"MUTE\":false,\"CHATUSER\":{\"USERID\":\"userid4\",\"NICKNAME\":\"picasso\",\"FOREGROUND_COLOR\":-8273,\"BACKGROUND_COLOR\":-23}}]"

        + "}}";

    CONFERENCE conference = (CONFERENCE) messagedecoder.decode(encoded);
    assertEquals(Command.CONFERENCE, conference.getCommand(), "COMMAND");
    assertEquals(HEADER.RESPONSE, conference.getHeader(), "header");
    assertEquals(Protocol.DATASET, conference.getDataset(), "dataset");

    assertEquals("Dienstag Treff", conference.getKonferenzname());
    assertEquals("anaconda", conference.getOrganisatorUid());
    KonferenzraumUser[] konfernzraumuser = conference.getKonferenzraumUser();
    assertEquals(-2839, konfernzraumuser[0].getForegroundColor());
    assertEquals(238, konfernzraumuser[0].getBackgroundColor());
    assertEquals("admin678", konfernzraumuser[0].getNickname());
    assertEquals("userid1", konfernzraumuser[0].getUserid());
    assertTrue(konfernzraumuser[0].isAnwesend());
    assertEquals("ab", konfernzraumuser[0].getSession());
    assertTrue(konfernzraumuser[0].isMute());
    assertEquals("admin678", conference.getUserid());
    assertTrue(conference.isOffhook());

  }



  @Test
  public void decodeCONFERENCE_RESPONSE_ohne_OFFHOOK() {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CONFERENCE\","
        + "\"DATASET\":{"
        + "\"KONFERENZNAME\":\"Dienstag Treff\","
        + "\"ORGANISATOR_UID\":\"anaconda\","
        + "\"USERID\":\"admin678\","
        + "\"KONFERENZRAUMUSER\":[{\"ANWESEND\":true,\"SESSION\":\"ab\",\"CHATUSER\":{\"USERID\":\"userid1\",\"NICKNAME\":\"admin678\",\"FOREGROUND_COLOR\":-2839,\"BACKGROUND_COLOR\":238}},"
        + "{\"ANWESEND\":true,\"SESSION\":\"cd\",\"CHATUSER\":{\"USERID\":\"userid2\",\"NICKNAME\":\"freigeist\",\"FOREGROUND_COLOR\":-9283,\"BACKGROUND_COLOR\":-293}},"
        + "{\"ANWESEND\":false,\"CHATUSER\":{\"USERID\":\"userid3\",\"NICKNAME\":\"Jester\",\"FOREGROUND_COLOR\":-7263,\"BACKGROUND_COLOR\":-9283}},"
        + "{\"ANWESEND\":false,\"CHATUSER\":{\"USERID\":\"userid4\",\"NICKNAME\":\"picasso\",\"FOREGROUND_COLOR\":-8273,\"BACKGROUND_COLOR\":-23}}]"

        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("DecodeException nicht geworfen, OFFHOOK fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }

  }



  @Test
  public void decodeCONFERENCE_RESPONSE_JERSEY() throws DecodeException {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CONFERENCE\","
        + "\"DATASET\":{"
        + "\"KONFERENZNAME\":\"Dienstag Treff\","
        + "\"ORGANISATOR_UID\":\"anaconda\","
        + "\"KONFERENZRAUMUSER\":[{\"ANWESEND\":true,\"SESSION\":\"2b\",\"CHATUSER\":{\"USERID\":\"userid1\",\"NICKNAME\":\"admin678\",\"FOREGROUND_COLOR\":-2839,\"BACKGROUND_COLOR\":238}},"
        + "{\"ANWESEND\":true,\"SESSION\":\"1a\",\"CHATUSER\":{\"USERID\":\"userid2\",\"NICKNAME\":\"freigeist\",\"FOREGROUND_COLOR\":-9283,\"BACKGROUND_COLOR\":-293}},"
        + "{\"ANWESEND\":false,\"CHATUSER\":{\"USERID\":\"userid3\",\"NICKNAME\":\"Jester\",\"FOREGROUND_COLOR\":-7263,\"BACKGROUND_COLOR\":-9283}},"
        + "{\"ANWESEND\":false,\"CHATUSER\":{\"USERID\":\"userid4\",\"NICKNAME\":\"picasso\",\"FOREGROUND_COLOR\":-8273,\"BACKGROUND_COLOR\":-23}}]"

        + "}}";

    GsonBuilder builder = new GsonBuilder();
    builder.registerTypeAdapter(CONFERENCE.class, new ConferenceDeserializer());
    Gson gson = builder.create();
    CONFERENCE conference = gson.fromJson(encoded, CONFERENCE.class);

    assertEquals(Command.CONFERENCE, conference.getCommand(), "COMMAND");
    assertEquals(HEADER.RESPONSE, conference.getHeader(), "header");
    assertEquals(Protocol.DATASET, conference.getDataset(), "dataset");

    assertEquals("Dienstag Treff", conference.getKonferenzname());
    assertEquals("anaconda", conference.getOrganisatorUid());
    KonferenzraumUser[] konfernzraumuser = conference.getKonferenzraumUser();
    assertEquals(-2839, konfernzraumuser[0].getForegroundColor());
    assertEquals(238, konfernzraumuser[0].getBackgroundColor());
    assertEquals("admin678", konfernzraumuser[0].getNickname());
    assertEquals("userid1", konfernzraumuser[0].getUserid());
    assertEquals("2b", konfernzraumuser[0].getSession());
    assertTrue(konfernzraumuser[0].isAnwesend());

  }



  @Test
  public void decodeCONFERENCE_RESPONSE_OHNE_KONFERENZRAUMUSER() throws DecodeException {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CONFERENCE\","
        + "\"DATASET\":{"
        + "\"KONFERENZNAME\":\"Dienstag Treff\","
        + "\"ORGANISATOR_UID\":\"anaconda\","
        + "\"USERID\":\"admin678\","
        + "\"OFFHOOK\":true,"
        + "\"KONFERENZRAUMUSER\":[]"
        + "}}";

    CONFERENCE conference = (CONFERENCE) messagedecoder.decode(encoded);
    assertEquals(Command.CONFERENCE, conference.getCommand(), "COMMAND");
    assertEquals(HEADER.RESPONSE, conference.getHeader(), "header");
    assertEquals(Protocol.DATASET, conference.getDataset(), "dataset");

    assertEquals("Dienstag Treff", conference.getKonferenzname());
    assertEquals("anaconda", conference.getOrganisatorUid());
    KonferenzraumUser[] konfernzraumuser = conference.getKonferenzraumUser();
    assertNotNull(konfernzraumuser);
    assertEquals(0, konfernzraumuser.length);
    assertTrue(conference.isOffhook());
  }



  @Test
  public void decodeCONFERENCE_RESPONSE_OHNE_KONFERENZNAME() {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CONFERENCE\","
        + "\"DATASET\":{"
        + "\"ORGANISATOR_UID\":\"anaconda\","
        + "\"KONFERENZRAUMUSER\":[]"
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("DecodeException nicht geworfen, KONFERENZNAME fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void decodeCONFERENCE_RESPONSE_OHNE_ORGANISATOR() {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CONFERENCE\","
        + "\"DATASET\":{"
        + "\"KONFERENZNAME\":\"Dienstag Treff\","
        + "\"KONFERENZRAUMUSER\":[]"
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("DecodeException nicht geworfen, ORGANISATOR fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void decodeCONFERENCEAUDIO() throws DecodeException {
//    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CONFERENCEAUDIO\","
//        + "\"DATASET\":{"
//        + "\"TARGET\":[\"ab\",\"a1\"],"
//        + "\"CONTENT\":\"ich bin ein base64 textstring\""
//        + "}}";

    String encoded = "{\"HEADER\":\"REQUEST\",\"COMMAND\":\"CONFERENCEAUDIO\"," + "\"DATASET\":{"
        + "\"TARGET\":[\"ab\",\"a1\"],"
        + "\"CONTENT\":\"aWNoIGJpbiBlaW4gYmFzZTY0IHRleHRzdHJpbmc=\","
        + "\"STREAM_ID\":\"streamid\""
        + "}}";

    CONFERENCEAUDIO conferenceaudio = (CONFERENCEAUDIO) messagedecoder.decode(encoded);
    assertEquals(Command.CONFERENCEAUDIO, conferenceaudio.getCommand(), "COMMAND");
    assertEquals(HEADER.REQUEST, conferenceaudio.getHeader(), "header");
    assertEquals(Protocol.DATASET, conferenceaudio.getDataset(), "dataset");

//    Arrays.equals(null, null)

    assertTrue(
        Arrays.equals(
            new byte[] {105, 99, 104, 32, 98, 105, 110, 32, 101, 105, 110, 32, 98, 97, 115, 101, 54, 52, 32,
                116, 101, 120, 116, 115, 116, 114, 105, 110, 103},
            conferenceaudio.getContent()
        ), "ich bin ein base64 textstring"
    );

    String[] target = conferenceaudio.getReceiverSessions();
    assertEquals("ab", target[0]);
    assertEquals("a1", target[1]);
    assertEquals("streamid", conferenceaudio.getStreamId());

  }



  @Test
  @DisplayName(value = "CONFERENCEAUDION | REQUEST | CONTENT | FEHLT")
  public void decodeCONFERENCEAUDIO_REQUEST_CONTENT_FEHLT() throws DecodeException {

    String encoded = "{\"HEADER\":\"REQUEST\",\"COMMAND\":\"CONFERENCEAUDIO\"," + "\"DATASET\":{"
        + "\"TARGET\":[\"ab\",\"a1\"],"
        + "\"STREAM_ID\":\"streamid\""
        + "}}";

    CONFERENCEAUDIO conferenceaudio = (CONFERENCEAUDIO) messagedecoder.decode(encoded);
    assertEquals(Command.CONFERENCEAUDIO, conferenceaudio.getCommand(), "COMMAND");
    assertEquals(HEADER.REQUEST, conferenceaudio.getHeader(), "header");
    assertEquals(Protocol.DATASET, conferenceaudio.getDataset(), "dataset");

    String[] target = conferenceaudio.getReceiverSessions();
    assertEquals("ab", target[0]);
    assertEquals("a1", target[1]);
    assertEquals("streamid", conferenceaudio.getStreamId());

  }



  @Test
  public void decodeCONFERENCEAUDIO_MIT_USERID() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CONFERENCEAUDIO\","
        + "\"DATASET\":{"
        + "\"USERID\":\"admin678\","
        + "\"TARGET\":[\"ab\",\"a1\"],"
        + "\"CONTENT\":\"aWNoIGJpbiBlaW4gYmFzZTY0IHRleHRzdHJpbmc=\"}}";

    CONFERENCEAUDIO conferenceaudio = (CONFERENCEAUDIO) messagedecoder.decode(encoded);
    assertEquals(Command.CONFERENCEAUDIO, conferenceaudio.getCommand(), "COMMAND");
    assertEquals(HEADER.REQUEST, conferenceaudio.getHeader(), "header");
    assertEquals(Protocol.DATASET, conferenceaudio.getDataset(), "dataset");

//    assertEquals("ich bin ein base64 textstring".getBytes(), conferenceaudio.getContent());
    assertTrue(
        Arrays.equals(
            new byte[] {105, 99, 104, 32, 98, 105, 110, 32, 101, 105, 110, 32, 98, 97, 115, 101, 54, 52, 32,
                116, 101, 120, 116, 115, 116, 114, 105, 110, 103},
            conferenceaudio.getContent()
        )
    );
    String[] target = conferenceaudio.getReceiverSessions();
    assertEquals("ab", target[0]);
    assertEquals("a1", target[1]);
    assertEquals("admin678", conferenceaudio.getUserid());

  }



  @Test
  public void decodeCONFERENCEAUDIO_MIT_ORIGIN() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CONFERENCEAUDIO\","
        + "\"DATASET\":{"
        + "\"TARGET\":[\"ab\",\"a1\"],"
        + "\"CONTENT\":\"aWNoIGJpbiBlaW4gYmFzZTY0IHRleHRzdHJpbmc=\"}}";

    CONFERENCEAUDIO conferenceaudio = (CONFERENCEAUDIO) messagedecoder.decode(encoded);
    assertEquals(Command.CONFERENCEAUDIO, conferenceaudio.getCommand(), "COMMAND");
    assertEquals(HEADER.REQUEST, conferenceaudio.getHeader(), "header");
    assertEquals(Protocol.DATASET, conferenceaudio.getDataset(), "dataset");

//    assertEquals("ich bin ein base64 textstring".getBytes(), conferenceaudio.getContent());
    assertTrue(
        Arrays.equals(
            new byte[] {105, 99, 104, 32, 98, 105, 110, 32, 101, 105, 110, 32, 98, 97, 115, 101, 54, 52, 32,
                116, 101, 120, 116, 115, 116, 114, 105, 110, 103},
            conferenceaudio.getContent()
        )
    );
    String[] target = conferenceaudio.getReceiverSessions();
    assertEquals("ab", target[0]);
    assertEquals("a1", target[1]);

  }



  @Test
  public void decodeCONFERENCEAUDIO_TARGET_NULL() {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CONFERENCEAUDIO\","
        + "\"DATASET\":{"
        + "\"CONTENT\":\"aWNoIGJpbiBlaW4gYmFzZTY0IHRleHRzdHJpbmc=\""
        + "}}";

    try {
      CONFERENCEAUDIO conferenceaudio = (CONFERENCEAUDIO) messagedecoder.decode(encoded);
      fail("Exception nicht geworfen");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }

  }



  @Test
  public void decodeCONFERENCEMUTE_REQUEST() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\",\"COMMAND\":\"CONFERENCEMUTE\"," + "\"DATASET\":{"
        + "\"KONFERENZRAUM\":\"Dienstag Treff\","
        + "\"ORGANISATOR\":\"admin678\","
        + "\"USERID\":\"max45678\","
        + "\"RECEIVER_SESSIONS\":[\"ab\",\"cd\"],"
        + "\"MUTE\":true"
        + "}}";

    CONFERENCEMUTE mute = (CONFERENCEMUTE) messagedecoder.decode(encoded);
    assertEquals(REQUEST, mute.getHeader(), "REQUEST");
    assertEquals(Command.CONFERENCEMUTE, mute.getCommand(), "COMMAND");
    assertEquals(Protocol.DATASET, mute.getDataset(), "DATASET");
    assertEquals("Dienstag Treff", mute.getKonferenzraum(), "KONFERENZRAUM");
    assertEquals("admin678", mute.getOrganisator(), "ORGANISATOR");
    assertEquals("max45678", mute.getUserid(), "USERID");
    assertEquals("ab", mute.getReceiverSessions()[0], "RECEIVER_SESSIONS");
    assertEquals("cd", mute.getReceiverSessions()[1], "RECEIVER_SESSIONS");
    assertEquals(Boolean.TRUE, mute.isMute(), "MUTE");

  }



  @Test
  public void decodeCONFERENCEMUTE_REQUEST_ohne_USERID() {
    String encoded = "{\"HEADER\":\"REQUEST\",\"COMMAND\":\"CONFERENCEMUTE\"," + "\"DATASET\":{"
        + "\"KONFERENZRAUM\":\"Dienstag Treff\","
        + "\"ORGANISATOR\":\"admin678\","
        + "\"RECEIVER_SESSIONS\":[\"ab\",\"cd\"],"
        + "\"MUTE\":true"
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("DecodeException nicht geworfen, USERID fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }

  }



  @Test
  public void decodeCONFERENCEMUTE_REQUEST_ohne_ORGANISATOR() {
    String encoded = "{\"HEADER\":\"REQUEST\",\"COMMAND\":\"CONFERENCEMUTE\"," + "\"DATASET\":{"
        + "\"KONFERENZRAUM\":\"Dienstag Treff\","
        + "\"USERID\":\"max45678\","
        + "\"RECEIVER_SESSIONS\":[\"ab\",\"cd\"],"
        + "\"MUTE\":true"
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("DecodeException nicht geworfen, ORGANISATOR fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }

  }



  @Test
  public void decodeCONFERENCEMUTE_REQUEST_ohne_KONFERENZRAUM() {
    String encoded = "{\"HEADER\":\"REQUEST\",\"COMMAND\":\"CONFERENCEMUTE\"," + "\"DATASET\":{"
        + "\"ORGANISATOR\":\"admin678\","
        + "\"USERID\":\"max45678\","
        + "\"RECEIVER_SESSIONS\":[\"ab\",\"cd\"],"
        + "\"MUTE\":true"
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("DecodeException nicht geworfen, KONFERENZRAUM fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }

  }



  @Test
  public void decodeCONFERENCEMUTE_REQUEST_ohne_MUTE() {
    String encoded = "{\"HEADER\":\"REQUEST\",\"COMMAND\":\"CONFERENCEMUTE\"," + "\"DATASET\":{"
        + "\"KONFERENZRAUM\":\"Dienstag Treff\","
        + "\"ORGANISATOR\":\"admin678\","
        + "\"USERID\":\"max45678\","
        + "\"RECEIVER_SESSIONS\":[\"ab\",\"cd\"]"
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("DecodeException nicht geworfen, MUTE fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void decodeCONFERENCEMUTE_REQUEST_ohne_RECEIVER_SESSIONS() {
    String encoded = "{\"HEADER\":\"REQUEST\",\"COMMAND\":\"CONFERENCEMUTE\"," + "\"DATASET\":{"
        + "\"KONFERENZRAUM\":\"Dienstag Treff\","
        + "\"ORGANISATOR\":\"admin678\","
        + "\"USERID\":\"max45678\","
        + "\"MUTE\":true"
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("DecodeException nicht geworfen, RECEIVER_SESSIONS() fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  @DisplayName(value = "CONFERENCEVIDEO ERROR")
  public void decodeCONFERENCEVIDEO_ERROR() {

    String encoded = "{\"HEADER\":\"ERROR\",\"COMMAND\":\"CONFERENCEVIDEO\"," + "\"DATASET\":{"
        + "\"MULTILINGUALKEY\":\"Server_Dritte_Person\""
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("DecodeException nicht geworfen, SHARING fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  @DisplayName(value = "CONFERENCEVIDEO ERROR mit SHARING")
  public void decodeCONFERENCEVIDEO_ERROR_SHARING() throws DecodeException {

    String encoded = "{\"HEADER\":\"ERROR\",\"COMMAND\":\"CONFERENCEVIDEO\"," + "\"DATASET\":{"
        + "\"MULTILINGUALKEY\":\"Server_Dritte_Person\","
        + "SHARING:false"
        + "}}";

    CONFERENCEVIDEO conferenceVideo = (CONFERENCEVIDEO) messagedecoder.decode(encoded);
    assertEquals(Command.CONFERENCEVIDEO, conferenceVideo.getCommand(), "COMMAND");
    assertEquals(ERROR, conferenceVideo.getHeader(), "header");
    assertEquals(Protocol.DATASET, conferenceVideo.getDataset(), "dataset");
    assertEquals(KEY.SERVER_DRITTE_PERSON, conferenceVideo.getMultilingualkey());
    assertEquals(false, conferenceVideo.isShared(), "SHARING");
  }



  @Test
  public void decodeCONFERENCEVIDEO() throws DecodeException {

    String encoded = "{\"HEADER\":\"REQUEST\",\"COMMAND\":\"CONFERENCEVIDEO\"," + "\"DATASET\":{"
        + "\"USERID\":\"anaconda\","
        + "\"NICKNAME\":\"erdbeere\","
        + "\"KONFERENZRAUM\":\"Dienstag Treff\","
        + "\"ORGANISATOR\":\"max45678\","
        + "\"RECEIVER_SESSIONS\":[\"ab\",\"a1\"],"
        + "\"SHARING\":true"
        + "}}";

    CONFERENCEVIDEO conferenceVideo = (CONFERENCEVIDEO) messagedecoder.decode(encoded);
    assertEquals(Command.CONFERENCEVIDEO, conferenceVideo.getCommand(), "COMMAND");
    assertEquals(HEADER.REQUEST, conferenceVideo.getHeader(), "header");
    assertEquals(Protocol.DATASET, conferenceVideo.getDataset(), "dataset");

    String[] receiver = conferenceVideo.getReceiverSessions();
    assertEquals("ab", receiver[0]);
    assertEquals("a1", receiver[1]);

    assertEquals("anaconda", conferenceVideo.getUserid(), "USERID");
    assertEquals("erdbeere", conferenceVideo.getNickname(), "NICKNAME");
    assertTrue(conferenceVideo.isShared(), "SHARING");
    assertEquals("max45678", conferenceVideo.getOrganisator());
    assertEquals("Dienstag Treff", conferenceVideo.getKonferenzraum());

  }



  @Test
  @DisplayName(value = "CONFERENCEVIDEO ohne NICKNAME")
  public void decodeCONFERENCEVIDEO_ohne_NICKNAME() {

    String encoded = "{\"HEADER\":\"REQUEST\",\"COMMAND\":\"CONFERENCEVIDEO\"," + "\"DATASET\":{"
        + "\"USERID\":\"anaconda\","
        + "\"KONFERENZRAUM\":\"Dienstag Treff\","
        + "\"ORGANISATOR\":\"max45678\","
        + "\"RECEIVER_SESSIONS\":[\"ab\",\"a1\"],"
        + "\"SHARING\":true"
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("Exception, NICKNAME nicht geworfen");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }

  }



  @Test
  @DisplayName(value = "CONFERENCEVIDEO ohne ORGANISATOR")
  public void decodeCONFERENCEVIDEO_ohne_ORGANISATOR() {

    String encoded = "{\"HEADER\":\"REQUEST\",\"COMMAND\":\"CONFERENCEVIDEO\"," + "\"DATASET\":{"
        + "\"RECEIVER_SESSIONS\":[\"ab\",\"a1\"],"
        + "\"CONNECT\":\"true\""
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("DecodeException nicht geworfen, ORGANISATOR fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }

  }



  @Test
  @DisplayName(value = "CONFERENCEVIDEO ohne RECEIVER_SESSIONS")
  public void decodeCONFERENCEVIDEO_ohne_Receiver() {

    String encoded = "{\"HEADER\":\"REQUEST\",\"COMMAND\":\"CONFERENCEVIDEO\"," + "\"DATASET\":{"
        + "\"ORGANISATOR\":\"max45678\","
        + "\"CONNECT\":\"true\""
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("DecodeException nicht geworfen, RECEIVER_SESSIONS fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }

  }



  @Test
  @DisplayName(value = "CONFERENCEVIDEO ohne CONNECT")
  public void decodeCONFERENCEVIDEO_ohne_CONNCET() {

    String encoded = "{\"HEADER\":\"REQUEST\",\"COMMAND\":\"CONFERENCEVIDEO\"," + "\"DATASET\":{"
        + "\"ORGANISATOR\":\"max45678\","
        + "\"RECEIVER_SESSIONS\":[\"ab\",\"a1\"]"
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("DecodeException nicht geworfen, CONNECT fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }

  }



  @Test
  public void decodeCREATETEMPROOM_ERROR() throws DecodeException {
    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"CREATETEMPROOM\","
        + "\"DATASET\":{"
        + "\"USER\":\"llange\","
        + "\"ROOM\":\"anaconda\","
        + "\"ROOMTYPE\":\"BESPRECHUNGSRAUM\","
        + "\"ERRORCODE\":1062,"
        + "\"ERRORMESSAGE\":\"duplicate key entry\""
        + "}}";

    CREATETEMPROOM temproom = (CREATETEMPROOM) messagedecoder.decode(encoded);
    assertEquals(Command.CREATETEMPROOM, temproom.getCommand(), "command");
    assertEquals(ERROR, temproom.getHeader(), "header");
    assertEquals(Protocol.DATASET, temproom.getDataset(), "dataset");
    assertEquals("anaconda", temproom.getRoom(), "room");
    assertEquals("llange", temproom.getUser(), "user");
    assertEquals(1062, temproom.getErrorCode(), "errorcode");
    assertEquals("duplicate key entry", temproom.getErrorMessage(), "user");
  }



  @Test
  public void decodeCREATETEMPROOM_ERROR_mit_ROOMTYPE() throws DecodeException {
    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"CREATETEMPROOM\","
        + "\"DATASET\":{"
        + "\"USER\":\"llange\","
        + "\"ROOM\":\"anaconda\","
        + "\"ROOMTYPE\":\"PAUSENRAUM\","
        + "\"ERRORCODE\":1062,"
        + "\"ERRORMESSAGE\":\"duplicate key entry\""
        + "}}";

    CREATETEMPROOM temproom = (CREATETEMPROOM) messagedecoder.decode(encoded);
    assertEquals(Command.CREATETEMPROOM, temproom.getCommand(), "command");
    assertEquals(ERROR, temproom.getHeader(), "header");
    assertEquals(Protocol.DATASET, temproom.getDataset(), "dataset");
    assertEquals("anaconda", temproom.getRoom(), "room");
    assertEquals("llange", temproom.getUser(), "user");
    assertEquals("PAUSENRAUM", temproom.getRoomtype().name(), "ROOMTYPE");
    assertEquals(1062, temproom.getErrorCode(), "errorcode");
    assertEquals("duplicate key entry", temproom.getErrorMessage(), "user");
  }



  @Test
  public void decodeCREATETEMPROOM_ERROR_MULTILINGUALKEY() throws DecodeException {
    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"CREATETEMPROOM\","
        + "\"DATASET\":{"
        + "\"USER\":\"llange\","
        + "\"ROOM\":\"anaconda\","
        + "\"ROOMTYPE\":\"BESPRECHUNGSRAUM\","
        + "\"ERRORCODE\":1062,"
        + "\"ERRORMESSAGE\":\"duplicate key entry\","
        + "\"MULTILINGUALKEY\":\"Server_Zu_kurzer_Raumname\""
        + "}}";

    CREATETEMPROOM temproom = (CREATETEMPROOM) messagedecoder.decode(encoded);
    assertEquals(Command.CREATETEMPROOM, temproom.getCommand(), "command");
    assertEquals(ERROR, temproom.getHeader(), "header");
    assertEquals(Protocol.DATASET, temproom.getDataset(), "dataset");
    assertEquals("anaconda", temproom.getRoom(), "room");
    assertEquals("llange", temproom.getUser(), "user");
    assertEquals(1062, temproom.getErrorCode(), "errorcode");
    assertEquals("duplicate key entry", temproom.getErrorMessage(), "user");
    assertEquals(KEY.SERVER_ZU_KURZER_RAUMNAME, temproom.getMultilingualkey(), "SERVER_KURZER_RAUMNAME");
  }



  @Test
  public void decodeCREATETEMPROOM_ERROR_PROTECTED() throws DecodeException {
    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"CREATETEMPROOM\","
        + "\"DATASET\":{"
        + "\"USER\":\"louisxiv\","
        + "\"ROOM\":\"anaconda\","
        + "\"ENTRY\":\"PROTECTED\","
        + "\"LIFETIME\":\"TEMPORARY\","
        + "\"ROOMTYPE\":\"GRUPPENRAUM\","
        + "\"ERRORMESSAGE\":\"The password is missing.\"}}";

    CREATETEMPROOM temproom = (CREATETEMPROOM) messagedecoder.decode(encoded);
    assertEquals(Command.CREATETEMPROOM, temproom.getCommand(), "command");
    assertEquals(ERROR, temproom.getHeader(), "header");
    assertEquals(Protocol.DATASET, temproom.getDataset(), "dataset");
    assertEquals("anaconda", temproom.getRoom(), "room");
    assertEquals("louisxiv", temproom.getUser(), "user");
    assertEquals("The password is missing.", temproom.getErrorMessage(), "errorMessage");
    assertEquals(Entry.PROTECTED, temproom.getEntry(), "entry");
    assertEquals(Lifetime.TEMPORARY, temproom.getLifetime(), "lifetime");
    assertNull(temproom.getErrorCode(), "errorcode");
  }



  @Test
  public void decodeCREATETEMPROOM_ERROR_TEXT() throws DecodeException {
    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"CREATETEMPROOM\","
        + "\"DATASET\":{"
        + "\"USER\":\"llange\","
        + "\"ROOM\":\"anaconda\","
        + "\"ROOMTYPE\":\"FORUM\","
        + "\"ERRORCODE\":1062,"
        + "\"ERRORMESSAGE\":\"duplicate key entry\","
        + "\"TEXT\":\"fehler\""
        + "}}";

    CREATETEMPROOM temproom = (CREATETEMPROOM) messagedecoder.decode(encoded);
    assertEquals(Command.CREATETEMPROOM, temproom.getCommand(), "command");
    assertEquals(ERROR, temproom.getHeader(), "header");
    assertEquals(Protocol.DATASET, temproom.getDataset(), "dataset");
    assertEquals("anaconda", temproom.getRoom(), "room");
    assertEquals("llange", temproom.getUser(), "user");
    assertEquals(1062, temproom.getErrorCode(), "errorcode");
    assertEquals("duplicate key entry", temproom.getErrorMessage(), "user");
    assertEquals("fehler", temproom.getText(), "text");
  }



  @Test
  public void decodeCREATETEMPROOM_REQUEST() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CREATETEMPROOM\","
        + "\"DATASET\":{"
        + "\"USER\":\"llange\","
        + "\"NICKNAME\":\"viper\","
        + "\"ROOM\":\"anaconda\","
        + "\"ENTRY\":\"PUBLIC\","
        + "\"LIFETIME\":\"TEMPORARY\","
        + "\"ROOMTYPE\":\"FORUM\""
        + "}}";

    CREATETEMPROOM temproom = (CREATETEMPROOM) messagedecoder.decode(encoded);
    assertEquals(Command.CREATETEMPROOM, temproom.getCommand(), "command");
    assertEquals(REQUEST, temproom.getHeader(), "header");
    assertEquals(Protocol.DATASET, temproom.getDataset(), "dataset");
    assertEquals("anaconda", temproom.getRoom(), "room");
    assertEquals("llange", temproom.getUser(), "user");
    assertEquals(Entry.PUBLIC, temproom.getEntry(), "PUBLIC");
    assertEquals(Lifetime.TEMPORARY, temproom.getLifetime(), "TEMPORARY");
    assertEquals("viper", temproom.getNickname(), "nickname");
    assertEquals("FORUM", temproom.getRoomtype().name(), "ROOMTYPE");
  }



  @Test
  public void decodeCREATETEMPROOM_REQUEST_DecodeException() {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CREATETEMPROOM\","
        + "\"DATASET\":{"
        + "\"USER\":\"llange\","
        + "\"ROOM\":\"anaconda\","
        + "\"ENTRY\":\"PUBLIC\","
        + "\"LIFETIME\":\"TEMPORARY\""
        + "}}";

    CREATETEMPROOM temproom;
    try {
      temproom = (CREATETEMPROOM) messagedecoder.decode(encoded);
      fail("DecodeException nicht geworfen, NICKNAME");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void decodeCREATETEMPROOM_REQUEST_PROTECTED_temporary() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CREATETEMPROOM\","
        + "\"DATASET\":{"
        + "\"USER\":\"llange\","
        + "\"NICKNAME\":\"viper\","
        + "\"ROOM\":\"anaconda\","
        + "\"ENTRY\":\"PROTECTED\","
        + "\"LIFETIME\":\"TEMPORARY\","
        + "\"PASSWORD\":\"geheim\","
        + "\"ROOMTYPE\":\"PAUSENRAUM\""
        + "}}";

    CREATETEMPROOM temproom = (CREATETEMPROOM) messagedecoder.decode(encoded);
    assertEquals(Command.CREATETEMPROOM, temproom.getCommand(), "command");
    assertEquals(REQUEST, temproom.getHeader(), "header");
    assertEquals(Protocol.DATASET, temproom.getDataset(), "dataset");
    assertEquals("anaconda", temproom.getRoom(), "room");
    assertEquals("llange", temproom.getUser(), "user");
    assertEquals("PROTECTED", temproom.getEntry().name());
    assertEquals("TEMPORARY", temproom.getLifetime().name(), "TEMPORARY");
    assertEquals("geheim", temproom.getPassword(), "PASSWORD");
    assertEquals("viper", temproom.getNickname(), "NICKNAME");
    assertEquals("PAUSENRAUM", temproom.getRoomtype().name(), "PAUSENRAUM");
  }



  @Test
  public void decodeCREATETEMPROOM_RESPONSE() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CREATETEMPROOM\","
        + "\"DATASET\":{"
        + "\"USER\":\"llange\","
        + "\"ROOM\":\"anaconda\","
        + "\"ENTRY\":\"PUBLIC\","
        + "\"LIFETIME\":\"TEMPORARY\","
        + "\"ROOMTYPE\":\"PAUSENRAUM\""
        + "}}";

    CREATETEMPROOM temproom = (CREATETEMPROOM) messagedecoder.decode(encoded);
    assertEquals(Command.CREATETEMPROOM, temproom.getCommand(), "command");
    assertEquals(RESPONSE, temproom.getHeader(), "header");
    assertEquals(Protocol.DATASET, temproom.getDataset(), "dataset");
    assertEquals("anaconda", temproom.getRoom(), "room");
    assertEquals("llange", temproom.getUser(), "user");
    assertEquals("PUBLIC", temproom.getEntry().name(), "entry");
    assertEquals("TEMPORARY", temproom.getLifetime().name(), "lifetime");
    assertEquals("PAUSENRAUM", temproom.getRoomtype().name(), "ROOMTYPE");
  }



  @Test
  public void decodeCREATETEMPROOM_RESPONSE_MULTILINGUALKEY() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CREATETEMPROOM\","
        + "\"DATASET\":{"
        + "\"USER\":\"llange\","
        + "\"ROOM\":\"anaconda\","
        + "\"ENTRY\":\"PUBLIC\","
        + "\"LIFETIME\":\"TEMPORARY\","
        + "\"ROOMTYPE\":\"PAUSENRAUM\","
        + "\"MULTILINGUALKEY\":\"Server_Der_Chatraum_wurde_angelegt\""
        + "}}";

    CREATETEMPROOM temproom = (CREATETEMPROOM) messagedecoder.decode(encoded);
    assertEquals(Command.CREATETEMPROOM, temproom.getCommand(), "command");
    assertEquals(RESPONSE, temproom.getHeader(), "header");
    assertEquals(Protocol.DATASET, temproom.getDataset(), "dataset");
    assertEquals("anaconda", temproom.getRoom(), "room");
    assertEquals("llange", temproom.getUser(), "user");
    assertEquals("PUBLIC", temproom.getEntry().name(), "entry");
    assertEquals("TEMPORARY", temproom.getLifetime().name(), "lifetime");
    assertEquals("PAUSENRAUM", temproom.getRoomtype().name(), "ROOMTYPE");
    assertEquals(KEY.SERVER_DER_CHATRAUM_WURDE_ANGELEGT, temproom.getMultilingualkey(), "MULTILINGUALKEY");
  }



  @Test
  public void decodeCREATETEMPROOM_RESPONSE_PROTECTED_temporary() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CREATETEMPROOM\","
        + "\"DATASET\":{"
        + "\"USER\":\"llange\","
        + "\"ROOM\":\"anaconda\","
        + "\"ENTRY\":\"PROTECTED\","
        + "\"LIFETIME\":\"TEMPORARY\","
        + "\"PASSWORD\":\"geheim\","
        + "\"ROOMTYPE\":\"PAUSENRAUM\""
        + "}}";

    CREATETEMPROOM temproom = (CREATETEMPROOM) messagedecoder.decode(encoded);
    assertEquals(Command.CREATETEMPROOM, temproom.getCommand(), "command");
    assertEquals(RESPONSE, temproom.getHeader(), "header");
    assertEquals(Protocol.DATASET, temproom.getDataset(), "dataset");
    assertEquals("anaconda", temproom.getRoom(), "room");
    assertEquals("llange", temproom.getUser(), "user");
    assertEquals(Entry.PROTECTED, temproom.getEntry(), "PROTECTED");
    assertEquals("TEMPORARY", temproom.getLifetime().name(), "TEMPORARY");
    assertEquals("geheim", temproom.getPassword(), "PASSWORD");
    assertEquals("PAUSENRAUM", temproom.getRoomtype().name(), "PAUSENRAUM");
  }



  @Test
  public void decodeCREATETEMPROOM_RESPONSE_TEXT() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CREATETEMPROOM\","
        + "\"DATASET\":{"
        + "\"USER\":\"llange\","
        + "\"ROOM\":\"anaconda\","
        + "\"ENTRY\":\"PUBLIC\","
        + "\"LIFETIME\":\"TEMPORARY\","
        + "\"ROOMTYPE\":\"PAUSENRAUM\","
        + "\"TEXT\":\"Hallo\""
        + "}}";

    CREATETEMPROOM temproom = (CREATETEMPROOM) messagedecoder.decode(encoded);
    assertEquals(Command.CREATETEMPROOM, temproom.getCommand(), "command");
    assertEquals(RESPONSE, temproom.getHeader(), "header");
    assertEquals(Protocol.DATASET, temproom.getDataset(), "dataset");
    assertEquals("anaconda", temproom.getRoom(), "room");
    assertEquals("llange", temproom.getUser(), "user");
    assertEquals("PUBLIC", temproom.getEntry().name(), "entry");
    assertEquals("TEMPORARY", temproom.getLifetime().name(), "lifetime");
    assertEquals("Hallo", temproom.getText(), "text");
    assertEquals("PAUSENRAUM", temproom.getRoomtype().name(), "ROOMTYPE");
  }



  @Test
  public void decodeDELETEROOM_ERROR() throws DecodeException {
    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"DELETEROOM\","
        + "\"DATASET\":{"
        + "\"USER\":\"louisxiv\","
        + "\"ROOM\":\"nexus\","
        + "\"TEXT\":\"the room does not exist.\""
        + "}}";

    DELETEROOM delete = (DELETEROOM) messagedecoder.decode(encoded);
    assertEquals(Command.DELETEROOM, delete.getCommand(), "command");
    assertEquals(ERROR, delete.getHeader(), "header");
    assertEquals("louisxiv", delete.getUserid(), "user");
    assertEquals("nexus", delete.getRoom(), "room");
    assertEquals("the room does not exist.", delete.getText(), "text");

  }



  @Test
  public void decodeDELETEROOM_REQUEST() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"DELETEROOM\","
        + "\"DATASET\":{"
        + "\"USER\":\"louisxiv\","
        + "\"ROOM\":\"nexus\""
        + "}}";

    DELETEROOM delete = (DELETEROOM) messagedecoder.decode(encoded);
    assertEquals(Command.DELETEROOM, delete.getCommand(), "command");
    assertEquals(REQUEST, delete.getHeader(), "header");
    assertEquals("louisxiv", delete.getUserid(), "user");
    assertEquals("nexus", delete.getRoom(), "room");

  }



  @Test
  public void decodeDELETEROOM_RESPONSE() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"DELETEROOM\","
        + "\"DATASET\":{"
        + "\"USER\":\"louisxiv\","
        + "\"ROOM\":\"nexus\""
        + "}}";

    DELETEROOM delete = (DELETEROOM) messagedecoder.decode(encoded);
    assertEquals(Command.DELETEROOM, delete.getCommand(), "command");
    assertEquals(RESPONSE, delete.getHeader(), "header");
    assertEquals("louisxiv", delete.getUserid(), "user");
    assertEquals("nexus", delete.getRoom(), "room");

  }



  @Test
  public void decodeDELETEROOM_RESPONSE_TEXT() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"DELETEROOM\","
        + "\"DATASET\":{"
        + "\"USER\":\"louisxiv\","
        + "\"ROOM\":\"nexus\","
        + "\"TEXT\":\"bla\""
        + "}}";

    DELETEROOM delete = (DELETEROOM) messagedecoder.decode(encoded);
    assertEquals(Command.DELETEROOM, delete.getCommand(), "command");
    assertEquals(RESPONSE, delete.getHeader(), "header");
    assertEquals("louisxiv", delete.getUserid(), "user");
    assertEquals("nexus", delete.getRoom(), "room");
    assertEquals("bla", delete.getText(), "text");

  }



  @Test
  public void decodeDELETEUPLOADFILES_CONFIRM() throws DecodeException {

    String encoded = "{\"HEADER\":\"CONFIRM\"," + "\"COMMAND\":\"DELETEUPLOADFILES\"" + "}";

    DELETEUPLOADFILES deletefiles = (DELETEUPLOADFILES) messagedecoder.decode(encoded);

    assertEquals(Command.DELETEUPLOADFILES, deletefiles.getCommand(), "command");
    assertEquals(CONFIRM, deletefiles.getHeader(), "header");
  }



  @Test
  public void decodeDELETEUPLOADFILES_ERROR() throws DecodeException {

    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"DELETEUPLOADFILES\","
        + "\"DATASET\":{"
        + "\"TEXT\":\"A database error has occured\","
        + "\"ERRORCODE\":34,"
        + "\"ERRORMESSAGE\":\"fehler\""
        + "}}";

    DELETEUPLOADFILES deletefiles = (DELETEUPLOADFILES) messagedecoder.decode(encoded);

    assertEquals(Command.DELETEUPLOADFILES, deletefiles.getCommand(), "command");
    assertEquals(ERROR, deletefiles.getHeader(), "header");
    assertEquals(Protocol.DATASET, deletefiles.getDataset(), "dataset");
    assertEquals(34, deletefiles.getErrorCode(), "errorcode");
    assertEquals("fehler", deletefiles.getErrorMessage(), "message");
    assertEquals("A database error has occured", deletefiles.getText(), "text");
  }



  @Test
  public void decodeDELETEUPLOADFILES_REQUEST() throws DecodeException {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"DELETEUPLOADFILES\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"IP\":\"192.168.1.41\""
        + "}}";

    DELETEUPLOADFILES deletefiles = (DELETEUPLOADFILES) messagedecoder.decode(encoded);

    assertEquals(Command.DELETEUPLOADFILES, deletefiles.getCommand(), "command");
    assertEquals(REQUEST, deletefiles.getHeader(), "header");
    assertEquals(Protocol.DATASET, deletefiles.getDataset(), "dataset");
    assertEquals("louisxiv", deletefiles.getUserid(), "userid");
    assertEquals("192.168.1.41", deletefiles.getIp(), "ip");
  }



  @Test
  public void decodeDELETEUSER_CONFIRM() throws DecodeException {
    String encoded = "{\"HEADER\":\"CONFIRM\"," + "\"COMMAND\":\"DELETE_USER\"" + "}";

    DELETEUSER deleteuser = (DELETEUSER) messagedecoder.decode(encoded);

    assertEquals(Command.DELETE_USER, deleteuser.getCommand(), "command");
    assertEquals(CONFIRM, deleteuser.getHeader(), "header");
  }



  @Test
  public void decodeDELETEUSER_ERROR() throws DecodeException {
    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"DELETE_USER\"" + "}";

    DELETEUSER deleteuser = (DELETEUSER) messagedecoder.decode(encoded);
    assertEquals(Command.DELETE_USER, deleteuser.getCommand(), "command");
    assertEquals(ERROR, deleteuser.getHeader(), "header");

  }



  @Test
  public void decodeDELETEUSER_REQUEST() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"DELETE_USER\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\""
        + "}}";

    DELETEUSER deleteuser = (DELETEUSER) messagedecoder.decode(encoded);

    assertEquals(Command.DELETE_USER, deleteuser.getCommand(), "command");
    assertEquals(REQUEST, deleteuser.getHeader(), "header");
    assertEquals(Protocol.DATASET, deleteuser.getDataset(), "dataset");
    assertEquals("louisxiv", deleteuser.getUserid(), "userid");
  }



  @Test
  public void decodeDELETEUSER_REQUEST_DecodeException() {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"DELETE_USER\","
        + "\"DATASE\":{"
        + "\"USERID\":\"louisxiv\""
        + "}}";

    DELETEUSER deleteuser;
    try {
      deleteuser = (DELETEUSER) messagedecoder.decode(encoded);
      fail("DecodeException nicht geworfen");
    }
    catch (DecodeException e) {
      // .error(e.getMessage(), e.getCause());
    }

    /* assertEquals("command", Command.DELETE_USER, deleteuser.getCommand());
     * assertEquals("header", REQUEST, deleteuser.getHeader());
     * assertEquals("dataset", Protocol.DATASET, deleteuser.getDataset());
     * assertEquals("userid", "louisxiv", deleteuser.getUserid()); */
  }



  @Test
  public void decodeDIAL_ERROR() throws DecodeException {
    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"DIAL\","
        + "\"DATASET\":{"
        + "\"CALLER_USERID\":\"12345678\","
        + "\"CALLER_NICKNAME\":\"anaconda\","
        + "\"CALLER_VOICE\":\"niedrig\","
        + "\"RECEIVER_PORT\":29477,"
        + "\"RECEIVER_IP\":\"localhost\","
        + "\"RECEIVER_NICKNAME\":\"javaanaconda\","
        + "\"ERRORMESSAGE\":\"Der Anwender hat in der Zwischenzeit seinen Nickname geändert.\""
        + "}}";

    DIAL dial = (DIAL) messagedecoder.decode(encoded);
    assertEquals(Command.DIAL, dial.getCommand(), "COMMAND");
    assertEquals(
        "Der Anwender hat in der Zwischenzeit seinen Nickname geändert.", dial.getErrorMessage(),
        "ErrorMessage"
    );
    assertEquals(ERROR, dial.getHeader(), "HEADER");
    assertEquals("localhost", dial.getReceiverIp(), "RECEIVER_IP");
    assertEquals("anaconda", dial.getCallerNickname(), "CALLER_NICKNAME");
    assertEquals(29477, dial.getReceiverPort(), "RECEIVER_PORT");
    assertEquals("12345678", dial.getCallerUserid(), "CALLER_USERID");
    assertEquals("niedrig", dial.getCallerVoice(), "CALLER_VOICE");
    assertEquals("javaanaconda", dial.getReceiverNickname(), "RECEIVER_NICKNAME");
    assertNull(dial.getReceiverUserid(), "RECEIVER_USERID");
  }



  @Test
  public void decodeDIAL_ERROR_database_1() throws DecodeException {
    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"DIAL\","
        + "\"DATASET\":{"
        + "\"CALLER_USERID\":\"12345678\","
        + "\"CALLER_NICKNAME\":\"anaconda\","
        + "\"CALLER_VOICE\":\"niedrig\","
        + "\"RECEIVER_NICKNAME\":\"jester\","
        + "\"ERRORMESSAGE\":\"Der Anwender hat in der Zwischenzeit seinen Nickname geändert.\""
        + "}}";

    DIAL dial = (DIAL) messagedecoder.decode(encoded);
    assertEquals(Command.DIAL, dial.getCommand(), "COMMAND");
    assertEquals(
        "Der Anwender hat in der Zwischenzeit seinen Nickname geändert.", dial.getErrorMessage(),
        "ErrorMessage"
    );
    assertEquals(ERROR, dial.getHeader(), "HEADER");
    assertNull(dial.getReceiverIp(), "RECEIVER_IP");
    assertEquals("anaconda", dial.getCallerNickname(), "CALLER_NICKNAME");
    assertNull(dial.getReceiverPort(), "RECEIVER_PORT");
    assertEquals("12345678", dial.getCallerUserid(), "CALLER_USERID");
    assertEquals("niedrig", dial.getCallerVoice(), "CALLER_VOICE");
    assertEquals("jester", dial.getReceiverNickname(), "RECEIVER_NICKNAME");
    assertNull(dial.getReceiverUserid(), "RECEIVER_USERID");
  }



  @Test
  public void decodeDIAL_ERROR_MULTILINGUALKEY() throws DecodeException {
    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"DIAL\","
        + "\"DATASET\":{"
        + "\"CALLER_USERID\":\"12345678\","
        + "\"CALLER_NICKNAME\":\"anaconda\","
        + "\"CALLER_VOICE\":\"niedrig\","
        + "\"RECEIVER_PORT\":29477,"
        + "\"RECEIVER_IP\":\"localhost\","
        + "\"RECEIVER_NICKNAME\":\"javaanaconda\","
        + "\"ERRORMESSAGE\":\"Der Anwender hat in der Zwischenzeit seinen Nickname geändert.\","
        + "\"MULTILINGUALKEY\":\"String_nimmt_keine_Anrufe_entgegen\""
        + "}}";

    DIAL dial = (DIAL) messagedecoder.decode(encoded);
    assertEquals(Command.DIAL, dial.getCommand(), "COMMAND");
    assertEquals(
        "Der Anwender hat in der Zwischenzeit seinen Nickname geändert.", dial.getErrorMessage(),
        "ErrorMessage"
    );
    assertEquals(ERROR, dial.getHeader(), "HEADER");
    assertEquals("localhost", dial.getReceiverIp(), "RECEIVER_IP");
    assertEquals("anaconda", dial.getCallerNickname(), "CALLER_NICKNAME");
    assertEquals(29477, dial.getReceiverPort(), "RECEIVER_PORT");
    assertEquals("12345678", dial.getCallerUserid(), "CALLER_USERID");
    assertEquals("niedrig", dial.getCallerVoice(), "CALLER_VOICE");
    assertEquals("javaanaconda", dial.getReceiverNickname(), "RECEIVER_NICKNAME");
    assertEquals(KEY.STRING_NIMMT_KEINE_ANRUFE_ENTGEGEN, dial.getMultilingualkey(), "MULTILINGUALKEY");
    assertNull(dial.getReceiverUserid(), "RECEIVER_USERID");
  }



  @Test
  public void decodeDIAL_ERROR_MULTILINGUALKEY_ohne_ERRORMESSAGE() throws DecodeException {
    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"DIAL\","
        + "\"DATASET\":{"
        + "\"CALLER_USERID\":\"12345678\","
        + "\"CALLER_NICKNAME\":\"anaconda\","
        + "\"CALLER_VOICE\":\"niedrig\","
        + "\"RECEIVER_PORT\":29477,"
        + "\"RECEIVER_IP\":\"localhost\","
        + "\"RECEIVER_NICKNAME\":\"javaanaconda\","
        + "\"MULTILINGUALKEY\":\"String_nimmt_keine_Anrufe_entgegen\""
        + "}}";

    DIAL dial = (DIAL) messagedecoder.decode(encoded);
    assertEquals(Command.DIAL, dial.getCommand(), "COMMAND");
    assertNull(dial.getErrorMessage(), "ERRORMESSAGE");
    assertEquals(ERROR, dial.getHeader(), "HEADER");
    assertEquals("localhost", dial.getReceiverIp(), "RECEIVER_IP");
    assertEquals("anaconda", dial.getCallerNickname(), "CALLER_NICKNAME");
    assertEquals(29477, dial.getReceiverPort(), "RECEIVER_PORT");
    assertEquals("12345678", dial.getCallerUserid(), "CALLER_USERID");
    assertEquals("niedrig", dial.getCallerVoice(), "CALLER_VOICE");
    assertEquals("javaanaconda", dial.getReceiverNickname(), "RECEIVER_NICKNAME");
    assertEquals(KEY.STRING_NIMMT_KEINE_ANRUFE_ENTGEGEN, dial.getMultilingualkey(), "MULTILINGUALKEY");
    assertNull(dial.getReceiverUserid(), "RECEIVER_USERID");
  }



  @Test
  public void decodeDIAL_REQUEST() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"DIAL\","
        + "\"DATASET\":{"
        + "\"CALLER_USERID\":\"12345678\","
        + "\"CALLER_NICKNAME\":\"anaconda\","
        + "\"CALLER_VOICE\":\"niedrig\","
        + "\"RECEIVER_NICKNAME\":\"javaanaconda\""
        + "}}";

    DIAL dial = (DIAL) messagedecoder.decode(encoded);
    assertEquals(Command.DIAL, dial.getCommand(), "COMMAND");
    assertEquals(REQUEST, dial.getHeader(), "HEADER");
    assertNull(dial.getReceiverIp(), "RECEIVER_IP");
    assertEquals("anaconda", dial.getCallerNickname(), "CALLER_NICKNAME");
    assertNull(dial.getReceiverPort(), "RECEIVER_PORT");
    assertEquals("12345678", dial.getCallerUserid(), "CALLER_USERID");
    assertEquals("niedrig", dial.getCallerVoice(), "CALLER_VOICE");
    assertEquals("javaanaconda", dial.getReceiverNickname(), "RECEIVER_NICKNAME");
    assertNull(dial.getReceiverUserid(), "RECEIVER_USERID");
  }



  @Test
  public void decodeDIAL_REQUEST_ohne_RECEIVER_NICKNAME() {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"DIAL\","
        + "\"DATASET\":{"
        + "\"CALLER_USERID\":\"louisxiv\","
        + "\"CALLER_NICKNAME\":\"anaconda\","
        + "\"CALLER_VOICE\":\"16000Hz\""
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("DecodeException nicht geworfen, RECEIVER_NICKNAME fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void decodeDIAL_REQUEST_ohne_CALLERVOICE() {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"DIAL\","
        + "\"DATASET\":{"
        + "\"CALLER_USERID\":\"12345678\","
        + "\"CALLER_NICKNAME\":\"anaconda\","
        + "\"RECEIVER_NICKNAME\":\"javaanaconda\""
        + "}}";

    DIAL dial;
    try {
      dial = (DIAL) messagedecoder.decode(encoded);
      fail("Exception nicht geworfen");
    }
    catch (DecodeException e) {
      assert true;
    }
  }



  @Test
  public void decodeDIAL_RESPONSE() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"DIAL\","
        + "\"DATASET\":{"
        + "\"CALLER_USERID\":\"12345678\","
        + "\"CALLER_NICKNAME\":\"anaconda\","
        + "\"CALLER_VOICE\":\"niedrig\","
        + "\"RECEIVER_PORT\":29477,"
        + "\"RECEIVER_IP\":\"localhost\","
        + "\"RECEIVER_NICKNAME\":\"javaanaconda\","
        + "\"RECEIVER_USERID\":\"java123\","
        + "\"CALLER_SESSION\":\"4d\","
        + "\"RECEIVER_SESSION\":\"4e\""
        + "}}";

    DIAL dial = (DIAL) messagedecoder.decode(encoded);
    assertEquals(Command.DIAL, dial.getCommand(), "COMMAND");
    assertEquals(RESPONSE, dial.getHeader(), "HEADER");
    assertEquals("localhost", dial.getReceiverIp(), "RECEIVER_IP");
    assertEquals("anaconda", dial.getCallerNickname(), "CALLER_NICKNAME");
    assertEquals(29477, dial.getReceiverPort(), "RECEIVER_PORT");
    assertEquals("12345678", dial.getCallerUserid(), "CALLER_USERID");
    assertEquals("niedrig", dial.getCallerVoice(), "CALLER_VOICE");
    assertEquals("javaanaconda", dial.getReceiverNickname(), "RECEIVER_NICKNAME");
    assertEquals("java123", dial.getReceiverUserid(), "RECEIVER_USERID");
    assertEquals("4d", dial.getCallerSession(), "CALLER_SESSION");
    assertEquals("4e", dial.getReceiverSession(), "RECEIVER_SESSION");
  }



  @Test
  public void decodeDIAL_RESPONSE_mit_SESSION() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"DIAL\","
        + "\"DATASET\":{"
        + "\"CALLER_USERID\":\"12345678\","
        + "\"CALLER_NICKNAME\":\"anaconda\","
        + "\"CALLER_VOICE\":\"niedrig\","
        + "\"CALLER_SESSION\":\"8b\","
        + "\"RECEIVER_IP\":\"localhost\","
        + "\"RECEIVER_PORT\":29477,"
        + "\"RECEIVER_IP\":\"localhost\","
        + "\"RECEIVER_NICKNAME\":\"javaanaconda\","
        + "\"RECEIVER_USERID\":\"java123\","
        + "\"RECEIVER_SESSION\":\"7c\""
        + "}}";

    DIAL dial = (DIAL) messagedecoder.decode(encoded);
    assertEquals(Command.DIAL, dial.getCommand(), "COMMAND");
    assertEquals(RESPONSE, dial.getHeader(), "HEADER");
    assertEquals("localhost", dial.getReceiverIp(), "RECEIVER_IP");
    assertEquals("anaconda", dial.getCallerNickname(), "CALLER_NICKNAME");
    assertEquals(29477, dial.getReceiverPort(), "RECEIVER_PORT");
    assertEquals("12345678", dial.getCallerUserid(), "CALLER_USERID");
    assertEquals("niedrig", dial.getCallerVoice(), "CALLER_VOICE");
    assertEquals("javaanaconda", dial.getReceiverNickname(), "RECEIVER_NICKNAME");
    assertEquals("java123", dial.getReceiverUserid(), "RECEIVER_USERID");
    assertEquals("7c", dial.getReceiverSession(), "SESSION");
    assertEquals("8b", dial.getCallerSession(), "CALLER_SESSION");
  }



  @Test
  public void decodeDIAL_RESPONSE_ohne_CALLERVOICE() {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"DIAL\","
        + "\"DATASET\":{"
        + "\"CALLER_USERID\":\"12345678\","
        + "\"CALLER_NICKNAME\":\"anaconda\","
        + "\"RECEIVER_PORT\":29477,"
        + "\"RECEIVER_IP\":\"localhost\","
        + "\"RECEIVER_NICKNAME\":\"javaanaconda\","
        + "\"RECEIVER_USERID\":\"java123\""
        + "}}";

    DIAL dial;
    try {
      dial = (DIAL) messagedecoder.decode(encoded);
      fail("Excrption nicht geworfen");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void decodeDIAL_RESPONSE_OHNE_IP_UND_OHNE_PORT() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"DIAL\","
        + "\"DATASET\":{"
        + "\"CALLER_USERID\":\"12345678\","
        + "\"CALLER_NICKNAME\":\"anaconda\","
        + "\"CALLER_VOICE\":\"niedrig\","
        + "\"RECEIVER_NICKNAME\":\"javaanaconda\","
        + "\"RECEIVER_USERID\":\"java123\","
        + "\"CALLER_SESSION\":\"3c\","
        + "\"RECEIVER_SESSION\":\"3d\""
        + "}}";

    DIAL dial = (DIAL) messagedecoder.decode(encoded);
    assertEquals(Command.DIAL, dial.getCommand(), "COMMAND");
    assertEquals(RESPONSE, dial.getHeader(), "HEADER");
    assertNull(dial.getReceiverIp(), "RECEIVER_IP");
    assertEquals("anaconda", dial.getCallerNickname(), "CALLER_NICKNAME");
    assertNull(dial.getReceiverPort(), "RECEIVER_PORT");
    assertEquals("12345678", dial.getCallerUserid(), "CALLER_USERID");
    assertEquals("niedrig", dial.getCallerVoice(), "CALLER_VOICE");
    assertEquals("javaanaconda", dial.getReceiverNickname(), "RECEIVER_NICKNAME");
    assertEquals("java123", dial.getReceiverUserid(), "RECEIVER_USERID");
    assertEquals("3c", dial.getCallerSession(), "CALLER_SESSION");
    assertEquals("3d", dial.getReceiverSession(), "RECEIVER_SESSION");
  }



  @Test
  public void decodeDIAL_RESPONSE_ohne_PORT() {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"DIAL\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"NICKNAME\":\"anaconda\","
        + "\"IP\":\"0.0.0.0\","
        + "\"VOICE\":\"niedrig\""
        + "}}";

    try {
      DIAL dial = (DIAL) messagedecoder.decode(encoded);
      fail("Exception für fehlenden PORT wurde nicht geworfen");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void decodeDOWNLOAD_ERROR() throws DecodeException {

    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"DOWNLOAD\","
        + "\"DATASET\":{"
        + "\"USERID\":\"python\","
        + "\"SLOT\":1,"
        + "\"FILENAME\":\"hallo.mp3\","
        + "\"TEXT\":\"The file\","
        + "\"INDEX\":2873}"
        + "}";

    DOWNLOAD download = (DOWNLOAD) messagedecoder.decode(encoded);

    assertEquals(Command.DOWNLOAD, download.getCommand(), "command");
    assertEquals(ERROR, download.getHeader(), "header");
    assertEquals(Protocol.DATASET, download.getDataset(), "dataset");
    assertEquals("python", download.getUserid(), "python");
    assertEquals(1, download.getSlot(), "slot");
    assertEquals("hallo.mp3", download.getFilename(), "filename");
    assertEquals(2873, download.getIndex(), "index");
    assertEquals("The file", download.getText(), "text");
  }



  @Test
  public void decodeDOWNLOAD_REQUEST() throws DecodeException {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"DOWNLOAD\","
        + "\"DATASET\":{"
        + "\"USERID\":\"python\","
        + "\"SLOT\":1,"
        + "\"FILENAME\":\"hallo.mp3\","
        + "\"INDEX\":2873}"
        + "}";

    DOWNLOAD download = (DOWNLOAD) messagedecoder.decode(encoded);

    assertEquals(Command.DOWNLOAD, download.getCommand(), "command");
    assertEquals(REQUEST, download.getHeader(), "header");
    assertEquals(Protocol.DATASET, download.getDataset(), "dataset");
    assertEquals("python", download.getUserid(), "python");
    assertEquals(1, download.getSlot(), "slot");
    assertEquals("hallo.mp3", download.getFilename(), "filename");
    assertEquals(2873, download.getIndex(), "index");
  }



  @Test
  public void decodeDOWNLOAD_RESPONSE() {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"DOWNLOAD\","
        + "\"DATASET\":{"
        + "\"USERID\":\"python\","
        + "\"SLOT\":1,"
        + "\"FILENAME\":\"hallo.mp3\","
        + "\"IP\":\"127.0.0.1\","
        + "\"PATHFILE\":\"c:/hallo.mp3\","
        + "\"INDEX\":2873}"
        + "}";

    DOWNLOAD download;
    try {
      download = (DOWNLOAD) messagedecoder.decode(encoded);
      fail("DecodeException nicht geworfen");
    }
    catch (DecodeException e) {}

  }



  @Test
  public void decodeDOWNLOAD_RESPONSE_PORT() throws DecodeException {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"DOWNLOAD\","
        + "\"DATASET\":{"
        + "\"USERID\":\"python\","
        + "\"SLOT\":1,"
        + "\"FILENAME\":\"hallo.mp3\","
        + "\"IP\":\"127.0.0.1\","
        + "\"PATHFILE\":\"c:/hallo.mp3\","
        + "\"INDEX\":2873,"
        + "\"PORT\":8443"
        + "}"
        + "}";

    DOWNLOAD download = (DOWNLOAD) messagedecoder.decode(encoded);

    assertEquals(Command.DOWNLOAD, download.getCommand(), "command");
    assertEquals(RESPONSE, download.getHeader(), "header");
    assertEquals(Protocol.DATASET, download.getDataset(), "dataset");
    assertEquals("python", download.getUserid(), "python");
    assertEquals(1, download.getSlot(), "slot");
    assertEquals("hallo.mp3", download.getFilename(), "filename");
    assertEquals(2873, download.getIndex(), "index");
    assertEquals("c:/hallo.mp3", download.getPathfile(), "pathfile");
    assertEquals("127.0.0.1", download.getIp(), "index");
    assertEquals(8443, download.getPort(), "port");
  }



  @Test
  @DisplayName(value = "ENTERROOM | ERROR")
  public void decodeENTERROOM_ERROR() throws DecodeException {
    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"ENTERROOM\","
        + "\"DATASET\":{"
        + "\"MULTILINGUALKEY\":\"Server_Zutrittsberechtigung\","
        + "\"TEXT\":\"Du hast für diesen Raum keine Zutrittsberechtigung.\""
        + "}}";

    ENTERROOM enterroom = (ENTERROOM) messagedecoder.decode(encoded);
    assertEquals(Command.ENTERROOM, enterroom.getCommand(), "COMMAND");
    assertEquals(ERROR, enterroom.getHeader(), "HEADER");
    assertEquals(Protocol.DATASET, enterroom.getDataset(), "DATASET");
    assertEquals("Du hast für diesen Raum keine Zutrittsberechtigung.", enterroom.getText());
    assertEquals("Server_Zutrittsberechtigung", enterroom.getMultilingualkey().value());
  }



  @Test
  @DisplayName(value = "ENTERROOM | ERROR | MULTILINGUALKEY | FEHLT")
  public void decodeENTERROOM_ERROR_MULTILINGUALKEY_FEHLT() {
    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"ENTERROOM\","
        + "\"DATASET\":{"
        + "\"TEXT\":\"Du hast für diesen Raum keine Zutrittsberechtigung.\""
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("Exception nicht geworfen, MULTILINGUALKEY fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }

  }



  @Test
  @DisplayName(value = "ENTERROOM | ERROR | TEXT | FEHLT")
  public void decodeENTERROOM_ERROR_TEXT_FEHLT() throws DecodeException {
    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"ENTERROOM\","
        + "\"DATASET\":{"
        + "\"MULTILINGUALKEY\":\"Server_Zutrittsberechtigung\""
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("Exception nicht geworfen, MULTILINGUALKEY fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void decodeENTERROOM_REQUEST() {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"ENTERROOM\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"ROOM\":\"anaconda\""
        + "}}";

    try {
      ENTERROOM enterroom = (ENTERROOM) messagedecoder.decode(encoded);
      fail("DecodeException nicht geworfen, NICKNAME fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void decodeENTERROOM_REQUEST_Nickname() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"ENTERROOM\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"NICKNAME\":\"viper\","
        + "\"ROOM\":\"anaconda\""
        + "}}";

    ENTERROOM enterroom = (ENTERROOM) messagedecoder.decode(encoded);
    assertEquals(Command.ENTERROOM, enterroom.getCommand(), "command");
    assertEquals(REQUEST, enterroom.getHeader(), "header");
    assertEquals(Protocol.DATASET, enterroom.getDataset(), "dataset");
    assertEquals("12345678", enterroom.getUserid(), "userid");
    assertEquals("viper", enterroom.getNickname(), "nickname");
    assertEquals("anaconda", enterroom.getRoom(), "room");
  }



  @Test
  public void decodeENTERROOM_RESPONSE() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"ENTERROOM\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"NICKNAME\":\"viper\","
        + "\"ROOM\":\"anaconda\","
        + "\"ROOMTYPE\":\"PAUSENRAUM\""
        + "}}";

    ENTERROOM enterroom = (ENTERROOM) messagedecoder.decode(encoded);
    assertEquals(Command.ENTERROOM, enterroom.getCommand(), "command");
    assertEquals(RESPONSE, enterroom.getHeader(), "header");
    assertEquals(Protocol.DATASET, enterroom.getDataset(), "dataset");
    assertEquals("12345678", enterroom.getUserid(), "userid");
    assertEquals("viper", enterroom.getNickname(), "nickname");
    assertEquals("anaconda", enterroom.getRoom(), "room");
  }



  @Test
  public void decodeENTERROOM_RESPONSE_DecodeException() {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"ENTERROOM\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"ROOM\":\"anaconda\""
        + "}}";

    ENTERROOM enterroom;
    try {
      enterroom = (ENTERROOM) messagedecoder.decode(encoded);
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void decodeENTERROOM_RESPONSE_ROOMTYPE() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"ENTERROOM\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"NICKNAME\":\"viper\","
        + "\"ROOM\":\"anaconda\","
        + "\"ROOMTYPE\":\"BESPRECHUNGSRAUM\""
        + "}}";

    ENTERROOM enterroom = (ENTERROOM) messagedecoder.decode(encoded);
    assertEquals(Command.ENTERROOM, enterroom.getCommand(), "command");
    assertEquals(RESPONSE, enterroom.getHeader(), "header");
    assertEquals(Protocol.DATASET, enterroom.getDataset(), "dataset");
    assertEquals("12345678", enterroom.getUserid(), "userid");
    assertEquals("viper", enterroom.getNickname(), "nickname");
    assertEquals("anaconda", enterroom.getRoom(), "room");
    assertEquals(Roomtype.BESPRECHUNGSRAUM, enterroom.getRoomtype(), "roomtype");
  }



  @Test
  public void decodeFILETYPES_REQUEST() throws DecodeException {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"FILETYPES\","
        + "\"DATASET\":{"
        + "\"USERID\":\"userid123\""
        + "}"
        + "}";

    FILETYPES filetypes = (FILETYPES) messagedecoder.decode(encoded);
    assertEquals(Command.FILETYPES, filetypes.getCommand(), "Command");
    assertEquals(REQUEST, filetypes.getHeader(), "REQUEST");
    assertEquals("userid123", filetypes.getUserid(), "USERID");
  }



  @Test
  public void decodeFILETYPES_RESPONSE() throws DecodeException {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"FILETYPES\","
        + "\"DATASET\":{"
        + "\"USERID\":\"userid123\","
        + "\"FILETYPES\":["
        + "{\"SUFFIX\":\"pdf\",\"DESCRIPTION\":\"Adobe\",CHECKED:true},"
        + "{\"SUFFIX\":\"txt\",\"DESCRIPTION\":\"Simple Text Format\",CHECKED:false}"
        + "]}"
        + "}";

    FILETYPES filetypes = (FILETYPES) messagedecoder.decode(encoded);
    assertEquals(Command.FILETYPES, filetypes.getCommand(), "Command");
    assertEquals(RESPONSE, filetypes.getHeader(), "RESPONSE");
    assertEquals("userid123", filetypes.getUserid(), "USERID");

    BlacklistTypes pdf = filetypes.getFiletypes()[0];
    BlacklistTypes txt = filetypes.getFiletypes()[1];

    assertEquals("pdf", pdf.getSuffix(), "pdfSuffix");
    assertEquals("Adobe", pdf.getDescription(), "Adobe");
    assertEquals(true, pdf.getChecked(), "pdfChecked");

    assertEquals("txt", txt.getSuffix(), "txtSuffix");
    assertEquals("Simple Text Format", txt.getDescription(), "Simple Text Format");
    assertEquals(false, txt.getChecked(), "txtChecked");

  }



  @Test
  public void decodeFTSDOWNLOAD_ERROR() throws DecodeException {

    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"FTSDOWNLOAD\","
        + "\"DATASET\":{"
        + "\"SLOT\":1,"
        + "\"FILENAME\":\"hallo.mp3\","
        + "\"PATH\":\"c:/\","
        + "\"ERRORMESSAGE\":\"errormeaage\","
        + "\"TEXT\":\"ich bin ein text\""
        + "}"
        + "}";

    FTSDOWNLOAD fts = (FTSDOWNLOAD) messagedecoder.decode(encoded);

    assertEquals(Command.FTSDOWNLOAD, fts.getCommand(), "command");
    assertEquals(ERROR, fts.getHeader(), "header");
    assertEquals(Protocol.DATASET, fts.getDataset(), "dataset");
    assertEquals(1, fts.getSlot(), "slot");
    assertEquals("hallo.mp3", fts.getFilename(), "filename");
    assertEquals("c:/", fts.getPath(), "pathfile");
    assertEquals("errormeaage", fts.getErrorMessage(), "errorMessage");
    assertEquals("ich bin ein text", fts.getText(), "freier text");
  }



  @Test
  public void decodeFTSDOWNLOAD_REQUEST() throws DecodeException {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"FTSDOWNLOAD\","
        + "\"DATASET\":{"
        + "\"SLOT\":1,"
        + "\"FILENAME\":\"hallo.mp3\","
        + "\"PATH\":\"c:/\"}"
        + "}";

    FTSDOWNLOAD fts = (FTSDOWNLOAD) messagedecoder.decode(encoded);

    assertEquals(Command.FTSDOWNLOAD, fts.getCommand(), "command");
    assertEquals(REQUEST, fts.getHeader(), "header");
    assertEquals(Protocol.DATASET, fts.getDataset(), "dataset");
    assertEquals(1, fts.getSlot(), "slot");
    assertEquals("hallo.mp3", fts.getFilename(), "filename");
    assertEquals("c:/", fts.getPath(), "pathfile");
  }



  @Test
  public void decodeFTSDOWNLOAD_REQUEST_ERROR() {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"FTSDOWNLOAD\","
        + "\"DATASET\":{\"SLOT\":0,"
        + "\"FILENAME\":\"wir_sind_helden2.mp4\"}}";

    try {
      messagedecoder.decode(encoded);
      fail("DecodeException nicht geworfen");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }

  }



  @Test
  public void decodeFTSDOWNLOAD_RESPONSE() throws DecodeException {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"FTSDOWNLOAD\","
        + "\"DATASET\":{"
        + "\"SLOT\":1,"
        + "\"FILENAME\":\"hallo.mp3\","
        + "\"PATH\":\"c:/\","
        + "\"MAXDATABLOCKS\":848"
        + "}"
        + "}";

    FTSDOWNLOAD fts = (FTSDOWNLOAD) messagedecoder.decode(encoded);

    assertEquals(Command.FTSDOWNLOAD, fts.getCommand(), "command");
    assertEquals(RESPONSE, fts.getHeader(), "header");
    assertEquals(Protocol.DATASET, fts.getDataset(), "dataset");
    assertEquals(1, fts.getSlot(), "slot");
    assertEquals("hallo.mp3", fts.getFilename(), "filename");
    assertEquals("c:/", fts.getPath(), "pathfile");
    assertEquals(848, fts.getMaxDatablocks(), "maxdatablocks");
  }



  @Test
  public void decodeHISTORYMESSAGE_REQUEST() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"HISTORYMESSAGE\","
        + "\"DATASET\":{"
        + "\"ROOM\":\"room\"}}";

    HISTORYMESSAGE history = (HISTORYMESSAGE) messagedecoder.decode(encoded);

    assertEquals(Command.HISTORYMESSAGE, history.getCommand(), "command");
    assertEquals(REQUEST, history.getHeader(), "header");
    assertEquals(Protocol.DATASET, history.getDataset(), "dataset");
    assertEquals("room", history.getRoom(), "room");

  }



  @Test
  public void decodeHISTORYMESSAGE_RESPONSE_1_Nachricht() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"HISTORYMESSAGE\","
        + "\"DATASET\":{"
        + "\"ROOM\":\"room\","
        + "\"RECORD\":["
        + "{\"ROOM\":\"chat\",\"DATETIME\":666,\"MESSAGE\":\"Hallo\",\"CHATUSER\":{\"BACKGROUND_COLOR\":-1,\"FOREGROUND_COLOR\":-2,\"NICKNAME\":\"python\",\"USERID\":\"louisxiv\"}}"
        + "]"
        + "}}";

    HISTORYMESSAGE history = (HISTORYMESSAGE) messagedecoder.decode(encoded);

    assertEquals(Command.HISTORYMESSAGE, history.getCommand(), "command");
    assertEquals(RESPONSE, history.getHeader(), "header");
    assertEquals(Protocol.DATASET, history.getDataset(), "dataset");
    assertEquals("room", history.getRoom(), "room");
    RecordInterface[] record = history.getAttachment();
    assertEquals(1, record.length, "1 Datensatz");
    assertEquals(666, record[0].getDatetime(), "datetime");
    assertEquals("Hallo", record[0].getMessage(), "message");
    assertEquals("chat", record[0].getRoom(), "room");
    ChatUser chatuser = record[0].getChatUser();
    assertEquals(-1, chatuser.getBackgroundColor(), "bgColor");
    assertEquals(-2, chatuser.getForegroundColor(), "fgColor");
    assertEquals("python", chatuser.getNickname(), "fgColor");
    assertEquals("louisxiv", chatuser.getUserid(), "fgColor");

  }



  @Test
  public void decodeHISTORYMESSAGE_RESPONSE_1_Nachricht_JERSEY() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"HISTORYMESSAGE\","
        + "\"DATASET\":{"
        + "\"ROOM\":\"room\","
        + "\"RECORD\":["
        + "{\"ROOM\":\"chat\",\"DATETIME\":666,\"MESSAGE\":\"Hallo\",\"CHATUSER\":{\"BACKGROUND_COLOR\":-1,\"FOREGROUND_COLOR\":-2,\"NICKNAME\":\"python\",\"USERID\":\"louisxiv\"}}"
        + "]"
        + "}}";

    GsonBuilder builder = new GsonBuilder();
    builder.registerTypeAdapter(HISTORYMESSAGE.class, new HistoryMessageDeserializer());
    Gson gson = builder.create();
    HISTORYMESSAGE history = gson.fromJson(encoded, HISTORYMESSAGE.class);

    assertEquals(Command.HISTORYMESSAGE, history.getCommand(), "command");
    assertEquals(RESPONSE, history.getHeader(), "header");
    assertEquals(Protocol.DATASET, history.getDataset(), "dataset");
    assertEquals("room", history.getRoom(), "room");
    RecordInterface[] record = history.getAttachment();
    assertEquals(1, record.length, "1 Datensatz");
    assertEquals(666, record[0].getDatetime(), "datetime");
    assertEquals("Hallo", record[0].getMessage(), "message");
    assertEquals("chat", record[0].getRoom(), "room");
    ChatUser chatuser = record[0].getChatUser();
    assertEquals(-1, chatuser.getBackgroundColor(), "bgColor");
    assertEquals(-2, chatuser.getForegroundColor(), "fgColor");
    assertEquals("python", chatuser.getNickname(), "fgColor");
    assertEquals("louisxiv", chatuser.getUserid(), "fgColor");

  }



  @Test
  public void decodeHISTORYMESSAGE_RESPONSE_ATTACHEMENT() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"HISTORYMESSAGE\","
        + "\"DATASET\":{"
        + "\"ROOM\":\"room\","
        + "\"RECORD\":["
        + "{\"ROOM\":\"chat\","
        + "\"DATETIME\":1458579290065,"
        + "\"MESSAGE\":\"Hallo\","
        + "\"ATTACHMENT\":99,"
        + "\"FILENAME\":\"name.txt\","
        + "\"FILESIZE\":"
        + String.valueOf(Long.MAX_VALUE)
        + ","
        + "\"CHATUSER\":{"
        + "\"BACKGROUND_COLOR\":-1,"
        + "\"FOREGROUND_COLOR\":-2,"
        + "\"NICKNAME\":\"python\","
        + "\"USERID\":\"louisxiv\""
        + "}}"
        + "]"
        + "}}";

    HISTORYMESSAGE history = (HISTORYMESSAGE) messagedecoder.decode(encoded);

    assertEquals(Command.HISTORYMESSAGE, history.getCommand(), "command");
    assertEquals(RESPONSE, history.getHeader(), "header");
    assertEquals(Protocol.DATASET, history.getDataset(), "dataset");
    assertEquals("room", history.getRoom(), "room");
    Attachment[] record = history.getAttachment();
    assertEquals(1, record.length, "1 Datensatz");
    assertEquals(1458579290065L, record[0].getDatetime(), "datetime");
    assertEquals("Hallo", record[0].getMessage(), "message");
    assertEquals("chat", record[0].getRoom(), "room");
    ChatUser chatuser = record[0].getChatUser();
    assertEquals(-1, chatuser.getBackgroundColor(), "bgColor");
    assertEquals(-2, chatuser.getForegroundColor(), "fgColor");
    assertEquals("python", chatuser.getNickname(), "fgColor");
    assertEquals("louisxiv", chatuser.getUserid(), "fgColor");

    assertEquals("name.txt", record[0].getFilename());
    assertEquals(Long.MAX_VALUE, record[0].getFilesize());
    assertEquals(99, record[0].getAttachment());

  }



  @Test
  public void decodeHISTORYMESSAGE_RESPONSE_ATTACHEMENT_JERSEY() {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"HISTORYMESSAGE\","
        + "\"DATASET\":{"
        + "\"ROOM\":\"room\","
        + "\"RECORD\":["
        + "{\"ROOM\":\"chat\","
        + "\"DATETIME\":1458579290065,"
        + "\"MESSAGE\":\"Hallo\","
        + "\"ATTACHMENT\":99,"
        + "\"FILENAME\":\"name.txt\","
        + "\"FILESIZE\":"
        + String.valueOf(Long.MAX_VALUE)
        + ","
        + "\"CHATUSER\":{"
        + "\"BACKGROUND_COLOR\":-1,"
        + "\"FOREGROUND_COLOR\":-2,"
        + "\"NICKNAME\":\"python\","
        + "\"USERID\":\"louisxiv\""
        + "}}"
        + "]"
        + "}}";

    GsonBuilder builder = new GsonBuilder();
    builder.registerTypeAdapter(HISTORYMESSAGE.class, new HistoryMessageDeserializer());
    Gson gson = builder.create();
    HISTORYMESSAGE history = gson.fromJson(encoded, HISTORYMESSAGE.class);

    assertNotNull(history);
    assertEquals(Command.HISTORYMESSAGE, history.getCommand(), "command");
    assertEquals(RESPONSE, history.getHeader(), "header");
    assertEquals(Protocol.DATASET, history.getDataset(), "dataset");
    assertEquals("room", history.getRoom(), "room");
    Attachment[] record = history.getAttachment();
    assertEquals(1, record.length, "1 Datensatz");
    assertEquals(1458579290065L, record[0].getDatetime(), "datetime");
    assertEquals("Hallo", record[0].getMessage(), "message");
    assertEquals("chat", record[0].getRoom(), "room");
    ChatUser chatuser = record[0].getChatUser();
    assertEquals(-1, chatuser.getBackgroundColor(), "bgColor");
    assertEquals(-2, chatuser.getForegroundColor(), "fgColor");
    assertEquals("python", chatuser.getNickname(), "fgColor");
    assertEquals("louisxiv", chatuser.getUserid(), "fgColor");

    assertEquals("name.txt", record[0].getFilename(), "FILENAME");
    assertEquals(Long.MAX_VALUE, record[0].getFilesize(), "FILESIZE");
    assertEquals(99, record[0].getAttachment(), "ATTACHMENT");

  }



  @Test
  public void decodeIMAGE_ERROR() throws DecodeException {
    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"IMAGE\","
        + "\"DATASET\":{"
        + "\"SENDERUID\":\"louisxiv\","
        + "\"SENDERNICKNAME\":\"anaconda\","
        + "\"RECEIVERNICKNAME\":\"freigeißt\","
        + "\"MULTILINGUALKEY\":\"Server_Der_Empfaenger_ist_verschwunden\""
        + "}}";

    IMAGE image = (IMAGE) messagedecoder.decode(encoded);

    assertEquals(Command.IMAGE, image.getCommand(), "command");
    assertEquals(ERROR, image.getHeader(), "header");
    assertEquals(Protocol.DATASET, image.getDataset(), "dataset");
    assertEquals("anaconda", image.getSenderNickname());
    assertEquals("louisxiv", image.getSenderUid());
    assertEquals("freigeißt", image.getReceiverNickname());
    assertEquals(KEY.SERVER_DER_EMPFAENGER_IST_VERSCHWUNDEN, image.getMultilingualkey());
  }



  @Test
  public void decodeIMAGE_REQUEST() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"IMAGE\","
        + "\"DATASET\":{"
        + "\"SENDERUID\":\"louisxiv\","
        + "\"SENDERNICKNAME\":\"anaconda\","
        + "\"RECEIVERNICKNAME\":\"freigeißt\""
        + "}}";

    IMAGE image = (IMAGE) messagedecoder.decode(encoded);

    assertEquals(Command.IMAGE, image.getCommand(), "command");
    assertEquals(REQUEST, image.getHeader(), "header");
    assertEquals(Protocol.DATASET, image.getDataset(), "dataset");
    assertEquals("anaconda", image.getSenderNickname());
    assertEquals("louisxiv", image.getSenderUid());
    assertEquals("freigeißt", image.getReceiverNickname());
  }



  @Test
  @DisplayName(value = "IMAGE | RESPONSE")
  public void decodeIMAGE_RESPONSE() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"IMAGE\","
        + "\"DATASET\":{"
        + "\"SENDERUID\":\"louisxiv\","
        + "\"SENDERNICKNAME\":\"anaconda\","
        + "\"RECEIVERNICKNAME\":\"freigeißt\","
        + "\"RECEIVERUID\":\"user5678\","
        + "\"RECEIVER_SESSION\":\"123\""
        + "}}";

    IMAGE image = (IMAGE) messagedecoder.decode(encoded);

    assertEquals(Command.IMAGE, image.getCommand(), "command");
    assertEquals(RESPONSE, image.getHeader(), "header");
    assertEquals(Protocol.DATASET, image.getDataset(), "dataset");
    assertEquals("anaconda", image.getSenderNickname());
    assertEquals("louisxiv", image.getSenderUid());
    assertEquals("freigeißt", image.getReceiverNickname());
    assertEquals("user5678", image.getReceiverUid());
    assertEquals("123", image.getReceiverSession());

  }



  @Test
  @DisplayName(value = "IMAGE | RESPONSE | RECEIVER_SESSION | FEHLT")
  public void decodeIMAGE_RESPONSE_RECEIVER_SESSION_FEHLT() {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"IMAGE\","
        + "\"DATASET\":{"
        + "\"SENDERUID\":\"louisxiv\","
        + "\"SENDERNICKNAME\":\"anaconda\","
        + "\"RECEIVERNICKNAME\":\"freigeißt\","
        + "\"RECEIVERUID\":\"user5678\""
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("Exception nicht geworfen, RECEIVER_SESSION fehlt");
    }
    catch (DecodeException e) {
      assert (true);
    }

  }



  @Test
  @DisplayName(value = "INCOMINGCALL | CONFIRM")
  public void decodeINCOMINGCALL_CONFIRM() throws DecodeException {
    String encoded = "{\"HEADER\":\"CONFIRM\"," + "\"COMMAND\":\"INCOMINGCALL\","
        + "\"DATASET\":{"
        + "\"CALLER_NICKNAME\":\"hallo123\","
        + "\"CALLER_USERID\":\"anaconda\","
        + "\"CALLER_SESSION\":\"11\","
        + "\"CALLER_VOICE\":\"niedrig\","
        + "\"RECEIVER_USERID\":\"uid123\","
        + "\"RECEIVER_NICKNAME\":\"java\","
        + "\"RECEIVER_SESSION\":\"34\""
        + "}}";

    INCOMINGCALL incoming = (INCOMINGCALL) messagedecoder.decode(encoded);
    assertEquals(Command.INCOMINGCALL, incoming.getCommand(), "COMMAND");
    assertEquals(CONFIRM, incoming.getHeader(), "CONFIRM");
    assertEquals("anaconda", incoming.getCallerUserid(), "CALLER_USERID");
    assertEquals("hallo123", incoming.getCallerNickname(), "CALLER_NICKNAME");
    assertEquals("uid123", incoming.getReceiverUserid(), "RECEIVER_USERID");
    assertEquals("java", incoming.getReceiverNickname(), "RECEIVER_NICKNAME");
    assertEquals("niedrig", incoming.getCallerVoice(), "CALLER_VOICE");
    assertEquals("34", incoming.getReceiverSession(), "RECEIVER_SESSION");
    assertEquals("11", incoming.getCallerSession(), "CALLER_SESSION");
  }



  @Test
  @DisplayName(value = "INCOMINGCALL | CONFIRM | RECEIVER_SESSION | FEHLT")
  public void decodeINCOMINGCALL_CONFIRM_RECEIVER_SESSION_FEHLT() {
    String encoded = "{\"HEADER\":\"CONFIRM\"," + "\"COMMAND\":\"INCOMINGCALL\","
        + "\"DATASET\":{"
        + "\"CALLER_NICKNAME\":\"hallo123\","
        + "\"CALLER_USERID\":\"anaconda\","
        + "\"CALLER_SESSION\":\"11\","
        + "\"CALLER_VOICE\":\"niedrig\","
        + "\"RECEIVER_USERID\":\"uid123\","
        + "\"RECEIVER_NICKNAME\":\"java\""
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("Exception nicht geworfen, RECEIVER_SESSION fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void decodeINCOMINGCALL_ERROR() throws DecodeException {
    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"INCOMINGCALL\","
        + "\"DATASET\":{"
        + "\"RECEIVER_NICKNAME\":\"llange\","
        + "\"MULTILINGUALKEY\":\"String_ist_nicht_mehr_online\""
        + "}}";

    INCOMINGCALL incoming = (INCOMINGCALL) messagedecoder.decode(encoded);
    assertEquals(Command.INCOMINGCALL, incoming.getCommand(), "COMMAND");
    assertEquals(ERROR, incoming.getHeader(), "HEADER");
    assertEquals("llange", incoming.getReceiverNickname(), "RECEIVER_NICKNAME");
    assertEquals(KEY.STRING_IST_NICHT_MEHR_ONLINE, incoming.getMultilingualkey(), "MULTILINGUALKEY");
  }



  @Test
  @DisplayName(value = "INCOMINGCALL | ERROR | MULTILINGUALKEY | FEHLT")
  public void decodeINCOMINGCALL_ERROR_MULTILINGUALKEY_FEHLT() {
    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"INCOMINGCALL\","
        + "\"DATASET\":{"
        + "\"RECEIVER_NICKNAME\":\"llange\""
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("Exception nicht geworfen, MULTILINGUALKEY fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  @DisplayName(value = "INCOMINGCALL | ERROR | RECEIVER_NICKNAME | FEHLT")
  public void decodeINCOMINGCALL_ERROR_RECEIVER_NICKNAME_FEHLT() throws DecodeException {
    String encoded = "{\"HEADER\":\"ERROR\",\"COMMAND\":\"INCOMINGCALL\",\"DATASET\":{"
        + "\"MULTILINGUALKEY\":\"String_hat_aufgelegt\""
        + "}}";

    INCOMINGCALL incoming = (INCOMINGCALL) messagedecoder.decode(encoded);
    assertEquals(Command.INCOMINGCALL, incoming.getCommand(), "COMMAND");
    assertEquals(ERROR, incoming.getHeader(), "HEADER");
    assertEquals(KEY.STRING_HAT_AUFGELEGT, incoming.getMultilingualkey(), "MULTILINGUALKEY");
  }



  @Test
  public void decodeINCOMINGCALL_ERROR_MULTILINGUALKEY() throws DecodeException {
    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"INCOMINGCALL\","
        + "\"DATASET\":{"
        + "\"RECEIVER_NICKNAME\":\"llange\","
        + "\"ERRORMESSAGE\":\"'llange' ist nicht mehr online.\","
        + "\"MULTILINGUALKEY\":\"String_ist_nicht_mehr_online\""
        + "}}";

    INCOMINGCALL incoming = (INCOMINGCALL) messagedecoder.decode(encoded);
    assertEquals(Command.INCOMINGCALL, incoming.getCommand(), "COMMAND");
    assertEquals(ERROR, incoming.getHeader(), "HEADER");
    assertEquals("llange", incoming.getReceiverNickname(), "RECEIVER_NICKNAME");
    assertEquals("'llange' ist nicht mehr online.", incoming.getErrorMessage(), "ERRORMESSAGE");
    assertEquals(KEY.STRING_IST_NICHT_MEHR_ONLINE, incoming.getMultilingualkey(), "MULTILINGUALKEY");
  }



  @Test
  public void decodeINCOMINGCALL_REQUEST() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"INCOMINGCALL\","
        + "\"DATASET\":{"
        + "\"CALLER_NICKNAME\":\"hallo123\","
        + "\"CALLER_USERID\":\"anaconda\","
        + "\"CALLER_VOICE\":\"niedrig\","
        + "\"CALLER_SESSION\":\"7b\","
        + "\"RECEIVER_USERID\":\"uid123\","
        + "\"RECEIVER_NICKNAME\":\"java\","
        + "\"RECEIVER_SESSION\":\"1a\""
        + "}}";

    INCOMINGCALL incoming = (INCOMINGCALL) messagedecoder.decode(encoded);
    assertEquals(Command.INCOMINGCALL, incoming.getCommand(), "COMMAND");
    assertEquals(REQUEST, incoming.getHeader(), "HEADER");
    assertEquals("anaconda", incoming.getCallerUserid(), "CALLER_USERID");
    assertEquals("hallo123", incoming.getCallerNickname(), "CALLER_NICKNAME");
    assertEquals("niedrig", incoming.getCallerVoice(), "CALLER_VOICE");
    assertEquals("uid123", incoming.getReceiverUserid(), "RECEIVER_USERID");
    assertEquals("java", incoming.getReceiverNickname(), "RECEIVER_NICKNAME");
    assertEquals("7b", incoming.getCallerSession(), "CALLER_SESSION");
  }



  @Test
  @DisplayName(value = "INCOMINGCALL | REQUEST | RECEIVER_UID | FEHLT")
  public void decodeINCOMINGCALL_REQUEST_RECEIVER_UID_FEHLT() {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"INCOMINGCALL\","
        + "\"DATASET\":{"
        + "\"CALLER_NICKNAME\":\"hallo123\","
        + "\"CALLER_VOICE\":\"niedrig\","
        + "\"CALLER_SESSION\":\"7b\","
        + "\"RECEIVER_NICKNAME\":\"java\","
        + "\"RECEIVER_SESSION\":\"1a\""
        + "}}";

    try {
      messagedecoder.decode(encoded);
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  @DisplayName(value = "INCOMINGCALL | REQUEST | CALLER_VOICE | FEHLT")
  public void decodeINCOMINGCALL_REQUEST_CALLERVOICE_FEHLT() {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"INCOMINGCALL\","
        + "\"DATASET\":{"
        + "\"CALLER_NICKNAME\":\"hallo123\","
        + "\"CALLER_USERID\":\"anaconda\","
        + "\"CALLER_SESSION\":\"7b\","
        + "\"RECEIVER_USERID\":\"uid123\","
        + "\"RECEIVER_NICKNAME\":\"java\","
        + "\"RECEIVER_SESSION\":\"7c\""
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("Exception nicht geworfen, CALLER_VOICE fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  @DisplayName(value = "INCOMINGCALL | REQUEST | RECEIVER_SESSION | FEHLT")
  public void decodeINCOMINGCALL_REQUEST_RECEIVER_SESSION_FEHLT() {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"INCOMINGCALL\","
        + "\"DATASET\":{"
        + "\"CALLER_NICKNAME\":\"hallo123\","
        + "\"CALLER_USERID\":\"anaconda\","
        + "\"CALLER_VOICE\":\"niedrig\","
        + "\"CALLER_IP\":\"192.168.1.67\","
        + "\"CALLER_PORT\":29477,"
        + "\"CALLER_SESSION\":\"7b\","
        + "\"RECEIVER_USERID\":\"uid123\","
        + "\"RECEIVER_NICKNAME\":\"java\""
        + "}}";

    try {
      messagedecoder.decode(encoded);
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void decodeINCOMINGCALL_RESPONSE() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"INCOMINGCALL\","
        + "\"DATASET\":{"
        + "\"CALLER_NICKNAME\":\"hallo123\","
        + "\"CALLER_USERID\":\"anaconda\","
        + "\"CALLER_VOICE\":niedrig,"
        + "\"CALLER_SESSION\":\"7b\","
        + "\"RECEIVER_USERID\":\"uid123\","
        + "\"RECEIVER_NICKNAME\":\"java\","
        + "\"RECEIVER_SESSION\":\"33\","
        + "\"RECEIVER_VOICE\":\"hoch\""
        + "}}";

    INCOMINGCALL incoming = (INCOMINGCALL) messagedecoder.decode(encoded);
    assertEquals(Command.INCOMINGCALL, incoming.getCommand(), "COMMAND");
    assertEquals(RESPONSE, incoming.getHeader(), "HEADER");
    assertEquals("anaconda", incoming.getCallerUserid(), "CALLER_USERID");
    assertEquals("niedrig", incoming.getCallerVoice(), "CALLER_VOICE");
    assertEquals("7b", incoming.getCallerSession(), "CALLER_SESSION");
    assertEquals("hallo123", incoming.getCallerNickname(), "CALLER_NICKNAME");
    assertEquals("uid123", incoming.getReceiverUserid(), "RECEIVER_USERID");
    assertEquals("java", incoming.getReceiverNickname(), "RECEIVER_NICKNAME");
    assertEquals("33", incoming.getReceiverSession(), "RECEIVER_SESSION");
  }



  @Test
  @DisplayName(value = "INCOMINGCALL | RESPONSE | RECEIVER_SESSION | FEHLT")
  public void decodeINCOMINGCALL_RESPONSE_RECEIVER_SESSION_FEHLT() {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"INCOMINGCALL\","
        + "\"DATASET\":{"
        + "\"CALLER_NICKNAME\":\"hallo123\","
        + "\"CALLER_USERID\":\"anaconda\","
        + "\"CALLER_VOICE\":niedrig,"
        + "\"CALLER_SESSION\":\"7b\","
        + "\"RECEIVER_USERID\":\"uid123\","
        + "\"RECEIVER_NICKNAME\":\"java\","
        + "\"RECEIVER_VOICE\":\"hoch\""
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("Exception nicht geworfen, RECEIVER_SESSION fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  @DisplayName(value = "INCOMINGCALL | RESPONSE | CALLER_VOICE | FEHLT")
  public void decodeINCOMINGCALL_RESPONSE_CALLER_VOICE_FEHLT() {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"INCOMINGCALL\","
        + "\"DATASET\":{"
        + "\"CALLER_NICKNAME\":\"hallo123\","
        + "\"CALLER_USERID\":\"anaconda\","
        + "\"CALLER_SESSION\":\"7b\","
        + "\"RECEIVER_USERID\":\"uid123\","
        + "\"RECEIVER_NICKNAME\":\"java\","
        + "\"RECEIVER_SESSION\":\"33\","
        + "\"RECEIVER_VOICE\":\"hoch\""
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("Exception wurde nicht geworfen, CALLER_VOICE nicht geworfen");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void decodeKEEPALIVE_REQUEST() throws DecodeException {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"KEEPALIVE\"}";

    KEEPALIVE keepalive = (KEEPALIVE) messagedecoder.decode(encoded);
    assertEquals(Command.KEEPALIVE, keepalive.getCommand(), "command");
    assertEquals(REQUEST, keepalive.getHeader(), "header");
    assertEquals(null, keepalive.getDataset(), "dataset");
  }



  @Test
  public void decodeKEEPALIVE_REQUEST_JERSEY() throws DecodeException {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"KEEPALIVE\"}";

    Gson gson = new GsonBuilder().registerTypeAdapter(KEEPALIVE.class, new KeepaliveDeserializer()).create();
    KEEPALIVE keepalive = gson.fromJson(encoded, KEEPALIVE.class);
    assertEquals(Command.KEEPALIVE, keepalive.getCommand(), "command");
    assertEquals(REQUEST, keepalive.getHeader(), "header");
    assertNull(keepalive.getDataset(), "DATASET vorhanden");
  }



  @Test
  public void decodeKEEPALIVE_REQUEST_ohne_Attribut_SESSIONID() {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"KEEPALIVE\"," + "\"DATASET\":{" + "}}";

    try {
      KEEPALIVE keepalive = (KEEPALIVE) messagedecoder.decode(encoded);
      fail("DecodeException nicht geworfen");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }

  }



  @Test
  public void decodeKEEPALIVE_SESSION_REQUEST() throws DecodeException {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"KEEPALIVE\","
        + "\"DATASET\":{"
        + "\"SESSIONID\":\"e\""
        + "}}";

    KEEPALIVE keepalive = (KEEPALIVE) messagedecoder.decode(encoded);
    assertEquals(Command.KEEPALIVE, keepalive.getCommand(), "command");
    assertEquals(REQUEST, keepalive.getHeader(), "header");
    assertEquals("e", keepalive.getSessionId(), "header");
  }



  @Test
  public void decodeKEEPALIVE_SESSION_REQUEST_JERSEY() throws DecodeException {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"KEEPALIVE\","
        + "\"DATASET\":{"
        + "\"SESSIONID\":\"e\""
        + "}}";

    Gson gson = new GsonBuilder().registerTypeAdapter(KEEPALIVE.class, new KeepaliveDeserializer()).create();
    KEEPALIVE keepalive = gson.fromJson(encoded, KEEPALIVE.class);
    assertEquals(Command.KEEPALIVE, keepalive.getCommand());
    assertEquals(REQUEST, keepalive.getHeader());
    assertEquals("e", keepalive.getSessionId(), "SESSIONID fehlt");
  }



  @Test
  public void decodeLEAVEPRIVATECHAT_REQUEST() throws DecodeException {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"LEAVEPRIVATECHAT\","
        + "\"DATASET\":{"
        + "\"GONE_SESSIONID\":\"5\"}}";

    LEAVEPRIVATECHAT chat = (LEAVEPRIVATECHAT) messagedecoder.decode(encoded);

    assertEquals(Command.LEAVEPRIVATECHAT, chat.getCommand(), "command");
    assertEquals(REQUEST, chat.getHeader(), "header");
    assertEquals(Protocol.DATASET, chat.getDataset(), "dataset");
    assertEquals("5", chat.getGoneSessionid(), "errorMessage");

  }



  @Test
  public void decodeLEAVEPRIVATECHAT_RESPONSE() throws DecodeException {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"LEAVEPRIVATECHAT\","
        + "\"DATASET\":{"
        + "\"REMOTE_SESSIONID\":\"6\","
        + "\"GONE_SESSIONID\":\"5\"}}";

    LEAVEPRIVATECHAT chat = (LEAVEPRIVATECHAT) messagedecoder.decode(encoded);

    assertEquals(Command.LEAVEPRIVATECHAT, chat.getCommand(), "command");
    assertEquals(RESPONSE, chat.getHeader(), "header");
    assertEquals(Protocol.DATASET, chat.getDataset(), "dataset");
    assertEquals("5", chat.getGoneSessionid(), "gone");
    assertEquals("6", chat.getRemotesession(), "remote");

  }



  @Test
  public void decodeLEAVEROOM_ERROR() throws DecodeException {
    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"LEAVEROOM\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"ROOM\":\"anaconda\""
        + "}}";

    LEAVEROOM leaveroom = (LEAVEROOM) messagedecoder.decode(encoded);
    assertEquals(Command.LEAVEROOM, leaveroom.getCommand(), "command");
    assertEquals(ERROR, leaveroom.getHeader(), "header");
    assertEquals(Protocol.DATASET, leaveroom.getDataset(), "dataset");
    assertEquals("12345678", leaveroom.getUserid(), "userid");
    assertEquals("anaconda", leaveroom.getRoom(), "room");
  }



  @Test
  public void decodeLEAVEROOM_REQUEST() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"LEAVEROOM\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"ROOM\":\"anaconda\""
        + "}}";

    LEAVEROOM leaveroom = (LEAVEROOM) messagedecoder.decode(encoded);
    assertEquals(Command.LEAVEROOM, leaveroom.getCommand(), "command");
    assertEquals(REQUEST, leaveroom.getHeader(), "header");
    assertEquals(Protocol.DATASET, leaveroom.getDataset(), "dataset");
    assertEquals("12345678", leaveroom.getUserid(), "userid");
    assertEquals("anaconda", leaveroom.getRoom(), "room");
  }



  @Test
  public void decodeLEAVEROOM_RESPONSE() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"LEAVEROOM\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"ROOM\":\"anaconda\""
        + "}}";

    LEAVEROOM leaveroom = (LEAVEROOM) messagedecoder.decode(encoded);
    assertEquals(Command.LEAVEROOM, leaveroom.getCommand(), "command");
    assertEquals(RESPONSE, leaveroom.getHeader(), "header");
    assertEquals(Protocol.DATASET, leaveroom.getDataset(), "dataset");
    assertEquals("12345678", leaveroom.getUserid(), "userid");
    assertEquals("anaconda", leaveroom.getRoom(), "room");
  }



  @Test
  public void decodeMICROERROR() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"MICROERROR\","
        + "\"DATASET\":{"
        + "\"FROM_NICKNAME\":\"anaconda\","
        + "\"FROM_USERID\":\"jester\","
        + "\"TEXT\":\"Die Gegenseite ist nicht mehr online\","
        + "\"TO_USERID\":\"userid123\""
        + "}}";

    MICROERROR microerror = (MICROERROR) messagedecoder.decode(encoded);
    assertEquals(Command.MICROERROR, microerror.getCommand(), "MICROERROR");
    assertEquals(REQUEST, microerror.getHeader(), "REQUEST");
    assertEquals("anaconda", microerror.getFromNickname(), "FROM_NICKNAME");
    assertEquals("jester", microerror.getFromUserid(), "FROM_USERID");
    assertEquals("Die Gegenseite ist nicht mehr online", microerror.getText(), "TEXT");
    assertEquals("userid123", microerror.getToUserid(), "TO_USERID");

  }



  @Test
  @DisplayName(value = "NEWIMAGE | REQUEST")
  public void decodeNEWIMAGE_REQUEST() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"NEWIMAGE\","
        + "\"DATASET\":{"
        + "\"USERID\":\"anaconda\""
        + "}}";

    NEWIMAGE newimage = (NEWIMAGE) messagedecoder.decode(encoded);
    assertEquals(Command.NEWIMAGE, newimage.getCommand(), "NEWIMAGE");
    assertEquals(REQUEST, newimage.getHeader(), "REQUEST");
    assertEquals("anaconda", newimage.getUserid(), "USERID");

  }



  @Test
  @DisplayName(value = "NEWIMAGE | REQUEST | RECEIVER_SESSIONS | FEHLT")
  public void decodeNEWIMAGE_REQUEST_RECEIVER_SESSIONS_FEHLT() {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"NEWIMAGE\"," + "\"DATASET\":{" + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("Exception nicht geworfen, USERID fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }

  }



  @Test
  @DisplayName(value = "NEWMOUSEPOINTER | REQUEST | X | Y")
  public void encodeNEWMOUSEPOINTER_REQUEST() throws DecodeException {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"NEWMOUSEPOINTER\","
        + "\"DATASET\":{"
        + "\"X\":19,"
        + "\"Y\":67"
        + "}}";

    NEWMOUSEPOINTER mousepointer = (NEWMOUSEPOINTER) messagedecoder.decode(encoded);
    assertEquals(REQUEST, mousepointer.getHeader(), "REQUEST");
    assertEquals(Command.NEWMOUSEPOINTER, mousepointer.getCommand(), "NEWMOUSEPOINTER");
    assertEquals(19, mousepointer.getX(), "X");
    assertEquals(67, mousepointer.getY(), "Y");
  }



  @Test
  @DisplayName(value = "NEWMOUSEPOINTER | REQUEST | X FEHLT")
  public void encodeNEWMOUSEPOINTER_REQUEST_X_FEHLT() {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"NEWMOUSEPOINTER\","
        + "\"DATASET\":{"
        + "\"Y\":67"
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("Exception nicht geworfen, X fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  @DisplayName(value = "NEWMOUSEPOINTER | REQUEST | Y FEHLT")
  public void encodeNEWMOUSEPOINTER_REQUEST_Y_FEHLT() {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"NEWMOUSEPOINTER\","
        + "\"DATASET\":{"
        + "\"X\":19"
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("Exception nicht geworfen, Y fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void decodeONCALL_REQUEST() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"ONCALL\","
        + "\"DATASET\":{"
        + "\"USERID\":\"anaconda\","
        + "\"ONCALL\":true"
        + "}}";

    ONCALL oncall = (ONCALL) messagedecoder.decode(encoded);
    assertEquals(Command.ONCALL, oncall.getCommand(), "COMMAND");
    assertEquals(REQUEST, oncall.getHeader(), "HEADER");
    assertEquals("anaconda", oncall.getUserid(), "userid");
    assertEquals(true, oncall.isOnCall(), "onCall");
  }



  @Test
  public void decodeONCALL_RESPONSE() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"ONCALL\","
        + "\"DATASET\":{"
        + "\"USERID\":\"anaconda\","
        + "\"ONCALL\":false"
        + "}}";

    ONCALL oncall = (ONCALL) messagedecoder.decode(encoded);
    assertEquals(Command.ONCALL, oncall.getCommand(), "COMMAND");
    assertEquals(RESPONSE, oncall.getHeader(), "HEADER");
    assertEquals("anaconda", oncall.getUserid(), "userid");
    assertEquals(false, oncall.isOnCall(), "onCall");
  }



  @Test
  public void decodeONHOOK_REQUEST() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"ONHOOK\","
        + "\"DATASET\":{"
        + "\"RECEIVER_USERID\":\"jester\","
        + "\"CALLER_USERID\":\"anaconda\","
        + "\"TO\":true,"
        + "\"SIGNAL\":true"
        + "}}";
    ONHOOK onhook = (ONHOOK) messagedecoder.decode(encoded);
    assertEquals(Command.ONHOOK, onhook.getCommand(), "ONHOOK");
    assertEquals(REQUEST, onhook.getHeader(), "REQUEST");
    assertEquals("jester", onhook.getReceiverUserid(), "RECEIVER_USERID");
    assertEquals("anaconda", onhook.getCallerUserid(), "CALLER_USERID");
    assertEquals(true, onhook.isTo(), "TO");
  }



  @Test
  public void decodeONHOOK_REQUEST_ohne_TO() {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"ONHOOK\","
        + "\"DATASET\":{"
        + "\"RECEIVER_USERID\":\"jester\","
        + "\"CALLER_USERID\":\"anaconda\""
        + "}}";
    try {
      ONHOOK onhook = (ONHOOK) messagedecoder.decode(encoded);
      fail("Exception nicht geworfen, TO");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void decodeONHOOK_REQUEST_SIGNAL() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"ONHOOK\","
        + "\"DATASET\":{"
        + "\"RECEIVER_USERID\":\"jester\","
        + "\"CALLER_USERID\":\"anaconda\","
        + "\"TO\":true,"
        + "\"SIGNAL\":true"
        + "}}";
    ONHOOK onhook = (ONHOOK) messagedecoder.decode(encoded);
    assertEquals(Command.ONHOOK, onhook.getCommand(), "ONHOOK");
    assertEquals(REQUEST, onhook.getHeader(), "REQUEST");
    assertEquals("jester", onhook.getReceiverUserid(), "RECEIVER_USERID");
    assertEquals("anaconda", onhook.getCallerUserid(), "CALLER_USERID");
    assertEquals(true, onhook.isTo(), "TO");
    assertEquals(true, onhook.isSignal(), "SIGNAL");
  }



  @Test
  public void decodeONHOOK_RESPONSE() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"ONHOOK\","
        + "\"DATASET\":{"
        + "\"RECEIVER_USERID\":\"jester\","
        + "\"CALLER_USERID\":\"anaconda\","
        + "\"TO\":true"
        + "}}";
    ONHOOK onhook = (ONHOOK) messagedecoder.decode(encoded);
    assertEquals(Command.ONHOOK, onhook.getCommand(), "ONHOOK");
    assertEquals(RESPONSE, onhook.getHeader(), "RESPONSE");
    assertEquals("jester", onhook.getReceiverUserid(), "RECEIVER_USERID");
    assertEquals("anaconda", onhook.getCallerUserid(), "CALLER_USERID");
    assertEquals(true, onhook.isTo(), "TO");
  }



  @Test
  public void decodePRIVATECHATFILE_ERROR() throws DecodeException {

    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"PRIVATECHATFILE\"," + "\"DATASET\":{}}";

    PRIVATECHATFILE chatfile = (PRIVATECHATFILE) messagedecoder.decode(encoded);

    assertEquals(Command.PRIVATECHATFILE, chatfile.getCommand(), "command");
    assertEquals(ERROR, chatfile.getHeader(), "header");
    assertEquals(Protocol.DATASET, chatfile.getDataset(), "dataset");
  }



  @Test
  public void decodePRIVATECHATFILE_REQUEST() throws DecodeException {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"PRIVATECHATFILE\","
        + "\"DATASET\":{"
        + "\"USERID\":\"anaconda\","
        + "\"FILENAME\":\"dateiname\","
        + "\"REMOTE_SESSION_ID\":\"6\","
        + "\"REMOTE_NICKNAME\":\"anaconda\","
        + "\"ABSOLUTE_FILENAME\":\"/absoluter/dateiname\","
        + "\"FILESIZE\":0,"
        + "\"MESSAGE\":\"\"}}";

    PRIVATECHATFILE chatfile = (PRIVATECHATFILE) messagedecoder.decode(encoded);

    assertEquals(Command.PRIVATECHATFILE, chatfile.getCommand(), "command");
    assertEquals(REQUEST, chatfile.getHeader(), "header");
    assertEquals(Protocol.DATASET, chatfile.getDataset(), "dataset");
    assertEquals("anaconda", chatfile.getUserid());
    assertEquals("dateiname", chatfile.getFilename());
    assertEquals("6", chatfile.getRemoteSessionId());
    assertEquals("/absoluter/dateiname", chatfile.getAbsoluteFilename());
    assertEquals(0, chatfile.getFilesize());
    assertEquals("", chatfile.getMessage());
    assertEquals("anaconda", chatfile.getRemoteNickname());

  }



  @Test
  public void decodePRIVATECHATFILE_REQUEST_OHNE_MESSAGE() {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"PRIVATECHATFILE\","
        + "\"DATASET\":{"
        + "\"NUMBER\":33,"
        + "\"USERID\":\"anaconda\","
        + "\"FILENAME\":\"dateiname\","
        + "\"REMOTE_SESSION_ID\":\"6\","
        + "\"ABSOLUTE_FILENAME\":\"/absoluter/dateiname\","
        + "\"FILESIZE\":0"
        + "}}";

    try {
      PRIVATECHATFILE chatfile = (PRIVATECHATFILE) messagedecoder.decode(encoded);
      fail("Exception nicht geworfen");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }

  }



  @Test
  public void decodePRIVATECHATFILE_RESPONSE() throws DecodeException {

    String encoded = "{\"HEADER\":\"RESPONSE\",\"COMMAND\":\"PRIVATECHATFILE\",\"DATASET\":"
        + "{\"NUMBER\":31,"
        + "\"REMOTE_NICKNAME\":\"nickname\","
        + "\"RECEIVER_USERID\":\"louisxiv\","
        + "\"USERID\":\"4nsb1j9r\","
        + "\"FILENAME\":\"98   Tampa Curhat Beat _ Karl Wine   Baila Remix DVJ CraSh_.mp4\","
        + "\"ABSOLUTE_FILENAME\":\"C:\\\\Users\\\\llange\\\\Videos\\\\98   Tampa Curhat Beat _ Karl Wine   Baila Remix DVJ CraSh_.mp4\","
        + "\"FILESIZE\":11319456,"
        + "\"MESSAGE\":\"123\","
        + "\"REMOTE_SESSION_ID\":\"6a\""
        + "}}";

    PRIVATECHATFILE chatfile = (PRIVATECHATFILE) messagedecoder.decode(encoded);

    assertEquals(Command.PRIVATECHATFILE, chatfile.getCommand(), "command");
    assertEquals(RESPONSE, chatfile.getHeader(), "header");
    assertEquals(Protocol.DATASET, chatfile.getDataset(), "dataset");
    assertEquals("4nsb1j9r", chatfile.getUserid());
    assertEquals("98   Tampa Curhat Beat _ Karl Wine   Baila Remix DVJ CraSh_.mp4", chatfile.getFilename());
    assertEquals(
        "C:\\Users\\llange\\Videos\\98   Tampa Curhat Beat _ Karl Wine   Baila Remix DVJ CraSh_.mp4",
        chatfile.getAbsoluteFilename()
    );
    assertEquals(11319456, chatfile.getFilesize());
    assertEquals("123", chatfile.getMessage());
    assertEquals("6a", chatfile.getRemoteSessionId());
    assertEquals(31, chatfile.getNumber());
    assertEquals("nickname", chatfile.getRemoteNickname());

  }



  @Test
  public void decodePRIVATECHATFILE_RESPONSE_ohne_MESSAGE() throws DecodeException {

    String encoded = "{\"HEADER\":\"RESPONSE\",\"COMMAND\":\"PRIVATECHATFILE\",\"DATASET\":{"
        + "\"NUMBER\":77,"
        + "\"REMOTE_NICKNAME\":\"bla\","
        + "\"RECEIVER_USERID\":\"louisxiv\","
        + "\"USERID\":\"4nsb1j9r\","
        + "\"FILENAME\":\"98   Tampa Curhat Beat _ Karl Wine   Baila Remix DVJ CraSh_.mp4\","
        + "\"ABSOLUTE_FILENAME\":\"C:\\\\Users\\\\llange\\\\Videos\\\\98   Tampa Curhat Beat _ Karl Wine   Baila Remix DVJ CraSh_.mp4\","
        + "\"FILESIZE\":11319456,"
        + "\"REMOTE_SESSION_ID\":\"6a\""
        + "}}";

    PRIVATECHATFILE chatfile = (PRIVATECHATFILE) messagedecoder.decode(encoded);

    assertEquals(Command.PRIVATECHATFILE, chatfile.getCommand(), "command");
    assertEquals(RESPONSE, chatfile.getHeader(), "header");
    assertEquals(Protocol.DATASET, chatfile.getDataset(), "dataset");
    assertEquals("4nsb1j9r", chatfile.getUserid());
    assertEquals("98   Tampa Curhat Beat _ Karl Wine   Baila Remix DVJ CraSh_.mp4", chatfile.getFilename());
    assertEquals(
        "C:\\Users\\llange\\Videos\\98   Tampa Curhat Beat _ Karl Wine   Baila Remix DVJ CraSh_.mp4",
        chatfile.getAbsoluteFilename()
    );
    assertEquals(11319456, chatfile.getFilesize());
    assertEquals("6a", chatfile.getRemoteSessionId());
    assertEquals(null, chatfile.getMessage());
    assertEquals("bla", chatfile.getRemoteNickname());

  }



  @Test
  public void decodePRIVATECHATFILE_RESPONSE_OHNE_REMOTE_SESSION_ID() throws DecodeException {

    String encoded = "{\"HEADER\":\"RESPONSE\",\"COMMAND\":\"PRIVATECHATFILE\",\"DATASET\":"
        + "{\"NUMBER\":32,"
        + "\"REMOTE_NICKNAME\":\"\","
        + "\"RECEIVER_USERID\":\"louisxiv\","
        + "\"USERID\":\"4nsb1j9r\","
        + "\"FILENAME\":\"98   Tampa Curhat Beat _ Karl Wine   Baila Remix DVJ CraSh_.mp4\","
        + "\"ABSOLUTE_FILENAME\":\"C:\\\\Users\\\\llange\\\\Videos\\\\98   Tampa Curhat Beat _ Karl Wine   Baila Remix DVJ CraSh_.mp4\","
        + "\"FILESIZE\":11319456,"
        + "\"MESSAGE\":\"123\"}}";

    PRIVATECHATFILE chatfile = (PRIVATECHATFILE) messagedecoder.decode(encoded);

    assertEquals(Command.PRIVATECHATFILE, chatfile.getCommand(), "command");
    assertEquals(RESPONSE, chatfile.getHeader(), "header");
    assertEquals(Protocol.DATASET, chatfile.getDataset(), "dataset");
    assertEquals("4nsb1j9r", chatfile.getUserid());
    assertEquals("98   Tampa Curhat Beat _ Karl Wine   Baila Remix DVJ CraSh_.mp4", chatfile.getFilename());
    assertEquals(
        "C:\\Users\\llange\\Videos\\98   Tampa Curhat Beat _ Karl Wine   Baila Remix DVJ CraSh_.mp4",
        chatfile.getAbsoluteFilename()
    );
    assertEquals(11319456, chatfile.getFilesize());
    assertEquals("123", chatfile.getMessage());
    assertEquals("", chatfile.getRemoteNickname());

  }



  @Test
  public void decodePRIVATEMESSAGE_REQUEST() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"PRIVATEMESSAGE\","
        + "\"DATASET\":{"
        + "\"LOCAL_SESSIONID\":\"5\","
        + "\"REMOTE_SESSIONID\":\"6\","
        + "\"DATETIME\":123,"
        + "\"MESSAGE\":\"Hallo\"}}";

    PRIVATEMESSAGE chatmessage = (PRIVATEMESSAGE) messagedecoder.decode(encoded);

    assertEquals(Command.PRIVATEMESSAGE, chatmessage.getCommand(), "command");
    assertEquals(REQUEST, chatmessage.getHeader(), "header");
    assertEquals(Protocol.DATASET, chatmessage.getDataset(), "dataset");
    assertEquals("5", chatmessage.getLocalSessionid(), "local");
    assertEquals("6", chatmessage.getRemoteSessionid(), "remote");
    assertEquals("Hallo", chatmessage.getMessage(), "message");
    assertEquals(123, chatmessage.getDatetime(), "datetime");
    assertNull(chatmessage.getAttachment());
    assertNull(chatmessage.getFilename());
    assertNull(chatmessage.getFilesize());
  }



  @Test
  public void decodePRIVATEMESSAGE_REQUEST_2() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"PRIVATEMESSAGE\","
        + "\"DATASET\":{"
        + "\"SENDER_UID\":\"python\","
        + "\"REMOTE_SESSIONID\":\"6\","
        + "\"DATETIME\":123,"
        + "\"MESSAGE\":\"Hallo\"}}";

    PRIVATEMESSAGE chatmessage = (PRIVATEMESSAGE) messagedecoder.decode(encoded);

    assertEquals(Command.PRIVATEMESSAGE, chatmessage.getCommand(), "command");
    assertEquals(REQUEST, chatmessage.getHeader(), "header");
    assertEquals(Protocol.DATASET, chatmessage.getDataset(), "dataset");
    assertEquals("python", chatmessage.getSenderUID(), "local");
    assertEquals("6", chatmessage.getRemoteSessionid(), "remote");
    assertEquals("Hallo", chatmessage.getMessage(), "message");
    assertEquals(123, chatmessage.getDatetime(), "datetime");

  }



  @Test
  public void decodePRIVATEMESSAGE_REQUEST_3() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"PRIVATEMESSAGE\","
        + "\"DATASET\":{"
        + "\"SENDER_UID\":\"python\","
        + "\"REMOTE_SESSIONID\":\"6\","
        + "\"DATETIME\":123,"
        + "\"MESSAGE\":\"Hallo\"}}";

    PRIVATEMESSAGE chatmessage = (PRIVATEMESSAGE) messagedecoder.decode(encoded);

    assertEquals(Command.PRIVATEMESSAGE, chatmessage.getCommand(), "command");
    assertEquals(REQUEST, chatmessage.getHeader(), "header");
    assertEquals(Protocol.DATASET, chatmessage.getDataset(), "dataset");
    assertEquals("python", chatmessage.getSenderUID(), "local");
    assertEquals("6", chatmessage.getRemoteSessionid(), "remote");
    assertEquals("Hallo", chatmessage.getMessage(), "message");
    assertEquals(123, chatmessage.getDatetime(), "datetime");

  }



  @Test
  public void decodePRIVATEMESSAGE_REQUEST_4() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"PRIVATEMESSAGE\","
        + "\"DATASET\":{"
        + "\"SENDER_UID\":\"louisxiv\","
        + "\"REMOTE_SESSIONID\":\"6\","
        + "\"MESSAGE\":\"Hallo\"}}";

    PRIVATEMESSAGE chatmessage = (PRIVATEMESSAGE) messagedecoder.decode(encoded);

    assertEquals(Command.PRIVATEMESSAGE, chatmessage.getCommand(), "command");
    assertEquals(REQUEST, chatmessage.getHeader(), "header");
    assertEquals(Protocol.DATASET, chatmessage.getDataset(), "dataset");
    assertEquals("louisxiv", chatmessage.getSenderUID(), "senderUID");
    assertEquals("6", chatmessage.getRemoteSessionid(), "remote");
    assertEquals("Hallo", chatmessage.getMessage(), "message");
    assertNull(chatmessage.getDatetime(), "datetime");

  }



  @Test
  public void decodePRIVATEMESSAGE_REQUEST_5() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"PRIVATEMESSAGE\","
        + "\"DATASET\":{"
        + "\"SENDER_UID\":\"louisxiv\","
        + "\"REMOTE_SESSIONID\":\"6\","
        + "\"MESSAGE\":\"Hallo\","
        + "\"CHATUSER\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"NICKNAME\":\"anaconda\","
        + "\"FOREGROUND_COLOR\":-2,"
        + "\"BACKGROUND_COLOR\":-3"

        + "}}}";

    PRIVATEMESSAGE chatmessage = (PRIVATEMESSAGE) messagedecoder.decode(encoded);

    assertEquals(Command.PRIVATEMESSAGE, chatmessage.getCommand(), "command");
    assertEquals(REQUEST, chatmessage.getHeader(), "header");
    assertEquals(Protocol.DATASET, chatmessage.getDataset(), "dataset");
    assertEquals("louisxiv", chatmessage.getSenderUID(), "senderUID");
    assertEquals("6", chatmessage.getRemoteSessionid(), "remote");
    assertEquals("Hallo", chatmessage.getMessage(), "message");
    assertNull(chatmessage.getDatetime(), "datetime");
    ChatUser chatuser = chatmessage.getChatUser();
    assertEquals("louisxiv", chatuser.getUserid(), "uid");
    assertEquals("anaconda", chatuser.getNickname(), "nickname");
    assertEquals(-2, chatuser.getForegroundColor(), "forground");
    assertEquals(-3, chatuser.getBackgroundColor(), "background");
  }



  @Test
  public void decodePRIVATEMESSAGE_REQUEST_ATTACHMENT() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"PRIVATEMESSAGE\","
        + "\"DATASET\":{"
        + "\"LOCAL_SESSIONID\":\"5\","
        + "\"REMOTE_SESSIONID\":\"6\","
        + "\"DATETIME\":123,"
        + "\"MESSAGE\":\"Hallo\","
        + "\"ATTACHMENT\":33,"
        + "\"FILENAME\":\"einTest.mp3\","
        + "\"FILESIZE\":8192"
        + "}}";

    PRIVATEMESSAGE chatmessage = (PRIVATEMESSAGE) messagedecoder.decode(encoded);

    assertEquals(Command.PRIVATEMESSAGE, chatmessage.getCommand(), "command");
    assertEquals(REQUEST, chatmessage.getHeader(), "header");
    assertEquals(Protocol.DATASET, chatmessage.getDataset(), "dataset");
    assertEquals("5", chatmessage.getLocalSessionid(), "local");
    assertEquals("6", chatmessage.getRemoteSessionid(), "remote");
    assertEquals("Hallo", chatmessage.getMessage(), "message");
    assertEquals(123, chatmessage.getDatetime(), "datetime");
    assertEquals(33, chatmessage.getAttachment(), "NUMBER");
    assertEquals("einTest.mp3", chatmessage.getFilename(), "FILENAME");
    assertEquals(8192, chatmessage.getFilesize(), "FILESIZE");

  }



  @Test
  public void decodePRIVATEMESSAGE_REQUEST_OHNE_REMOTESESSIONID() {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"PRIVATEMESSAGE\","
        + "\"DATASET\":{"
        + "\"LOCAL_SESSIONID\":\"5\","
        + "\"DATETIME\":123,"
        + "\"MESSAGE\":\"Hallo\","
        + "\"NUMBER\":33,"
        + "\"FILENAME\":\"einTest.mp3\","
        + "\"FILESIZE\":8192"
        + "}}";

    try {
      PRIVATEMESSAGE chatmessage = (PRIVATEMESSAGE) messagedecoder.decode(encoded);
      fail("Exception nicht geworfen, obwohl REMOTE_SESSIONID fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }

  }



  @Test
  public void decodePRIVATEMESSAGE_RESPONSE() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"PRIVATEMESSAGE\","
        + "\"DATASET\":{"
        + "\"LOCAL_SESSIONID\":\"5\","
        + "\"REMOTE_SESSIONID\":\"6\","
        + "\"DATETIME\":123,"
        + "\"MESSAGE\":\"Hallo\""
        + "}}";

    PRIVATEMESSAGE chatmessage = (PRIVATEMESSAGE) messagedecoder.decode(encoded);

    assertEquals(Command.PRIVATEMESSAGE, chatmessage.getCommand(), "command");
    assertEquals(RESPONSE, chatmessage.getHeader(), "header");
    assertEquals(Protocol.DATASET, chatmessage.getDataset(), "dataset");
    assertEquals("5", chatmessage.getLocalSessionid(), "local");
    assertEquals("6", chatmessage.getRemoteSessionid(), "remote");
    assertEquals("Hallo", chatmessage.getMessage(), "message");
    assertEquals(123, chatmessage.getDatetime(), "datetime");
    assertNull(chatmessage.getAttachment());
    assertNull(chatmessage.getFilename());
    assertNull(chatmessage.getFilesize());

  }



  @Test
  public void decodePRIVATEMESSAGE_RESPONSE_2() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"PRIVATEMESSAGE\","
        + "\"DATASET\":{"
        + "\"SENDER_UID\":\"anaconda\","
        + "\"REMOTE_SESSIONID\":\"6\","
        + "\"DATETIME\":123,"
        + "\"MESSAGE\":\"Hallo\"}}";

    PRIVATEMESSAGE chatmessage = (PRIVATEMESSAGE) messagedecoder.decode(encoded);

    assertEquals(Command.PRIVATEMESSAGE, chatmessage.getCommand(), "command");
    assertEquals(RESPONSE, chatmessage.getHeader(), "header");
    assertEquals(Protocol.DATASET, chatmessage.getDataset(), "dataset");
    assertEquals("anaconda", chatmessage.getSenderUID(), "local");
    assertEquals("6", chatmessage.getRemoteSessionid(), "remote");
    assertEquals("Hallo", chatmessage.getMessage(), "message");
    assertEquals(123, chatmessage.getDatetime(), "datetime");

  }



  @Test
  public void decodePRIVATEMESSAGE_RESPONSE_5() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"PRIVATEMESSAGE\","
        + "\"DATASET\":{"
        + "\"SENDER_UID\":\"louisxiv\","
        + "\"REMOTE_SESSIONID\":\"6\","
        + "\"MESSAGE\":\"Hallo\","
        + "\"CHATUSER\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"NICKNAME\":\"anaconda\","
        + "\"FOREGROUND_COLOR\":-2,"
        + "\"BACKGROUND_COLOR\":-3"

        + "}}}";

    PRIVATEMESSAGE chatmessage = (PRIVATEMESSAGE) messagedecoder.decode(encoded);

    assertEquals(Command.PRIVATEMESSAGE, chatmessage.getCommand(), "command");
    assertEquals(RESPONSE, chatmessage.getHeader(), "header");
    assertEquals(Protocol.DATASET, chatmessage.getDataset(), "dataset");
    assertEquals("louisxiv", chatmessage.getSenderUID(), "senderUID");
    assertEquals("6", chatmessage.getRemoteSessionid(), "remote");
    assertEquals("Hallo", chatmessage.getMessage(), "message");
    assertNull(chatmessage.getDatetime(), "datetime");
    ChatUser chatuser = chatmessage.getChatUser();
    assertEquals("louisxiv", chatuser.getUserid(), "uid");
    assertEquals("anaconda", chatuser.getNickname(), "nickname");
    assertEquals(-2, chatuser.getForegroundColor(), "forground");
    assertEquals(-3, chatuser.getBackgroundColor(), "background");
  }



  @Test
  public void decodePRIVATEMESSAGE_RESPONSE_ATTACHMENT() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"PRIVATEMESSAGE\","
        + "\"DATASET\":{"
        + "\"LOCAL_SESSIONID\":\"5\","
        + "\"REMOTE_SESSIONID\":\"6\","
        + "\"DATETIME\":123,"
        + "\"MESSAGE\":\"Hallo\","
        + "\"NUMBER\":33,"
        + "\"FILENAME\":\"einTest.mp3\","
        + "\"FILESIZE\":8192"
        + "}}";

    PRIVATEMESSAGE chatmessage = (PRIVATEMESSAGE) messagedecoder.decode(encoded);

    assertEquals(Command.PRIVATEMESSAGE, chatmessage.getCommand(), "command");
    assertEquals(RESPONSE, chatmessage.getHeader(), "header");
    assertEquals(Protocol.DATASET, chatmessage.getDataset(), "dataset");
    assertEquals("5", chatmessage.getLocalSessionid(), "local");
    assertEquals("6", chatmessage.getRemoteSessionid(), "remote");
    assertEquals("Hallo", chatmessage.getMessage(), "message");
    assertEquals(123, chatmessage.getDatetime(), "datetime");

  }



  @Test
  public void decodePROYECTORCLOSING_REQUEST() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"PROYECTORCLOSING\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\""
        + "}}";

    PROYECTORCLOSING proyector = (PROYECTORCLOSING) messagedecoder.decode(encoded);
    assertEquals(Command.PROYECTORCLOSING, proyector.getCommand(), "PROYECTORCLOSING");
    assertEquals(REQUEST, proyector.getHeader(), "REQUEST");
    assertEquals(Protocol.DATASET, proyector.getDataset(), "DATASET");
    assertEquals("louisxiv", proyector.getUserid(), "userid");
  }



  @Test
  public void decodePROYECTORCLOSING_RESPONSE() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"PROYECTORCLOSING\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\""
        + "}}";

    PROYECTORCLOSING proyector = (PROYECTORCLOSING) messagedecoder.decode(encoded);
    assertEquals(Command.PROYECTORCLOSING, proyector.getCommand(), "PROYECTORCLOSING");
    assertEquals(RESPONSE, proyector.getHeader(), "RESPONSE");
    assertEquals(Protocol.DATASET, proyector.getDataset(), "DATASET");
    assertEquals("louisxiv", proyector.getUserid(), "userid");
  }



  @Test
  public void decodeREADROOMS_REQUEST() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"READROOMS\","
        + "\"DATASET\":{"
        + "\"USERID\":\"anaconda\","
        + "\"ROOMTYPE\":4"
        + "}}";

    READROOMS readrooms = (READROOMS) messagedecoder.decode(encoded);
    assertEquals(Command.READROOMS, readrooms.getCommand(), "READROOMS");
    assertEquals(REQUEST, readrooms.getHeader(), "REQUEST");
    assertEquals("anaconda", readrooms.getUserid(), "USERID");
    assertEquals(Roomtype.GRUPPENRAUM, readrooms.getRoomtype(), "ROOMTYPE");
  }



  @Test
  public void decodeREADROOMS_RESPONSE() throws DecodeException {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"READROOMS\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"ROOMTYPE\":4,"
        + "\"ROOMS\":[\"schach\",\"radsport\"]"
        + "}}";

    READROOMS readrooms = (READROOMS) messagedecoder.decode(encoded);
    assertEquals(Command.READROOMS, readrooms.getCommand(), "READROOMS");
    assertEquals(RESPONSE, readrooms.getHeader(), "RESPONSE");
    assertEquals("louisxiv", readrooms.getUserid(), "USERID");
    assertEquals(Roomtype.GRUPPENRAUM, readrooms.getRoomtype(), "ROOMTYPE");
    assertEquals(2, readrooms.getRooms().length, "2 Räume");
  }



  @Test
  public void decodeREADROOMS_RESPONSE_EMPTY() throws DecodeException {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"READROOMS\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"ROOMTYPE\":4,"
        + "\"ROOMS\":[]"
        + "}}";

    READROOMS readrooms = (READROOMS) messagedecoder.decode(encoded);
    assertEquals(Command.READROOMS, readrooms.getCommand(), "READROOMS");
    assertEquals(RESPONSE, readrooms.getHeader(), "RESPONSE");
    assertEquals("louisxiv", readrooms.getUserid(), "USERID");
    assertEquals(Roomtype.GRUPPENRAUM, readrooms.getRoomtype(), "ROOMTYPE");
    assertEquals(0, readrooms.getRooms().length, "0 Räume");
  }



  @Test
  public void decodeREADTOPICROOMOWNER_REQUEST_OWNER() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"READTOPICROOMOWNER\","
        + "\"DATASET\":"
        + "{\"OWNER\":\"viper\","
        + "\"USERID\":\"12345678\""
        + "}}";

    READTOPICROOMOWNER roomlist = (READTOPICROOMOWNER) messagedecoder.decode(encoded);
    assertEquals(Command.READTOPICROOMOWNER, roomlist.getCommand(), "command");
    assertEquals(REQUEST, roomlist.getHeader(), "header");
    assertEquals("viper", roomlist.getOwner(), "owner");
    assertEquals("12345678", roomlist.getUserid(), "usweid");
  }



  @Test
  public void decodeREADTOPICROOMOWNER_RESPONSE_LIST() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"READTOPICROOMOWNER\","
        + "\"DATASET\":"
        + "{\"OWNER\":\"viper\","
        + "\"USERID\":\"12345678\","
        + "\"ROOMLIST\":"
        + "["
        + "{\"ROOM\":\"JFZI\",\"ENTRY\":\"PROTECTED\",\"LIFETIME\":\"PERMANENT\",\"OWNER\":\"viper\"}"
        + "]"
        + "}}";

    READTOPICROOMOWNER roomlist = (READTOPICROOMOWNER) messagedecoder.decode(encoded);
    assertEquals(Command.READTOPICROOMOWNER, roomlist.getCommand(), "command");
    assertEquals(RESPONSE, roomlist.getHeader(), "header");
    assertEquals("viper", roomlist.getOwner(), "owner");
    assertEquals("12345678", roomlist.getUserid(), "userid");
    Room[] rooms = roomlist.getRoom();
    assertEquals(1, rooms.length, "roomlist 0 size");
    assertEquals("JFZI", rooms[0].getChatid(), "chatid");
    assertEquals("PROTECTED", rooms[0].getEntry().name(), "ENTRY");
    assertEquals(Lifetime.PERMANENT, rooms[0].getLifetime(), "LIFETIME");
    assertEquals("viper", rooms[0].getOwner(), "OWNER");
  }



  @Test
  public void decodeREADTOPICROOMOWNER_RESPONSE_OWNER() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"READTOPICROOMOWNER\","
        + "\"DATASET\":"
        + "{\"OWNER\":\"viper\","
        + "\"USERID\":\"12345678\","
        + "\"ROOMLIST\":[]"
        + "}}";

    READTOPICROOMOWNER roomlist = (READTOPICROOMOWNER) messagedecoder.decode(encoded);
    assertEquals(Command.READTOPICROOMOWNER, roomlist.getCommand(), "command");
    assertEquals(RESPONSE, roomlist.getHeader(), "header");
    assertEquals("viper", roomlist.getOwner(), "owner");
    assertEquals("12345678", roomlist.getUserid(), "usweid");
    assertNotNull(roomlist.getRoom(), "roomlist not null");
    assertEquals(0, roomlist.getRoom().length, "roomlist 0 size");
  }



  @Test
  public void decodeROOMLIST_REQUEST()
      throws DecodeException, JsonSyntaxException, UnsupportedEncodingException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"ROOMLIST\"" + "}";

    ROOMLIST roomlist = (ROOMLIST) messagedecoder.decodeROOMLIST(encoded);
    assertEquals(Command.ROOMLIST, roomlist.getCommand(), "command");
    assertEquals(REQUEST, roomlist.getHeader(), "header");

  }



  @Test
  public void decodeROOMLIST_RESPONSE()
      throws DecodeException, JsonSyntaxException, UnsupportedEncodingException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"ROOMLIST\","
        + "\"DATASET\":{\"ROOMLIST\":["
        + "{\"ENTRY\":\"PUBLIC\",\"LIFETIME\":\"PERMANENT\",\"ROOM\":\"Everyone\",\"ROOMTYPE\":\"BESPRECHUNGSRAUM\"},"
        + "{\"ENTRY\":\"PROTECTED\",\"LIFETIME\":\"TEMPORARY\",\"ROOM\":\"Developer\",\"ROOMTYPE\":\"PAUSENRAUM\"}"
        + "]}}";

    ROOMLIST roomlist = (ROOMLIST) messagedecoder.decodeROOMLIST(encoded);
    assertEquals(Command.ROOMLIST, roomlist.getCommand(), "command");
    assertEquals(RESPONSE, roomlist.getHeader(), "header");
    assertEquals(Protocol.DATASET, roomlist.getDataset(), "dataset");
    assertEquals(2, roomlist.getRoom().length, "2 Felder");
    assertEquals(Lifetime.TEMPORARY, roomlist.getRoom()[1].getLifetime(), "TEMPORARY");
    assertEquals(Lifetime.PERMANENT, roomlist.getRoom()[0].getLifetime(), "PERMANENT");

    assertEquals(Roomtype.BESPRECHUNGSRAUM, roomlist.getRoom()[0].getRoomtype(), "");
    assertEquals(Roomtype.PAUSENRAUM, roomlist.getRoom()[1].getRoomtype(), "PAUSENRAUM");

  }



  @Test
  public void decodeROOMLIST_RESPONSE_ROOM_FEHLT() {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"ROOMLIST\","
        + "\"DATASET\":{\"ROOMLIST\":["
        + "{\"ENTRY\":\"PUBLIC\",\"LIFETIME\":\"PERMANENT\",\"ROOM\":\"Everyone\",\"ROOMTYPE\":\"BESPRECHUNGSRAUM\"},"
        + "{\"ENTRY\":\"PROTECTED\",\"LIFETIME\":\"TEMPORARY\",\"ROOMTYPE\":\"PAUSENRAUM\"}"
        + "]}}";

    try {
      messagedecoder.decodeROOMLIST(encoded);
      fail("Exception nicht geworfen, ROOM fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }

  }



  @Test
  public void decodeSCREENCAST_ERROR() throws DecodeException {

    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"SCREENCAST\","
        + "\"DATASET\":"
        + "{\"RECEIVER_NICKNAME\":\"anaconda\","
        + "\"ERRORMESSAGE\":\"freigeist ist nicht mehr online.\""
        + "}}";

    SCREENCAST screencast = (SCREENCAST) messagedecoder.decode(encoded);

    assertEquals("freigeist ist nicht mehr online.", screencast.getErrorMessage(), "errormessage");
    assertEquals("anaconda", screencast.getReceiverNickname(), "RECEIVER_NICKNAME");

  }



  @Test
  public void decodeSCREENCAST_ERROR_ohne_RECEIVER_NICKNAME() {

    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"SCREENCAST\","
        + "\"DATASET\":{"
        + "\"RECEIVER_USERID\":\"admin678\","
        + "\"ERRORMESSAGE\":\"freigeist ist nicht mehr online.\""
        + "}}";

    SCREENCAST screencast;
    try {
      screencast = (SCREENCAST) messagedecoder.decode(encoded);
      fail("Exception nicht geworfen");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }

  }



  @Test
  public void decodeSCREENCAST_ERROR_RECEIVERUSERID() throws DecodeException {

    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"SCREENCAST\","
        + "\"DATASET\":"
        + "{\"RECEIVER_NICKNAME\":\"anaconda\","
        + "\"RECEIVER_USERID\":\"admin678\","
        + "\"ERRORMESSAGE\":\"freigeist ist nicht mehr online.\""
        + "}}";

    SCREENCAST screencast = (SCREENCAST) messagedecoder.decode(encoded);

    assertEquals("freigeist ist nicht mehr online.", screencast.getErrorMessage(), "errormessage");
    assertEquals("anaconda", screencast.getReceiverNickname(), "RECEIVER_NICKNAME");
    assertEquals("admin678", screencast.getReceiverUserid(), "RECEIVER_USERID");

  }



  @Test
  public void decodeSCREENCAST_REQUEST() throws DecodeException {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"SCREENCAST\","
        + "\"DATASET\":"
        + "{\"CALLER_USERID\":\"admin678\","
        + "\"CALLER_NICKNAME\":\"anaconda\","
        + "\"RECEIVER_NICKNAME\":\"freigeist\","
        + "\"CALLER_IP\":\"javacomm.net\","
        + "\"CALLER_PORT\":29478"
        + "}}";

    SCREENCAST screencast = (SCREENCAST) messagedecoder.decode(encoded);

    assertEquals("anaconda", screencast.getCallerNickname(), "nickname");
    assertEquals("admin678", screencast.getCallerUserid(), "userid");
    assertEquals("freigeist", screencast.getReceiverNickname(), "RECEIVER_NICKNAME");

  }



  @Test
  public void decodeSCREENCAST_REQUEST_DecoderException() throws DecodeException {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"SCREENCAST\"," + "\"DATASET\":{}}";

    try {
      SCREENCAST screencast = (SCREENCAST) messagedecoder.decode(encoded);
      fail("DecodeException wurde nicht geworfen");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void decodeSCREENCAST_RESPONSE() throws DecodeException {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"SCREENCAST\","
        + "\"DATASET\":"
        + "{\"CALLER_USERID\":\"caller78\","
        + "\"RECEIVER_USERID\":\"admin678\","
        + "\"RECEIVER_NICKNAME\":\"anaconda\""
        + "}}";

    SCREENCAST screencast = (SCREENCAST) messagedecoder.decode(encoded);

    assertEquals("caller78", screencast.getCallerUserid(), "caller userid");
    assertEquals("anaconda", screencast.getReceiverNickname(), "nickname");
    assertEquals("admin678", screencast.getReceiverUserid(), "userid");

  }



  @Test
  public void decodeSEARCHFILES_REQUEST() throws DecodeException {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"SEARCHFILES\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"SEARCHFILES\":["
        + "{\"FILE\":\"abc.123\"},"
        + "{\"FILE\":\"hallo.456\"}]}}";

    SEARCHFILES searchfiles = (SEARCHFILES) messagedecoder.decode(encoded);
    String[] files = searchfiles.getPatterns();

    assertEquals(Command.SEARCHFILES, searchfiles.getCommand(), "command");
    assertEquals(REQUEST, searchfiles.getHeader(), "header");
    assertEquals(Protocol.DATASET, searchfiles.getDataset(), "dataset");
    assertEquals("louisxiv", searchfiles.getUserid(), "userid");
    assertEquals("abc.123", files[0], "files1");
    assertEquals("hallo.456", files[1], "files1");
  }



  @Test
  public void decodeSEARCHFILES_REQUEST_2() throws DecodeException {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"SEARCHFILES\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"SEARCHFILES\":["
        + "]}}";

    SEARCHFILES searchfiles = (SEARCHFILES) messagedecoder.decode(encoded);
    String[] files = searchfiles.getPatterns();

    assertEquals(Command.SEARCHFILES, searchfiles.getCommand(), "command");
    assertEquals(REQUEST, searchfiles.getHeader(), "header");
    assertEquals(Protocol.DATASET, searchfiles.getDataset(), "dataset");
    assertEquals("louisxiv", searchfiles.getUserid(), "userid");
    assertEquals(0, files.length, "0");
  }



  @Test
  public void decodeSEARCHFILES_RESPONSE() throws DecodeException {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"SEARCHFILES\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"START\":true,"
        + "\"SEARCHFILE\":["
        + "{\"INDEX\":1,"
        + "\"FILENAME\":\"abc.123\","
        + "\"FILESIZE\":123},"
        + "{\"INDEX\":2,"
        + "\"FILENAME\":\"hallo.456\","
        + "\"FILESIZE\":456}"
        + "]}}";

    SEARCHFILES searchfiles = (SEARCHFILES) messagedecoder.decode(encoded);
    Searchfile[] files = searchfiles.getSearchfiles();

    assertEquals(Command.SEARCHFILES, searchfiles.getCommand(), "command");
    assertEquals(RESPONSE, searchfiles.getHeader(), "header");
    assertEquals(Protocol.DATASET, searchfiles.getDataset(), "dataset");
    assertEquals("louisxiv", searchfiles.getUserid(), "userid");
    assertEquals(true, searchfiles.hasStarted(), "start");

    assertEquals("abc.123", files[0].getFilename(), "files1");
    assertEquals(1, files[0].getIndex(), "index1");
    assertEquals("hallo.456", files[1].getFilename(), "files2");
    assertEquals(2, files[1].getIndex(), "index2");
  }



  @Test
  public void decodeSEARCHFILES_RESPONSE_JERSEY() {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"SEARCHFILES\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"START\":true,"
        + "\"SEARCHFILE\":["
        + "{\"INDEX\":1,"
        + "\"FILENAME\":\"abc.123\","
        + "\"FILESIZE\":123},"
        + "{\"INDEX\":2,"
        + "\"FILENAME\":\"hallo.456\","
        + "\"FILESIZE\":456}"
        + "]}}";

    Gson gson = new GsonBuilder().registerTypeAdapter(SEARCHFILES.class, new SearchFilesDeserializer())
        .create();
    SEARCHFILES searchfiles = gson.fromJson(encoded, SEARCHFILES.class);

    Searchfile[] files = searchfiles.getSearchfiles();

    assertEquals(Command.SEARCHFILES, searchfiles.getCommand(), "COMMAND");
    assertEquals(RESPONSE, searchfiles.getHeader(), "HEADER");
    assertEquals(Protocol.DATASET, searchfiles.getDataset(), "DATASET");
    assertEquals("louisxiv", searchfiles.getUserid(), "USERID");
    assertEquals(true, searchfiles.hasStarted(), "START");

    assertEquals("abc.123", files[0].getFilename(), "FILENAME 1");
    assertEquals(1, files[0].getIndex(), "INDEX 1");
    assertEquals(123, files[0].getFilesize(), "FILESIZE 1");

    assertEquals("hallo.456", files[1].getFilename(), "FILENAME 2");
    assertEquals(2, files[1].getIndex(), "INDEX 2");
    assertEquals(456, files[1].getFilesize(), "FILESIZE 2");
  }



  @Test
  public void decodeSEARCHFILES_RESPONSE_ohne_START() throws DecodeException {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"SEARCHFILES\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"SEARCHFILE\":["
        + "{\"INDEX\":1,"
        + "\"FILENAME\":\"abc.123\","
        + "\"FILESIZE\":123},"
        + "{\"INDEX\":2,"
        + "\"FILENAME\":\"hallo.456\","
        + "\"FILESIZE\":890}"
        + "]}}";

    SEARCHFILES searchfiles = (SEARCHFILES) messagedecoder.decode(encoded);
    Searchfile[] files = searchfiles.getSearchfiles();

    assertEquals(Command.SEARCHFILES, searchfiles.getCommand(), "command");
    assertEquals(RESPONSE, searchfiles.getHeader(), "header");
    assertEquals(Protocol.DATASET, searchfiles.getDataset(), "dataset");
    assertEquals("louisxiv", searchfiles.getUserid(), "userid");
    assertEquals(false, searchfiles.hasStarted(), "start");

    assertEquals("abc.123", files[0].getFilename(), "files1");
    assertEquals(1, files[0].getIndex(), "index1");
    assertEquals(123, files[0].getFilesize(), "filesize1");

    assertEquals("hallo.456", files[1].getFilename(), "files2");
    assertEquals(2, files[1].getIndex(), "index2");
    assertEquals(890, files[1].getFilesize(), "filesize2");
  }



  @Test
  public void decodeSEARCHFILES_RESPONSE_ohne_START_JERSEY() {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"SEARCHFILES\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"SEARCHFILE\":["
        + "{\"INDEX\":1,"
        + "\"FILENAME\":\"abc.123\","
        + "\"FILESIZE\":123},"
        + "{\"INDEX\":2,"
        + "\"FILENAME\":\"hallo.456\","
        + "\"FILESIZE\":890}"
        + "]}}";

    Gson gson = new GsonBuilder().registerTypeAdapter(SEARCHFILES.class, new SearchFilesDeserializer())
        .create();

    SEARCHFILES searchfiles = gson.fromJson(encoded, SEARCHFILES.class);
    Searchfile[] files = searchfiles.getSearchfiles();

    assertEquals(Command.SEARCHFILES, searchfiles.getCommand(), "command");
    assertEquals(RESPONSE, searchfiles.getHeader(), "header");
    assertEquals(Protocol.DATASET, searchfiles.getDataset(), "dataset");
    assertEquals("louisxiv", searchfiles.getUserid(), "userid");
    assertEquals(true, searchfiles.hasStarted(), "start");

    assertEquals("abc.123", files[0].getFilename(), "files1");
    assertEquals(1, files[0].getIndex(), "index1");
    assertEquals(123, files[0].getFilesize(), "filesize1");

    assertEquals("hallo.456", files[1].getFilename(), "files2");
    assertEquals(2, files[1].getIndex(), "index2");
    assertEquals(890, files[1].getFilesize(), "filesize2");
  }



  @Test
  public void decodeSEARCHFILES_RESPONSE_OHNE_USERID_JERSEY() {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"SEARCHFILES\","
        + "\"DATASET\":{"
        + "\"START\":true,"
        + "\"SEARCHFILE\":["
        + "{\"INDEX\":1,"
        + "\"FILENAME\":\"abc.123\","
        + "\"FILESIZE\":123},"
        + "{\"INDEX\":2,"
        + "\"FILENAME\":\"hallo.456\","
        + "\"FILESIZE\":456}"
        + "]}}";

    Gson gson = new GsonBuilder().registerTypeAdapter(SEARCHFILES.class, new SearchFilesDeserializer())
        .create();

    SEARCHFILES searchfiles = gson.fromJson(encoded, SEARCHFILES.class);

    Searchfile[] files = searchfiles.getSearchfiles();

    assertEquals(Command.SEARCHFILES, searchfiles.getCommand(), "command");
    assertEquals(RESPONSE, searchfiles.getHeader(), "header");
    assertEquals(Protocol.DATASET, searchfiles.getDataset(), "dataset");
    assertNull(searchfiles.getUserid(), "USERID");
    assertEquals(true, searchfiles.hasStarted(), "start");

    assertEquals("abc.123", files[0].getFilename(), "files1");
    assertEquals(1, files[0].getIndex(), "index1");
    assertEquals("hallo.456", files[1].getFilename(), "files2");
    assertEquals(2, files[1].getIndex(), "index2");
  }



  @Test
  public void decodeSEARCHFILES_RESPONSE2_JERSEY() {

    String encoded = "{\"HEADER\":\"RESPONSE\",\"COMMAND\":\"SEARCHFILES\",\"DATASET\":{\"START\":true,\"SEARCHFILE\":[{\"INDEX\":0,\"FILENAME\":\"1.mp4\",\"FILESIZE\":666},{\"INDEX\":1,\"FILENAME\":\"2.mp4\",\"FILESIZE\":666},{\"INDEX\":2,\"FILENAME\":\"3.mp4\",\"FILESIZE\":666}]}}";

    Gson gson = new GsonBuilder().registerTypeAdapter(SEARCHFILES.class, new SearchFilesDeserializer())
        .create();

    SEARCHFILES result = gson.fromJson(encoded, SEARCHFILES.class);

    Searchfile[] files = result.getSearchfiles();

    assertEquals(3, result.getSearchfiles().length, "3 Dateien");

  }



  @Test
  @DisplayName(value = "SIGNIN | CONFIRM")
  public void decodeSIGNIN_CONFIRM() throws DecodeException, EncodeException, IOException {

    String encoded = "{\"HEADER\":\"CONFIRM\"," + "\"COMMAND\":\"SIGNIN\","
        + "\"DATASET\": {"
        + "\"EMAIL\":\"team@javacomm.net\","
        + "\"NICKNAME\":\"nickname\","
        + "\"PASSWORD\":\"password\","
        + "\"USERID\":\"userid\","
        + "\"FOREGROUND_COLOR\":33,"
        + "\"BACKGROUND_COLOR\":99"
        + "}}";

    SIGNIN result = (SIGNIN) messagedecoder.decode(encoded);
    assertEquals(CONFIRM, result.getHeader(), "HEADER");
    assertEquals(Command.SIGNIN, result.getCommand(), "COMMAND");
    assertEquals("team@javacomm.net", result.getEmail(), "EMAIL");
    assertEquals("nickname", result.getNickname(), "NICKNAME");
    assertEquals("password", result.getPassword(), "PASSWORD");
    assertEquals("userid", result.getUserid(), "USERID");
    assertEquals(33, result.getForegroundColor(), "FOREGROUND");
    assertEquals(99, result.getBackgroundColor(), "BACKGROUND");
    assertTrue(true);
  }



  @Test
  @DisplayName(value = "SIGNIN | CONFIRM | BACKGROUND | FEHLT")
  public void decodeSIGNIN_CONFIRM_BACKGROUND_FEHLT() {

    String encoded = "{\"HEADER\":\"CONFIRM\"," + "\"COMMAND\":\"SIGNIN\","
        + "\"DATASET\": {"
        + "\"EMAIL\":\"team@javacomm.net\","
        + "\"NICKNAME\":\"nickname\","
        + "\"PASSWORD\":\"password\","
        + "\"USERID\":\"userid\","
        + "\"FOREGROUND_COLOR\":33"
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("Exception nicht geworfen, BACKGROUND fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  @DisplayName("SIGIN CONFIRM | MULTILINGUALKEY")
  public void decodeSIGNIN_CONFIRM_MULTILINGUALKEY() throws DecodeException, EncodeException, IOException {

    String encoded = "{\"HEADER\":\"CONFIRM\"," + "\"COMMAND\":\"SIGNIN\","
        + "\"DATASET\":{"
        + "\"EMAIL\":\"team@javacomm.net\","
        + "\"NICKNAME\":\"nickname\","
        + "\"PASSWORD\":\"password\","
        + "\"USERID\":\"userid\","
        + "\"IP\":\"localhost\","
        + "\"FOREGROUND_COLOR\":33,"
        + "\"BACKGROUND_COLOR\":99,"
        + "\"MULTILINGUALKEY\":\"Server_confirm_Passwort\""
        + "}}";

    SIGNIN result = (SIGNIN) messagedecoder.decode(encoded);
    assertEquals(CONFIRM, result.getHeader(), "header");
    assertEquals(Command.SIGNIN, result.getCommand(), "command");
    assertEquals("team@javacomm.net", result.getEmail(), "eMail");
    assertEquals("password", result.getPassword(), "password");
    assertEquals("userid", result.getUserid(), "userid");
    assertEquals("nickname", result.getNickname(), "nickname");
    assertEquals(33, result.getForegroundColor(), "foreground");
    assertEquals(99, result.getBackgroundColor(), "ip");
    assertEquals(KEY.SERVER_CONFIRM_PASSWORT, result.getMultilingualkey(), "MULTILINGUALKEY");
    assertTrue(true);
  }



  @Test
  public void decodeSIGNIN_CONFIRM_USERID_FEHLT() {
    String encoded = "{\"HEADER\":\"CONFIRM\"," + "\"COMMAND\":\"SIGNIN\","
        + "\"DATASET\": {"
        + "\"EMAIL\":\"team@javacomm.net\","
        + "\"NICKNAME\":\"nickname\","
        + "\"PASSWORD\":\"password\","
        + "\"FOREGROUND_COLOR\":33,"
        + "\"BACKGROUND_COLOR\":99"
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("Exception nicht geworfen, USERID fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }

  }



  @Test
  public void decodeSIGNIN_CONFIRM_TEXT() throws DecodeException, EncodeException, IOException {

    String encoded = "{\"HEADER\":\"CONFIRM\"," + "\"COMMAND\":\"SIGNIN\","
        + "\"DATASET\":"
        + "{\"EMAIL\":\"team@javacomm.net\","
        + "\"NICKNAME\":\"nickname\","
        + "\"PASSWORD\":\"password\","
        + "\"USERID\":\"userid\","
        + "\"IP\":\"localhost\","
        + "\"FOREGROUND_COLOR\":33,"
        + "\"BACKGROUND_COLOR\":99,"
        + "\"TEXT\":\"ich bin ein Test\""
        + "}}";

    SIGNIN result = (SIGNIN) messagedecoder.decode(encoded);
    assertEquals(CONFIRM, result.getHeader(), "header");
    assertEquals(Command.SIGNIN, result.getCommand(), "command");
    assertEquals("team@javacomm.net", result.getEmail(), "eMail");
    assertEquals("password", result.getPassword(), "password");
    assertEquals("userid", result.getUserid(), "userid");
    assertEquals("nickname", result.getNickname(), "nickname");
    assertEquals(33, result.getForegroundColor(), "foreground");
    assertEquals(99, result.getBackgroundColor(), "ip");
    assertEquals("ich bin ein Test", result.getText(), "TEXT");
    assertTrue(true);
  }



  @Test
  @DisplayName(value = "SIGNIN | ERROR")
  public void decodeSIGNIN_ERROR() throws DecodeException {

    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"SIGNIN\","
        + "\"DATASET\":{"
        + "\"EMAIL\":\"admin@javacomm.net\","
        + "\"ERRORCODE\":33,"
        + "\"ERRORMESSAGE\":\"123\","
        + "\"TEXT\":\"User could not be added.\","
        + "\"MULTILINGUALKEY\":"
        + "\"Server_Email_invalid\""
        + "}}";

    SIGNIN signin = (SIGNIN) messagedecoder.decodeSIGNIN(encoded);

    assertEquals("admin@javacomm.net", signin.getEmail(), "EMAIL");
    assertEquals(33, signin.getErrorCode(), "ERRORCODE");
    assertEquals("123", signin.getErrorMessage(), "ERRORMESSAGE");
    assertEquals("User could not be added.", signin.getText(), "TEXT");
    assertEquals("Server_Email_invalid", signin.getMultilingualkey().value(), "MULTILINGUALKEY");

  }



  @Test
  @DisplayName(value = "SIGNIN | ERROR | ERRORMESSAGE | FEHLT")
  public void decodeSIGNIN_ERROR_ERRORMESSAGE_FEHLT() throws DecodeException {

    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"SIGNIN\","
        + "\"DATASET\":{"
        + "\"EMAIL\":\"admin@javacomm.net\","
        + "\"ERRORCODE\":33,"
        + "\"TEXT\":\"User could not be added.\","
        + "\"MULTILINGUALKEY\":"
        + "\"Server_Email_invalid\""
        + "}}";

    SIGNIN signin = (SIGNIN) messagedecoder.decodeSIGNIN(encoded);

    assertEquals("admin@javacomm.net", signin.getEmail(), "EMAIL");
    assertEquals(33, signin.getErrorCode(), "ERRORCODE");
    assertEquals("User could not be added.", signin.getText(), "TEXT");
    assertEquals("Server_Email_invalid", signin.getMultilingualkey().value(), "MULTILINGUALKEY");
    assertNull(signin.getErrorMessage(), "ERRORMESSAGE");

  }



  @Test
  @DisplayName(value = "SIGNIN | ERROR | TEXT | FEHLT")
  public void decodeSIGNIN_ERROR_TEXT_FEHLT() throws DecodeException, EncodeException, IOException {

    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"SIGNIN\","
        + "\"DATASET\":{"
        + "\"EMAIL\":\"admin@javacomm.net\","
        + "\"ERRORCODE\":33,"
        + "\"ERRORMESSAGE\":\"123\","
        + "\"MULTILINGUALKEY\":\"Server_Email_invalid\""
        + "}}";

    SIGNIN result = (SIGNIN) messagedecoder.decodeSIGNIN(encoded);
    assertEquals(Command.SIGNIN, result.getCommand(), "COMMAND");
    assertEquals(ERROR, result.getHeader(), "HEADER");
    assertEquals("admin@javacomm.net", result.getEmail(), "EMAIL");
    assertEquals(33, result.getErrorCode(), "ERRORCODE");
    assertEquals("123", result.getErrorMessage(), "ERRORMESSAGE");

  }



  @Test
  public void decodeSIGNIN_ERROR_URL() throws DecodeException, EncodeException, IOException {

    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"SIGNIN\","
        + "\"DATASET\":{\"EMAIL\":\"abcd\","
        + "\"ERRORCODE\":0,"
        + "\"ERRORMESSAGE\":\"Die EMail-Adresse ist ungültig.\","
        + "\"TEXT\":\"Deine Email-Adresse ist ung%C3%BCltig.\"}}";

    SIGNIN result = (SIGNIN) messagedecoder.decodeSIGNIN(encoded);
    assertEquals(Command.SIGNIN, result.getCommand(), "command");
    assertEquals(ERROR, result.getHeader(), "header");
    assertEquals("abcd", result.getEmail(), "email");
    assertEquals(0, result.getErrorCode(), "errorcode");
    assertEquals(
        "Die EMail-Adresse ist ungültig.", result.getErrorMessage(), "Die EMail-Adresse ist ungültig."
    );
    assertEquals("Deine Email-Adresse ist ung%C3%BCltig.", result.getText(), "text");

  }



  @Test
  @DisplayName(value = "SIGNIN | REQUEST")
  public void decodeSIGNIN_REQUEST() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"SIGNIN\","
        + "\"DATASET\":{"
        + "\"EMAIL\":\"admin@javacomm.net\""
        + "}}";

    SIGNIN signin = (SIGNIN) messagedecoder.decode(encoded);
    assertEquals(Command.SIGNIN, signin.getCommand(), "command");
    assertEquals(REQUEST, signin.getHeader(), "header");
    assertEquals(Protocol.DATASET, signin.getDataset(), "dataset");
    assertEquals("admin@javacomm.net", signin.getEmail(), "email");
  }



  @Test
  @DisplayName(value = "SIGNIN | RESPONSE")
  public void decodeSIGNIN_RESPONSE() throws DecodeException {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"SIGNIN\","
        + "\"DATASET\":{"
        + "\"USERID\":\"userid\","
        + "\"NEUES_PASSWORD\":\"pass1234\","
        + "\"CONFIRMATION_KEY\":\"key12345\","
        + "\"NICKNAME\":\"anaconda\","
        + "\"BACKGROUND_COLOR\":-127,"
        + "\"FOREGROUND_COLOR\":-111,"
        + "\"MULTILINGUALKEY\":\"Server_neues_Passwort\","
        + "\"TEXT\":\"Server_neues_Passwort\""
        + "}}";

    SIGNIN result = (SIGNIN) messagedecoder.decode(encoded);
    assertEquals(RESPONSE, result.getHeader(), "header");
    assertEquals(Command.SIGNIN, result.getCommand(), "command");
    assertEquals("userid", result.getUserid(), "userid");
    assertEquals("pass1234", result.getNeuesPassword(), "neuesPassword");
    assertEquals("key12345", result.getConfirmationKey(), "confirmationKey");
    assertEquals(KEY.SERVER_NEUES_PASSWORT.value(), result.getText(), "TEXT");
    assertEquals(-127, result.getBackgroundColor(), "BACKGROUND_COLOR");
    assertEquals(-111, result.getForegroundColor(), "FOREGROUND_COLOR");
    assertTrue(true);
  }



  @Test
  @DisplayName(value = "SIGNIN | RESPONSE | BACKGROUND_COLOR | FEHLT")
  public void decodeSIGNIN_RESPONSE_BACKGROUND_COLOR_FEHLT() {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"SIGNIN\","
        + "\"DATASET\":{"
        + "\"USERID\":\"userid\","
        + "\"NEUES_PASSWORD\":\"pass1234\","
        + "\"CONFIRMATION_KEY\":\"key12345\","
        + "\"NICKNAME\":\"anaconda\","
        + "\"FOREGROUND_COLOR\":-111,"
        + "\"MULTILINGUALKEY\":\"Server_neues_Passwort\","
        + "\"TEXT\":\"Server_neues_Passwort\""
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("Exception nicht geworfen, BACKGROUND_COLOR fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  @DisplayName(value = "SIGNIN | RESPONSE | NICKNAME | FEHLT")
  public void decodeSIGNIN_RESPONSE_NICKNAME_FEHLT() {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"SIGNIN\","
        + "\"DATASET\":{"
        + "\"USERID\":\"userid\","
        + "\"NEUES_PASSWORD\":\"pass1234\","
        + "\"CONFIRMATION_KEY\":\"key12345\","
        + "\"BACKGROUND_COLOR\":-127,"
        + "\"FOREGROUND_COLOR\":-111,"
        + "\"MULTILINGUALKEY\":\"Server_neues_Passwort\","
        + "\"TEXT\":\"Server_neues_Passwort\""
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("Exception nicht geworfen, NICKNAME fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  @DisplayName(value = "SIGNIN | RESPONSE | CONFIRMATION_KEY | FEHLT")
  public void decodeSIGNIN_RESPONSE_CONFIRMATION_KEY_FEHLT() {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"SIGNIN\","
        + "\"DATASET\":{"
        + "\"USERID\":\"userid\","
        + "\"NEUES_PASSWORD\":\"pass1234\","
        + "\"NICKNAME\":\"anaconda\","
        + "\"BACKGROUND_COLOR\":-127,"
        + "\"FOREGROUND_COLOR\":-111,"
        + "\"MULTILINGUALKEY\":\"Server_neues_Passwort\","
        + "\"TEXT\":\"Server_neues_Passwort\""
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("Exception nicht geworfen, CONFIRMATION_KEY fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  @DisplayName(value = "SIGNIN | RESPONSE | MULTILINGUALKEY | FEHLT")
  public void decodeSIGNIN_RESPONSE_MULTILINGUALKEY_FEHLT() {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"SIGNIN\","
        + "\"DATASET\":"
        + "{\"EMAIL\":\"team@javacomm.net\","
        + "\"NICKNAME\":\"nickname\","
        + "\"PASSWORD\":\"password\","
        + "\"USERID\":\"userid\","
        + "\"IP\":\"localhost\","
        + "\"FOREGROUND_COLOR\":33,"
        + "\"BACKGROUND_COLOR\":99,"
        + "\"NEUES_PASSWORD\":\"pass1234\","
        + "\"CONFIRMATION_KEY\":\"key12345\","
        + "\"TEXT\":\"Server_neues_Passwort\""
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("Exception wurde nicht geworfen, MULTILINGUALKEY fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  @DisplayName(value = "SIGNIN | REQUEST | DATASET | FEHLT")
  public void decodeSIGNIN_REQUEST_DATASET_FEHLT() {
    String encoded = "{\"HEADER\":\"REQUEST\",\"COMMAND\":\"SIGNIN\"}";
    try {
      messagedecoder.decode(encoded);
      fail("DecodeException nicht geworfen, DATASET fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  @DisplayName(value = "SIGNIN | REQUEST | EMAIL | FEHLT")
  public void decodeSIGNIN_REQUEST_EMAIL_FEHLT() {
    String encoded = "{\"HEADER\":\"REQUEST\",\"COMMAND\":\"SIGNIN\"," + "\"DATASET\":{}}";
    try {
      messagedecoder.decode(encoded);
      fail("Exception nicht geworfen, EMAIL fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  @DisplayName(value = "SIGNIN")
  public void decodeSIGNIN() {
    String encoded = "";
    try {
      messagedecoder.decode(encoded);
      fail("DecodeException nicht geworfen");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  @DisplayName(value = "STOPSENDINGVIDEO REQUEST")
  public void decodeSTOPSENDINGVIDEO_REQUEST() throws DecodeException {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"STOPSENDINGVIDEO\","
        + "\"DATASET\":{"
        + "\"USERID\":\"fritz678\""
        + "}}";
    STOPSENDINGVIDEO result = (STOPSENDINGVIDEO) messagedecoder.decodeSTOPSENDINGVIDEO(encoded);
    assertEquals(REQUEST, result.getHeader());
    assertEquals(Command.STOPSENDINGVIDEO, result.getCommand());
    assertEquals(Protocol.DATASET, result.getDataset(), "dataset");
    assertEquals("fritz678", result.getUserid(), STOPSENDINGVIDEO.USERID);

  }

//  @Test
//  public void decodeTCPAUDIO() throws DecodeException, UnsupportedEncodingException {
//
//    final String polyfon = "Polyfon zwitschernd aßen Mäxchens Vögel Rüben, Joghurt und Quark";
//    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"TCPAUDIO\","
//        + "\"DATASET\":{"
//        + "\"TARGET\":\"8c\","
//        + "\"ORIGIN\":\"7b\","
//        + "\"CONTENT\":[80,111,108,121,102,111,110,32,122,119,105,116,115,99,104,101,114,110,100,32,97,-61,-97,101,110,32,77,-61,-92,120,99,104,101,110,115,32,86,-61,-74,103,101,108,32,82,-61,-68,98,101,110,44,32,74,111,103,104,117,114,116,32,117,110,100,32,81,117,97,114,107]"
//        + "}}";
//
//    TCPAUDIO result = (TCPAUDIO) messagedecoder.decodeTCPAUDIO(encoded);
//    assertEquals(REQUEST, result.getHeader());
//    assertEquals(Command.TCPAUDIO, result.getCommand());
//    assertEquals(Protocol.DATASET, result.getDataset(), "dataset");
//    assertEquals("8c", result.getTarget(), "TARGET");
//    assertEquals("7b", result.getOrigin(), "ORIGIN");
//    assertEquals(polyfon, new String(result.getContent()));
//  }



  @Test
  public void decodeTOPICMEMBER_Exception() {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"TOPICMEMBER\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"ROOM\":\"fussball\","
        + "\"TOPICMEMBER\":["
        + "{\"USERID\":\"jester\",\"NICKNAME\":\"anaconda\"},"
        + "{\"USERID\":\"luis\",\"NICKNAME\":\"viper\""
        + "}]"
        + "}}";

    try {
      TOPICMEMBER result = (TOPICMEMBER) messagedecoder.decode(encoded);
      fail("DecodeException wurde nicht geworfen, ROOMTYPE fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }

  }



  @Test
  public void decodeTOPICMEMBER_REQUEST() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"TOPICMEMBER\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"ROOM\":\"fussball\"}"
        + "}";

    TOPICMEMBER result = (TOPICMEMBER) messagedecoder.decode(encoded);
    assertEquals("louisxiv", result.getUserid(), "userid");
    assertEquals("fussball", result.getRoom(), "room");

  }



  @Test
  public void decodeTOPICMEMBER_RESPONSE() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"TOPICMEMBER\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"ROOM\":\"fussball\","
        + "\"ROOMTYPE\":\"FORUM\","
        + "\"TOPICMEMBER\":["
        + "{\"USERID\":\"jester\",\"NICKNAME\":\"anaconda\"},"
        + "{\"USERID\":\"luis\",\"NICKNAME\":\"viper\""
        + "}]"
        + "}}";

    TOPICMEMBER result = (TOPICMEMBER) messagedecoder.decode(encoded);
    assertEquals("louisxiv", result.getUserid(), "userid");
    assertEquals("fussball", result.getRoom(), "room");
    ChatUser[] chatuser = result.getChatUser();
    assertEquals(2, chatuser.length, "2 Datensätze");
    assertEquals("jester", chatuser[0].getUserid(), "userid0");
    assertEquals("anaconda", chatuser[0].getNickname(), "nickname0");
    assertEquals("luis", chatuser[1].getUserid(), "userid1");
    assertEquals("viper", chatuser[1].getNickname(), "nickname1");
    assertEquals("FORUM", result.getRoomtype().toString(), "FORUM");

  }



  @Test
  public void decodeTRANSFER_ERROR() throws DecodeException {

    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"TRANSFER\","
        + "\"DATASET\":{\"SLOT\":1,"
        + "\"FILENAME\":\"hallo.txt\","
        + "\"PATHFILE\":\"/root/path/to/file\","
        + "\"ERRORMESSAGE\":\"Die Datei ist ungültig\"}}";

    TRANSFER transfer = (TRANSFER) messagedecoder.decode(encoded);

    assertEquals(Command.TRANSFER, transfer.getCommand(), "command");
    assertEquals(ERROR, transfer.getHeader(), "header");
    assertEquals(Protocol.DATASET, transfer.getDataset(), "dataset");
    assertEquals(1, transfer.getSlot(), "slot");
    assertEquals("hallo.txt", transfer.getFilename(), "filename");
    assertEquals("/root/path/to/file", transfer.getPathfile(), "pathfile");
    assertEquals("Die Datei ist ungültig", transfer.getErrorMessage(), "errorMessage");

  }



  @Test
  public void decodeTRANSFER_REQUEST() throws DecodeException {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"TRANSFER\","
        + "\"DATASET\":{"
        + "\"SLOT\":1,"
        + "\"BLOCKINDEX\":33,"
        + "\"FILENAME\":\"hallo.mp3\","
        + "\"PATHFILE\":\"c:/\","
        + "\"MAXDATABLOCKS\":848"
        + "}"
        + "}";

    TRANSFER transfer = (TRANSFER) messagedecoder.decode(encoded);

    assertEquals(Command.TRANSFER, transfer.getCommand(), "command");
    assertEquals(REQUEST, transfer.getHeader(), "header");
    assertEquals(Protocol.DATASET, transfer.getDataset(), "dataset");
    assertEquals(1, transfer.getSlot(), "slot");
    assertEquals("hallo.mp3", transfer.getFilename(), "filename");
    assertEquals("c:/", transfer.getPathfile(), "pathfile");

    assertEquals(33, transfer.getBlockindex(), "blockindex");
    assertEquals(848, transfer.getMaxDatablocks(), "maxdatablcoks");

  }



  @Test
  public void decodeTRANSFER_RESPONSE() throws DecodeException {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"TRANSFER\","
        + "\"DATASET\":{"
        + "\"SLOT\":1,"
        + "\"BLOCKINDEX\":33,"
        + "\"FILENAME\":\"hallo.mp3\","
        + "\"PATHFILE\":\"c:/\","
        + "\"ENDFLAG\":true,"
        + "\"PAYLOAD\":\"273237\","
        + "\"MAXDATABLOCKS\":848"
        + "}"
        + "}";

    TRANSFER transfer = (TRANSFER) messagedecoder.decode(encoded);

    assertEquals(Command.TRANSFER, transfer.getCommand(), "command");
    assertEquals(RESPONSE, transfer.getHeader(), "header");
    assertEquals(Protocol.DATASET, transfer.getDataset(), "dataset");
    assertEquals(1, transfer.getSlot(), "slot");
    assertEquals("hallo.mp3", transfer.getFilename(), "filename");
    assertEquals("c:/", transfer.getPathfile(), "pathfile");

    assertEquals(33, transfer.getBlockindex(), "blockindex");
    assertEquals(true, transfer.getEndflag(), "endflag");
    assertEquals("273237", transfer.getPayload(), "pathfile");
    assertEquals(848, transfer.getMaxDatablocks(), "maxdatablocks");

  }



  @Test
  public void decodeUPDATEFILETYPES_CONFIRM() throws DecodeException {

    String encoded = "{\"HEADER\":\"CONFIRM\"," + "\"COMMAND\":\"UPDATEFILETYPES\","
        + "\"DATASET\":{"
        + "\"USERID\":\"userid123\""
        + "}"
        + "}";

    UPDATEFILETYPES updatetypes = (UPDATEFILETYPES) messagedecoder.decode(encoded);
    assertEquals(Command.UPDATEFILETYPES, updatetypes.getCommand(), "Command");
    assertEquals(CONFIRM, updatetypes.getHeader(), "CONFIRM");
    assertEquals("userid123", updatetypes.getUserid(), "USERID");
  }



  @Test
  public void decodeUPDATEFILETYPES_REQUEST() throws DecodeException {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"UPDATEFILETYPES\","
        + "\"DATASET\":{"
        + "\"USERID\":\"userid123\","
        + "\"FILETYPES\":["
        + "{\"SUFFIX\":\"pdf\",\"DESCRIPTION\":\"Adobe\",CHECKED:true},"
        + "{\"SUFFIX\":\"txt\",\"DESCRIPTION\":\"Simple Text Format\",CHECKED:false}"
        + "]}"
        + "}";

    UPDATEFILETYPES filetypes = (UPDATEFILETYPES) messagedecoder.decode(encoded);
    assertEquals(Command.UPDATEFILETYPES, filetypes.getCommand(), "Command");
    assertEquals(REQUEST, filetypes.getHeader(), "REQUEST");
    assertEquals("userid123", filetypes.getUserid(), "USERID");

    BlacklistTypes pdf = filetypes.getFiletypes()[0];
    BlacklistTypes txt = filetypes.getFiletypes()[1];

    assertEquals("pdf", pdf.getSuffix(), "pdfSuffix");
    assertEquals("Adobe", pdf.getDescription(), "Adobe");
    assertEquals(true, pdf.getChecked(), "pdfChecked");

    assertEquals("txt", txt.getSuffix(), "txtSuffix");
    assertEquals("Simple Text Format", txt.getDescription(), "Simple Text Format");
    assertEquals(false, txt.getChecked(), "txtChecked");
  }



  @Test
  public void decodeUPDATEPHONE_ERROR() throws DecodeException {
    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"UPDATEPHONE\"" + "}";

    UPDATEPHONE phone = (UPDATEPHONE) messagedecoder.decode(encoded);
    assertEquals(Command.UPDATEPHONE, phone.getCommand(), "COMMAND");
    assertEquals(ERROR, phone.getHeader(), "HEADER");

  }



  @Test
  public void decodeUPDATEPHONE_REQUEST() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"UPDATEPHONE\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"IP\":\"127.0.0.1\","
        + "\"PORT\":29477,"
        + "\"ONCALL\":false,"
        + "\"ONBUSY\":false"
        + "}}";

    UPDATEPHONE phone = (UPDATEPHONE) messagedecoder.decode(encoded);

    assertEquals(Command.UPDATEPHONE, phone.getCommand(), "COMMAND");
    assertEquals(REQUEST, phone.getHeader(), "HEADER");
    assertEquals("12345678", phone.getUserid(), "userid");
    assertEquals("127.0.0.1", phone.getIp(), "ip");
    assertEquals(29477, phone.getPort(), "port");
    assertEquals(false, phone.isOnBusy(), "onBusy");
  }



  @Test
  public void decodeUPDATEPHONE_REQUEST_OHNE_PORT() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"UPDATEPHONE\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"ONCALL\":false,"
        + "\"ONBUSY\":false"
        + "}}";

    UPDATEPHONE phone = (UPDATEPHONE) messagedecoder.decode(encoded);

    assertEquals(Command.UPDATEPHONE, phone.getCommand(), "COMMAND");
    assertEquals(REQUEST, phone.getHeader(), "HEADER");
    assertEquals("12345678", phone.getUserid(), "userid");
    assertEquals(false, phone.isOnBusy(), "onBusy");
  }



  @Test
  public void decodeUPDATEPHONE_REQUEST_ONCALL() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"UPDATEPHONE\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"IP\":\"127.0.0.1\","
        + "\"PORT\":29477,"
        + "\"ONCALL\":true,"
        + "\"ONBUSY\":true"
        + "}}";

    UPDATEPHONE phone = (UPDATEPHONE) messagedecoder.decode(encoded);

    assertEquals(Command.UPDATEPHONE, phone.getCommand(), "COMMAND");
    assertEquals(REQUEST, phone.getHeader(), "HEADER");
    assertEquals("12345678", phone.getUserid(), "userid");
    assertEquals("127.0.0.1", phone.getIp(), "ip");
    assertEquals(29477, phone.getPort(), "port");
    assertEquals(true, phone.isOnBusy(), "onBusy");
  }



  @Test
  public void decodeUPDATEPHONE_REQUEST_Schlechtfall() {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"UPDATEPHONE\","
        + "\"DATASET\":{"
        + "\"IP\":\"127.0.0.1\","
        + "\"PORT\":29477"
        + "}}";

    try {
      UPDATEPHONE phone = (UPDATEPHONE) messagedecoder.decode(encoded);
      fail("Exception nicht geworfen");
    }
    catch (DecodeException e) {
      assertEquals("Das Attribut USERID wurde nicht definiert.", e.getMessage());
    }

  }



  @Test
  public void decodeUPDATEPHONE_RESPONSE() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"UPDATEPHONE\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"IP\":\"93.0.0.1\","
        + "\"PORT\":29467,"
        + "\"ONCALL\":true,"
        + "\"ONBUSY\":true"
        + "}}";

    UPDATEPHONE phone = (UPDATEPHONE) messagedecoder.decode(encoded);

    assertEquals(Command.UPDATEPHONE, phone.getCommand(), "COMMAND");
    assertEquals(RESPONSE, phone.getHeader(), "HEADER");
    assertEquals("12345678", phone.getUserid(), "userid");
    assertEquals("93.0.0.1", phone.getIp(), "ip");
    assertEquals(29467, phone.getPort(), "port");
    assertEquals(true, phone.isOnBusy(), "onbusy");

  }



  @Test
  public void decodeUPDATEPHONE_RESPONSE_OHNE_ONCALL() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"UPDATEPHONE\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"ONBUSY\":true"
        + "}}";

    UPDATEPHONE phone = (UPDATEPHONE) messagedecoder.decode(encoded);

    assertEquals(Command.UPDATEPHONE, phone.getCommand(), "COMMAND");
    assertEquals(RESPONSE, phone.getHeader(), "HEADER");
    assertEquals("12345678", phone.getUserid(), "userid");
    assertEquals(true, phone.isOnBusy(), "onbusy");

  }



  @Test
  public void decodeUPDATEPHONE_RESPONSE_OHNE_PORT() throws DecodeException {
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"UPDATEPHONE\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"ONCALL\":true,"
        + "\"ONBUSY\":true"
        + "}}";

    UPDATEPHONE phone = (UPDATEPHONE) messagedecoder.decode(encoded);

    assertEquals(Command.UPDATEPHONE, phone.getCommand(), "COMMAND");
    assertEquals(RESPONSE, phone.getHeader(), "HEADER");
    assertEquals("12345678", phone.getUserid(), "userid");
    assertEquals(true, phone.isOnBusy(), "onbusy");

  }



  @Test
  public void decodeUPDATEPORT_REQUEST() throws DecodeException {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"UPDATEPORT\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"IP\":\"localhost\","
        + "\"PORT\":8443"
        + "}}";

    UPDATEPORT updateport = (UPDATEPORT) messagedecoder.decode(encoded);

    assertEquals(Command.UPDATEPORT, updateport.getCommand(), "command");
    assertEquals(REQUEST, updateport.getHeader(), "header");
    assertEquals(Protocol.DATASET, updateport.getDataset(), "dataset");
    assertEquals("louisxiv", updateport.getUserid(), "userid");
    assertEquals("localhost", updateport.getIp(), "ip");
    assertEquals(8443, updateport.getPort(), "port");
  }



  @Test
  public void decodeUPDATEPORT_RESPONSE() throws DecodeException {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"UPDATEPORT\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"IP\":\"localhost\","
        + "\"PORT\":8443"
        + "}}";

    UPDATEPORT updateport = (UPDATEPORT) messagedecoder.decode(encoded);

    assertEquals(Command.UPDATEPORT, updateport.getCommand(), "command");
    assertEquals(RESPONSE, updateport.getHeader(), "header");
    assertEquals(Protocol.DATASET, updateport.getDataset(), "dataset");
    assertEquals("louisxiv", updateport.getUserid(), "userid");
    assertEquals("localhost", updateport.getIp(), "ip");
    assertEquals(8443, updateport.getPort(), "port");
  }



  @Test
  @DisplayName(value = "UPDATEUSER | CONFIRM")
  public void decodeUPDATEUSER_CONFIRM() throws DecodeException {
    String encoded = "{\"HEADER\":\"CONFIRM\"," + "\"COMMAND\":\"UPDATEUSER\","
        + "\"DATASET\":{"
        + "\"IDENTITY\":\"ded\","
        + "\"FOREGROUND_COLOR\":3,"
        + "\"BACKGROUND_COLOR\":4,"
        + "\"NICKNAME\":\"anaconda\""
        + "}}";

    UPDATEUSER updateuser = (UPDATEUSER) messagedecoder.decode(encoded);

    assertEquals(Command.UPDATEUSER, updateuser.getCommand(), "command");
    assertEquals(CONFIRM, updateuser.getHeader(), "header");
    assertEquals(Protocol.DATASET, updateuser.getDataset(), "dataset");
    assertEquals(4, updateuser.getBackgroundColor(), "BACKGROUND_COLOR");
    assertEquals(3, updateuser.getForegroundColor(), "FOREGROUND_COLOR");
    assertEquals("anaconda", updateuser.getNickname(), "NICKNAME");
    assertEquals("ded", updateuser.getIdentity(), "IDENTITY");

  }



  @Test
  @DisplayName(value = "UPDATEUSER | CONFIRM | BACKGROUND_COLOR | FEHLT")
  public void decodeUPDATEUSER_CONFIRM_BACKGROUND_COLOR_FEHLT() {
    String encoded = "{\"HEADER\":\"CONFIRM\"," + "\"COMMAND\":\"UPDATE_USER\","
        + "\"DATASET\":{"
        + "\"IDENTITY\":\"ded\","
        + "\"FOREGROUND_COLOR\":3,"
        + "\"NICKNAME\":\"anaconda\","
        + "\"VOLUME\":89,"
        + "\"ONCALL\":true"
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("Exception nicht geworfen, BACKGROUND_COLOR fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }

  }



  @Test
  @DisplayName(value = "UPDATEUSER | CONFIRM | ONCALL | FEHLT")
  public void decodeUPDATEUSER_CONFIRM_ONCALL_FEHLT() {
    String encoded = "{\"HEADER\":\"CONFIRM\"," + "\"COMMAND\":\"UPDATE_USER\","
        + "\"DATASET\":{"
        + "\"IDENTITY\":\"ded\","
        + "\"FOREGROUND_COLOR\":3,"
        + "\"BACKGROUND_COLOR\":4,"
        + "\"NICKNAME\":\"anaconda\","
        + "\"VOLUME\":89"
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("Exception nicht geworfen, ONCALL fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }

  }



  @Test
  @DisplayName(value = "UPDATEUSER | CONFIRM | NICKNAME | FEHLT")
  public void decodeUPDATEUSER_CONFIRM_NICKNAME_FEHLT() {
    String encoded = "{\"HEADER\":\"CONFIRM\"," + "\"COMMAND\":\"UPDATE_USER\","
        + "\"DATASET\":{"
        + "\"IDENTITY\":\"ded\","
        + "\"FOREGROUND_COLOR\":3,"
        + "\"BACKGROUND_COLOR\":4,"
        + "\"VOLUME\":89,"
        + "\"ONCALL\":true"
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("Exception nicht geworfen, NICKNAME fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }

  }



  @Test
  @DisplayName(value = "UPDATEUSER | CONFIRM | IDENTITY | FEHLT")
  public void decodeUPDATEUSER_CONFIRM_IDENTITY_FEHLT() {
    String encoded = "{\"HEADER\":\"CONFIRM\"," + "\"COMMAND\":\"UPDATE_USER\","
        + "\"DATASET\":{"
        + "\"FOREGROUND_COLOR\":3,"
        + "\"BACKGROUND_COLOR\":4,"
        + "\"NICKNAME\":\"anaconda\","
        + "\"VOLUME\":89,"
        + "\"ONCALL\":true"
        + "}}";

    try {
      messagedecoder.decode(encoded);
      fail("Exception nicht geworfen, IDENTITY fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }

  }



  @Test
  @DisplayName(value = "UPDATEUSER | ERROR")
  public void decodeUPDATEUSER_ERROR() throws DecodeException {
    String encoded = "{\"HEADER\":\"ERROR\",\"COMMAND\":\"UPDATEUSER\",\"DATASET\":{"
        + "\"TEXT\":\"Hallo Freunde\""
        + "}}";

    UPDATEUSER updateuser = (UPDATEUSER) messagedecoder.decode(encoded);
    assertEquals("Hallo Freunde", updateuser.getText(), "TEXT");

  }



  @Test
  @DisplayName(value = "UPDATEUSER | ERROR | TEXT | FEHLT")
  public void decodeUPDATEUSER_ERROR_TEXT_FEHLT() {
    String encoded = "{\"HEADER\":\"ERROR\",\"COMMAND\":\"UPDATEUSER\",\"DATASET\":{}}";

    try {
      messagedecoder.decode(encoded);
      fail("Exception nicht geworfen, TEXT fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }

  }



  @Test
  @DisplayName(value = "UPDATEUSER | REQUEST")
  public void decodeUPDATEUSER_REQUEST() throws DecodeException {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"UPDATEUSER\","
        + "\"DATASET\":{"
        + "\"IDENTITY\":\"diejd\","
        + "\"FOREGROUND_COLOR\":-16777216,"
        + "\"BACKGROUND_COLOR\":-1,"
        + "\"NICKNAME\":\"anaconda\"}}";

    UPDATEUSER updateuser = (UPDATEUSER) messagedecoder.decode(encoded);
    assertEquals("diejd", updateuser.getIdentity(), "IDENTITY");
    assertEquals(-16777216, updateuser.getForegroundColor(), "FOREGROUND_COLOR");
    assertEquals(-1, updateuser.getBackgroundColor(), "BACKGROUND_COLOR");
    assertEquals("anaconda", updateuser.getNickname(), "NICKNAME");
  }



  @Test
  @DisplayName(value = "UPDATEUSER | REQUEST | IDENTITY | FEHLT ")
  public void decodeUPDATEUSER_REQUEST_IDENTITY_FEHLT() {
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"UPDATE_USER\","
        + "\"DATASET\":{"
        + "\"FOREGROUND_COLOR\":-16777216,"
        + "\"BACKGROUND_COLOR\":-1,"
        + "\"NICKNAME\":\"anaconda\"}}";

    try {
      messagedecoder.decode(encoded);
      fail("Exception nicht geworfen, IDENTITY fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void decodeUPLOADFILES_REQUEST() {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"UPLOADFILES\","
        + "\"DATASET\":{"
        + "\"START\":true,"
        + "\"USERID\":\"louisxiv\","
        + "\"IP\":\"192.168.1.41\","
        + "\"UPLOADFILES\":["
        + "{\"FILE\":\"abc.123\"},"
        + "{\"FILE\":\"hallo.456\"}]}}";

    try {
      UPLOADFILES uploadfiles = (UPLOADFILES) messagedecoder.decode(encoded);
      fail("DecodeException nicht geworfen");
    }
    catch (DecodeException e) {}

  }



  @Test
  public void decodeUPLOADFILES_REQUEST_PORT() throws DecodeException {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"UPLOADFILES\","
        + "\"DATASET\":{"
        + "\"START\":true,"
        + "\"USERID\":\"louisxiv\","
        + "\"IP\":\"192.168.1.41\","
        + "\"PORT\":8443,"
        + "\"UPLOADFILES\":["
        + "{\"FILENAME\":\"abc.txt\","
        + "\"FILESIZE\":1000},"
        + "{\"FILENAME\":\"zwei.txt\","
        + "\"FILESIZE\":1234}"
        + "]}}";

    UPLOADFILES uploadfiles = (UPLOADFILES) messagedecoder.decode(encoded);
    Uploadfile[] files = uploadfiles.getUploadfile();

    assertEquals(Command.UPLOADFILES, uploadfiles.getCommand(), "command");
    assertEquals(REQUEST, uploadfiles.getHeader(), "header");
    assertEquals(Protocol.DATASET, uploadfiles.getDataset(), "dataset");
    assertEquals("louisxiv", uploadfiles.getUserid(), "userid");
    assertEquals("192.168.1.41", uploadfiles.getIp(), "ip");
    assertEquals("abc.txt", files[0].getFilename(), "filename");
    assertEquals(1000, files[0].getFilesize(), "filesize");
    assertEquals("zwei.txt", files[1].getFilename(), "filename2");
    assertEquals(1234, files[1].getFilesize(), "filesize2");
    assertEquals(true, uploadfiles.hasStarted(), "strat");
    assertEquals(8443, uploadfiles.getPort(), "port");
  }



  @Test
  public void decodeUPLOADFILES_RESPONSE() throws DecodeException {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"UPLOADFILES\","
        + "\"DATASET\":{"
        + "\"START\":true,"
        + "\"USERID\":\"louisxiv\","
        + "\"IP\":\"192.168.1.41\""
        + "}}";

    UPLOADFILES uploadfiles = (UPLOADFILES) messagedecoder.decode(encoded);
    Uploadfile[] files = uploadfiles.getUploadfile();

    assertEquals(Command.UPLOADFILES, uploadfiles.getCommand(), "command");
    assertEquals(RESPONSE, uploadfiles.getHeader(), "header");
    assertEquals(Protocol.DATASET, uploadfiles.getDataset(), "dataset");
    assertEquals("louisxiv", uploadfiles.getUserid(), "userid");
    assertEquals("192.168.1.41", uploadfiles.getIp(), "ip");
    assertEquals(true, uploadfiles.hasStarted(), "strat");
  }



  @Test
  public void decodeUSERONLINELIST_REQUEST() throws DecodeException {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"USERONLINELIST\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\""
        + "}}";

    USERONLINELIST useronline = (USERONLINELIST) messagedecoder.decode(encoded);
    assertEquals(Command.USERONLINELIST, useronline.getCommand(), "command");
    assertEquals(REQUEST, useronline.getHeader(), "header");
    assertEquals(Protocol.DATASET, useronline.getDataset(), "dataset");
    assertEquals("louisxiv", useronline.getUserid(), "userid");

  }



  @Test
  public void decodeUSERONLINELIST_REQUEST_OHNE_USERID() throws DecodeException {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"USERONLINELIST\"," + "\"DATASET\":{" + "}}";

    USERONLINELIST useronline = (USERONLINELIST) messagedecoder.decode(encoded);
    assertEquals(Command.USERONLINELIST, useronline.getCommand(), "command");
    assertEquals(REQUEST, useronline.getHeader(), "header");
    assertEquals(Protocol.DATASET, useronline.getDataset(), "dataset");
    assertNull(useronline.getUserid(), "userid");

  }



  @Test
  public void decodeUSERONLINELIST_RESPONSE() throws DecodeException {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"USERONLINELIST\","
        + "\"DATASET\":{"
        + "\"USERID\":\"python\","
        + "\"USERONLINELIST\":["
        + "{\"IP\":\"localhost\","
        + "\"USERID\":\"louisxiv\","
        + "\"NICKNAME\":\"anaconda\"}]}}";

    USERONLINELIST useronline = (USERONLINELIST) messagedecoder.decode(encoded);

    assertEquals(Command.USERONLINELIST, useronline.getCommand(), "command");
    assertEquals(RESPONSE, useronline.getHeader(), "header");
    assertEquals(Protocol.DATASET, useronline.getDataset(), "dataset");
    UserOnline online[] = useronline.getUserOnline();
    assertEquals("localhost", online[0].getIp(), "ip");
    assertEquals("anaconda", online[0].getNickname(), "nickname");
    assertEquals("louisxiv", online[0].getUserid(), "userid");
    assertEquals("python", useronline.getUserid(), "sender");

  }



  @Test
  public void decodeUSERONLINELIST_RESPONSE_OHNE_USERID() throws DecodeException {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"USERONLINELIST\","
        + "\"DATASET\":{"
        + "\"USERONLINELIST\":["
        + "{\"IP\":\"localhost\","
        + "\"USERID\":\"louisxiv\","
        + "\"NICKNAME\":\"anaconda\"}]}}";

    USERONLINELIST useronline = (USERONLINELIST) messagedecoder.decode(encoded);

    assertEquals(Command.USERONLINELIST, useronline.getCommand(), "command");
    assertEquals(RESPONSE, useronline.getHeader(), "header");
    assertEquals(Protocol.DATASET, useronline.getDataset(), "dataset");
    UserOnline online[] = useronline.getUserOnline();
    assertEquals("localhost", online[0].getIp(), "ip");
    assertEquals("anaconda", online[0].getNickname(), "nickname");
    assertEquals("louisxiv", online[0].getUserid(), "userid");
    assertNull(useronline.getUserid(), "sender Userid");

  }



  @Test
  public void decodeUSERONLINELIST_RESPONSE_ohne_IP() throws DecodeException {

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"USERONLINELIST\","
        + "\"DATASET\":{"
        + "\"USERID\":\"python\","
        + "\"USERONLINELIST\":["
        + "{\"USERID\":\"louisxiv\","
        + "\"NICKNAME\":\"anaconda\"}]}}";

    USERONLINELIST useronline = (USERONLINELIST) messagedecoder.decode(encoded);

    assertEquals(Command.USERONLINELIST, useronline.getCommand(), "command");
    assertEquals(RESPONSE, useronline.getHeader(), "header");
    assertEquals(Protocol.DATASET, useronline.getDataset(), "dataset");
    UserOnline online[] = useronline.getUserOnline();
    assertNull(online[0].getIp(), "ip");
    assertEquals("anaconda", online[0].getNickname(), "nickname");
    assertEquals("louisxiv", online[0].getUserid(), "userid");
    assertEquals("python", useronline.getUserid(), "sender");

  }



  @Test
  public void decodeUSRLOGIN() {
    try {
      messagedecoder.decode(null);
      fail("DecodeException nicht geworfen");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @DisplayName(value = "USRLOGIN | ERROR")
  @Test
  public void descodeUSRLOGIN_ERROR() throws DecodeException {
    String encoded = "{\"HEADER\":\"ERROR\",\"COMMAND\":\"USRLOGIN\",\"DATASET\":{"
        + "\"TEXT\":\"Hallo Freunde\""
        + "}}";

    USRLOGIN useronline = (USRLOGIN) messagedecoder.decode(encoded);

    assertNull(useronline.getMultilingualkey());
    assertNotNull(useronline.getDataset());
    assertEquals(Command.USRLOGIN, useronline.getCommand(), "USRLOGIN");
    assertEquals("Hallo Freunde", useronline.getText(), "TEXT");

  }



  @DisplayName(value = "USRLOGIN | ERROR | DATASET | FEHLT")
  @Test
  public void descodeUSRLOGIN_ERROR_DATASET_FEHLT() {
    String encoded = "{\"HEADER\":\"ERROR\",\"COMMAND\":\"USRLOGIN\"}";

    try {
      messagedecoder.decode(encoded);
      fail("Exception nicht geworfen, DATASET fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }

  }



  @DisplayName(value = "USRLOGIN | ERROR | MULTILINGUALKEY")
  @Test
  public void descodeUSRLOGIN_ERROR_MULTILINGUALKEY() throws DecodeException {
    String encoded = "{\"HEADER\":\"ERROR\",\"COMMAND\":\"USRLOGIN\",\"DATASET\":{"
        + "\"MULTILINGUALKEY\":\"String_Konto_Deaktiviert\","
        + "\"TEXT\":\"String_Konto_Deaktiviert\""
        + "}}";

    USRLOGIN useronline = (USRLOGIN) messagedecoder.decode(encoded);

    assertNotNull(useronline.getMultilingualkey());
    assertNotNull(useronline.getDataset());
    assertEquals(Command.USRLOGIN, useronline.getCommand(), "USRLOGIN");
    assertEquals(KEY.STRING_KONTO_DEAKTIVIERT, useronline.getMultilingualkey(), "MULTILINGUALKEY");
    assertEquals("String_Konto_Deaktiviert", useronline.getText(), "TEXT");

  }



  @Test
  @DisplayName(value = "USRLOGIN | AGENT | IDENTITY")
  public void decodeUSRLOGIN_REQUEST_AGENT_IDENTITY() throws DecodeException {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"USRLOGIN\","
        + "\"DATASET\":"
        + "{\"IP\":\"localhost\","
        + "\"AGENT\":\"SmartPhone\","
        + "\"IDENTITY\":\"xWKuCO0i8qGv7MuKKfruEY+hkjbNgzumJP0P4yv7n+KNWJ2MzZdQ8zXr5Cnom0eF5MQpbbM9hHtH01Mjr+Cmy50jEk62FatmTPt/Y6KZbt0brYpXxXYFc29ME5/CTyp8Fd0mlYRAGg\\u003d\\u003d\""
        + "}}";

    USRLOGIN result = (USRLOGIN) messagedecoder.decodeUSRLOGIN(encoded);
    assertEquals(Command.USRLOGIN, result.getCommand(), "COMMAND");
    assertEquals(REQUEST, result.getHeader(), "HEADER");
    assertEquals(Protocol.DATASET, result.getDataset(), "DATASET");
    assertEquals(Agent.SmartPhone, result.getAgent(), "AGENT");
    assertEquals(
        "xWKuCO0i8qGv7MuKKfruEY+hkjbNgzumJP0P4yv7n+KNWJ2MzZdQ8zXr5Cnom0eF5MQpbbM9hHtH01Mjr+Cmy50jEk62FatmTPt/Y6KZbt0brYpXxXYFc29ME5/CTyp8Fd0mlYRAGg\u003d\u003d",
        result.getIdentity(), "IDENTITY"
    );
  }



  @Test
  @DisplayName(value = "USRLOGIN | IDENTITY")
  public void decodeUSRLOGIN_REQUEST_IP_IDENTITY() throws DecodeException {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"USRLOGIN\","
        + "\"DATASET\":{"
        + "\"AGENT\":\"Desktop\","
        + "\"IDENTITY\":\"xWKuCO0i8qGv7MuKKfruEY+hkjbNgzumJP0P4yv7n+KNWJ2MzZdQ8zXr5Cnom0eF5MQpbbM9hHtH01Mjr+Cmy50jEk62FatmTPt/Y6KZbt0brYpXxXYFc29ME5/CTyp8Fd0mlYRAGg\\u003d\\u003d\""
        + "}}";

    USRLOGIN result = (USRLOGIN) messagedecoder.decodeUSRLOGIN(encoded);
    assertEquals(Command.USRLOGIN, result.getCommand(), "COMMAND");
    assertEquals(REQUEST, result.getHeader(), "HEADER");
    assertEquals(Protocol.DATASET, result.getDataset(), "DATASET");
    assertEquals(Agent.Desktop, result.getAgent(), "AGENT");
    assertEquals(
        "xWKuCO0i8qGv7MuKKfruEY+hkjbNgzumJP0P4yv7n+KNWJ2MzZdQ8zXr5Cnom0eF5MQpbbM9hHtH01Mjr+Cmy50jEk62FatmTPt/Y6KZbt0brYpXxXYFc29ME5/CTyp8Fd0mlYRAGg\u003d\u003d",
        result.getIdentity(), "IDENTITY"
    );

  }



  @Test
  @DisplayName(value = "USRLOGIN | AGENT | IDENTITY | FEHLT")
  public void decodeUSRLOGIN_REQUEST_AGENT_IDENTITY_FEHLT() {

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"USRLOGIN\","
        + "\"DATASET\":{"
        + "\"AGENT\":\"SmartPhone\""
        + "}}";

    try {
      messagedecoder.decodeUSRLOGIN(encoded);
      fail("Exception nicht geworfen, IDENTITY fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }

  }



  @Test
  public void decodeUSRLOGIN_CONFIRM()
      throws DecodeException, JsonSyntaxException, UnsupportedEncodingException {

    String encoded = "{\"HEADER\":\"CONFIRM\"," + "\"COMMAND\":\"USRLOGIN\","
        + "\"DATASET\":"
        + "{\"SESSION\":\"ab\","
        + "\"IDENTITY\":\"jeidj\","
        + "\"FOREGROUND_COLOR\":-16777216,"
        + "\"BACKGROUND_COLOR\":-1,"
        + "\"ONCALL\":true,"
        + "\"VOLUME\":100,"
        + "\"NICKNAME\":\"anaconda\"}}";

    USRLOGIN result = (USRLOGIN) messagedecoder.decodeUSRLOGIN(encoded);
    assertEquals(Command.USRLOGIN, result.getCommand(), "COMMAND");
    assertEquals(CONFIRM, result.getHeader(), "HEADER");
    assertEquals(Protocol.DATASET, result.getDataset(), "DATASET");

    assertEquals("ab", result.getSession(), "SESSION");
    assertEquals("jeidj", result.getIdentity(), "IDENTITY");
    assertEquals(-16777216, result.getForegroundColor(), "FGCOLOR");
    assertEquals(-1, result.getBackgroundColor(), "BGCOLOR");
    assertEquals(true, result.getOncall(), "ONCALL");
    assertEquals(100, result.getVolume(), "VOLUME");
    assertEquals("anaconda", result.getNickname(), "NICKNAME");
  }



  @Test
  @DisplayName(value = "USRLOGIN | CONFRIM | NICKNAME | FEHLT")
  public void decodeUSRLOGIN_CONFIRM_NICKNAME_FEHLT() {

    String encoded = "{\"HEADER\":\"CONFIRM\"," + "\"COMMAND\":\"USRLOGIN\","
        + "\"DATASET\":"
        + "{\"SESSION\":\"ab\","
        + "\"IP\":\"localhost\","
        + "\"IDENTITY\":\"jeidj\","
        + "\"FOREGROUND_COLOR\":-16777216,"
        + "\"BACKGROUND_COLOR\":-1,"
        + "\"ONCALL\":true,"
        + "\"VOLUME\":100"
        + "}}";

    try {
      messagedecoder.decodeUSRLOGIN(encoded);
      fail("Exception nicht geworfen, NICKNAME fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  @DisplayName(value = "USRLOGIN | CONFIRM | SESSION | FEHLT")
  public void decodeUSRLOGIN_CONFIRM_SESSION_FEHLT() {

    String encoded = "{\"HEADER\":\"CONFIRM\"," + "\"COMMAND\":\"USRLOGIN\","
        + "\"DATASET\":"
        + "{\"IP\":\"localhost\","
        + "\"IDENTITY\":\"jeidj\","
        + "\"FOREGROUND_COLOR\":-16777216,"
        + "\"BACKGROUND_COLOR\":-1,"
        + "\"ONCALL\":true,"
        + "\"VOLUME\":100,"
        + "\"NICKNAME\":\"anaconda\"}}";

    try {
      messagedecoder.decodeUSRLOGIN(encoded);
      fail("Exception nicht geworfen, SESSION fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void decodeUSRLOGIN_CONFIRM_IDENTITY_FEHLT() {

    String encoded = "{\"HEADER\":\"CONFIRM\"," + "\"COMMAND\":\"USRLOGIN\","
        + "\"DATASET\":"
        + "{\"SESSION\":\"ab\","
        + "\"IP\":\"localhost\","
        + "\"FOREGROUND_COLOR\":-16777216,"
        + "\"BACKGROUND_COLOR\":-1,"
        + "\"ONCALL\":true,"
        + "\"VOLUME\":100,"
        + "\"NICKNAME\":\"anaconda\"}}";

    try {
      messagedecoder.decodeUSRLOGIN(encoded);
      fail("Exception geworfen, IDENTITY fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  @DisplayName(value = "USRLOGIN | CONFIRM | VOLUME | FEHLT")
  public void decodeUSRLOGIN_CONFIRM_VOLUME_FEHLT() {

    String encoded = "{\"HEADER\":\"CONFIRM\"," + "\"COMMAND\":\"USRLOGIN\","
        + "\"DATASET\":"
        + "{\"SESSION\":\"ab\","
        + "\"IP\":\"localhost\","
        + "\"IDENTITY\":\"jeidj\","
        + "\"FOREGROUND_COLOR\":-16777216,"
        + "\"BACKGROUND_COLOR\":-1,"
        + "\"ONCALL\":true,"
        + "\"NICKNAME\":\"anaconda\"}}";

    try {
      messagedecoder.decodeUSRLOGIN(encoded);
      fail("Exception nicht geworfen, VOLUME fehlt");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void kein_COMMAND() {
    String encoded = "{\"HEADER\":\"REQUEST\",\"kein_COMMAND\":\"USRLOGIN\",\"DATASET\":{\"EMAIL\":\"team@javacomm.net\"}}";
    try {
      MESSAGE result = messagedecoder.decode(encoded);
      fail("DecodeException nicht geworfen");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void ohne_COMMAND() {
    String encoded = "{\"HEADER\":\"REQUEST\":\"USRLOGIN\",\"DATASET\":{\"EMAIL\":\"team@javacomm.net\"}}";
    try {
      MESSAGE result = messagedecoder.decode(encoded);
      fail("DecodeException nicht geworfen");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void ohne_COMMAND_ohne_REQUEST() {
    String encoded = "{\"HEADER\":\"DATASET\":{\"EMAIL\":\"team@javacomm.net\"}}";
    try {
      MESSAGE result = messagedecoder.decode(encoded);
      fail("DecodeException nicht geworfen");
    }
    catch (DecodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void patternJSON() {
    Matcher matcher1 = Protocol.PATTERN_JSON.matcher("{\"HEADER\"abcd\"COMMAND\" }");
    assertTrue(matcher1.matches());

    Matcher matcher2 = Protocol.PATTERN_JSON.matcher("{\"HEADER\"     \"COMMAND\"       }");
    assertTrue(matcher2.matches());

    Matcher matcher3 = Protocol.PATTERN_JSON.matcher("{");
    assertFalse(matcher3.matches());

    Matcher matcher4 = Protocol.PATTERN_JSON.matcher("}");
    assertFalse(matcher4.matches());

    Matcher matcher5 = Protocol.PATTERN_JSON.matcher("");
    assertFalse(matcher5.matches());

    String bla = "{\"HEADER\":REQUEST,\"COMMAND\":SIGNIN,DATASET:{EMAIL:team@javacomm.net}}";

    Matcher matcher7 = Protocol.PATTERN_JSON.matcher(bla);
    assertTrue(matcher7.matches());

    Matcher matcher6 = Protocol.PATTERN_JSON.matcher(
        "{\"HEADER\":\"REQUEST\",\"COMMAND\":\"SIGNIN\",\"DATASET\":{\"EMAIL\":\"team@javacomm.net\"}}"
    );
    assertTrue(matcher6.matches());

  }



  @BeforeEach
  public void setUp() {
    messagedecoder = new MessageDecoder();
    messagedecoder.init(null);
  }

}
