package net.javacomm.protocol;

import static net.javacomm.protocol.HEADER.CONFIRM;
import static net.javacomm.protocol.HEADER.ERROR;
import static net.javacomm.protocol.HEADER.REQUEST;
import static net.javacomm.protocol.HEADER.RESPONSE;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertTrue;
import static org.junit.jupiter.api.Assertions.fail;
import jakarta.websocket.DecodeException;
import jakarta.websocket.EncodeException;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.UUID;
import javax.crypto.SecretKey;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;
import net.javacomm.multilingual.schema.KEY;
import net.javacomm.protocol.crypto.Crypto;



public class TestMessageEncoder {

  private final static Logger log = LogManager.getLogger(TestMessageEncoder.class);
  MessageEncoder encoder;

  @Test
  public void encodeBEAMEROFF_REQUEST() throws EncodeException {
    BEAMEROFF beamer = new BEAMEROFF();
    beamer.setCommand(Command.BEAMEROFF);
    beamer.setHeader(REQUEST);
    beamer.setDataset(Protocol.DATASET);
    beamer.setUserid("louisxiv");

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"BEAMEROFF\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\""
        + "}}";

    String result = encoder.encode(beamer);
    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeBEAMEROFF_RESPONSE() throws EncodeException {
    BEAMEROFF beamer = new BEAMEROFF();
    beamer.setCommand(Command.BEAMEROFF);
    beamer.setHeader(RESPONSE);
    beamer.setDataset(Protocol.DATASET);
    beamer.setUserid("louisxiv");

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"BEAMEROFF\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\""
        + "}}";

    String result = encoder.encode(beamer);
    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeBREAKROOMS_REQUEST() throws EncodeException {
    BREAKROOMS rooms = new BREAKROOMS();
    rooms.setCommand(Command.BREAKROOMS);
    rooms.setHeader(REQUEST);
    rooms.setDataset(Protocol.DATASET);
    rooms.setUserid("louisxiv");

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"BREAKROOMS\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\""
        + "}}";

    String result = encoder.encode(rooms);
    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeBREAKROOMS_RESPONSE() throws EncodeException {
    BREAKROOMS rooms = new BREAKROOMS();
    rooms.setCommand(Command.BREAKROOMS);
    rooms.setHeader(RESPONSE);
    rooms.setDataset(Protocol.DATASET);
    rooms.setUserid("louisxiv");
    rooms.setRooms(new String[] {"schach", "radsport"});

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"BREAKROOMS\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"ROOMS\":[\"schach\",\"radsport\"]"
        + "}}";

    String result = encoder.encode(rooms);
    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeBREAKROOMS_RESPONSE_ohne_Rooms() throws EncodeException {
    BREAKROOMS rooms = new BREAKROOMS();
    rooms.setCommand(Command.BREAKROOMS);
    rooms.setHeader(RESPONSE);
    rooms.setDataset(Protocol.DATASET);
    rooms.setUserid("louisxiv");
    rooms.setRooms(new String[] {});

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"BREAKROOMS\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"ROOMS\":[]"
        + "}}";

    String result = encoder.encode(rooms);
    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeCALLPRIVATECHAT_CONFIRM() throws EncodeException {
    CALLPRIVATECHAT call = new CALLPRIVATECHAT();
    call.setCommand(Command.CALLPRIVATECHAT);
    call.setHeader(CONFIRM);
    call.setDataset(Protocol.DATASET);
    call.setLocalNickname("anaconda");
    call.setRemoteNickname("Jester");
    call.setLocalSessionid("7");
    call.setRemoteSessionid("8");
    call.setSenderUID("hdh39678");

    String result = encoder.encode(call);

    String encoded = "{\"HEADER\":\"CONFIRM\"," + "\"COMMAND\":\"CALLPRIVATECHAT\","
        + "\"DATASET\":{"
        + "\"LOCAL_NICKNAME\":\"anaconda\","
        + "\"LOCAL_SESSIONID\":\"7\","
        + "\"REMOTE_NICKNAME\":\"Jester\","
        + "\"REMOTE_SESSIONID\":\"8\"}"
        + "}";

    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeCALLPRIVATECHAT_ERROR() throws EncodeException {
    CALLPRIVATECHAT call = new CALLPRIVATECHAT();
    call.setCommand(Command.CALLPRIVATECHAT);
    call.setHeader(ERROR);
    call.setDataset(Protocol.DATASET);
    call.setLocalNickname("anaconda");
    call.setRemoteNickname("anaconda");
    call.setLocalSessionid("7");
    call.setRemoteSessionid("7");
    call.setSenderUID("hdh39678");
    call.setErrorMessage("You cannot call yourself.");

    String result = encoder.encode(call);

    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"CALLPRIVATECHAT\","
        + "\"DATASET\":{"
        + "\"LOCAL_NICKNAME\":\"anaconda\","
        + "\"LOCAL_SESSIONID\":\"7\","
        + "\"REMOTE_NICKNAME\":\"anaconda\","
        + "\"REMOTE_SESSIONID\":\"7\","
        + "\"ERRORMESSAGE\":\"You cannot call yourself.\"}"
        + "}";

    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeCALLPRIVATECHAT_ERROR_MULTILINGUALKEY() throws EncodeException {
    CALLPRIVATECHAT call = new CALLPRIVATECHAT();
    call.setCommand(Command.CALLPRIVATECHAT);
    call.setHeader(ERROR);
    call.setDataset(Protocol.DATASET);
    call.setErrorMessage("anaconda kann keine weiteren Chats entgegennehmen.");
    call.setMultilingualkey(KEY.SERVER_CHAT_EXHAUSTED);

    String result = encoder.encode(call);

    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"CALLPRIVATECHAT\","
        + "\"DATASET\":{"
        + "\"ERRORMESSAGE\":\"anaconda kann keine weiteren Chats entgegennehmen.\","
        + "\"MULTILINGUALKEY\":\"Server_Chat_Exhausted\""
        + "}}";

    assertEquals(encoded, result, "JSON");
  }

//  {"header":"ERROR","command":"CALLPRIVATECHAT","dataset":"DATASET","errormessage":"anaconda kann keine weiteren Chats entgegennehmen.","multilingualkey":"SERVER_CHAT_EXHAUSTED"}



  @Test
  public void encodeCALLPRIVATECHAT_ERROR_OHNE_ErrorMessage() throws EncodeException {
    CALLPRIVATECHAT call = new CALLPRIVATECHAT();
    call.setCommand(Command.CALLPRIVATECHAT);
    call.setHeader(ERROR);
    call.setDataset(Protocol.DATASET);
    call.setLocalNickname("anaconda");
    call.setRemoteNickname("anaconda");
    call.setLocalSessionid("7");
    call.setRemoteSessionid("8");
    call.setText("einsame Seelen");
    call.setSenderUID("hdh39678");
    call.setMultilingualkey(KEY.STRING_EINSAME_SEELEN);

    String result = encoder.encode(call);

    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"CALLPRIVATECHAT\","
        + "\"DATASET\":{"
        + "\"LOCAL_NICKNAME\":\"anaconda\","
        + "\"LOCAL_SESSIONID\":\"7\","
        + "\"REMOTE_NICKNAME\":\"anaconda\","
        + "\"REMOTE_SESSIONID\":\"8\","
        + "\"TEXT\":\"einsame Seelen\","
        + "\"MULTILINGUALKEY\":\"String_Einsame_Seelen\""
        + "}}";

    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeCALLPRIVATECHAT_REQUEST() throws EncodeException {
    CALLPRIVATECHAT call = new CALLPRIVATECHAT();
    call.setCommand(Command.CALLPRIVATECHAT);
    call.setHeader(REQUEST);
    call.setDataset(Protocol.DATASET);
    call.setLocalNickname("anaconda");
    call.setRemoteNickname("jester");
    call.setSenderUID("hdh39678");

    String result = encoder.encode(call);

    String bla = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CALLPRIVATECHAT\","
        + "\"DATASET\":{"
        + "\"SENDER_UID\":\"hdh39678\","
        + "\"LOCAL_NICKNAME\":\"anaconda\","
        + "\"REMOTE_NICKNAME\":\"jester\"}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeCALLPRIVATECHAT_RESPONSE() throws EncodeException {
    CALLPRIVATECHAT call = new CALLPRIVATECHAT();
    call.setCommand(Command.CALLPRIVATECHAT);
    call.setHeader(RESPONSE);
    call.setDataset(Protocol.DATASET);
    call.setLocalNickname("anaconda");
    call.setRemoteNickname("Jester");
    call.setLocalSessionid("7");
    call.setRemoteSessionid("8");
    call.setSenderUID("hdh39678");
    call.setText("Der Anwender nimmt keine Anrufe entgegen.");

    String result = encoder.encode(call);

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CALLPRIVATECHAT\","
        + "\"DATASET\":{"
        + "\"LOCAL_NICKNAME\":\"anaconda\","
        + "\"LOCAL_SESSIONID\":\"7\","
        + "\"REMOTE_NICKNAME\":\"Jester\","
        + "\"REMOTE_SESSIONID\":\"8\","
        + "\"TEXT\":\"Der Anwender nimmt keine Anrufe entgegen.\"}"
        + "}";

    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeCALLPRIVATECHAT_RESPONSE_MILTILINGUALKEY() throws EncodeException {
    CALLPRIVATECHAT call = new CALLPRIVATECHAT();
    call.setCommand(Command.CALLPRIVATECHAT);
    call.setHeader(RESPONSE);
    call.setDataset(Protocol.DATASET);
    call.setLocalNickname("anaconda");
    call.setRemoteNickname("Jester");
    call.setLocalSessionid("7");
    call.setRemoteSessionid("8");
    call.setSenderUID("hdh39678");
    call.setText("Der Anwender nimmt keine Anrufe entgegen.");
    call.setMultilingualkey(KEY.SERVER_LEHNT_CHATANFRAGEN_AB);

    String result = encoder.encode(call);

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CALLPRIVATECHAT\","
        + "\"DATASET\":{"
        + "\"LOCAL_NICKNAME\":\"anaconda\","
        + "\"LOCAL_SESSIONID\":\"7\","
        + "\"REMOTE_NICKNAME\":\"Jester\","
        + "\"REMOTE_SESSIONID\":\"8\","
        + "\"TEXT\":\"Der Anwender nimmt keine Anrufe entgegen.\","
        + "\"MULTILINGUALKEY\":\"Server_Lehnt_Chatanfragen_Ab\""
        + "}}";

    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeCALLPRIVATECHAT_RESPONSE_ohne_TEXT() throws EncodeException {
    CALLPRIVATECHAT call = new CALLPRIVATECHAT();
    call.setCommand(Command.CALLPRIVATECHAT);
    call.setHeader(RESPONSE);
    call.setDataset(Protocol.DATASET);
    call.setLocalNickname("anaconda");
    call.setRemoteNickname("Jester");
    call.setLocalSessionid("7");
    call.setRemoteSessionid("8");
    call.setSenderUID("hdh39678");

    String result = encoder.encode(call);

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CALLPRIVATECHAT\","
        + "\"DATASET\":{"
        + "\"LOCAL_NICKNAME\":\"anaconda\","
        + "\"LOCAL_SESSIONID\":\"7\","
        + "\"REMOTE_NICKNAME\":\"Jester\","
        + "\"REMOTE_SESSIONID\":\"8\""
        + "}}";

    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeCALLREMOTEUSER_CONFIRM() throws EncodeException {
    CALLREMOTEUSER call = new CALLREMOTEUSER();
    call.setCommand(Command.CALLREMOTEUSER);
    call.setHeader(CONFIRM);
    call.setDataset(Protocol.DATASET);
    call.setLocalNickname("anaconda");
    call.setLocalSessionid("7");
    call.setRemoteNickname("Jester");
    call.setRemoteSessionid("8");

    String result = encoder.encode(call);

    String bla = "{\"HEADER\":\"CONFIRM\"," + "\"COMMAND\":\"CALLREMOTEUSER\","
        + "\"DATASET\":{"
        + "\"LOCAL_NICKNAME\":\"anaconda\","
        + "\"LOCAL_SESSIONID\":\"7\","
        + "\"REMOTE_NICKNAME\":\"Jester\","
        + "\"REMOTE_SESSIONID\":\"8\"}"
        + "}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeCALLREMOTEUSER_ERROR() throws EncodeException {
    CALLREMOTEUSER call = new CALLREMOTEUSER();
    call.setCommand(Command.CALLREMOTEUSER);
    call.setHeader(ERROR);
    call.setDataset(Protocol.DATASET);
    call.setErrorMessage("The user is offline.");

    String result = encoder.encode(call);

    String bla = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"CALLREMOTEUSER\","
        + "\"DATASET\":{"
        + "\"ERRORMESSAGE\":\"The user is offline.\"}"
        + "}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeCALLREMOTEUSER_REQUEST() throws EncodeException {
    CALLREMOTEUSER call = new CALLREMOTEUSER();
    call.setCommand(Command.CALLREMOTEUSER);
    call.setHeader(REQUEST);
    call.setDataset(Protocol.DATASET);
    call.setLocalNickname("anaconda");
    call.setLocalSessionid("7");
    call.setRemoteNickname("Jester");
    call.setRemoteSessionid("8");

    String result = encoder.encode(call);

    String bla = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CALLREMOTEUSER\","
        + "\"DATASET\":{"
        + "\"LOCAL_NICKNAME\":\"anaconda\","
        + "\"LOCAL_SESSIONID\":\"7\","
        + "\"REMOTE_NICKNAME\":\"Jester\","
        + "\"REMOTE_SESSIONID\":\"8\"}"
        + "}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeCALLREMOTEUSER_RESPONSE() throws EncodeException {
    CALLREMOTEUSER call = new CALLREMOTEUSER();
    call.setCommand(Command.CALLREMOTEUSER);
    call.setHeader(RESPONSE);
    call.setDataset(Protocol.DATASET);
    call.setLocalNickname("anaconda");
    call.setLocalSessionid("7");
    call.setRemoteNickname("Jester");
    call.setRemoteSessionid("8");

    String result = encoder.encode(call);

    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CALLREMOTEUSER\","
        + "\"DATASET\":{"
        + "\"LOCAL_NICKNAME\":\"anaconda\","
        + "\"LOCAL_SESSIONID\":\"7\","
        + "\"REMOTE_NICKNAME\":\"Jester\","
        + "\"REMOTE_SESSIONID\":\"8\"}"
        + "}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeCANDIDATETOPICMEMBER_REQUEST() throws EncodeException {
    CANDIDATETOPICMEMBER candidate = new CANDIDATETOPICMEMBER();
    candidate.setCommand(Command.CANDIDATETOPICMEMBER);
    candidate.setHeader(REQUEST);
    candidate.setDataset(Protocol.DATASET);
    candidate.setUserid("12345678");
    candidate.setRoom("Nexus");
    String result = encoder.encode(candidate);
    String expected = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CANDIDATETOPICMEMBER\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"ROOM\":\"Nexus\""
        + "}}";

    assertEquals(expected, result, "roomlist,request");
  }



  @Test
  public void encodeCANDIDATETOPICMEMBER_RESPONSE_mit_User2() throws EncodeException {
    CANDIDATETOPICMEMBER member = new CANDIDATETOPICMEMBER();
    member.setCommand(Command.CANDIDATETOPICMEMBER);
    member.setHeader(RESPONSE);
    member.setDataset(Protocol.DATASET);
    member.setUserid("louisxiv");
    member.setRoomtype(Roomtype.BESPRECHUNGSRAUM);
    member.setRoom("fussball");
    ChatUser[] chatuser = new ChatUser[2];
    chatuser[0] = new ChatUser();
    chatuser[0].setUserid("jester");
    chatuser[0].setNickname("python");

    chatuser[1] = new ChatUser();
    chatuser[1].setUserid("luis");
    chatuser[1].setNickname("viper");
    chatuser[1].setBackgroundColor(33);

    member.setChatUser(chatuser);

    String result = encoder.encode(member);

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CANDIDATETOPICMEMBER\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"ROOM\":\"fussball\","
        + "\"ROOMTYPE\":\"BESPRECHUNGSRAUM\","
        + "\"TOPICMEMBER\":["
        + "{\"USERID\":\"jester\",\"NICKNAME\":\"python\"},"
        + "{\"USERID\":\"luis\",\"NICKNAME\":\"viper\""
        + "}]"
        + "}}";
    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeCHANGETOPICMEMBER_ERROR() throws EncodeException {
    CHANGETOPICMEMBER change = new CHANGETOPICMEMBER();
    change.setCommand(Command.CHANGETOPICMEMBER);
    change.setHeader(ERROR);
    change.setDataset(Protocol.DATASET);
    change.setUserid("12345678");
    change.setRoom("Nexus");
    StringBuilder buffer = new StringBuilder();
    buffer.append("Room <");
    buffer.append(change.getRoom());
    buffer.append("> does not exist.");
    change.setErrorMessage(buffer.toString());

    String result = encoder.encode(change);
    String expected = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"CHANGETOPICMEMBER\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"ROOM\":\"Nexus\","
        + "\"ERRORMESSAGE\":\"Room <Nexus> does not exist.\""
        + "}}";

    assertEquals(expected, result, "roomlist,request");
  }



  @Test
  public void encodeCHANGETOPICMEMBER_REQUEST() throws EncodeException {
    CHANGETOPICMEMBER change = new CHANGETOPICMEMBER();
    change.setCommand(Command.CHANGETOPICMEMBER);
    change.setHeader(REQUEST);
    change.setDataset(Protocol.DATASET);
    change.setUserid("12345678");
    change.setRoom("Nexus");
    change.setRoomtype(Roomtype.BESPRECHUNGSRAUM);
    ChatUser[] value = new ChatUser[2];
    value[0] = new ChatUser();
    value[0].setUserid("jester");
    value[0].setNickname("python");

    value[1] = new ChatUser();
    value[1].setUserid("luis");
    value[1].setNickname("anaconda");

    change.setChatUser(value);
    String result = encoder.encode(change);
    String expected = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CHANGETOPICMEMBER\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"ROOM\":\"Nexus\","
        + "\"ROOMTYPE\":\"BESPRECHUNGSRAUM\","
        + "\"TOPICMEMBER\":["
        + "{\"USERID\":\"jester\",\"NICKNAME\":\"python\"},"
        + "{\"USERID\":\"luis\",\"NICKNAME\":\"anaconda\""
        + "}]"
        + "}}";

    assertEquals(expected, result, "roomlist,request");
  }



  @Test
  public void encodeCHANGETOPICMEMBER_REQUEST_NICKNAME() throws EncodeException {
    CHANGETOPICMEMBER change = new CHANGETOPICMEMBER();
    change.setCommand(Command.CHANGETOPICMEMBER);
    change.setHeader(REQUEST);
    change.setDataset(Protocol.DATASET);
    change.setUserid("12345678");
    change.setRoom("Nexus");
    change.setRoomtype(Roomtype.BESPRECHUNGSRAUM);
    ChatUser[] value = new ChatUser[0];

    change.setChatUser(value);
    String result = encoder.encode(change);
    String expected = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CHANGETOPICMEMBER\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"ROOM\":\"Nexus\","
        + "\"ROOMTYPE\":\"BESPRECHUNGSRAUM\","
        + "\"TOPICMEMBER\":["
        + "]"
        + "}}";

    assertEquals(expected, result, "roomlist,request");
  }



  @Test
  public void encodeCHANGETOPICMEMBER_REQUEST_ohne_NICKNAME() {
    CHANGETOPICMEMBER change = new CHANGETOPICMEMBER();
    change.setCommand(Command.CHANGETOPICMEMBER);
    change.setHeader(REQUEST);
    change.setDataset(Protocol.DATASET);
    change.setUserid("12345678");
    change.setRoom("Nexus");
    ChatUser[] value = new ChatUser[1];
    value[0] = new ChatUser();
    value[0].setUserid("luis");

    change.setChatUser(value);

    String result;
    try {
      result = encoder.encode(change);
      fail("EncodeException nicht geworfen");
    }
    catch (EncodeException e) {
      // TODO Auto-generated catch block
      assertTrue(true);
    }
  }



  @Test
  public void encodeCHANGETOPICMEMBER_REQUEST_ohne_ROOM_EncodeException() {
    CHANGETOPICMEMBER member = new CHANGETOPICMEMBER();
    member.setCommand(Command.CHANGETOPICMEMBER);
    member.setHeader(REQUEST);
    member.setDataset(Protocol.DATASET);
    member.setUserid("fussball");

    try {
      String result = encoder.encode(member);
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void encodeCHANGETOPICMEMBER_REQUEST_ohne_UID_EncodeException() {
    CHANGETOPICMEMBER member = new CHANGETOPICMEMBER();
    member.setCommand(Command.CHANGETOPICMEMBER);
    member.setHeader(REQUEST);
    member.setDataset(Protocol.DATASET);
    member.setRoom("fussball");

    try {
      String result = encoder.encode(member);
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void encodeCHANGETOPICMEMBER_REQUEST_ohne_USERID() throws EncodeException {
    CHANGETOPICMEMBER change = new CHANGETOPICMEMBER();
    change.setCommand(Command.CHANGETOPICMEMBER);
    change.setHeader(REQUEST);
    change.setDataset(Protocol.DATASET);
    change.setUserid("12345678");
    change.setRoom("Nexus");
    change.setRoomtype(Roomtype.PAUSENRAUM);
    ChatUser[] value = new ChatUser[1];
    value[0] = new ChatUser();
    value[0].setNickname("luis");

    change.setChatUser(value);
    String result = encoder.encode(change);
    String expected = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CHANGETOPICMEMBER\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"ROOM\":\"Nexus\","
        + "\"ROOMTYPE\":\"PAUSENRAUM\","
        + "\"TOPICMEMBER\":["
        + "{"
        + "\"NICKNAME\":\"luis\""
        + "}]"
        + "}}";

    assertEquals(expected, result, "roomlist,request");
  }



  @Test
  public void encodeCHANGETOPICMEMBER_RESPONSE() throws EncodeException {
    CHANGETOPICMEMBER member = new CHANGETOPICMEMBER();
    member.setCommand(Command.CHANGETOPICMEMBER);
    member.setHeader(RESPONSE);
    member.setDataset(Protocol.DATASET);
    member.setUserid("louisxiv");
    member.setRoom("fussball");
    member.setRoomtype(Roomtype.BESPRECHUNGSRAUM);
    member.setChatUser(new ChatUser[0]);

    String result = encoder.encode(member);

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CHANGETOPICMEMBER\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"ROOM\":\"fussball\","
        + "\"ROOMTYPE\":\"BESPRECHUNGSRAUM\","
        + "\"TOPICMEMBER\":[]"
        + "}}";
    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeCHANGETOPICMEMBER_RESPONSE_mit_User2() throws EncodeException {
    CHANGETOPICMEMBER member = new CHANGETOPICMEMBER();
    member.setCommand(Command.CHANGETOPICMEMBER);
    member.setHeader(RESPONSE);
    member.setDataset(Protocol.DATASET);
    member.setUserid("louisxiv");
    member.setRoom("fussball");
    member.setRoomtype(Roomtype.PAUSENRAUM);
    ChatUser[] chatuser = new ChatUser[2];
    chatuser[0] = new ChatUser();
    chatuser[0].setUserid("jester");
    chatuser[0].setNickname("python");

    chatuser[1] = new ChatUser();
    chatuser[1].setUserid("luis");
    chatuser[1].setNickname("viper");
    chatuser[1].setBackgroundColor(33);

    member.setChatUser(chatuser);

    String result = encoder.encode(member);

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CHANGETOPICMEMBER\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"ROOM\":\"fussball\","
        + "\"ROOMTYPE\":\"PAUSENRAUM\","
        + "\"TOPICMEMBER\":["
        + "{\"USERID\":\"jester\",\"NICKNAME\":\"python\"},"
        + "{\"USERID\":\"luis\",\"NICKNAME\":\"viper\""
        + "}]"
        + "}}";
    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeCHATMESSAGE_REQUEST() throws EncodeException {
    CHATMESSAGE userlist = new CHATMESSAGE();
    userlist.setCommand(Command.CHATMESSAGE);
    userlist.setHeader(REQUEST);
    userlist.setDataset(Protocol.DATASET);

    ChatUser chatuser = new ChatUser();
    chatuser.setUserid("louisxiv");
    chatuser.setForegroundColor(-16777216);
    chatuser.setBackgroundColor(-1);
    chatuser.setNickname("anaconda");
    userlist.setMessage("Hallo meine Freunde");
    userlist.setChatUser(chatuser);
    userlist.setRoom("room");

    String result = encoder.encode(userlist);

    String bla = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CHATMESSAGE\","
        + "\"DATASET\":{"
        + "\"ROOM\":\"room\","
        + "\"MESSAGE\":\"Hallo meine Freunde\","
        + "\"CHATUSER\":{"
        + "\"BACKGROUND_COLOR\":-1,"
        + "\"FOREGROUND_COLOR\":-16777216,"
        + "\"USERID\":\"louisxiv\","
        + "\"NICKNAME\":\"anaconda\"}}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeCHATMESSAGE_REQUEST_2() throws EncodeException {
    CHATMESSAGE userlist = new CHATMESSAGE();
    userlist.setCommand(Command.CHATMESSAGE);
    userlist.setHeader(REQUEST);
    userlist.setDataset(Protocol.DATASET);

    ChatUser chatuser = new ChatUser();
    chatuser.setUserid("louisxiv");
    chatuser.setForegroundColor(-14607386);
    chatuser.setBackgroundColor(-1772550);
    chatuser.setNickname("anaconda");
    userlist.setMessage("anhang");
    userlist.setChatUser(chatuser);
    userlist.setRoom("Tester");
    userlist.setAttachment(29098336975847574l);
    userlist.setFilename("Top 5 Highest Win Rate Openings for White (2025).mp4");
    userlist.setFilesize(67934764l);

    String result = encoder.encode(userlist);

    String encoded = "{\"HEADER\":\"REQUEST\",\"COMMAND\":\"CHATMESSAGE\",\"DATASET\":{"
        + "\"ROOM\":\"Tester\","
        + "\"MESSAGE\":\"anhang\","
        + "\"ATTACHMENT\":29098336975847574,"
        + "\"FILENAME\":\"Top 5 Highest Win Rate Openings for White (2025).mp4\","
        + "\"FILESIZE\":67934764,"
        + "\"CHATUSER\":{\"BACKGROUND_COLOR\":-1772550,"
        + "\"FOREGROUND_COLOR\":-14607386,"
        + "\"USERID\":\"louisxiv\","
        + "\"NICKNAME\":\"anaconda\"}}}";

    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeCHATMESSAGE_REQUEST_ATTACHMENT() throws EncodeException {
    CHATMESSAGE userlist = new CHATMESSAGE();
    userlist.setCommand(Command.CHATMESSAGE);
    userlist.setHeader(REQUEST);
    userlist.setDataset(Protocol.DATASET);

    ChatUser chatuser = new ChatUser();
    chatuser.setUserid("louisxiv");
    chatuser.setForegroundColor(-16777216);
    chatuser.setBackgroundColor(-1);
    chatuser.setNickname("anaconda");
    userlist.setMessage("Hallo meine Freunde");
    userlist.setChatUser(chatuser);
    userlist.setRoom("room");
    userlist.setAttachment(33l);
    userlist.setFilename("name.txt");
    userlist.setFilesize(1024l);

    String result = encoder.encode(userlist);

    String bla = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CHATMESSAGE\","
        + "\"DATASET\":{"
        + "\"ROOM\":\"room\","
        + "\"MESSAGE\":\"Hallo meine Freunde\","
        + "\"ATTACHMENT\":33,"
        + "\"FILENAME\":\"name.txt\","
        + "\"FILESIZE\":1024,"
        + "\"CHATUSER\":{"
        + "\"BACKGROUND_COLOR\":-1,"
        + "\"FOREGROUND_COLOR\":-16777216,"
        + "\"USERID\":\"louisxiv\","
        + "\"NICKNAME\":\"anaconda\"}}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeCHATMESSAGE_REQUEST_ATTACHMENT_JERSEY() {
    CHATMESSAGE userlist = new CHATMESSAGE();
    userlist.setCommand(Command.CHATMESSAGE);
    userlist.setHeader(REQUEST);
    userlist.setDataset(Protocol.DATASET);

    ChatUser chatuser = new ChatUser();
    chatuser.setUserid("louisxiv");
    chatuser.setForegroundColor(-16777216);
    chatuser.setBackgroundColor(-1);
    chatuser.setNickname("anaconda");
    userlist.setMessage("Hallo meine Freunde");
    userlist.setChatUser(chatuser);
    userlist.setRoom("room");
    userlist.setAttachment(33l);
    userlist.setFilename("name.txt");
    userlist.setFilesize(1024l);

    String result = userlist.toString();

    String bla = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CHATMESSAGE\","
        + "\"DATASET\":{"
        + "\"ROOM\":\"room\","
        + "\"MESSAGE\":\"Hallo meine Freunde\","
        + "\"ATTACHMENT\":33,"
        + "\"FILENAME\":\"name.txt\","
        + "\"FILESIZE\":1024,"
        + "\"CHATUSER\":{"
        + "\"BACKGROUND_COLOR\":-1,"
        + "\"FOREGROUND_COLOR\":-16777216,"
        + "\"USERID\":\"louisxiv\","
        + "\"NICKNAME\":\"anaconda\"}}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeCHATMESSAGE_REQUEST_Jersey() {
    CHATMESSAGE userlist = new CHATMESSAGE();
    userlist.setCommand(Command.CHATMESSAGE);
    userlist.setHeader(REQUEST);
    userlist.setDataset(Protocol.DATASET);

    ChatUser chatuser = new ChatUser();
    chatuser.setUserid("louisxiv");
    chatuser.setForegroundColor(-16777216);
    chatuser.setBackgroundColor(-1);
    chatuser.setNickname("anaconda");
    userlist.setMessage("Hallo meine Freunde");
    userlist.setChatUser(chatuser);
    userlist.setRoom("room");

    String result = userlist.toString();

    String bla = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CHATMESSAGE\","
        + "\"DATASET\":{"
        + "\"ROOM\":\"room\","
        + "\"MESSAGE\":\"Hallo meine Freunde\","
        + "\"CHATUSER\":{"
        + "\"BACKGROUND_COLOR\":-1,"
        + "\"FOREGROUND_COLOR\":-16777216,"
        + "\"USERID\":\"louisxiv\","
        + "\"NICKNAME\":\"anaconda\"}}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeCHATMESSAGE_REQUEST_UMLAUTE() throws EncodeException {
    CHATMESSAGE userlist = new CHATMESSAGE();
    userlist.setCommand(Command.CHATMESSAGE);
    userlist.setHeader(REQUEST);
    userlist.setDataset(Protocol.DATASET);

    ChatUser chatuser = new ChatUser();
    chatuser.setUserid("louisxiv");
    chatuser.setForegroundColor(-16777216);
    chatuser.setBackgroundColor(-1);
    chatuser.setNickname("anaconda");
    userlist.setMessage("äöüßéáúéíóú");
    userlist.setChatUser(chatuser);
    userlist.setRoom("room");
    userlist.setAttachment(33l);
    userlist.setFilename("name.txt");
    userlist.setFilesize(1024l);

    String result = encoder.encode(userlist);

    String bla = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CHATMESSAGE\","
        + "\"DATASET\":{"
        + "\"ROOM\":\"room\","
        + "\"MESSAGE\":\"äöüßéáúéíóú\","
        + "\"ATTACHMENT\":33,"
        + "\"FILENAME\":\"name.txt\","
        + "\"FILESIZE\":1024,"
        + "\"CHATUSER\":{"
        + "\"BACKGROUND_COLOR\":-1,"
        + "\"FOREGROUND_COLOR\":-16777216,"
        + "\"USERID\":\"louisxiv\","
        + "\"NICKNAME\":\"anaconda\"}}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeCHATMESSAGE_REQUEST_UMLAUTE_JERSEY() {
    CHATMESSAGE userlist = new CHATMESSAGE();
    userlist.setCommand(Command.CHATMESSAGE);
    userlist.setHeader(REQUEST);
    userlist.setDataset(Protocol.DATASET);

    ChatUser chatuser = new ChatUser();
    chatuser.setUserid("louisxiv");
    chatuser.setForegroundColor(-16777216);
    chatuser.setBackgroundColor(-1);
    chatuser.setNickname("anaconda");
    userlist.setMessage("äöüßéáúéíóú");
    userlist.setChatUser(chatuser);
    userlist.setRoom("room");
    userlist.setAttachment(33l);
    userlist.setFilename("name.txt");
    userlist.setFilesize(1024l);

    String result = userlist.toString();

    String bla = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CHATMESSAGE\","
        + "\"DATASET\":{"
        + "\"ROOM\":\"room\","
        + "\"MESSAGE\":\"äöüßéáúéíóú\","
        + "\"ATTACHMENT\":33,"
        + "\"FILENAME\":\"name.txt\","
        + "\"FILESIZE\":1024,"
        + "\"CHATUSER\":{"
        + "\"BACKGROUND_COLOR\":-1,"
        + "\"FOREGROUND_COLOR\":-16777216,"
        + "\"USERID\":\"louisxiv\","
        + "\"NICKNAME\":\"anaconda\"}}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeCHATMESSAGE_RESPONSE() throws EncodeException {
    CHATMESSAGE userlist = new CHATMESSAGE();
    userlist.setCommand(Command.CHATMESSAGE);
    userlist.setHeader(RESPONSE);
    userlist.setDataset(Protocol.DATASET);

    ChatUser chatuser = new ChatUser();
    chatuser.setUserid("louisxiv");
    chatuser.setForegroundColor(-16777216);
    chatuser.setBackgroundColor(-1);
    chatuser.setNickname("anaconda");
    userlist.setMessage("Hallo meine Freunde");
    userlist.setChatUser(chatuser);
    userlist.setRoom("room");
    userlist.setDatetime(666L);

    String result = encoder.encode(userlist);

    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CHATMESSAGE\","
        + "\"DATASET\":{"
        + "\"ROOM\":\"room\","
        + "\"DATETIME\":666,"
        + "\"MESSAGE\":\"Hallo meine Freunde\","
        + "\"CHATUSER\":{"
        + "\"BACKGROUND_COLOR\":-1,"
        + "\"FOREGROUND_COLOR\":-16777216,"
        + "\"USERID\":\"louisxiv\","
        + "\"NICKNAME\":\"anaconda\"}}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeCHATMESSAGE_RESPONSE_ATTACHMENT() throws EncodeException {
    CHATMESSAGE userlist = new CHATMESSAGE();
    userlist.setCommand(Command.CHATMESSAGE);
    userlist.setHeader(RESPONSE);
    userlist.setDataset(Protocol.DATASET);

    ChatUser chatuser = new ChatUser();
    chatuser.setUserid("louisxiv");
    chatuser.setForegroundColor(-16777216);
    chatuser.setBackgroundColor(-1);
    chatuser.setNickname("anaconda");
    userlist.setMessage("Hallo meine Freunde");
    userlist.setChatUser(chatuser);
    userlist.setRoom("room");
    userlist.setDatetime(666L);
    userlist.setAttachment(99l);
    userlist.setFilename("name.txt");
    userlist.setFilesize(Long.MAX_VALUE);

    String result = encoder.encode(userlist);

    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CHATMESSAGE\","
        + "\"DATASET\":{"
        + "\"ROOM\":\"room\","
        + "\"DATETIME\":666,"
        + "\"MESSAGE\":\"Hallo meine Freunde\","
        + "\"ATTACHMENT\":99,"
        + "\"FILENAME\":\"name.txt\","
        + "\"FILESIZE\":"
        + String.valueOf(Long.MAX_VALUE)
        + ","
        + "\"CHATUSER\":{"
        + "\"BACKGROUND_COLOR\":-1,"
        + "\"FOREGROUND_COLOR\":-16777216,"
        + "\"USERID\":\"louisxiv\","
        + "\"NICKNAME\":\"anaconda\"}}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeCHATMESSAGE_RESPONSE_JERSEY() throws EncodeException {
    CHATMESSAGE userlist = new CHATMESSAGE();
    userlist.setCommand(Command.CHATMESSAGE);
    userlist.setHeader(RESPONSE);
    userlist.setDataset(Protocol.DATASET);

    ChatUser chatuser = new ChatUser();
    chatuser.setUserid("louisxiv");
    chatuser.setForegroundColor(-16777216);
    chatuser.setBackgroundColor(-1);
    chatuser.setNickname("anaconda");
    userlist.setMessage("Hallo meine Freunde");
    userlist.setChatUser(chatuser);
    userlist.setRoom("room");
    userlist.setDatetime(666L);

    String result = userlist.toString();

    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CHATMESSAGE\","
        + "\"DATASET\":{"
        + "\"ROOM\":\"room\","
        + "\"DATETIME\":666,"
        + "\"MESSAGE\":\"Hallo meine Freunde\","
        + "\"CHATUSER\":{"
        + "\"BACKGROUND_COLOR\":-1,"
        + "\"FOREGROUND_COLOR\":-16777216,"
        + "\"USERID\":\"louisxiv\","
        + "\"NICKNAME\":\"anaconda\"}}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeCHATONLINELIST_REQUEST_0() throws EncodeException {
    CHATONLINELIST onlinelist = new CHATONLINELIST();
    onlinelist.setCommand(Command.CHATONLINELIST);
    onlinelist.setHeader(REQUEST);
    onlinelist.setDataset(Protocol.DATASET);
    ChatOnline[] value = new ChatOnline[0];
    onlinelist.setChatOnline(value);

    String result = encoder.encode(onlinelist);
    String bla = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CHATONLINELIST\","
        + "\"DATASET\":{"
        + "\"CHATONLINELIST\":["
        + "]}}";
    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeCHATONLINELIST_REQUEST_1() throws EncodeException {
    CHATONLINELIST onlinelist = new CHATONLINELIST();
    onlinelist.setCommand(Command.CHATONLINELIST);
    onlinelist.setHeader(REQUEST);
    onlinelist.setDataset(Protocol.DATASET);
    ChatOnline[] value = new ChatOnline[1];
    value[0] = new ChatOnline();
    value[0].setOnline(99);
    value[0].setRoom("everyone");
    onlinelist.setChatOnline(value);

    String result = encoder.encode(onlinelist);
    String bla = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CHATONLINELIST\","
        + "\"DATASET\":{"
        + "\"CHATONLINELIST\":["
        + "{\"ROOM\":\"everyone\",\"ONLINE\":99}"
        + "]}}";
    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeCHATONLINELIST_REQUEST_2() throws EncodeException {
    CHATONLINELIST onlinelist = new CHATONLINELIST();
    onlinelist.setCommand(Command.CHATONLINELIST);
    onlinelist.setHeader(REQUEST);
    onlinelist.setDataset(Protocol.DATASET);
    ChatOnline[] value = new ChatOnline[2];
    value[0] = new ChatOnline();
    value[0].setOnline(99);
    value[0].setRoom("everyone");
    value[1] = new ChatOnline();
    value[1].setOnline(7);
    value[1].setRoom("ingress");
    onlinelist.setChatOnline(value);

    String result = encoder.encode(onlinelist);
    String bla = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CHATONLINELIST\","
        + "\"DATASET\":{"
        + "\"CHATONLINELIST\":["
        + "{\"ROOM\":\"everyone\",\"ONLINE\":99},"
        + "{\"ROOM\":\"ingress\",\"ONLINE\":7}"
        + "]}}";
    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeCHATUSERLIST_ERROR() throws EncodeException {
    CHATUSERLIST userlist = new CHATUSERLIST();
    userlist.setCommand(Command.CHATUSERLIST);
    userlist.setHeader(ERROR);
    userlist.setDataset(Protocol.DATASET);
    userlist.setErrorCode(-1);
    userlist.setErrorMessage("error");

    String result = encoder.encode(userlist);

    String bla = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"CHATUSERLIST\","
        + "\"DATASET\":{\"ERRORCODE\":-1,\"ERRORMESSAGE\":\"error\"}}";
    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeCHATUSERLIST_ERROR_ohne_DATASET() throws EncodeException {
    CHATUSERLIST userlist = new CHATUSERLIST();
    userlist.setCommand(Command.CHATUSERLIST);
    userlist.setHeader(ERROR);

    String result = encoder.encode(userlist);

    String bla = "{\"HEADER\":\"ERROR\",\"COMMAND\":\"CHATUSERLIST\"}";
    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeCHATUSERLIST_REQUEST_Gutfall() throws EncodeException, IOException {
    CHATUSERLIST userlist = new CHATUSERLIST();
    userlist.setCommand(Command.CHATUSERLIST);
    userlist.setHeader(REQUEST);
    userlist.setDataset(Protocol.DATASET);
    userlist.setRoom("louisxiv");

    String result = encoder.encodeCHATUSERLIST(userlist);

    String bla = "{\"HEADER\":\"REQUEST\","

        + "\"COMMAND\":\"CHATUSERLIST\","
        + "\"DATASET\":"
        + "{\"ROOM\":\"louisxiv\"}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeCHATUSERLIST_RESPONSE() throws EncodeException, IOException {
    CHATUSERLIST userlist = new CHATUSERLIST();
    userlist.setCommand(Command.CHATUSERLIST);
    userlist.setHeader(RESPONSE);
    userlist.setDataset(Protocol.DATASET);

    ChatUser[] chatuser = new ChatUser[1];
    chatuser[0] = new ChatUser();
    chatuser[0].setUserid("louisxiv");
    chatuser[0].setForegroundColor(-16777216);
    chatuser[0].setBackgroundColor(-1);
    chatuser[0].setNickname("anaconda");

    userlist.setChatUser(chatuser);
    userlist.setRoom("room");

    String result = encoder.encodeCHATUSERLIST(userlist);

    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CHATUSERLIST\","
        + "\"DATASET\":{"
        + "\"ROOM\":\"room\","
        + "\"CHATUSERLIST\":["
        + "{\"USERID\":\"louisxiv\","
        + "\"FOREGROUND_COLOR\":-16777216,"
        + "\"BACKGROUND_COLOR\":-1,"
        + "\"NICKNAME\":\"anaconda\"}]}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeCHATUSERLIST_RESPONSE_ohneUserID() throws EncodeException, IOException {
    CHATUSERLIST userlist = new CHATUSERLIST();
    userlist.setCommand(Command.CHATUSERLIST);
    userlist.setHeader(RESPONSE);
    userlist.setDataset(Protocol.DATASET);

    ChatUser[] chatuser = new ChatUser[1];
    chatuser[0] = new ChatUser();
    chatuser[0].setForegroundColor(-16777216);
    chatuser[0].setBackgroundColor(-1);
    chatuser[0].setNickname("anaconda");

    userlist.setChatUser(chatuser);
    userlist.setRoom("room");

    String result = encoder.encodeCHATUSERLIST(userlist);

    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CHATUSERLIST\","
        + "\"DATASET\":{"
        + "\"ROOM\":\"room\","
        + "\"CHATUSERLIST\":["
        + "{"
        + "\"FOREGROUND_COLOR\":-16777216,"
        + "\"BACKGROUND_COLOR\":-1,"
        + "\"NICKNAME\":\"anaconda\"}]}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeCONFERENCE_REQUEST() throws IOException, EncodeException {
    CONFERENCE conference = new CONFERENCE();
    conference.setCommand(Command.CONFERENCE);
    conference.setHeader(HEADER.REQUEST);
    conference.setDataset(Protocol.DATASET);
    conference.setKonferenzname("Dienstag Treff");
    conference.setOrganisatorUid("anaconda");
    conference.setUserid("12345678");
    conference.setSession("3a");
    conference.setOffhook(true);
    String result = encoder.encode(conference);

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CONFERENCE\","
        + "\"DATASET\":{"
        + "\"KONFERENZNAME\":\"Dienstag Treff\","
        + "\"ORGANISATOR_UID\":\"anaconda\","
        + "\"USERID\":\"12345678\","
        + "\"SESSION\":\"3a\","
        + "\"OFFHOOK\":true"
        + "}}";

    assertEquals(encoded, result, "JSON");

  }



  @Test
  @DisplayName(value = "CONFERENCE mit Konferenzname NULL")
  public void encodeCONFERENCE_REQUEST_ERROR() {
    CONFERENCE conferencecall = new CONFERENCE();
    conferencecall.setCommand(Command.CONFERENCE);
    conferencecall.setHeader(HEADER.REQUEST);
    conferencecall.setDataset(Protocol.DATASET);
    conferencecall.setKonferenzname(null);
    conferencecall.setOrganisatorUid("anaconda");
    try {
      String result = encoder.encode(conferencecall);
      fail("Exception nicht geworfen");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }

  }



  @Test
  public void encodeCONFERENCE_REQUEST_ohne_SESSION() {
    CONFERENCE conference = new CONFERENCE();
    conference.setCommand(Command.CONFERENCE);
    conference.setHeader(HEADER.REQUEST);
    conference.setDataset(Protocol.DATASET);
    conference.setKonferenzname("Dienstag Treff");
    conference.setOrganisatorUid("anaconda");
    conference.setUserid("12345678");
    conference.setOffhook(true);
    try {
      encoder.encode(conference);
      fail("EncodeException nicht geworfen, SESSION fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void encodeCONFERENCE_REQUEST_ohne_USERID() {
    CONFERENCE conference = new CONFERENCE();
    conference.setCommand(Command.CONFERENCE);
    conference.setHeader(HEADER.REQUEST);
    conference.setDataset(Protocol.DATASET);
    conference.setKonferenzname("Dienstag Treff");
    conference.setOrganisatorUid("anaconda");
    conference.setSession("3a");
    conference.setOffhook(true);
    try {
      encoder.encode(conference);
      fail("EncodeException nicht geworfen, USERID fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }

  }



  @Test
  public void encodeCONFERENCE_RESPONSE() throws IOException, EncodeException {
    CONFERENCE conferencecall = new CONFERENCE();
    conferencecall.setCommand(Command.CONFERENCE);
    conferencecall.setHeader(HEADER.RESPONSE);
    conferencecall.setDataset(Protocol.DATASET);
    conferencecall.setKonferenzname("Dienstag Treff");
    conferencecall.setOrganisatorUid("anaconda");
    conferencecall.setUserid("admin678");
    conferencecall.setOffhook(true);

    KonferenzraumUser[] konferenzraumuser = new KonferenzraumUser[4];
    konferenzraumuser[0] = new KonferenzraumUser();
    konferenzraumuser[0].setBackgroundColor(238);
    konferenzraumuser[0].setForegroundColor(-2839);
    konferenzraumuser[0].setNickname("admin678");
    konferenzraumuser[0].setAnwesend(true);
    konferenzraumuser[0].setUserid("userid1");
    konferenzraumuser[0].setSession("ab");
    konferenzraumuser[0].setMute(true);

    konferenzraumuser[1] = new KonferenzraumUser();
    konferenzraumuser[1].setBackgroundColor(-293);
    konferenzraumuser[1].setForegroundColor(-9283);
    konferenzraumuser[1].setNickname("freigeist");
    konferenzraumuser[1].setAnwesend(true);
    konferenzraumuser[1].setUserid("userid2");
    konferenzraumuser[1].setSession("cd");
    konferenzraumuser[1].setMute(true);

    konferenzraumuser[2] = new KonferenzraumUser();
    konferenzraumuser[2].setBackgroundColor(-9283);
    konferenzraumuser[2].setForegroundColor(-7263);
    konferenzraumuser[2].setNickname("Jester");
    konferenzraumuser[2].setAnwesend(false);
    konferenzraumuser[2].setUserid("userid3");
    konferenzraumuser[2].setMute(false);

    konferenzraumuser[3] = new KonferenzraumUser();
    konferenzraumuser[3].setBackgroundColor(-23);
    konferenzraumuser[3].setForegroundColor(-8273);
    konferenzraumuser[3].setNickname("picasso");
    konferenzraumuser[3].setAnwesend(false);
    konferenzraumuser[3].setUserid("userid4");
    konferenzraumuser[3].setMute(false);

    conferencecall.setKonferenzraumUser(konferenzraumuser);

    String result = encoder.encode(conferencecall);

//    Gson gson = new Gson();
//    String result = gson.toJson(conferencecall);

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CONFERENCE\","
        + "\"DATASET\":{"
        + "\"KONFERENZNAME\":\"Dienstag Treff\","
        + "\"ORGANISATOR_UID\":\"anaconda\","
        + "\"USERID\":\"admin678\","
        + "\"OFFHOOK\":true,"
        + "\"KONFERENZRAUMUSER\":[{\"ANWESEND\":true,\"SESSION\":\"ab\",\"MUTE\":true,\"CHATUSER\":{\"USERID\":\"userid1\",\"NICKNAME\":\"admin678\",\"FOREGROUND_COLOR\":-2839,\"BACKGROUND_COLOR\":238}},"
        + "{\"ANWESEND\":true,\"SESSION\":\"cd\",\"MUTE\":true,\"CHATUSER\":{\"USERID\":\"userid2\",\"NICKNAME\":\"freigeist\",\"FOREGROUND_COLOR\":-9283,\"BACKGROUND_COLOR\":-293}},"
        + "{\"ANWESEND\":false,\"MUTE\":false,\"CHATUSER\":{\"USERID\":\"userid3\",\"NICKNAME\":\"Jester\",\"FOREGROUND_COLOR\":-7263,\"BACKGROUND_COLOR\":-9283}},"
        + "{\"ANWESEND\":false,\"MUTE\":false,\"CHATUSER\":{\"USERID\":\"userid4\",\"NICKNAME\":\"picasso\",\"FOREGROUND_COLOR\":-8273,\"BACKGROUND_COLOR\":-23}}]"

        + "}}";

    assertEquals(encoded, result, "JSON");

  }



  @Test
  public void encodeCONFERENCE_RESPONSE_OHNE_KONFERENZNAME() {
    CONFERENCE conferencecall = new CONFERENCE();
    conferencecall.setCommand(Command.CONFERENCE);
    conferencecall.setHeader(HEADER.RESPONSE);
    conferencecall.setDataset(Protocol.DATASET);
    conferencecall.setOrganisatorUid("anaconda");
    ArrayList<KonferenzraumUser> list = new ArrayList<>();
    conferencecall.setKonferenzraumUser(list.toArray(new KonferenzraumUser[list.size()]));

    try {
      encoder.encode(conferencecall);
      fail("EncodeException nicht geworfen, KONFERENZNAME fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void encodeCONFERENCE_RESPONSE_OHNE_OFFHOOK() {
    CONFERENCE conferencecall = new CONFERENCE();
    conferencecall.setCommand(Command.CONFERENCE);
    conferencecall.setHeader(HEADER.RESPONSE);
    conferencecall.setDataset(Protocol.DATASET);
    conferencecall.setKonferenzname("Dienstag Treff");
    conferencecall.setOrganisatorUid("anaconda");
    conferencecall.setUserid("admin678");
    ArrayList<KonferenzraumUser> list = new ArrayList<>();
    conferencecall.setKonferenzraumUser(list.toArray(new KonferenzraumUser[list.size()]));

    try {
      encoder.encode(conferencecall);
      fail("EncodeException nicht geworfen, OFFHOOK fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void encodeCONFERENCE_RESPONSE_OHNE_ORGANISATOR() {
    CONFERENCE conferencecall = new CONFERENCE();
    conferencecall.setCommand(Command.CONFERENCE);
    conferencecall.setHeader(HEADER.RESPONSE);
    conferencecall.setDataset(Protocol.DATASET);
    conferencecall.setKonferenzname("Dienstag Treff");
    ArrayList<KonferenzraumUser> list = new ArrayList<>();
    conferencecall.setKonferenzraumUser(list.toArray(new KonferenzraumUser[list.size()]));
    try {
      encoder.encode(conferencecall);
      fail("EncodeException fehlt, ORGANISATOR fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void encodeCONFERENCE_RESPONSE_OHNE_USER() {
    CONFERENCE conferencecall = new CONFERENCE();
    conferencecall.setCommand(Command.CONFERENCE);
    conferencecall.setHeader(HEADER.RESPONSE);
    conferencecall.setDataset(Protocol.DATASET);
    conferencecall.setKonferenzname("Dienstag Treff");
    conferencecall.setOrganisatorUid("anaconda");
    ArrayList<KonferenzraumUser> list = new ArrayList<>();
    conferencecall.setKonferenzraumUser(list.toArray(new KonferenzraumUser[list.size()]));

    try {
      encoder.encode(conferencecall);
      fail("EncodeException nicht geworfen, USERID fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void encodeCONFERENCEAUDIO_REQUEST() throws EncodeException {
    CONFERENCEAUDIO conferenceaudio = new CONFERENCEAUDIO();
    conferenceaudio.setCommand(Command.CONFERENCEAUDIO);
    conferenceaudio.setHeader(HEADER.REQUEST);
    conferenceaudio.setDataset(Protocol.DATASET);
    conferenceaudio.setContent("ich bin ein base64 textstring".getBytes());
    String streamid = UUID.randomUUID().toString();
    conferenceaudio.setStreamId(streamid);
    String[] sessionids = new String[2];
    sessionids[0] = "ab";
    sessionids[1] = "a1";
    conferenceaudio.setReceiverSessions(sessionids);
    String result = encoder.encode(conferenceaudio);

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CONFERENCEAUDIO\","
        + "\"DATASET\":{"
        + "\"TARGET\":[\"ab\",\"a1\"],"
        + "\"CONTENT\":\"aWNoIGJpbiBlaW4gYmFzZTY0IHRleHRzdHJpbmc=\","
        + "\"STREAM_ID\":".concat("\"").concat(streamid).concat("\"")
        + "}}";

    assertEquals(encoded, result, "JSON");

  }



  @Test
  @DisplayName(value = "CONFERENCEAUDIO | REQUEST | CONTENT | FEHLT")
  public void encodeCONFERENCEAUDIO_REQUEST_CONTENT_FEHLT() throws EncodeException {
    CONFERENCEAUDIO conferenceaudio = new CONFERENCEAUDIO();
    conferenceaudio.setCommand(Command.CONFERENCEAUDIO);
    conferenceaudio.setHeader(HEADER.REQUEST);
    conferenceaudio.setDataset(Protocol.DATASET);
    String streamid = UUID.randomUUID().toString();
    conferenceaudio.setStreamId(streamid);
    String[] sessionids = new String[2];
    sessionids[0] = "ab";
    sessionids[1] = "a1";
    conferenceaudio.setReceiverSessions(sessionids);
    String result = encoder.encode(conferenceaudio);

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CONFERENCEAUDIO\","
        + "\"DATASET\":{"
        + "\"TARGET\":[\"ab\",\"a1\"],"
        + "\"STREAM_ID\":".concat("\"").concat(streamid).concat("\"")
        + "}}";

    assertEquals(encoded, result, "JSON");

  }



  @Test
  public void encodeCONFERENCEAUDIO_REQUEST_EMPTY_TARGET() throws EncodeException {
    CONFERENCEAUDIO conferenceaudio = new CONFERENCEAUDIO();
    conferenceaudio.setCommand(Command.CONFERENCEAUDIO);
    conferenceaudio.setHeader(HEADER.REQUEST);
    conferenceaudio.setDataset(Protocol.DATASET);
    conferenceaudio.setContent("ich bin ein base64 textstring".getBytes());
    String[] sessionids = new String[0];
    conferenceaudio.setReceiverSessions(sessionids);
    String result = encoder.encode(conferenceaudio);

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CONFERENCEAUDIO\","
        + "\"DATASET\":{"
        + "\"TARGET\":[],"
        + "\"CONTENT\":\"aWNoIGJpbiBlaW4gYmFzZTY0IHRleHRzdHJpbmc=\"}}";

    assertEquals(encoded, result, "JSON");

  }



  @Test
  public void encodeCONFERENCEAUDIO_REQUEST_MIT_ORIGIN() throws EncodeException {
    CONFERENCEAUDIO conferenceaudio = new CONFERENCEAUDIO();
    conferenceaudio.setCommand(Command.CONFERENCEAUDIO);
    conferenceaudio.setHeader(HEADER.REQUEST);
    conferenceaudio.setDataset(Protocol.DATASET);
    conferenceaudio.setContent("ich bin ein base64 textstring".getBytes());
    String[] sessionids = new String[2];
    sessionids[0] = "ab";
    sessionids[1] = "a1";
    conferenceaudio.setReceiverSessions(sessionids);
    String result = encoder.encode(conferenceaudio);

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CONFERENCEAUDIO\","
        + "\"DATASET\":{"
        + "\"TARGET\":[\"ab\",\"a1\"],"
        + "\"CONTENT\":\"aWNoIGJpbiBlaW4gYmFzZTY0IHRleHRzdHJpbmc=\"}}";

    assertEquals(encoded, result, "JSON");

  }



  @Test
  public void encodeCONFERENCEAUDIO_REQUEST_mit_USERID() throws EncodeException {
    CONFERENCEAUDIO conferenceaudio = new CONFERENCEAUDIO();
    conferenceaudio.setCommand(Command.CONFERENCEAUDIO);
    conferenceaudio.setHeader(HEADER.REQUEST);
    conferenceaudio.setDataset(Protocol.DATASET);
    conferenceaudio.setUserid("admin678");
    conferenceaudio.setContent("ich bin ein base64 textstring".getBytes());
    String[] sessionids = new String[2];
    sessionids[0] = "ab";
    sessionids[1] = "a1";
    conferenceaudio.setReceiverSessions(sessionids);
    String result = encoder.encode(conferenceaudio);

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CONFERENCEAUDIO\","
        + "\"DATASET\":{"
        + "\"USERID\":\"admin678\","
        + "\"TARGET\":[\"ab\",\"a1\"],"
        + "\"CONTENT\":\"aWNoIGJpbiBlaW4gYmFzZTY0IHRleHRzdHJpbmc=\"}}";

    assertEquals(encoded, result, "JSON");

  }



  @Test
  public void encodeCONFERENCEAUDIO_REQUEST_NULL_TARGET() {
    CONFERENCEAUDIO conferenceaudio = new CONFERENCEAUDIO();
    conferenceaudio.setCommand(Command.CONFERENCEAUDIO);
    conferenceaudio.setHeader(HEADER.REQUEST);
    conferenceaudio.setDataset(Protocol.DATASET);
    conferenceaudio.setContent("ich bin ein base64 textstring".getBytes());
    try {
      encoder.encode(conferenceaudio);
      fail("Exception nicht geworfen");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }

  }



  @Test
  public void encodeCONFERENCEMUTE_REQUEST() throws EncodeException {
    CONFERENCEMUTE conferencemute = new CONFERENCEMUTE();
    conferencemute.setCommand(Command.CONFERENCEMUTE);
    conferencemute.setHeader(REQUEST);
    conferencemute.setDataset(Protocol.DATASET);
    conferencemute.setKonferenzraum("Dienstag Treff");
    conferencemute.setOrganisator("admin678");
    conferencemute.setUserid("max45678");
    conferencemute.setReceiverSessions(new String[] {"ab", "cd"});
    conferencemute.setMute(true);

    String result = encoder.encode(conferencemute);

    String encoded = "{\"HEADER\":\"REQUEST\",\"COMMAND\":\"CONFERENCEMUTE\"," + "\"DATASET\":{"
        + "\"KONFERENZRAUM\":\"Dienstag Treff\","
        + "\"ORGANISATOR\":\"admin678\","
        + "\"USERID\":\"max45678\","
        + "\"RECEIVER_SESSIONS\":[\"ab\",\"cd\"],"
        + "\"MUTE\":true"
        + "}}";

    assertEquals(result, encoded);

  }



  @Test
  public void encodeCONFERENCEMUTE_REQUEST_ohne_KONFERENZRAUM() {
    CONFERENCEMUTE conferencemute = new CONFERENCEMUTE();
    conferencemute.setCommand(Command.CONFERENCEMUTE);
    conferencemute.setHeader(REQUEST);
    conferencemute.setDataset(Protocol.DATASET);
    conferencemute.setOrganisator("admin678");
    conferencemute.setUserid("max45678");
    conferencemute.setReceiverSessions(new String[] {"ab", "cd"});
    conferencemute.setMute(false);

    try {
      encoder.encode(conferencemute);
      fail("EncodeException nicht geworfen, KONFERENZRAUM fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void encodeCONFERENCEMUTE_REQUEST_ohne_MUTE() {
    CONFERENCEMUTE conferencemute = new CONFERENCEMUTE();
    conferencemute.setCommand(Command.CONFERENCEMUTE);
    conferencemute.setHeader(REQUEST);
    conferencemute.setDataset(Protocol.DATASET);
    conferencemute.setKonferenzraum("Dienstag Treff");
    conferencemute.setOrganisator("admin678");
    conferencemute.setUserid("max45678");
    conferencemute.setReceiverSessions(new String[] {"ab", "cd"});

    try {
      encoder.encode(conferencemute);
      fail("EncodeException nicht geworfen, MUTE fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void encodeCONFERENCEMUTE_REQUEST_ohne_RECEIVER_SESSIONS() {
    CONFERENCEMUTE conferencemute = new CONFERENCEMUTE();
    conferencemute.setCommand(Command.CONFERENCEMUTE);
    conferencemute.setHeader(REQUEST);
    conferencemute.setDataset(Protocol.DATASET);
    conferencemute.setKonferenzraum("Dienstag Treff");
    conferencemute.setOrganisator("admin678");
    conferencemute.setUserid("max45678");
    conferencemute.setMute(true);

    try {
      encoder.encode(conferencemute);
      fail("EncodeException nicht geworfen, RECEIVER_SESSIONS fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }

  }



  @Test
  @DisplayName(value = "CONFERENCEVIDEO REQUEST")
  public void encodeCONFERENCEVIDEO() throws DecodeException, EncodeException {

    CONFERENCEVIDEO conferenceVideo = new CONFERENCEVIDEO();
    conferenceVideo.setCommand(Command.CONFERENCEVIDEO);
    conferenceVideo.setHeader(HEADER.REQUEST);
    conferenceVideo.setDataset(Protocol.DATASET);
    conferenceVideo.setOrganisator("max45678");
    conferenceVideo.setKonferenzraum("Dienstag Treff");
    String[] receiver = new String[] {"ab", "a1"};
    conferenceVideo.setReceiverSessions(receiver);
    conferenceVideo.setShared(true);
    conferenceVideo.setUserid("anaconda");
    conferenceVideo.setNickname("erdbeere");

    String encoded = "{\"HEADER\":\"REQUEST\",\"COMMAND\":\"CONFERENCEVIDEO\",\"DATASET\":{"
        + "\"USERID\":\"anaconda\","
        + "\"NICKNAME\":\"erdbeere\","
        + "\"ORGANISATOR\":\"max45678\","
        + "\"RECEIVER_SESSIONS\":[\"ab\",\"a1\"],"
        + "\"SHARING\":true,"
        + "\"KONFERENZRAUM\":\"Dienstag Treff\""
        + "}}";

    String result = encoder.encode(conferenceVideo);
    assertEquals(result, encoded);

  }



  @Test
  @DisplayName(value = "CONFERENCEVIDEO ERROR")
  public void encodeCONFERENCEVIDEO_ERROR_MULTILINGUALKEY() throws EncodeException {

    CONFERENCEVIDEO conferenceVideo = new CONFERENCEVIDEO();
    conferenceVideo.setCommand(Command.CONFERENCEVIDEO);
    conferenceVideo.setHeader(HEADER.ERROR);
    conferenceVideo.setDataset(Protocol.DATASET);
    conferenceVideo.setMultilingualkey(KEY.SERVER_DRITTE_PERSON);

    String encoded = "{\"HEADER\":\"ERROR\",\"COMMAND\":\"CONFERENCEVIDEO\"," + "\"DATASET\":{"
        + "\"MULTILINGUALKEY\":\"Server_Dritte_Person\""
        + "}}";

    String result = encoder.encode(conferenceVideo);
    assertEquals(result, encoded);

  }



  @Test
  @DisplayName(value = "CONFERENCEVIDEO ERROR mit SHARING")
  public void encodeCONFERENCEVIDEO_ERROR_SHARING() throws EncodeException {

    CONFERENCEVIDEO conferenceVideo = new CONFERENCEVIDEO();
    conferenceVideo.setCommand(Command.CONFERENCEVIDEO);
    conferenceVideo.setHeader(HEADER.ERROR);
    conferenceVideo.setDataset(Protocol.DATASET);
    conferenceVideo.setMultilingualkey(KEY.SERVER_DRITTE_PERSON);
    conferenceVideo.setShared(false);

    String encoded = "{\"HEADER\":\"ERROR\",\"COMMAND\":\"CONFERENCEVIDEO\"," + "\"DATASET\":{"
        + "\"MULTILINGUALKEY\":\"Server_Dritte_Person\","
        + "\"SHARING\":false"
        + "}}";

    String result = encoder.encode(conferenceVideo);
    assertEquals(result, encoded);

  }



  @Test
  @DisplayName(value = "CONFERENCEVIDEO ohne KONFERNEZRAUM")
  public void encodeCONFERENCEVIDEO_ohne_Konfernzraum() {

    CONFERENCEVIDEO conferenceVideo = new CONFERENCEVIDEO();
    conferenceVideo.setCommand(Command.CONFERENCEVIDEO);
    conferenceVideo.setHeader(HEADER.REQUEST);
    conferenceVideo.setDataset(Protocol.DATASET);
    conferenceVideo.setOrganisator("max45678");
    String[] receiver = new String[] {"ab", "a1"};
    conferenceVideo.setReceiverSessions(receiver);
    conferenceVideo.setShared(true);

    try {
      encoder.encode(conferenceVideo);
      fail("EncodeException nicht geworfen, KONFERNEZRAUM fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }

  }



  @Test
  public void encodeCONFERENCEVIDEO_ohne_NICKNAME() {

    CONFERENCEVIDEO conferenceVideo = new CONFERENCEVIDEO();
    conferenceVideo.setCommand(Command.CONFERENCEVIDEO);
    conferenceVideo.setHeader(HEADER.REQUEST);
    conferenceVideo.setDataset(Protocol.DATASET);
    conferenceVideo.setOrganisator("max45678");
    conferenceVideo.setKonferenzraum("Dienstag Treff");
    String[] receiver = new String[] {"ab", "a1"};
    conferenceVideo.setReceiverSessions(receiver);
    conferenceVideo.setShared(true);
    conferenceVideo.setUserid("anaconda");

    try {
      encoder.encode(conferenceVideo);
      fail("Exception nicht geworfen, NICKNAME fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }

  }



  @Test
  @DisplayName(value = "CONFERENCEVIDEO ohne ORGANISATOR")
  public void encodeCONFERENCEVIDEO_ohne_ORGANISATOR() {

    CONFERENCEVIDEO conferenceVideo = new CONFERENCEVIDEO();
    conferenceVideo.setCommand(Command.CONFERENCEVIDEO);
    conferenceVideo.setHeader(HEADER.REQUEST);
    conferenceVideo.setDataset(Protocol.DATASET);
    conferenceVideo.setShared(true);
    String[] receiver = new String[] {"ab", "a1"};
    conferenceVideo.setReceiverSessions(receiver);

    try {
      encoder.encode(conferenceVideo);
      fail("EncodeException nicht geworfen, ORGANISATOR fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }

  }



  @Test
  @DisplayName(value = "CONFERENCEVIDEO ohne RECEIVER_SESSIONS")
  public void encodeCONFERENCEVIDEO_ohne_Receiver() {

    CONFERENCEVIDEO conferenceVideo = new CONFERENCEVIDEO();
    conferenceVideo.setCommand(Command.CONFERENCEVIDEO);
    conferenceVideo.setHeader(HEADER.REQUEST);
    conferenceVideo.setDataset(Protocol.DATASET);
    conferenceVideo.setOrganisator("max45678");
    conferenceVideo.setShared(true);

    try {
      encoder.encode(conferenceVideo);
      fail("EncodeException nicht geworfen, RECEIVER_SESSIONS fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }

  }



  @Test
  @DisplayName(value = "CONFERENCEVIDEO ohne SHARING")
  public void encodeCONFERENCEVIDEO_ohne_SHARING() {
    CONFERENCEVIDEO conferenceVideo = new CONFERENCEVIDEO();
    conferenceVideo.setCommand(Command.CONFERENCEVIDEO);
    conferenceVideo.setHeader(HEADER.REQUEST);
    conferenceVideo.setDataset(Protocol.DATASET);
    conferenceVideo.setOrganisator("max45678");
    String[] receiver = new String[] {"ab", "a1"};
    conferenceVideo.setReceiverSessions(receiver);

    try {
      encoder.encode(conferenceVideo);
      fail("EncodeException nicht geworfen, connect fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }

  }



  @Test
  @DisplayName(value = "CONFERENCEVIDEO ohne USERID")
  public void encodeCONFERENCEVIDEO_ohne_USERID() {

    CONFERENCEVIDEO conferenceVideo = new CONFERENCEVIDEO();
    conferenceVideo.setCommand(Command.CONFERENCEVIDEO);
    conferenceVideo.setHeader(HEADER.REQUEST);
    conferenceVideo.setDataset(Protocol.DATASET);
    conferenceVideo.setOrganisator("max45678");
    conferenceVideo.setKonferenzraum("Dienstag Treff");
    String[] receiver = new String[] {"ab", "a1"};
    conferenceVideo.setReceiverSessions(receiver);
    conferenceVideo.setShared(true);

    try {
      encoder.encode(conferenceVideo);
      fail("EncodeException nicht geworfen, USERID fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }

  }



  @Test
  public void encodeCREATETEMPROOM_ERROR() throws EncodeException {
    CREATETEMPROOM temproom = new CREATETEMPROOM();
    temproom.setCommand(Command.CREATETEMPROOM);
    temproom.setHeader(ERROR);
    temproom.setDataset(Protocol.DATASET);
    temproom.setUser("louisxiv");
    temproom.setRoom("room");
    temproom.setRoomtype(Roomtype.FORUM);
    temproom.setErrorMessage("duplicate key entry");
    temproom.setErrorCode(1062);
    String result = encoder.encode(temproom);

    String bla = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"CREATETEMPROOM\","
        + "\"DATASET\":{"
        + "\"USER\":\"louisxiv\","
        + "\"ROOM\":\"room\","
        + "\"ROOMTYPE\":\"FORUM\","
        + "\"ERRORCODE\":1062,"
        + "\"ERRORMESSAGE\":\"duplicate key entry\"}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeCREATETEMPROOM_ERROR_MULTILINGUALKEY() throws EncodeException {
    CREATETEMPROOM temproom = new CREATETEMPROOM();
    temproom.setCommand(Command.CREATETEMPROOM);
    temproom.setHeader(ERROR);
    temproom.setDataset(Protocol.DATASET);
    temproom.setRoom("room");
    temproom.setUser("llange");
    temproom.setEntry(Entry.PUBLIC);
    temproom.setLifetime(Lifetime.TEMPORARY);
    temproom.setRoomtype(Roomtype.PAUSENRAUM);
    temproom.setMultilingualkey(KEY.SERVER_ZU_KURZER_RAUMNAME);

    String result = encoder.encode(temproom);
    String bla = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"CREATETEMPROOM\","
        + "\"DATASET\":{"
        + "\"USER\":\"llange\","
        + "\"ROOM\":\"room\","
        + "\"ROOMTYPE\":\"PAUSENRAUM\","
        + "\"ENTRY\":\"PUBLIC\","
        + "\"LIFETIME\":\"TEMPORARY\","
        + "\"MULTILINGUALKEY\":\"Server_Zu_kurzer_Raumname\""
        + "}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeCREATETEMPROOM_ERROR_ohne_ROOMTYPE() throws EncodeException {
    CREATETEMPROOM temproom = new CREATETEMPROOM();
    temproom.setCommand(Command.CREATETEMPROOM);
    temproom.setHeader(ERROR);
    temproom.setDataset(Protocol.DATASET);
    temproom.setUser("louisxiv");
    temproom.setRoom("room");
    temproom.setErrorMessage("duplicate key entry");
    temproom.setErrorCode(1062);
    String result = encoder.encode(temproom);

    String bla = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"CREATETEMPROOM\","
        + "\"DATASET\":{"
        + "\"USER\":\"louisxiv\","
        + "\"ROOM\":\"room\","
        + "\"ERRORCODE\":1062,"
        + "\"ERRORMESSAGE\":\"duplicate key entry\"}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeCREATETEMPROOM_ERROR_PROTECTED() throws EncodeException {
    CREATETEMPROOM temproom = new CREATETEMPROOM();
    temproom.setCommand(Command.CREATETEMPROOM);
    temproom.setHeader(ERROR);
    temproom.setDataset(Protocol.DATASET);
    temproom.setUser("louisxiv");
    temproom.setRoom("room");
    temproom.setRoomtype(Roomtype.BESPRECHUNGSRAUM);
    temproom.setErrorMessage("The password is missing.");
    temproom.setEntry(Entry.PROTECTED);
    temproom.setLifetime(Lifetime.TEMPORARY);
    String result = encoder.encode(temproom);

    String bla = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"CREATETEMPROOM\","
        + "\"DATASET\":{"
        + "\"USER\":\"louisxiv\","
        + "\"ROOM\":\"room\","
        + "\"ROOMTYPE\":\"BESPRECHUNGSRAUM\","
        + "\"ENTRY\":\"PROTECTED\","
        + "\"LIFETIME\":\"TEMPORARY\","
        + "\"ERRORMESSAGE\":\"The password is missing.\"}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeCREATETEMPROOM_ERROR_TEXT() throws EncodeException {
    CREATETEMPROOM temproom = new CREATETEMPROOM();
    temproom.setCommand(Command.CREATETEMPROOM);
    temproom.setHeader(ERROR);
    temproom.setDataset(Protocol.DATASET);
    temproom.setUser("louisxiv");
    temproom.setRoom("room");
    temproom.setErrorMessage("duplicate key entry");
    temproom.setErrorCode(1062);
    temproom.setText("fehler");
    String result = encoder.encode(temproom);

    String bla = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"CREATETEMPROOM\","
        + "\"DATASET\":{"
        + "\"USER\":\"louisxiv\","
        + "\"ROOM\":\"room\","
        + "\"ERRORCODE\":1062,"
        + "\"ERRORMESSAGE\":\"duplicate key entry\","
        + "\"TEXT\":\"fehler\""
        + "}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeCREATETEMPROOM_REQUEST() throws EncodeException {
    CREATETEMPROOM temproom = new CREATETEMPROOM();
    temproom.setCommand(Command.CREATETEMPROOM);
    temproom.setHeader(REQUEST);
    temproom.setDataset(Protocol.DATASET);
    temproom.setRoom("room");
    temproom.setUser("llange");
    temproom.setEntry(Entry.PUBLIC);
    temproom.setLifetime(Lifetime.TEMPORARY);
    temproom.setNickname("viper");
    temproom.setRoomtype(Roomtype.PAUSENRAUM);

    String result = encoder.encode(temproom);
    String bla = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CREATETEMPROOM\","
        + "\"DATASET\":{"
        + "\"USER\":\"llange\","
        + "\"NICKNAME\":\"viper\","
        + "\"ROOM\":\"room\","
        + "\"ROOMTYPE\":\"PAUSENRAUM\","
        + "\"ENTRY\":\"PUBLIC\","
        + "\"LIFETIME\":\"TEMPORARY\""
        + "}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeCREATETEMPROOM_REQUEST_EncodeException() {
    CREATETEMPROOM temproom = new CREATETEMPROOM();
    temproom.setCommand(Command.CREATETEMPROOM);
    temproom.setHeader(REQUEST);
    temproom.setDataset(Protocol.DATASET);
    temproom.setRoom("room");
    temproom.setUser("llange");
    temproom.setEntry(Entry.PUBLIC);
    temproom.setLifetime(Lifetime.TEMPORARY);
    temproom.setRoomtype(Roomtype.PAUSENRAUM);

    String result;
    try {
      result = encoder.encode(temproom);
      fail("EncodeException nicht geworfen, NICKNAME");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void encodeCREATETEMPROOM_REQUEST_PROTECTED_temporary() throws EncodeException {
    CREATETEMPROOM temproom = new CREATETEMPROOM();
    temproom.setCommand(Command.CREATETEMPROOM);
    temproom.setHeader(REQUEST);
    temproom.setDataset(Protocol.DATASET);
    temproom.setRoom("room");
    temproom.setUser("llange");
    temproom.setEntry(Entry.PROTECTED);
    temproom.setLifetime(Lifetime.TEMPORARY);
    temproom.setPassword("geheim");
    temproom.setNickname("viper");
    temproom.setRoomtype(Roomtype.PAUSENRAUM);

    String result = encoder.encode(temproom);
    String bla = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CREATETEMPROOM\","
        + "\"DATASET\":{"
        + "\"USER\":\"llange\","
        + "\"NICKNAME\":\"viper\","
        + "\"ROOM\":\"room\","
        + "\"ROOMTYPE\":\"PAUSENRAUM\","
        + "\"ENTRY\":\"PROTECTED\","
        + "\"LIFETIME\":\"TEMPORARY\","
        + "\"PASSWORD\":\"geheim\""
        + "}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeCREATETEMPROOM_RESPONSE() throws EncodeException {
    CREATETEMPROOM temproom = new CREATETEMPROOM();
    temproom.setCommand(Command.CREATETEMPROOM);
    temproom.setHeader(RESPONSE);
    temproom.setDataset(Protocol.DATASET);
    temproom.setRoom("room");
    temproom.setUser("llange");
    temproom.setEntry(Entry.PUBLIC);
    temproom.setLifetime(Lifetime.TEMPORARY);
    temproom.setRoomtype(Roomtype.PAUSENRAUM);

    String result = encoder.encode(temproom);
    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CREATETEMPROOM\","
        + "\"DATASET\":{"
        + "\"USER\":\"llange\","
        + "\"ROOM\":\"room\","
        + "\"ROOMTYPE\":\"PAUSENRAUM\","
        + "\"ENTRY\":\"PUBLIC\","
        + "\"LIFETIME\":\"TEMPORARY\""
        + "}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeCREATETEMPROOM_RESPONSE_MULTILINGUALKEY() throws EncodeException {
    CREATETEMPROOM temproom = new CREATETEMPROOM();
    temproom.setCommand(Command.CREATETEMPROOM);
    temproom.setHeader(RESPONSE);
    temproom.setDataset(Protocol.DATASET);
    temproom.setRoom("room");
    temproom.setUser("llange");
    temproom.setEntry(Entry.PUBLIC);
    temproom.setLifetime(Lifetime.TEMPORARY);
    temproom.setRoomtype(Roomtype.PAUSENRAUM);
    temproom.setMultilingualkey(KEY.SERVER_DER_CHATRAUM_WURDE_ANGELEGT);

    String result = encoder.encode(temproom);
    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CREATETEMPROOM\","
        + "\"DATASET\":{"
        + "\"USER\":\"llange\","
        + "\"ROOM\":\"room\","
        + "\"ROOMTYPE\":\"PAUSENRAUM\","
        + "\"ENTRY\":\"PUBLIC\","
        + "\"LIFETIME\":\"TEMPORARY\","
        + "\"MULTILINGUALKEY\":\"Server_Der_Chatraum_wurde_angelegt\""
        + "}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeCREATETEMPROOM_RESPONSE_PROTECTED_TEMPORARY() throws EncodeException {
    CREATETEMPROOM temproom = new CREATETEMPROOM();
    temproom.setCommand(Command.CREATETEMPROOM);
    temproom.setHeader(RESPONSE);
    temproom.setDataset(Protocol.DATASET);
    temproom.setRoom("room");
    temproom.setUser("llange");
    temproom.setEntry(Entry.PROTECTED);
    temproom.setLifetime(Lifetime.TEMPORARY);
    temproom.setRoomtype(Roomtype.PAUSENRAUM);

    String result = encoder.encode(temproom);
    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CREATETEMPROOM\","
        + "\"DATASET\":{"
        + "\"USER\":\"llange\","
        + "\"ROOM\":\"room\","
        + "\"ROOMTYPE\":\"PAUSENRAUM\","
        + "\"ENTRY\":\"PROTECTED\","
        + "\"LIFETIME\":\"TEMPORARY\""
        + "}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeCREATETEMPROOM_RESPONSE_TEXT() throws EncodeException {
    CREATETEMPROOM temproom = new CREATETEMPROOM();
    temproom.setCommand(Command.CREATETEMPROOM);
    temproom.setHeader(RESPONSE);
    temproom.setDataset(Protocol.DATASET);
    temproom.setRoom("room");
    temproom.setUser("llange");
    temproom.setEntry(Entry.PUBLIC);
    temproom.setLifetime(Lifetime.TEMPORARY);
    temproom.setText("Hallo meine Freunde");
    temproom.setRoomtype(Roomtype.PAUSENRAUM);

    String result = encoder.encode(temproom);
    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"CREATETEMPROOM\","
        + "\"DATASET\":{"
        + "\"USER\":\"llange\","
        + "\"ROOM\":\"room\","
        + "\"ROOMTYPE\":\"PAUSENRAUM\","
        + "\"ENTRY\":\"PUBLIC\","
        + "\"LIFETIME\":\"TEMPORARY\","
        + "\"TEXT\":\"Hallo meine Freunde\""
        + "}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeDELETEROOM_ERROR() throws EncodeException {
    DELETEROOM delete = new DELETEROOM();
    delete.setCommand(Command.DELETEROOM);
    delete.setHeader(ERROR);
    delete.setDataset(Protocol.DATASET);
    delete.setUserid("louisxiv");
    delete.setRoom("nexus");
    delete.setText("the room does not exist.");
    String result = encoder.encode(delete);
    String bla = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"DELETEROOM\","
        + "\"DATASET\":{"
        + "\"USER\":\"louisxiv\","
        + "\"ROOM\":\"nexus\","
        + "\"TEXT\":\"the room does not exist.\""
        + "}}";
    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeDELETEROOM_REQUEST() throws EncodeException {
    DELETEROOM delete = new DELETEROOM();
    delete.setCommand(Command.DELETEROOM);
    delete.setHeader(REQUEST);
    delete.setDataset(Protocol.DATASET);
    delete.setUserid("louisxiv");
    delete.setRoom("nexus");
    String result = encoder.encode(delete);
    String bla = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"DELETEROOM\","
        + "\"DATASET\":{"
        + "\"USER\":\"louisxiv\","
        + "\"ROOM\":\"nexus\""
        + "}}";
    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeDELETEROOM_REQUEST_ohne_Room() {
    DELETEROOM delete = new DELETEROOM();
    delete.setCommand(Command.DELETEROOM);
    delete.setHeader(REQUEST);
    delete.setDataset(Protocol.DATASET);
    delete.setUserid("louisxiv");
    try {
      String result = encoder.encode(delete);
      fail("EncodeException nicht geworfen");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void encodeDELETEROOM_REQUEST_ohne_User() {
    DELETEROOM delete = new DELETEROOM();
    delete.setCommand(Command.DELETEROOM);
    delete.setHeader(REQUEST);
    delete.setDataset(Protocol.DATASET);
    delete.setRoom("nexus");
    try {
      String result = encoder.encode(delete);
      fail("EncodeException nicht geworfen");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void encodeDELETEROOM_RESPONSE() throws EncodeException {
    DELETEROOM delete = new DELETEROOM();
    delete.setCommand(Command.DELETEROOM);
    delete.setHeader(RESPONSE);
    delete.setDataset(Protocol.DATASET);
    delete.setUserid("louisxiv");
    delete.setRoom("nexus");
    String result = encoder.encode(delete);
    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"DELETEROOM\","
        + "\"DATASET\":{"
        + "\"USER\":\"louisxiv\","
        + "\"ROOM\":\"nexus\""
        + "}}";
    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeDELETEROOM_RESPONSE_ohne_Room() {
    DELETEROOM delete = new DELETEROOM();
    delete.setCommand(Command.DELETEROOM);
    delete.setHeader(RESPONSE);
    delete.setDataset(Protocol.DATASET);
    delete.setUserid("louisxiv");
    try {
      String result = encoder.encode(delete);
      fail("EncodeException nicht geworfen");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void encodeDELETEROOM_RESPONSE_ohne_User() {
    DELETEROOM delete = new DELETEROOM();
    delete.setCommand(Command.DELETEROOM);
    delete.setHeader(RESPONSE);
    delete.setDataset(Protocol.DATASET);
    delete.setRoom("nexus");
    try {
      String result = encoder.encode(delete);
      fail("EncodeException nicht geworfen");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void encodeDELETEROOM_RESPONSE_TEXT() throws EncodeException {
    DELETEROOM delete = new DELETEROOM();
    delete.setCommand(Command.DELETEROOM);
    delete.setHeader(RESPONSE);
    delete.setDataset(Protocol.DATASET);
    delete.setUserid("louisxiv");
    delete.setRoom("nexus");
    delete.setText("has been deleted");
    String result = encoder.encode(delete);
    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"DELETEROOM\","
        + "\"DATASET\":{"
        + "\"USER\":\"louisxiv\","
        + "\"ROOM\":\"nexus\","
        + "\"TEXT\":\"has been deleted\""
        + "}}";
    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeDELETEUPLOADFILES_CONFIRM() throws EncodeException {
    DELETEUPLOADFILES deleteUploadfiles = new DELETEUPLOADFILES();
    deleteUploadfiles.setCommand(Command.DELETEUPLOADFILES);
    deleteUploadfiles.setHeader(CONFIRM);

    String result = encoder.encode(deleteUploadfiles);

    String bla = "{\"HEADER\":\"CONFIRM\"," + "\"COMMAND\":\"DELETEUPLOADFILES\"" + "}";
    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeDELETEUPLOADFILES_ERROR() throws EncodeException {
    DELETEUPLOADFILES deleteUploadfiles = new DELETEUPLOADFILES();
    deleteUploadfiles.setCommand(Command.DELETEUPLOADFILES);
    deleteUploadfiles.setHeader(ERROR);
    deleteUploadfiles.setDataset(Protocol.DATASET);
    deleteUploadfiles.setErrorCode(34);
    deleteUploadfiles.setErrorMessage("fehler");
    deleteUploadfiles.setText("A database error has occured");

    String result = encoder.encode(deleteUploadfiles);

    String bla = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"DELETEUPLOADFILES\","
        + "\"DATASET\":{"
        + "\"TEXT\":\"A database error has occured\","
        + "\"ERRORCODE\":34,"
        + "\"ERRORMESSAGE\":\"fehler\"}}";
    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeDELETEUPLOADFILES_REQUEST() throws EncodeException {
    DELETEUPLOADFILES deleteUploadfiles = new DELETEUPLOADFILES();
    deleteUploadfiles.setCommand(Command.DELETEUPLOADFILES);
    deleteUploadfiles.setHeader(REQUEST);
    deleteUploadfiles.setDataset(Protocol.DATASET);
    deleteUploadfiles.setUserid("louisxiv");
    deleteUploadfiles.setIp("192.168.1.41");

    String result = encoder.encode(deleteUploadfiles);

    String bla = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"DELETEUPLOADFILES\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"IP\":\"192.168.1.41\"}}";
    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeDIAL_ERROR() throws EncodeException {
    DIAL dial = new DIAL();
    dial.setHeader(ERROR);
    dial.setCommand(Command.DIAL);
    dial.setDataset(Protocol.DATASET);
    dial.setReceiverIp("localhost");
    dial.setCallerNickname("anaconda");
    dial.setReceiverPort(29477);
    dial.setCallerUserid("12345678");
    dial.setCallerVoice("niedrig");
    dial.setReceiverNickname("javaanaconda");
    dial.setErrorMessage("Der Anwender hat in der Zwischenzeit seinen Nickname geändert.");

    String result = encoder.encode(dial);
    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"DIAL\","
        + "\"DATASET\":{"
        + "\"CALLER_USERID\":\"12345678\","
        + "\"CALLER_NICKNAME\":\"anaconda\","
        + "\"CALLER_VOICE\":\"niedrig\","
        + "\"RECEIVER_IP\":\"localhost\","
        + "\"RECEIVER_PORT\":29477,"
        + "\"RECEIVER_NICKNAME\":\"javaanaconda\","
        + "\"ERRORMESSAGE\":\"Der Anwender hat in der Zwischenzeit seinen Nickname geändert.\""
        + "}}";
    assertEquals(encoded, result, "encodeDIAL_REQUEST");
  }



  @Test
  public void encodeDIAL_ERROR_database_1() throws EncodeException {
    DIAL dial = new DIAL();
    dial.setHeader(ERROR);
    dial.setCommand(Command.DIAL);
    dial.setDataset(Protocol.DATASET);

    dial.setCallerNickname("anaconda");
    dial.setCallerUserid("12345678");
    dial.setCallerVoice("niedrig");
    dial.setErrorMessage("Der Anwender hat in der Zwischenzeit seinen Nickname geändert.");
    dial.setReceiverNickname("jester");

    String result = encoder.encode(dial);
    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"DIAL\","
        + "\"DATASET\":{"
        + "\"CALLER_USERID\":\"12345678\","
        + "\"CALLER_NICKNAME\":\"anaconda\","
        + "\"CALLER_VOICE\":\"niedrig\","
        + "\"RECEIVER_NICKNAME\":\"jester\","
        + "\"ERRORMESSAGE\":\"Der Anwender hat in der Zwischenzeit seinen Nickname geändert.\""
        + "}}";
    assertEquals(encoded, result, "encodeDIAL_REQUEST");
  }



  @Test
  public void encodeDIAL_ERROR_MULTILINGUALKEY() throws EncodeException {
    DIAL dial = new DIAL();
    dial.setHeader(ERROR);
    dial.setCommand(Command.DIAL);
    dial.setDataset(Protocol.DATASET);
    dial.setReceiverIp("localhost");
    dial.setCallerNickname("anaconda");
    dial.setReceiverPort(29477);
    dial.setCallerUserid("12345678");
    dial.setCallerVoice("niedrig");
    dial.setReceiverNickname("javaanaconda");
    dial.setErrorMessage("Der Anwender hat in der Zwischenzeit seinen Nickname geändert.");
    dial.setMultilingualkey(KEY.STRING_NIMMT_KEINE_ANRUFE_ENTGEGEN);

    String result = encoder.encode(dial);
    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"DIAL\","
        + "\"DATASET\":{"
        + "\"CALLER_USERID\":\"12345678\","
        + "\"CALLER_NICKNAME\":\"anaconda\","
        + "\"CALLER_VOICE\":\"niedrig\","
        + "\"RECEIVER_IP\":\"localhost\","
        + "\"RECEIVER_PORT\":29477,"
        + "\"RECEIVER_NICKNAME\":\"javaanaconda\","
        + "\"ERRORMESSAGE\":\"Der Anwender hat in der Zwischenzeit seinen Nickname geändert.\","
        + "\"MULTILINGUALKEY\":\"String_nimmt_keine_Anrufe_entgegen\""
        + "}}";
    assertEquals(encoded, result, "MULTILINGUALKEY");
  }



  @Test
  public void encodeDIAL_ERROR_ohne_ErrorMessage() throws EncodeException {
    DIAL dial = new DIAL();
    dial.setHeader(ERROR);
    dial.setCommand(Command.DIAL);
    dial.setDataset(Protocol.DATASET);
    dial.setReceiverIp("localhost");
    dial.setCallerNickname("anaconda");
    dial.setReceiverPort(29477);
    dial.setCallerUserid("12345678");
    dial.setCallerVoice("niedrig");
    dial.setReceiverNickname("javaanaconda");
    dial.setMultilingualkey(KEY.STRING_NIMMT_KEINE_ANRUFE_ENTGEGEN);

    String result = encoder.encode(dial);
    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"DIAL\","
        + "\"DATASET\":{"
        + "\"CALLER_USERID\":\"12345678\","
        + "\"CALLER_NICKNAME\":\"anaconda\","
        + "\"CALLER_VOICE\":\"niedrig\","
        + "\"RECEIVER_IP\":\"localhost\","
        + "\"RECEIVER_PORT\":29477,"
        + "\"RECEIVER_NICKNAME\":\"javaanaconda\","
        + "\"MULTILINGUALKEY\":\"String_nimmt_keine_Anrufe_entgegen\""
        + "}}";
    assertEquals(encoded, result, "MULTILINGUALKEY");
  }



  @Test
  public void encodeDIAL_REQUEST() throws EncodeException {
    DIAL dial = new DIAL();
    dial.setHeader(REQUEST);
    dial.setCommand(Command.DIAL);
    dial.setDataset(Protocol.DATASET);

    dial.setCallerNickname("anaconda");
    dial.setCallerUserid("12345678");
    dial.setCallerVoice("niedrig");
    dial.setReceiverNickname("javaanaconda");

    String result = encoder.encode(dial);
    String expected = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"DIAL\","
        + "\"DATASET\":{"
        + "\"CALLER_USERID\":\"12345678\","
        + "\"CALLER_NICKNAME\":\"anaconda\","
        + "\"CALLER_VOICE\":\"niedrig\","
        + "\"RECEIVER_NICKNAME\":\"javaanaconda\""
        + "}}";
    assertEquals(expected, result, "encodeDIAL_REQUEST");
  }



  @Test
  public void encodeDIAL_REQUEST_ohne_CALLERVOICE() {
    DIAL dial = new DIAL();
    dial.setHeader(REQUEST);
    dial.setCommand(Command.DIAL);
    dial.setDataset(Protocol.DATASET);

    dial.setCallerNickname("anaconda");
    dial.setCallerUserid("12345678");
    dial.setReceiverNickname("javaanaconda");

    String result;
    try {
      result = encoder.encode(dial);
      fail("Exception nicht geworfen");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
//    String expected = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"DIAL\","
//        + "\"DATASET\":{"
//        + "\"CALLER_USERID\":\"12345678\","
//        + "\"CALLER_NICKNAME\":\"anaconda\","
//        + "\"RECEIVER_NICKNAME\":\"javaanaconda\""
//        + "}}";
//    assertEquals(expected, result, "encodeDIAL_REQUEST");
  }



  @Test
  public void encodeDIAL_REQUEST_ohne_Nickname() {
    DIAL dial = new DIAL();
    dial.setHeader(REQUEST);
    dial.setCommand(Command.DIAL);
    dial.setDataset(Protocol.DATASET);
    try {
      String result = encoder.encode(dial);
      fail("keine Exception geworfen");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  @DisplayName(value = "DIAL REQUEST ohne RECEIVER_NICKNAME")
  public void encodeDIAL_REQUEST_ohne_RECEIVER_NICKNAME() {
    DIAL dial = new DIAL();
    dial.setHeader(REQUEST);
    dial.setCommand(Command.DIAL);
    dial.setDataset(Protocol.DATASET);

    dial.setCallerNickname("anaconda");
    dial.setCallerUserid("12345678");
    dial.setCallerVoice("niedrig");

    try {
      encoder.encode(dial);
      fail("EncodeException nicht geworfen, RECEIVER_NICKNAME fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }

  }



  @Test
  public void encodeDIAL_RESPONSE() throws EncodeException {
    DIAL dial = new DIAL();
    dial.setHeader(RESPONSE);
    dial.setCommand(Command.DIAL);
    dial.setDataset(Protocol.DATASET);
    dial.setReceiverIp("localhost");
    dial.setCallerNickname("anaconda");
    dial.setReceiverPort(29477);
    dial.setCallerUserid("12345678");
    dial.setCallerVoice("niedrig");
    dial.setReceiverNickname("javaanaconda");
    dial.setReceiverUserid("java123");
    dial.setCallerSession("4e");
    dial.setReceiverSession("4f");
    String result = encoder.encode(dial);
    String expected = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"DIAL\","
        + "\"DATASET\":{"
        + "\"CALLER_USERID\":\"12345678\","
        + "\"CALLER_NICKNAME\":\"anaconda\","
        + "\"RECEIVER_IP\":\"localhost\","
        + "\"RECEIVER_PORT\":29477,"
        + "\"CALLER_VOICE\":\"niedrig\","
        + "\"RECEIVER_NICKNAME\":\"javaanaconda\","
        + "\"RECEIVER_USERID\":\"java123\","
        + "\"CALLER_SESSION\":\"4e\","
        + "\"RECEIVER_SESSION\":\"4f\""
        + "}}";
    assertEquals(expected, result, "encodeDIAL_REQUEST");
  }



  @Test
  public void encodeDIAL_RESPONSE_ohne_CALLERVOICE() {
    DIAL dial = new DIAL();
    dial.setHeader(RESPONSE);
    dial.setCommand(Command.DIAL);
    dial.setDataset(Protocol.DATASET);
    dial.setReceiverIp("localhost");
    dial.setCallerNickname("anaconda");
    dial.setReceiverPort(29477);
    dial.setCallerUserid("12345678");
    dial.setReceiverNickname("javaanaconda");
    dial.setReceiverUserid("java123");
    String result;
    try {
      result = encoder.encode(dial);
      fail("Exception nicht geworfen");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void encodeDIAL_RESPONSE_OHNE_IP_UND_PORT() throws EncodeException {
    DIAL dial = new DIAL();
    dial.setHeader(RESPONSE);
    dial.setCommand(Command.DIAL);
    dial.setDataset(Protocol.DATASET);
    dial.setCallerNickname("anaconda");
    dial.setCallerUserid("12345678");
    dial.setCallerVoice("niedrig");
    dial.setReceiverNickname("javaanaconda");
    dial.setReceiverUserid("java123");
    dial.setCallerSession("3c");
    dial.setReceiverSession("3d");
    String result = encoder.encode(dial);
    String expected = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"DIAL\","
        + "\"DATASET\":{"
        + "\"CALLER_USERID\":\"12345678\","
        + "\"CALLER_NICKNAME\":\"anaconda\","
        + "\"CALLER_VOICE\":\"niedrig\","
        + "\"RECEIVER_NICKNAME\":\"javaanaconda\","
        + "\"RECEIVER_USERID\":\"java123\","
        + "\"CALLER_SESSION\":\"3c\","
        + "\"RECEIVER_SESSION\":\"3d\""
        + "}}";
    assertEquals(expected, result, "encodeDIAL_RESPONSE");
  }



  @Test
  public void encodeDIAL_RESPONSE_SESSION() throws EncodeException {
    DIAL dial = new DIAL();
    dial.setHeader(RESPONSE);
    dial.setCommand(Command.DIAL);
    dial.setDataset(Protocol.DATASET);
    dial.setReceiverIp("localhost");
    dial.setCallerNickname("anaconda");
    dial.setReceiverPort(29477);
    dial.setCallerUserid("12345678");
    dial.setCallerVoice("niedrig");
    dial.setReceiverNickname("javaanaconda");
    dial.setReceiverUserid("java123");
    dial.setReceiverSession("7c");
    dial.setCallerSession("8b");

    String result = encoder.encode(dial);
    String expected = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"DIAL\","
        + "\"DATASET\":{"
        + "\"CALLER_USERID\":\"12345678\","
        + "\"CALLER_NICKNAME\":\"anaconda\","
        + "\"RECEIVER_IP\":\"localhost\","
        + "\"RECEIVER_PORT\":29477,"
        + "\"CALLER_VOICE\":\"niedrig\","
        + "\"RECEIVER_NICKNAME\":\"javaanaconda\","
        + "\"RECEIVER_USERID\":\"java123\","
        + "\"CALLER_SESSION\":\"8b\","
        + "\"RECEIVER_SESSION\":\"7c\""
        + "}}";
    assertEquals(expected, result, "encodeDIAL_REQUEST");
  }



  @Test
  public void encodeDOWNLOAD_ERROR() throws EncodeException {
    DOWNLOAD download = new DOWNLOAD();
    download.setCommand(Command.DOWNLOAD);
    download.setHeader(ERROR);
    download.setDataset(Protocol.DATASET);
    download.setFilename("hallo.mp3");
    download.setIndex(2873);
    download.setSlot(1);
    download.setUserid("python");
    download.setText("The file");

    String result = encoder.encode(download);

    String bla = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"DOWNLOAD\","
        + "\"DATASET\":{"
        + "\"USERID\":\"python\","
        + "\"SLOT\":1,"
        + "\"FILENAME\":\"hallo.mp3\","
        + "\"TEXT\":\"The file\","
        + "\"INDEX\":2873}"
        + "}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeDOWNLOAD_REQUEST() throws EncodeException {
    DOWNLOAD download = new DOWNLOAD();
    download.setCommand(Command.DOWNLOAD);
    download.setHeader(REQUEST);
    download.setDataset(Protocol.DATASET);
    download.setFilename("hallo.mp3");
    download.setIndex(2873);
    download.setSlot(1);
    download.setUserid("python");

    String result = encoder.encode(download);

    String bla = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"DOWNLOAD\","
        + "\"DATASET\":{"
        + "\"USERID\":\"python\","
        + "\"SLOT\":1,"
        + "\"FILENAME\":\"hallo.mp3\","
        + "\"INDEX\":2873}"
        + "}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeDOWNLOAD_RESPONSE() {
    DOWNLOAD download = new DOWNLOAD();
    download.setCommand(Command.DOWNLOAD);
    download.setHeader(RESPONSE);
    download.setDataset(Protocol.DATASET);
    download.setFilename("hallo.mp3");
    download.setIndex(2873);
    download.setSlot(1);
    download.setUserid("python");
    download.setIp("127.0.0.1");
    download.setPathfile("c:/hallo.mp3");

    String result;
    try {
      result = encoder.encode(download);
      fail("EncodeException nicht geworfen");
    }
    catch (EncodeException e) {
      // TODO Auto-generated catch block
    }
  }



  @Test
  public void encodeDOWNLOAD_RESPONSE_PORT() throws EncodeException {
    DOWNLOAD download = new DOWNLOAD();
    download.setCommand(Command.DOWNLOAD);
    download.setHeader(RESPONSE);
    download.setDataset(Protocol.DATASET);
    download.setFilename("hallo.mp3");
    download.setIndex(2873);
    download.setSlot(1);
    download.setUserid("python");
    download.setIp("127.0.0.1");
    download.setPathfile("c:/hallo.mp3");
    download.setPort(8443);

    String result = encoder.encode(download);

    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"DOWNLOAD\","
        + "\"DATASET\":{"
        + "\"USERID\":\"python\","
        + "\"SLOT\":1,"
        + "\"FILENAME\":\"hallo.mp3\","
        + "\"IP\":\"127.0.0.1\","
        + "\"PATHFILE\":\"c:/hallo.mp3\","
        + "\"INDEX\":2873,"
        + "\"PORT\":8443"
        + "}"
        + "}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  @DisplayName(value = "ENTERROOM | ERROR")
  public void encodeENTERROOM_ERROR() throws EncodeException {
    ENTERROOM enterroom = new ENTERROOM();
    enterroom.setCommand(Command.ENTERROOM);
    enterroom.setHeader(ERROR);
    enterroom.setDataset(Protocol.DATASET);
    enterroom.setMultilingualkey(KEY.SERVER_ZUTRITTSBERECHTIGUNG);
    enterroom.setText("Du hast für diesen Raum keine Zutrittsberechtigung.");
    String expected = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"ENTERROOM\","
        + "\"DATASET\":{"
        + "\"MULTILINGUALKEY\":\"Server_Zutrittsberechtigung\","
        + "\"TEXT\":\"Du hast für diesen Raum keine Zutrittsberechtigung.\""
        + "}}";
    String result = encoder.encode(enterroom);
    assertEquals(expected, result, "enterroom,request");
  }



  @Test
  @DisplayName(value = "ENTERROOM | REQUEST")
  public void encodeENTERROOM_REQUEST() throws EncodeException {
    ENTERROOM enterroom = new ENTERROOM();
    enterroom.setCommand(Command.ENTERROOM);
    enterroom.setHeader(REQUEST);
    enterroom.setDataset(Protocol.DATASET);
    enterroom.setRoom("anaconda");
    enterroom.setUserid("12345678");
    enterroom.setNickname("viper");

    String expected = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"ENTERROOM\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"NICKNAME\":\"viper\","
        + "\"ROOM\":\"anaconda\""
        + "}}";

    String result = encoder.encode(enterroom);
    assertEquals(expected, result, "enterroom,request");
  }



  @Test
  @DisplayName(value = "ENTERROOM | REQUEST | NICKNAME | FEHLT")
  public void encodeENTERROOM_REQUEST_NICKNAME_FEHLT() {
    ENTERROOM enterroom = new ENTERROOM();
    enterroom.setCommand(Command.ENTERROOM);
    enterroom.setHeader(REQUEST);
    enterroom.setDataset(Protocol.DATASET);
    enterroom.setRoom("anaconda");
    enterroom.setUserid("12345678");

    try {
      String result = encoder.encode(enterroom);
      fail("EncodeException nicht geworfen, Nickname fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }

  }



  @Test
  @DisplayName(value = "ENTERROOM | RESPONSE")
  public void encodeENTERROOM_RESPONSE() throws EncodeException {
    ENTERROOM enterroom = new ENTERROOM();
    enterroom.setCommand(Command.ENTERROOM);
    enterroom.setHeader(RESPONSE);
    enterroom.setDataset(Protocol.DATASET);
    enterroom.setRoom("anaconda");
    enterroom.setUserid("12345678");
    enterroom.setNickname("viper");
    enterroom.setRoomtype(Roomtype.FORUM);

    String expected = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"ENTERROOM\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"NICKNAME\":\"viper\","
        + "\"ROOM\":\"anaconda\","
        + "\"ROOMTYPE\":\"FORUM\""
        + "}}";

    String result = encoder.encode(enterroom);

    assertEquals(expected, result, "enterroom,request");

  }



  @Test
  public void encodeENTERROOM_RESPONSE_error() {
    ENTERROOM enterroom = new ENTERROOM();
    enterroom.setCommand(Command.ENTERROOM);
    enterroom.setHeader(RESPONSE);
    enterroom.setDataset(Protocol.DATASET);
    enterroom.setRoom("anaconda");
    enterroom.setUserid("12345678");

    String expected = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"ENTERROOM\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"ROOM\":\"anaconda\""
        + "}}";

    try {
      String result = encoder.encode(enterroom);
      fail("EncodeException nicht geworfen");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }

  }

  /* @Test public void encodeTRANSFER_REQUEST() throws EncodeException { TRANSFER
   * transfer = new TRANSFER(); transfer.setCommand(Command.TRANSFER);
   * transfer.setHeader(REQUEST); transfer.setDataset(Protocol.DATASET);
   * transfer.setFilename("hallo.mp3"); transfer.setPathfile("c:/");
   * transfer.setSlot(1); transfer.setBlockindex(33); transfer.setEndflag(true);
   * transfer.setPayload("273237");
   * 
   * 
   * String result = encoder.encode(transfer);
   * 
   * String bla = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"TRANSFER\"," +
   * "\"DATASET\":{" + "\"SLOT\":1," + "\"BLOCKINDEX\":33," +
   * "\"FILENAME\":\"hallo.mp3\"," + "\"PATHFILE\":\"c:/\"," + "\"ENDFLAG\":true,"
   * + "\"PAYLOAD\":\"273237\"" + "}" + "}";
   * 
   * assertEquals(bla, result, "JSON"); } */



  @Test
  public void encodeENTERROOM_RESPONSE_FORUM() throws EncodeException {
    ENTERROOM enterroom = new ENTERROOM();
    enterroom.setCommand(Command.ENTERROOM);
    enterroom.setHeader(RESPONSE);
    enterroom.setDataset(Protocol.DATASET);
    enterroom.setRoom("anaconda");
    enterroom.setUserid("12345678");
    enterroom.setNickname("viper");
    enterroom.setRoomtype(Roomtype.FORUM);

    String expected = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"ENTERROOM\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"NICKNAME\":\"viper\","
        + "\"ROOM\":\"anaconda\","
        + "\"ROOMTYPE\":\"FORUM\""
        + "}}";

    String result = encoder.encode(enterroom);

    assertEquals(expected, result, "enterroom,request");

  }



  @Test
  @DisplayName(value = "ENTERROOM | RESPONSE | ROOMTYPE | FEHLT")
  public void encodeENTERROOM_RESPONSE_ROOMTYPE_FEHLT() {
    ENTERROOM enterroom = new ENTERROOM();
    enterroom.setCommand(Command.ENTERROOM);
    enterroom.setHeader(RESPONSE);
    enterroom.setDataset(Protocol.DATASET);
    enterroom.setRoom("anaconda");
    enterroom.setUserid("12345678");
    enterroom.setNickname("viper");

    try {
      encoder.encode(enterroom);
      fail("EncodeException nicht geworfen, Roomtype ist nicht definiert.");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }

  }



  @Test
  public void encodeFILETYPES_REQUEST() throws EncodeException {
    FILETYPES filetypes = new FILETYPES();
    filetypes.setCommand(Command.FILETYPES);
    filetypes.setHeader(REQUEST);
    filetypes.setDataset(Protocol.DATASET);
    filetypes.setUserid("userid123");

    String result = encoder.encode(filetypes);

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"FILETYPES\","
        + "\"DATASET\":{"
        + "\"USERID\":\"userid123\""
        + "}"
        + "}";

    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeFILETYPES_RESPONSE() throws EncodeException {
    FILETYPES filetypes = new FILETYPES();
    filetypes.setCommand(Command.FILETYPES);
    filetypes.setHeader(RESPONSE);
    filetypes.setDataset(Protocol.DATASET);
    filetypes.setUserid("userid123");
    BlacklistTypes[] list = new BlacklistTypes[2];
    list[0] = new BlacklistTypes();
    list[0].setDescription("Adobe");
    list[0].setSuffix("pdf");
    list[0].setChecked(true);
    list[1] = new BlacklistTypes();
    list[1].setDescription("Simple Text Format");
    list[1].setSuffix("txt");
    list[1].setChecked(false);
    filetypes.setFiletypes(list);

    String result = encoder.encode(filetypes);

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"FILETYPES\","
        + "\"DATASET\":{"
        + "\"USERID\":\"userid123\","
        + "\"FILETYPES\":["
        + "{\"SUFFIX\":\"pdf\",\"DESCRIPTION\":\"Adobe\",\"CHECKED\":true},"
        + "{\"SUFFIX\":\"txt\",\"DESCRIPTION\":\"Simple Text Format\",\"CHECKED\":false}"
        + "]}"
        + "}";

    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeFTSDOWNLOAD_ERROR() throws EncodeException {
    FTSDOWNLOAD fts = new FTSDOWNLOAD();
    fts.setCommand(Command.FTSDOWNLOAD);
    fts.setHeader(ERROR);
    fts.setDataset(Protocol.DATASET);
    fts.setErrorMessage("errormeaage");
    fts.setText("ich bin ein text");
    fts.setFilename("hallo.mp3");
    fts.setPath("c:/");
    fts.setSlot(1);

    String result = encoder.encode(fts);

    String expected = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"FTSDOWNLOAD\","
        + "\"DATASET\":{"
        + "\"SLOT\":1,"
        + "\"FILENAME\":\"hallo.mp3\","
        + "\"PATH\":\"c:/\","
        + "\"ERRORMESSAGE\":\"errormeaage\","
        + "\"TEXT\":\"ich bin ein text\""
        + "}"
        + "}";

    assertEquals(expected, result, "JSON");
  }



  @Test
  public void encodeFTSDOWNLOAD_REQUEST() throws EncodeException {
    FTSDOWNLOAD fts = new FTSDOWNLOAD();
    fts.setCommand(Command.FTSDOWNLOAD);
    fts.setHeader(REQUEST);
    fts.setDataset(Protocol.DATASET);
    fts.setFilename("hallo.mp3");
    fts.setPath("c:/");
    fts.setSlot(1);

    String result = encoder.encode(fts);

    String bla = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"FTSDOWNLOAD\","
        + "\"DATASET\":{"
        + "\"SLOT\":1,"
        + "\"FILENAME\":\"hallo.mp3\","
        + "\"PATH\":\"c:/\"}"
        + "}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeFTSDOWNLOAD_RESPONSE() throws EncodeException {
    FTSDOWNLOAD fts = new FTSDOWNLOAD();
    fts.setCommand(Command.FTSDOWNLOAD);
    fts.setHeader(RESPONSE);
    fts.setDataset(Protocol.DATASET);
    fts.setFilename("hallo.mp3");
    fts.setPath("c:/");
    fts.setSlot(1);
    fts.setMaxDatablocks(848);

    String result = encoder.encode(fts);

    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"FTSDOWNLOAD\","
        + "\"DATASET\":{"
        + "\"SLOT\":1,"
        + "\"FILENAME\":\"hallo.mp3\","
        + "\"PATH\":\"c:/\","
        + "\"MAXDATABLOCKS\":848"
        + "}"
        + "}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeHISTORY_REQUEST() throws EncodeException {
    HISTORYMESSAGE history = new HISTORYMESSAGE();
    history.setCommand(Command.HISTORYMESSAGE);
    history.setHeader(REQUEST);
    history.setDataset(Protocol.DATASET);
    history.setRoom("room");

    String result = encoder.encode(history);
    String bla = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"HISTORYMESSAGE\","
        + "\"DATASET\":{"
        + "\"ROOM\":\"room\"}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeHISTORY_RESPONSE_0_Nachricht() throws EncodeException {
    HISTORYMESSAGE history = new HISTORYMESSAGE();
    history.setCommand(Command.HISTORYMESSAGE);
    history.setHeader(RESPONSE);
    history.setDataset(Protocol.DATASET);
    history.setRoom("room");
    AttachmentImpl[] record = new AttachmentImpl[1];
    record[0] = new AttachmentImpl();
    ChatUser chatuser = new ChatUser();
    chatuser.setBackgroundColor(-1);
    chatuser.setForegroundColor(-2);
    chatuser.setNickname("python");
    chatuser.setUserid("louisxiv");
    record[0].setChatUser(chatuser);
    record[0].setDatetime(666l);
    record[0].setMessage("Hallo");
    record[0].setRoom("chat");

    // history.setRecord(record);
    history.setAttachment(new AttachmentImpl[0]);

    String result = encoder.encode(history);
    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"HISTORYMESSAGE\","
        + "\"DATASET\":{"
        + "\"ROOM\":\"room\","
        + "\"RECORD\":[]"
        + "}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeHISTORY_RESPONSE_0_Nachricht_JERSEY() {
    HISTORYMESSAGE history = new HISTORYMESSAGE();
    history.setCommand(Command.HISTORYMESSAGE);
    history.setHeader(RESPONSE);
    history.setDataset(Protocol.DATASET);
    history.setRoom("room");
    AttachmentImpl[] record = new AttachmentImpl[1];
    record[0] = new AttachmentImpl();
    ChatUser chatuser = new ChatUser();
    chatuser.setBackgroundColor(-1);
    chatuser.setForegroundColor(-2);
    chatuser.setNickname("python");
    chatuser.setUserid("louisxiv");
    record[0].setChatUser(chatuser);
    record[0].setDatetime(666l);
    record[0].setMessage("Hallo");
    record[0].setRoom("chat");

    history.setAttachment(new AttachmentImpl[0]);

    String result = history.toString();
    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"HISTORYMESSAGE\","
        + "\"DATASET\":{"
        + "\"ROOM\":\"room\","
        + "\"RECORD\":[]"
        + "}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeHISTORY_RESPONSE_1_Nachricht() throws EncodeException {
    HISTORYMESSAGE history = new HISTORYMESSAGE();
    history.setCommand(Command.HISTORYMESSAGE);
    history.setHeader(RESPONSE);
    history.setDataset(Protocol.DATASET);
    history.setRoom("room");
    AttachmentImpl[] record = new AttachmentImpl[1];
    record[0] = new AttachmentImpl();
    ChatUser chatuser = new ChatUser();
    chatuser.setBackgroundColor(-1);
    chatuser.setForegroundColor(-2);
    chatuser.setNickname("python");
    chatuser.setUserid("louisxiv");
    record[0].setChatUser(chatuser);
    record[0].setDatetime(1458579290065L);
    record[0].setMessage("Hallo");
    record[0].setRoom("chat");
    history.setAttachment(record);
    String result = encoder.encode(history);
    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"HISTORYMESSAGE\","
        + "\"DATASET\":{"
        + "\"ROOM\":\"room\","
        + "\"RECORD\":["
        + "{\"ROOM\":\"chat\",\"DATETIME\":1458579290065,\"MESSAGE\":\"Hallo\",\"CHATUSER\":{\"BACKGROUND_COLOR\":-1,\"FOREGROUND_COLOR\":-2,\"NICKNAME\":\"python\",\"USERID\":\"louisxiv\"}}"
        + "]"
        + "}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeHISTORY_RESPONSE_1_Nachricht_JERSEY() {
    HISTORYMESSAGE history = new HISTORYMESSAGE();
    history.setCommand(Command.HISTORYMESSAGE);
    history.setHeader(RESPONSE);
    history.setDataset(Protocol.DATASET);
    history.setRoom("room");
    AttachmentImpl[] record = new AttachmentImpl[1];
    record[0] = new AttachmentImpl();
    ChatUser chatuser = new ChatUser();
    chatuser.setBackgroundColor(-1);
    chatuser.setForegroundColor(-2);
    chatuser.setNickname("python");
    chatuser.setUserid("louisxiv");
    record[0].setChatUser(chatuser);
    record[0].setDatetime(1458579290065L);
    record[0].setMessage("Hallo");
    record[0].setRoom("chat");
    history.setAttachment(record);
    String result = history.toString();
    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"HISTORYMESSAGE\","
        + "\"DATASET\":{"
        + "\"ROOM\":\"room\","
        + "\"RECORD\":["
        + "{\"ROOM\":\"chat\",\"DATETIME\":1458579290065,\"MESSAGE\":\"Hallo\",\"CHATUSER\":{\"BACKGROUND_COLOR\":-1,\"FOREGROUND_COLOR\":-2,\"USERID\":\"louisxiv\",\"NICKNAME\":\"python\"}}"
        + "]"
        + "}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeHISTORY_RESPONSE_nullRecord_Nachricht() throws EncodeException {
    HISTORYMESSAGE history = new HISTORYMESSAGE();
    history.setCommand(Command.HISTORYMESSAGE);
    history.setHeader(RESPONSE);
    history.setDataset(Protocol.DATASET);
    history.setRoom("room");
    ChatUser chatuser = new ChatUser();
    chatuser.setBackgroundColor(-1);
    chatuser.setForegroundColor(-2);
    chatuser.setNickname("python");
    chatuser.setUserid("louisxiv");

    // history.setRecord(record);
    // history.setRecord(new Record[0]);

    String result = encoder.encode(history);
    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"HISTORYMESSAGE\","
        + "\"DATASET\":{"
        + "\"ROOM\":\"room\","
        + "\"RECORD\":[]"
        + "}}";
    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeHISTORY_RESPONSE_ohne_RECORD_JERSEY() {
    HISTORYMESSAGE history = new HISTORYMESSAGE();
    history.setCommand(Command.HISTORYMESSAGE);
    history.setHeader(RESPONSE);
    history.setDataset(Protocol.DATASET);
    history.setRoom("room");

//    history.setAttachment(null); fehlt

    ChatUser chatuser = new ChatUser();
    chatuser.setBackgroundColor(-1);
    chatuser.setForegroundColor(-2);
    chatuser.setNickname("python");
    chatuser.setUserid("louisxiv");

    String result = history.toString();
    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"HISTORYMESSAGE\","
        + "\"DATASET\":{"
        + "\"ROOM\":\"room\","
        + "\"RECORD\":[]"
        + "}}";
    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeHISTORYMESSAGE_RESPONSE_ATTACHMENT() throws EncodeException {
    HISTORYMESSAGE history = new HISTORYMESSAGE();
    history.setCommand(Command.HISTORYMESSAGE);
    history.setHeader(RESPONSE);
    history.setDataset(Protocol.DATASET);
    history.setRoom("room");
    AttachmentImpl[] record = new AttachmentImpl[1];
    record[0] = new AttachmentImpl();
    ChatUser chatuser = new ChatUser();
    chatuser.setBackgroundColor(-1);
    chatuser.setForegroundColor(-2);
    chatuser.setNickname("python");
    chatuser.setUserid("louisxiv");
    record[0].setChatUser(chatuser);
    record[0].setDatetime(1458579290065L);
    record[0].setMessage("Hallo");
    record[0].setRoom("chat");
    record[0].setAttachment(99l);
    record[0].setFilename("name.txt");
    record[0].setFilesize(Long.MAX_VALUE);

    history.setAttachment(record);
    String result = encoder.encode(history);
    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"HISTORYMESSAGE\","
        + "\"DATASET\":{"
        + "\"ROOM\":\"room\","
        + "\"RECORD\":["
        + "{\"ROOM\":\"chat\","
        + "\"DATETIME\":1458579290065,"
        + "\"MESSAGE\":\"Hallo\","
        + "\"ATTACHMENT\":99,"
        + "\"FILENAME\":\"name.txt\","
        + "\"FILESIZE\":"
        + String.valueOf(Long.MAX_VALUE)
        + ","
        + "\"CHATUSER\":{"
        + "\"BACKGROUND_COLOR\":-1,"
        + "\"FOREGROUND_COLOR\":-2,"
        + "\"NICKNAME\":\"python\","
        + "\"USERID\":\"louisxiv\""
        + "}}"
        + "]"
        + "}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeHISTORYMESSAGE_RESPONSE_ATTACHMENT_JERSEY() {
    HISTORYMESSAGE history = new HISTORYMESSAGE();
    history.setCommand(Command.HISTORYMESSAGE);
    history.setHeader(RESPONSE);
    history.setDataset(Protocol.DATASET);
    history.setRoom("room");
    AttachmentImpl[] record = new AttachmentImpl[1];
    record[0] = new AttachmentImpl();
    ChatUser chatuser = new ChatUser();
    chatuser.setBackgroundColor(-1);
    chatuser.setForegroundColor(-2);
    chatuser.setNickname("python");
    chatuser.setUserid("louisxiv");
    record[0].setChatUser(chatuser);
    record[0].setDatetime(1458579290065L);
    record[0].setMessage("Hallo");
    record[0].setRoom("chat");
    record[0].setAttachment(99l);
    record[0].setFilename("name.txt");
    record[0].setFilesize(Long.MAX_VALUE);

    history.setAttachment(record);
    String result = history.toString();
    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"HISTORYMESSAGE\","
        + "\"DATASET\":{"
        + "\"ROOM\":\"room\","
        + "\"RECORD\":["
        + "{\"ROOM\":\"chat\","
        + "\"DATETIME\":1458579290065,"
        + "\"MESSAGE\":\"Hallo\","
        + "\"ATTACHMENT\":99,"
        + "\"FILENAME\":\"name.txt\","
        + "\"FILESIZE\":"
        + String.valueOf(Long.MAX_VALUE)
        + ","
        + "\"CHATUSER\":{"
        + "\"BACKGROUND_COLOR\":-1,"
        + "\"FOREGROUND_COLOR\":-2,"
        + "\"USERID\":\"louisxiv\","
        + "\"NICKNAME\":\"python\""
        + "}}"
        + "]"
        + "}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeHISTORYMESSAGE_RESPONSE_ohne_FILENAME() {
    HISTORYMESSAGE history = new HISTORYMESSAGE();
    history.setCommand(Command.HISTORYMESSAGE);
    history.setHeader(RESPONSE);
    history.setDataset(Protocol.DATASET);
    history.setRoom("room");
    AttachmentImpl[] record = new AttachmentImpl[1];
    record[0] = new AttachmentImpl();
    ChatUser chatuser = new ChatUser();
    chatuser.setBackgroundColor(-1);
    chatuser.setForegroundColor(-2);
    chatuser.setNickname("python");
    chatuser.setUserid("louisxiv");
    record[0].setChatUser(chatuser);
    record[0].setDatetime(1458579290065L);
    record[0].setMessage("Hallo");
    record[0].setRoom("chat");
    record[0].setAttachment(3l);

    history.setAttachment(record);
    try {
      encoder.encode(history);
      fail("Exception nicht geworfen");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }

  }



  @Test
  public void encodeIMAGE_ERROR() throws EncodeException, UnsupportedEncodingException {
    IMAGE image = new IMAGE();
    image.setCommand(Command.IMAGE);
    image.setHeader(ERROR);
    image.setDataset(Protocol.DATASET);
    image.setSenderUid("louisxiv");
    image.setSenderNickname("anaconda");
    image.setReceiverNickname("freigeißt");
    image.setMultilingualkey(KEY.SERVER_DER_EMPFAENGER_IST_VERSCHWUNDEN);

    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"IMAGE\","
        + "\"DATASET\":{"
        + "\"SENDERUID\":\"louisxiv\","
        + "\"SENDERNICKNAME\":\"anaconda\","
        + "\"RECEIVERNICKNAME\":\"freigeißt\","
        + "\"MULTILINGUALKEY\":\"Server_Der_Empfaenger_ist_verschwunden\""
        + "}}";

    String result = encoder.encode(image);
    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeIMAGE_ERROR_1() throws UnsupportedEncodingException {
    IMAGE error = new IMAGE();
    error.setCommand(Command.IMAGE);
    error.setHeader(ERROR);
    error.setDataset(Protocol.DATASET);
    error.setReceiverNickname("freigeist");
    error.setMultilingualkey(KEY.SERVER_MUSS_NOCH_SEINEN_PROJEKTOR_AUSSCHALTEN);

    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"IMAGE\","
        + "\"DATASET\":{"
        + "\"RECEIVERNICKNAME\":\"freigeist\","
        + "\"MULTILINGUALKEY\":\"Server_Der_Empfaenger_ist_verschwunden\""
        + "}}";

    try {
      encoder.encode(error);
      fail("EncodeException wurd nicht geworfen - SENDERUSERID fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void encodeIMAGE_REQUEST() throws EncodeException, UnsupportedEncodingException {
    IMAGE image = new IMAGE();
    image.setCommand(Command.IMAGE);
    image.setHeader(REQUEST);
    image.setDataset(Protocol.DATASET);
    image.setSenderUid("louisxiv");
    image.setSenderNickname("anaconda");
    image.setReceiverNickname("freigeißt");

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"IMAGE\","
        + "\"DATASET\":{"
        + "\"SENDERUID\":\"louisxiv\","
        + "\"SENDERNICKNAME\":\"anaconda\","
        + "\"RECEIVERNICKNAME\":\"freigeißt\""
        + "}}";

    String result = encoder.encode(image);
    assertEquals(encoded, result, "JSON");
  }



  @Test
  @DisplayName(value = "IMAGE | RESPONSE")
  public void encodeIMAGE_RESPONSE() throws EncodeException {
    IMAGE image = new IMAGE();
    image.setCommand(Command.IMAGE);
    image.setHeader(RESPONSE);
    image.setDataset(Protocol.DATASET);
    image.setSenderUid("louisxiv");
    image.setSenderNickname("anaconda");
    image.setReceiverNickname("freigeißt");
    image.setReceiverUid("user5678");
    image.setReceiverSession("123");

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"IMAGE\","
        + "\"DATASET\":{"
        + "\"SENDERUID\":\"louisxiv\","
        + "\"SENDERNICKNAME\":\"anaconda\","
        + "\"RECEIVERNICKNAME\":\"freigeißt\","
        + "\"RECEIVERUID\":\"user5678\","
        + "\"RECEIVER_SESSION\":\"123\""
        + "}}";

    String result = encoder.encode(image);
    assertEquals(encoded, result, "JSON");
  }



  @Test
  @DisplayName(value = "IMAGE | RESPONSE | RECEIVER_SESSION | FEHLT")
  public void encodeIMAGE_RESPONSE_RECEIVER_SESSION_FEHLT() {
    IMAGE image = new IMAGE();
    image.setCommand(Command.IMAGE);
    image.setHeader(RESPONSE);
    image.setDataset(Protocol.DATASET);
    image.setSenderUid("louisxiv");
    image.setSenderNickname("anaconda");
    image.setReceiverNickname("freigeißt");
    image.setReceiverUid("user5678");

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"IMAGE\","
        + "\"DATASET\":{"
        + "\"SENDERUID\":\"louisxiv\","
        + "\"SENDERNICKNAME\":\"anaconda\","
        + "\"RECEIVERNICKNAME\":\"freigeißt\","
        + "\"RECEIVERUID\":\"user5678\""
        + "\"RECEIVER_SESSION\":\"123\""
        + "}}";

    try {
      encoder.encode(image);
      fail("Exception nicht geworfen, RECEIVER_SESSION fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  @DisplayName(value = "INCOMINGCALL | CONFIRM")
  public void encodeINCOMINGCALL_CONFIRM() throws EncodeException {
    INCOMINGCALL incoming = new INCOMINGCALL();
    incoming.setHeader(CONFIRM);
    incoming.setCommand(Command.INCOMINGCALL);
    incoming.setDataset(Protocol.DATASET);
    incoming.setReceiverUserid("uid123");
    incoming.setReceiverNickname("java");
    incoming.setReceiverSession("2a");
    incoming.setCallerUserid("anaconda");
    incoming.setCallerNickname("hallo123");
    incoming.setCallerVoice("niedrig");

    String result = encoder.encode(incoming);
    String encoded = "{\"HEADER\":\"CONFIRM\"," + "\"COMMAND\":\"INCOMINGCALL\","
        + "\"DATASET\":{"
        + "\"CALLER_NICKNAME\":\"hallo123\","
        + "\"CALLER_USERID\":\"anaconda\","
        + "\"CALLER_VOICE\":\"niedrig\","
        + "\"RECEIVER_USERID\":\"uid123\","
        + "\"RECEIVER_NICKNAME\":\"java\","
        + "\"RECEIVER_SESSION\":\"2a\""
        + "}}";
    assertEquals(encoded, result, "encodeINCOMINGCALL_CONFIRM");
  }



  @Test
  @DisplayName(value = "INCOMINGCALL | CONFIRIM | CALLER_VOICE | FEHLT")
  public void encodeINCOMINGCALL_CONFIRM_CALLERVOICE_FEHLT() {
    INCOMINGCALL incoming = new INCOMINGCALL();
    incoming.setHeader(CONFIRM);
    incoming.setCommand(Command.INCOMINGCALL);
    incoming.setDataset(Protocol.DATASET);
    incoming.setReceiverUserid("uid123");
    incoming.setReceiverNickname("java");
    incoming.setReceiverSession("2a");
    incoming.setCallerUserid("anaconda");
    incoming.setCallerNickname("hallo123");

    try {
      encoder.encode(incoming);
      fail("Exception wurde nicht geworfen, CALLER_VOICE fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  @DisplayName(value = "INCOMINGCALL | ERROR")
  public void encodeINCOMINGCALL_ERROR() throws EncodeException {
    INCOMINGCALL incoming = new INCOMINGCALL();
    incoming.setHeader(ERROR);
    incoming.setCommand(Command.INCOMINGCALL);
    incoming.setDataset(Protocol.DATASET);
    incoming.setReceiverNickname("llange");
    incoming.setMultilingualkey(KEY.STRING_IST_NICHT_MEHR_ONLINE);

    String result = encoder.encode(incoming);
    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"INCOMINGCALL\","
        + "\"DATASET\":{"
        + "\"RECEIVER_NICKNAME\":\"llange\","
        + "\"MULTILINGUALKEY\":\"String_ist_nicht_mehr_online\""
        + "}}";
    assertEquals(encoded, result, "encodeINCOMINGCALL_REQUEST");
  }



  @Test
  @DisplayName(value = "INCOMINGCALL | ERROR | RECEIVER_NICKNAME | FEHLT")
  public void encodeINCOMINGCALL_ERROR_RECEIVER_NICKNAME_FEHLT() throws EncodeException {
    INCOMINGCALL incoming = new INCOMINGCALL();
    incoming.setHeader(ERROR);
    incoming.setCommand(Command.INCOMINGCALL);
    incoming.setDataset(Protocol.DATASET);
    incoming.setMultilingualkey(KEY.STRING_IST_NICHT_MEHR_ONLINE);

    String result = encoder.encode(incoming);
    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"INCOMINGCALL\","
        + "\"DATASET\":{"
        + "\"MULTILINGUALKEY\":\"String_ist_nicht_mehr_online\""
        + "}}";
    assertEquals(encoded, result, "encodeINCOMINGCALL_REQUEST");
  }



  @Test
  @DisplayName(value = "INCOMINGCALL | REQUEST | RECEIVER_SESSION | FEHLT")
  public void encodeINCOMINGCALL_REQUEST_RECEIVER_SESSION_FEHLT() {
    INCOMINGCALL incoming = new INCOMINGCALL();
    incoming.setHeader(REQUEST);
    incoming.setCommand(Command.INCOMINGCALL);
    incoming.setDataset(Protocol.DATASET);
    incoming.setCallerNickname("hallo123");
    incoming.setCallerUserid("anaconda");
    incoming.setCallerVoice("niedrig");
    incoming.setCallerSession("7b");
    incoming.setReceiverUserid("uid123");
    incoming.setReceiverNickname("java");
    incoming.setReceiverVoice("high");

    try {
      encoder.encode(incoming);
      fail("Exception nicht geworfen, RECEIVER_SESSION fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  @DisplayName(value = "INCOMINGCALL | REQUEST | CALLERVOICE | FEHLT")
  public void encodeINCOMINGCALL_REQUEST_CALLERVOICE_FEHLT() {
    INCOMINGCALL incoming = new INCOMINGCALL();
    incoming.setHeader(REQUEST);
    incoming.setCommand(Command.INCOMINGCALL);
    incoming.setDataset(Protocol.DATASET);
    incoming.setCallerNickname("hallo123");
    incoming.setCallerUserid("anaconda");
    incoming.setCallerSession("7b");
    incoming.setReceiverUserid("uid123");
    incoming.setReceiverNickname("java");
    incoming.setReceiverSession("1a");
    incoming.setReceiverVoice("hoch");

    try {
      encoder.encode(incoming);
      fail("Exception nicht geworfen, CALLERVOICE");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  @DisplayName(value = "INCOMINGCALL | REQUEST")
  public void encodeINCOMINGCALL_REQUEST() throws EncodeException {
    INCOMINGCALL incoming = new INCOMINGCALL();
    incoming.setHeader(REQUEST);
    incoming.setCommand(Command.INCOMINGCALL);
    incoming.setDataset(Protocol.DATASET);
    incoming.setCallerNickname("hallo123");
    incoming.setCallerUserid("anaconda");
    incoming.setCallerVoice("niedrig");
    incoming.setCallerSession("7b");
    incoming.setReceiverUserid("uid123");
    incoming.setReceiverNickname("java");
    incoming.setReceiverSession("1a");
    incoming.setReceiverVoice("hoch");

    String result = encoder.encode(incoming);
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"INCOMINGCALL\","
        + "\"DATASET\":{"
        + "\"CALLER_NICKNAME\":\"hallo123\","
        + "\"CALLER_USERID\":\"anaconda\","
        + "\"CALLER_VOICE\":\"niedrig\","
        + "\"CALLER_SESSION\":\"7b\","
        + "\"RECEIVER_USERID\":\"uid123\","
        + "\"RECEIVER_NICKNAME\":\"java\","
        + "\"RECEIVER_SESSION\":\"1a\""
        + "}}";
    assertEquals(encoded, result, "encodeINCOMINGCALL_REQUEST");

  }



  @Test
  @DisplayName(value = "INCOMINGCALL | RESPONSE")
  public void encodeINCOMINGCALL_RESPONSE() throws EncodeException {
    INCOMINGCALL incoming = new INCOMINGCALL();
    incoming.setHeader(RESPONSE);
    incoming.setCommand(Command.INCOMINGCALL);
    incoming.setDataset(Protocol.DATASET);
    incoming.setCallerNickname("hallo123");
    incoming.setCallerUserid("anaconda");
    incoming.setCallerVoice("niedrig");
    incoming.setCallerSession("7b");

    incoming.setReceiverUserid("uid123");
    incoming.setReceiverNickname("java");
    incoming.setReceiverVoice("hoch");
    incoming.setReceiverSession("1a");

    String result = encoder.encode(incoming);
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"INCOMINGCALL\","
        + "\"DATASET\":{"
        + "\"CALLER_NICKNAME\":\"hallo123\","
        + "\"CALLER_USERID\":\"anaconda\","
        + "\"CALLER_VOICE\":\"niedrig\","
        + "\"CALLER_SESSION\":\"7b\","
        + "\"RECEIVER_USERID\":\"uid123\","
        + "\"RECEIVER_NICKNAME\":\"java\","
        + "\"RECEIVER_SESSION\":\"1a\","
        + "\"RECEIVER_VOICE\":\"hoch\""
        + "}}";
    assertEquals(encoded, result, "encodeINCOMINGCALL_RESPONSE");
  }



  @Test
  public void encodeINCOMINGCALL_RESPONSE_ohne_CALLERVOICE() {
    INCOMINGCALL incoming = new INCOMINGCALL();
    incoming.setHeader(RESPONSE);
    incoming.setCommand(Command.INCOMINGCALL);
    incoming.setDataset(Protocol.DATASET);
    incoming.setCallerNickname("hallo123");
    incoming.setCallerUserid("anaconda");
    incoming.setReceiverUserid("uid123");
    incoming.setReceiverNickname("java");

    String result;
    try {
      result = encoder.encode(incoming);
      fail("Exception wurde nicht geworfen");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void encodeINCOMINGCALL_RESPONSE_ohne_RECEIVERVOICE() {
    INCOMINGCALL incoming = new INCOMINGCALL();
    incoming.setHeader(RESPONSE);
    incoming.setCommand(Command.INCOMINGCALL);
    incoming.setDataset(Protocol.DATASET);
    incoming.setCallerNickname("hallo123");
    incoming.setCallerUserid("anaconda");
    incoming.setCallerVoice("niedrig");
    incoming.setReceiverUserid("uid123");
    incoming.setReceiverNickname("java");

    try {
      String result = encoder.encode(incoming);
      fail("Exception nicht geworfen");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  @DisplayName(value = "INCOMINGCALL | RESPONSE | RECEIVER_SESSION | FEHLT")
  public void encodeINCOMINGCALL_RESPONSE_RECEIVERSESSION() {
    INCOMINGCALL incoming = new INCOMINGCALL();
    incoming.setHeader(RESPONSE);
    incoming.setCommand(Command.INCOMINGCALL);
    incoming.setDataset(Protocol.DATASET);
    incoming.setCallerNickname("hallo123");
    incoming.setCallerUserid("anaconda");
    incoming.setCallerVoice("niedrig");
    incoming.setReceiverUserid("uid123");
    incoming.setReceiverNickname("java");
    incoming.setCallerSession("8b");
    incoming.setReceiverVoice("hoch");

    try {
      encoder.encode(incoming);
      fail("Exception nicht geworfen, RECEIVER_SESSION fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void encodeKEEPALIVE_REQUEST() throws EncodeException {
    KEEPALIVE history = new KEEPALIVE();
    history.setCommand(Command.KEEPALIVE);
    history.setHeader(REQUEST);

    String result = encoder.encode(history);
    String bla = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"KEEPALIVE\"}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeKEEPALIVE_REQUEST_JERSEY() throws EncodeException {
    KEEPALIVE history = new KEEPALIVE();
    history.setCommand(Command.KEEPALIVE);
    history.setHeader(REQUEST);

    String result = history.toString();
    String bla = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"KEEPALIVE\"}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeKEEPALIVE_REQUEST_SESSIONID() throws EncodeException {
    KEEPALIVE keepalive = new KEEPALIVE();
    keepalive.setCommand(Command.KEEPALIVE);
    keepalive.setHeader(REQUEST);
    keepalive.setDataset(Protocol.DATASET);
    keepalive.setSessionId("e");

    String result = encoder.encode(keepalive);

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"KEEPALIVE\","
        + "\"DATASET\":{"
        + "\"SESSIONID\":\"e\""
        + "}}";

    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeKEEPALIVE_REQUEST_SESSIONID_JERSEY() throws EncodeException {
    KEEPALIVE keepalive = new KEEPALIVE();
    keepalive.setCommand(Command.KEEPALIVE);
    keepalive.setHeader(REQUEST);
    keepalive.setDataset(Protocol.DATASET);
    keepalive.setSessionId("e");

    String result = keepalive.toString();

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"KEEPALIVE\","
        + "\"DATASET\":{"
        + "\"SESSIONID\":\"e\""
        + "}}";

    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeLEAVEPRIVATECHAT_REQUEST() throws EncodeException {
    LEAVEPRIVATECHAT chat = new LEAVEPRIVATECHAT();
    chat.setCommand(Command.LEAVEPRIVATECHAT);
    chat.setHeader(REQUEST);
    chat.setDataset(Protocol.DATASET);
    chat.setGoneSessionid("5");
    String result = encoder.encode(chat);

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"LEAVEPRIVATECHAT\","
        + "\"DATASET\":{"
        + "\"GONE_SESSIONID\":\"5\"}}";

    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeLEAVEPRIVATECHAT_RESPONSE() throws EncodeException {
    LEAVEPRIVATECHAT chat = new LEAVEPRIVATECHAT();
    chat.setCommand(Command.LEAVEPRIVATECHAT);
    chat.setHeader(RESPONSE);
    chat.setDataset(Protocol.DATASET);
    chat.setGoneSessionid("5");
    chat.setRemotesession("6");
    String result = encoder.encode(chat);

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"LEAVEPRIVATECHAT\","
        + "\"DATASET\":{"
        + "\"REMOTE_SESSIONID\":\"6\","
        + "\"GONE_SESSIONID\":\"5\"}}";

    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeLEAVEROOM_ERROR() throws EncodeException {
    LEAVEROOM leaveroom = new LEAVEROOM();
    leaveroom.setCommand(Command.LEAVEROOM);
    leaveroom.setHeader(ERROR);
    leaveroom.setDataset(Protocol.DATASET);
    leaveroom.setRoom("anaconda");
    leaveroom.setUserid("12345678");
    String result = encoder.encode(leaveroom);
    String expected = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"LEAVEROOM\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"ROOM\":\"anaconda\""
        + "}}";
    assertEquals(expected, result, "ERROR");
  }



  @Test
  public void encodeLEAVEROOM_REQUEST() throws EncodeException {
    LEAVEROOM leaveroom = new LEAVEROOM();
    leaveroom.setCommand(Command.LEAVEROOM);
    leaveroom.setHeader(REQUEST);
    leaveroom.setDataset(Protocol.DATASET);
    leaveroom.setRoom("anaconda");
    leaveroom.setUserid("12345678");
    String result = encoder.encode(leaveroom);
    String expected = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"LEAVEROOM\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"ROOM\":\"anaconda\""
        + "}}";
    assertEquals(expected, result, "REQUEST");
  }



  @Test
  public void encodeLEAVEROOM_RESPONSE() throws EncodeException {
    LEAVEROOM leaveroom = new LEAVEROOM();
    leaveroom.setCommand(Command.LEAVEROOM);
    leaveroom.setHeader(RESPONSE);
    leaveroom.setDataset(Protocol.DATASET);
    leaveroom.setRoom("anaconda");
    leaveroom.setUserid("12345678");
    String result = encoder.encode(leaveroom);
    String expected = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"LEAVEROOM\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"ROOM\":\"anaconda\""
        + "}}";
    assertEquals(expected, result, "RESPONSE");
  }



  @Test
  public void encodeMICROERROR() throws EncodeException {
    MICROERROR microerror = new MICROERROR();
    microerror.setCommand(Command.MICROERROR);
    microerror.setHeader(REQUEST);
    microerror.setDataset(Protocol.DATASET);
    microerror.setFromNickname("anaconda");
    microerror.setFromUserid("jester");
    microerror.setText("Die Gegenseite ist nicht mehr online");
    microerror.setToUserid("userid123");

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"MICROERROR\","
        + "\"DATASET\":{"
        + "\"FROM_NICKNAME\":\"anaconda\","
        + "\"FROM_USERID\":\"jester\","
        + "\"TEXT\":\"Die Gegenseite ist nicht mehr online\","
        + "\"TO_USERID\":\"userid123\""
        + "}}";
    String result = encoder.encode(microerror);
    assertEquals(encoded, result, "JSON");
  }



  @Test
  @DisplayName(value = "NEWMOUSEPOINTER | REQUEST | X | Y")
  public void encodeNEWMOUSEPOINTER_REQUEST() throws EncodeException {
    NEWMOUSEPOINTER mousepointer = new NEWMOUSEPOINTER();
    mousepointer.setCommand(Command.NEWMOUSEPOINTER);
    mousepointer.setHeader(REQUEST);
    mousepointer.setDataset(Protocol.DATASET);
    mousepointer.setX(19);
    mousepointer.setY(67);

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"NEWMOUSEPOINTER\","
        + "\"DATASET\":{"
        + "\"X\":19,"
        + "\"Y\":67"
        + "}}";
    String result = encoder.encode(mousepointer);
    assertEquals(encoded, result, "JSON");
  }



  @Test
  @DisplayName(value = "NEWMOUSEPOINTER | REQUEST | X | Y_FEHLT")
  public void encodeNEWMOUSEPOINTER_REQUEST_Y_FEHLT() {
    NEWMOUSEPOINTER mousepointer = new NEWMOUSEPOINTER();
    mousepointer.setCommand(Command.NEWMOUSEPOINTER);
    mousepointer.setHeader(REQUEST);
    mousepointer.setDataset(Protocol.DATASET);
    mousepointer.setX(19);

    try {
      encoder.encode(mousepointer);
      fail("Exception nicht geworfen, Y fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  @DisplayName(value = "NEWMOUSEPOINTER | REQUEST | X_FEHLT")
  public void encodeNEWMOUSEPOINTER_REQUEST_X_FEHLT() {
    NEWMOUSEPOINTER mousepointer = new NEWMOUSEPOINTER();
    mousepointer.setCommand(Command.NEWMOUSEPOINTER);
    mousepointer.setHeader(REQUEST);
    mousepointer.setDataset(Protocol.DATASET);
    mousepointer.setY(67);
    try {
      encoder.encode(mousepointer);
      fail("Exception nicht geworfen, Y fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  @DisplayName(value = "NEWIMAGE | REQUEST")
  public void encodeNEWIMAGE_REQUEST() throws EncodeException {
    NEWIMAGE newimage = new NEWIMAGE();
    newimage.setCommand(Command.NEWIMAGE);
    newimage.setHeader(REQUEST);
    newimage.setDataset(Protocol.DATASET);
    newimage.setUserid("anaconda");

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"NEWIMAGE\","
        + "\"DATASET\":{"
        + "\"USERID\":\"anaconda\""
        + "}}";
    String result = encoder.encode(newimage);
    assertEquals(encoded, result, "JSON");
  }



  @Test
  @DisplayName(value = "NEWIMAGE | REQUEST | USERID | FEHLT")
  public void encodeNEWIMAGE_REQUEST_RECEIVER_SESSIONS_FEHLT() {
    NEWIMAGE newimage = new NEWIMAGE();
    newimage.setCommand(Command.NEWIMAGE);
    newimage.setHeader(REQUEST);
    newimage.setDataset(Protocol.DATASET);

    try {
      encoder.encode(newimage);
      fail("Exception nicht geworfen, USERID fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void encodeONCALL_REQUEST() throws EncodeException {
    ONCALL oncall = new ONCALL();
    oncall.setHeader(REQUEST);
    oncall.setCommand(Command.ONCALL);
    oncall.setDataset(Protocol.DATASET);
    oncall.setUserid("anaconda");
    oncall.setOnCall(true);

    String result = encoder.encode(oncall);
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"ONCALL\","
        + "\"DATASET\":{"
        + "\"USERID\":\"anaconda\","
        + "\"ONCALL\":true"
        + "}}";
    assertEquals(encoded, result, "encodeONCALL");
  }



  @Test
  public void encodeONCALL_RESPONSE() throws EncodeException {
    ONCALL oncall = new ONCALL();
    oncall.setHeader(RESPONSE);
    oncall.setCommand(Command.ONCALL);
    oncall.setDataset(Protocol.DATASET);
    oncall.setUserid("anaconda");
    oncall.setOnCall(false);

    String result = encoder.encode(oncall);
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"ONCALL\","
        + "\"DATASET\":{"
        + "\"USERID\":\"anaconda\","
        + "\"ONCALL\":false"
        + "}}";
    assertEquals(encoded, result, "encodeONCALL_RESPONSE");
  }



  @Test
  public void encodeONCALL_RESPONSE_ONCALL_NULL() throws EncodeException {
    ONCALL oncall = new ONCALL();
    oncall.setHeader(RESPONSE);
    oncall.setCommand(Command.ONCALL);
    oncall.setDataset(Protocol.DATASET);
    oncall.setUserid("anaconda");

    String result = encoder.encode(oncall);
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"ONCALL\","
        + "\"DATASET\":{"
        + "\"USERID\":\"anaconda\","
        + "\"ONCALL\":false"
        + "}}";
    assertEquals(encoded, result, "encodeONCALL_RESPONSE");
  }



  @Test
  public void encodeONHOOK_REQUEST() throws EncodeException {
    ONHOOK onhook = new ONHOOK();
    onhook.setCommand(Command.ONHOOK);
    onhook.setHeader(REQUEST);
    onhook.setDataset(Protocol.DATASET);
    onhook.setCallerUserid("anaconda");
    onhook.setReceiverUserid("jester");
    onhook.setTo(true);
    onhook.setSignal(true);
    String result = encoder.encode(onhook);
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"ONHOOK\","
        + "\"DATASET\":{"
        + "\"RECEIVER_USERID\":\"jester\","
        + "\"CALLER_USERID\":\"anaconda\","
        + "\"TO\":true,"
        + "\"SIGNAL\":true"
        + "}}";
    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeONHOOK_REQUEST_ohne_TO() {
    ONHOOK onhook = new ONHOOK();
    onhook.setCommand(Command.ONHOOK);
    onhook.setHeader(REQUEST);
    onhook.setDataset(Protocol.DATASET);
    onhook.setCallerUserid("anaconda");
    onhook.setReceiverUserid("jester");
    try {
      String result = encoder.encode(onhook);
      fail("Exception nicht geworfen, TO fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void encodeONHOOK_REQUEST_SIGNAL() throws EncodeException {
    ONHOOK onhook = new ONHOOK();
    onhook.setCommand(Command.ONHOOK);
    onhook.setHeader(REQUEST);
    onhook.setDataset(Protocol.DATASET);
    onhook.setCallerUserid("anaconda");
    onhook.setReceiverUserid("jester");
    onhook.setTo(true);
    onhook.setSignal(true);
    String result = encoder.encode(onhook);
    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"ONHOOK\","
        + "\"DATASET\":{"
        + "\"RECEIVER_USERID\":\"jester\","
        + "\"CALLER_USERID\":\"anaconda\","
        + "\"TO\":true,"
        + "\"SIGNAL\":true"
        + "}}";
    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeONHOOK_RESPONSE() throws EncodeException {
    ONHOOK onhook = new ONHOOK();
    onhook.setCommand(Command.ONHOOK);
    onhook.setHeader(RESPONSE);
    onhook.setDataset(Protocol.DATASET);
    onhook.setCallerUserid("anaconda");
    onhook.setReceiverUserid("jester");
    onhook.setTo(true);
    String result = encoder.encode(onhook);
    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"ONHOOK\","
        + "\"DATASET\":{"
        + "\"RECEIVER_USERID\":\"jester\","
        + "\"CALLER_USERID\":\"anaconda\","
        + "\"TO\":true"
        + "}}";
    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodePRIVATECHATFILE_ERROR() throws EncodeException {
    PRIVATECHATFILE chatfile = new PRIVATECHATFILE();

    chatfile.setCommand(Command.PRIVATECHATFILE);
    chatfile.setHeader(ERROR);
    chatfile.setDataset(Protocol.DATASET);

    String result = encoder.encode(chatfile);

    String bla = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"PRIVATECHATFILE\"," + "\"DATASET\":{}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodePRIVATECHATFILE_REQUEST() throws EncodeException {
    PRIVATECHATFILE chatfile = new PRIVATECHATFILE();

    chatfile.setCommand(Command.PRIVATECHATFILE);
    chatfile.setHeader(REQUEST);
    chatfile.setDataset(Protocol.DATASET);
    chatfile.setAbsoluteFilename("/absoluter/dateiname");
    chatfile.setFilename("dateiname");
    chatfile.setRemoteSessionId("6");
    chatfile.setUserid("anaconda");
    chatfile.setMessage("");
    chatfile.setRemoteNickname("anaconda");

    String result = encoder.encode(chatfile);

    String bla = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"PRIVATECHATFILE\","
        + "\"DATASET\":{"
        + "\"USERID\":\"anaconda\","
        + "\"FILENAME\":\"dateiname\","
        + "\"REMOTE_SESSION_ID\":\"6\","
        + "\"REMOTE_NICKNAME\":\"anaconda\","
        + "\"ABSOLUTE_FILENAME\":\"/absoluter/dateiname\","
        + "\"MESSAGE\":\"\""
        + "}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodePRIVATECHATFILE_RESPONSE() throws EncodeException {
    PRIVATECHATFILE chatfile = new PRIVATECHATFILE();

    chatfile.setCommand(Command.PRIVATECHATFILE);
    chatfile.setHeader(RESPONSE);
    chatfile.setDataset(Protocol.DATASET);
    chatfile.setAbsoluteFilename("/absoluter/dateiname");
    chatfile.setFilename("dateiname");
    chatfile.setRemoteSessionId("6");
    chatfile.setUserid("anaconda");
    chatfile.setMessage("");
    chatfile.setNumber(1l);
    chatfile.setRemoteNickname("anaconda");

    String result = encoder.encode(chatfile);

    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"PRIVATECHATFILE\","
        + "\"DATASET\":{"
        + "\"NUMBER\":1,"
        + "\"REMOTE_NICKNAME\":\"anaconda\","
        + "\"ABSOLUTE_FILENAME\":\"/absoluter/dateiname\","
        + "\"USERID\":\"anaconda\","
        + "\"FILENAME\":\"dateiname\","
        + "\"REMOTE_SESSION_ID\":\"6\","
        + "\"MESSAGE\":\"\""
        + "}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodePRIVATECHATFILE_RESPONSE_MANDOTORY() throws EncodeException {
    PRIVATECHATFILE chatfile = new PRIVATECHATFILE();

    chatfile.setCommand(Command.PRIVATECHATFILE);
    chatfile.setHeader(RESPONSE);
    chatfile.setDataset(Protocol.DATASET);
    chatfile.setAbsoluteFilename("/absoluter/dateiname");
    chatfile.setFilename("dateiname");
    chatfile.setUserid("anaconda");
    chatfile.setNumber(2l);
    chatfile.setRemoteNickname("anaconda");

    String result = encoder.encode(chatfile);

    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"PRIVATECHATFILE\","
        + "\"DATASET\":{"
        + "\"NUMBER\":2,"
        + "\"REMOTE_NICKNAME\":\"anaconda\","
        + "\"ABSOLUTE_FILENAME\":\"/absoluter/dateiname\","
        + "\"USERID\":\"anaconda\","
        + "\"FILENAME\":\"dateiname\""
        + "}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  @DisplayName(value = "PRIVATECHATFILE ohne setAbsoluteFilename")
  public void encodePRIVATECHATFILE_RESPONSE_ohne_ABSOLUTE_FILENAME() throws EncodeException {
    PRIVATECHATFILE chatfile = new PRIVATECHATFILE();

    chatfile.setCommand(Command.PRIVATECHATFILE);
    chatfile.setHeader(RESPONSE);
    chatfile.setDataset(Protocol.DATASET);
    chatfile.setFilename("Top 10 Chess Endgame Principles (Crash Course).mp4");
    chatfile.setFilesize(26581987l);
    chatfile.setRemoteSessionId("1cb");
    chatfile.setUserid("m7gittqa");
    chatfile.setMessage("eine nachricht mit Anhang");
    chatfile.setNumber(29098336975848210l);
    chatfile.setRemoteNickname("gmail");

    try {
      encoder.encode(chatfile);
      fail("EncodeException nicht geworfen, setAbsoluteFilename wurde nicht gesetzt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }

  }



  @Test
  public void encodePRIVATECHATFILE_RESPONSE_OHNE_REMOTE_SESSION_ID() throws EncodeException {
    PRIVATECHATFILE chatfile = new PRIVATECHATFILE();

    chatfile.setCommand(Command.PRIVATECHATFILE);
    chatfile.setHeader(RESPONSE);
    chatfile.setDataset(Protocol.DATASET);
    chatfile.setAbsoluteFilename("/absoluter/dateiname");
    chatfile.setFilename("dateiname");
    chatfile.setUserid("anaconda");
    chatfile.setMessage("");
    chatfile.setNumber(Long.MAX_VALUE);
    chatfile.setRemoteNickname("anaconda");

    String result = encoder.encode(chatfile);

    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"PRIVATECHATFILE\","
        + "\"DATASET\":{"
        + "\"NUMBER\":9223372036854775807,"
        + "\"REMOTE_NICKNAME\":\"anaconda\","
        + "\"ABSOLUTE_FILENAME\":\"/absoluter/dateiname\","
        + "\"USERID\":\"anaconda\","
        + "\"FILENAME\":\"dateiname\","
        + "\"MESSAGE\":\"\""
        + "}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodePRIVATEMESSAGE_REQUEST() throws EncodeException {
    PRIVATEMESSAGE privatemessage = new PRIVATEMESSAGE();
    privatemessage.setCommand(Command.PRIVATEMESSAGE);
    privatemessage.setHeader(REQUEST);
    privatemessage.setDataset(Protocol.DATASET);
    privatemessage.setLocalSessionid("5");
    privatemessage.setMessage("Hallo");
    privatemessage.setRemoteSessionid("6");
    privatemessage.setDatetime(6666l);

    String result = encoder.encode(privatemessage);

    String bla = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"PRIVATEMESSAGE\","
        + "\"DATASET\":{"
        + "\"REMOTE_SESSIONID\":\"6\","
        + "\"LOCAL_SESSIONID\":\"5\","
        + "\"DATETIME\":6666,"
        + "\"MESSAGE\":\"Hallo\"}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodePRIVATEMESSAGE_REQUEST_2() throws EncodeException {
    PRIVATEMESSAGE privatemessage = new PRIVATEMESSAGE();
    privatemessage.setCommand(Command.PRIVATEMESSAGE);
    privatemessage.setHeader(REQUEST);
    privatemessage.setDataset(Protocol.DATASET);
    privatemessage.setSenderUID("louisxiv");
    privatemessage.setMessage("Hallo");
    privatemessage.setRemoteSessionid("6");
    privatemessage.setDatetime(6666l);

    String result = encoder.encode(privatemessage);

    String bla = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"PRIVATEMESSAGE\","
        + "\"DATASET\":{"
        + "\"REMOTE_SESSIONID\":\"6\","
        + "\"SENDER_UID\":\"louisxiv\","
        + "\"DATETIME\":6666,"
        + "\"MESSAGE\":\"Hallo\"}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodePRIVATEMESSAGE_REQUEST_3() throws EncodeException {
    PRIVATEMESSAGE privatemessage = new PRIVATEMESSAGE();
    privatemessage.setCommand(Command.PRIVATEMESSAGE);
    privatemessage.setHeader(REQUEST);
    privatemessage.setDataset(Protocol.DATASET);
    privatemessage.setSenderUID("louisxiv");
    privatemessage.setMessage("Hallo");
    privatemessage.setRemoteSessionid("6");
    privatemessage.setDatetime(34345l);

    String result = encoder.encode(privatemessage);

    String bla = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"PRIVATEMESSAGE\","
        + "\"DATASET\":{"
        + "\"REMOTE_SESSIONID\":\"6\","
        + "\"SENDER_UID\":\"louisxiv\","
        + "\"DATETIME\":34345,"
        + "\"MESSAGE\":\"Hallo\"}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodePRIVATEMESSAGE_REQUEST_4() throws EncodeException {
    PRIVATEMESSAGE privatemessage = new PRIVATEMESSAGE();
    privatemessage.setCommand(Command.PRIVATEMESSAGE);
    privatemessage.setHeader(REQUEST);
    privatemessage.setDataset(Protocol.DATASET);
    privatemessage.setSenderUID("louisxiv");
    privatemessage.setMessage("Hallo");
    privatemessage.setRemoteSessionid("6");

    String result = encoder.encode(privatemessage);

    String bla = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"PRIVATEMESSAGE\","
        + "\"DATASET\":{"
        + "\"REMOTE_SESSIONID\":\"6\","
        + "\"SENDER_UID\":\"louisxiv\","
        + "\"MESSAGE\":\"Hallo\"}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodePRIVATEMESSAGE_REQUEST_5() throws EncodeException {
    PRIVATEMESSAGE privatemessage = new PRIVATEMESSAGE();
    privatemessage.setCommand(Command.PRIVATEMESSAGE);
    privatemessage.setHeader(REQUEST);
    privatemessage.setDataset(Protocol.DATASET);
    privatemessage.setSenderUID("louisxiv");
    privatemessage.setMessage("Hallo");
    privatemessage.setRemoteSessionid("6");
    ChatUser value = new ChatUser();
    value.setBackgroundColor(-3);
    value.setForegroundColor(-2);
    value.setNickname("anaconda");
    value.setUserid("louisxiv");
    privatemessage.setChatUser(value);

    String result = encoder.encode(privatemessage);

    String bla = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"PRIVATEMESSAGE\","
        + "\"DATASET\":{"
        + "\"REMOTE_SESSIONID\":\"6\","
        + "\"SENDER_UID\":\"louisxiv\","
        + "\"MESSAGE\":\"Hallo\","
        + "\"CHATUSER\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"NICKNAME\":\"anaconda\","
        + "\"FOREGROUND_COLOR\":-2,"
        + "\"BACKGROUND_COLOR\":-3"

        + "}}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodePRIVATEMESSAGE_REQUEST_ATTACHMENT() throws EncodeException {
    PRIVATEMESSAGE privatemessage = new PRIVATEMESSAGE();
    privatemessage.setCommand(Command.PRIVATEMESSAGE);
    privatemessage.setHeader(REQUEST);
    privatemessage.setDataset(Protocol.DATASET);
    privatemessage.setLocalSessionid("5");
    privatemessage.setMessage("Hallo");
    privatemessage.setRemoteSessionid("6");
    privatemessage.setDatetime(6666l);
    privatemessage.setAttachment(33l);
    privatemessage.setFilename("einTest.mp3");
    privatemessage.setFilesize(8192l);

    String result = encoder.encode(privatemessage);

    String bla = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"PRIVATEMESSAGE\","
        + "\"DATASET\":{"
        + "\"REMOTE_SESSIONID\":\"6\","
        + "\"LOCAL_SESSIONID\":\"5\","
        + "\"DATETIME\":6666,"
        + "\"MESSAGE\":\"Hallo\","
        + "\"ATTACHMENT\":33,"
        + "\"FILENAME\":\"einTest.mp3\","
        + "\"FILESIZE\":8192"
        + "}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodePRIVATEMESSAGE_REQUEST_OHNE_REMOTE_SESSIONID() {
    PRIVATEMESSAGE privatemessage = new PRIVATEMESSAGE();
    privatemessage.setCommand(Command.PRIVATEMESSAGE);
    privatemessage.setHeader(REQUEST);
    privatemessage.setDataset(Protocol.DATASET);
    privatemessage.setSenderUID("4nsb1j9r");
    privatemessage.setMessage("jdij");

    ChatUser chatuser = new ChatUser();
    chatuser.setUserid("4nsb1j9r");
    chatuser.setNickname("freigeist");
    chatuser.setForegroundColor(-11653814);
    chatuser.setBackgroundColor(-6302541);
    privatemessage.setChatUser(chatuser);

    try {
      String result = encoder.encode(privatemessage);
      fail("Exception nicht geworfen");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void encodePRIVATEMESSAGE_RESPONSE() throws EncodeException {
    PRIVATEMESSAGE privatemessage = new PRIVATEMESSAGE();
    privatemessage.setCommand(Command.PRIVATEMESSAGE);
    privatemessage.setHeader(RESPONSE);
    privatemessage.setDataset(Protocol.DATASET);
    privatemessage.setLocalSessionid("5");
    privatemessage.setRemoteSessionid("6");
    privatemessage.setMessage("Hallo");
    privatemessage.setDatetime(111l);

    String result = encoder.encode(privatemessage);

    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"PRIVATEMESSAGE\","
        + "\"DATASET\":{"
        + "\"LOCAL_SESSIONID\":\"5\","
        + "\"REMOTE_SESSIONID\":\"6\","
        + "\"DATETIME\":111,"
        + "\"MESSAGE\":\"Hallo\"}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodePRIVATEMESSAGE_RESPONSE_2() throws EncodeException {
    PRIVATEMESSAGE privatemessage = new PRIVATEMESSAGE();
    privatemessage.setCommand(Command.PRIVATEMESSAGE);
    privatemessage.setHeader(RESPONSE);
    privatemessage.setDataset(Protocol.DATASET);
    privatemessage.setSenderUID("python");
    privatemessage.setMessage("Hallo");
    privatemessage.setRemoteSessionid("6");
    privatemessage.setDatetime(111l);

    String result = encoder.encode(privatemessage);

    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"PRIVATEMESSAGE\","
        + "\"DATASET\":{"
        + "\"SENDER_UID\":\"python\","
        + "\"REMOTE_SESSIONID\":\"6\","
        + "\"DATETIME\":111,"
        + "\"MESSAGE\":\"Hallo\"}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodePRIVATEMESSAGE_RESPONSE_5() throws EncodeException {
    PRIVATEMESSAGE privatemessage = new PRIVATEMESSAGE();
    privatemessage.setCommand(Command.PRIVATEMESSAGE);
    privatemessage.setHeader(RESPONSE);
    privatemessage.setDataset(Protocol.DATASET);
    privatemessage.setSenderUID("louisxiv");
    privatemessage.setMessage("Hallo");
    privatemessage.setRemoteSessionid("6");
    ChatUser value = new ChatUser();
    value.setBackgroundColor(-3);
    value.setForegroundColor(-2);
    value.setNickname("anaconda");
    value.setUserid("louisxiv");
    privatemessage.setChatUser(value);

    String result = encoder.encode(privatemessage);

    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"PRIVATEMESSAGE\","
        + "\"DATASET\":{"
        + "\"SENDER_UID\":\"louisxiv\","
        + "\"REMOTE_SESSIONID\":\"6\","
        + "\"MESSAGE\":\"Hallo\","
        + "\"CHATUSER\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"NICKNAME\":\"anaconda\","
        + "\"FOREGROUND_COLOR\":-2,"
        + "\"BACKGROUND_COLOR\":-3"

        + "}}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodePRIVATEMESSAGE_RESPONSE_ATTACHMENT() throws EncodeException {
    PRIVATEMESSAGE privatemessage = new PRIVATEMESSAGE();
    privatemessage.setCommand(Command.PRIVATEMESSAGE);
    privatemessage.setHeader(RESPONSE);
    privatemessage.setDataset(Protocol.DATASET);
    privatemessage.setLocalSessionid("5");
    privatemessage.setRemoteSessionid("6");
    privatemessage.setMessage("Hallo");
    privatemessage.setDatetime(111l);
    privatemessage.setAttachment(33l);
    privatemessage.setFilename("einTest.mp3");
    privatemessage.setFilesize(8192l);

    String result = encoder.encode(privatemessage);

    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"PRIVATEMESSAGE\","
        + "\"DATASET\":{"
        + "\"LOCAL_SESSIONID\":\"5\","
        + "\"REMOTE_SESSIONID\":\"6\","
        + "\"DATETIME\":111,"
        + "\"MESSAGE\":\"Hallo\","
        + "\"ATTACHMENT\":33,"
        + "\"FILENAME\":\"einTest.mp3\","
        + "\"FILESIZE\":8192"
        + "}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodePROYECTORCLOSING_REQUEST() throws EncodeException {
    PROYECTORCLOSING proyector = new PROYECTORCLOSING();
    proyector.setCommand(Command.PROYECTORCLOSING);
    proyector.setHeader(REQUEST);
    proyector.setDataset(Protocol.DATASET);
    proyector.setUserid("louisxiv");

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"PROYECTORCLOSING\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\""
        + "}}";

    String result = encoder.encode(proyector);
    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodePROYECTORCLOSING_RESPONSE() throws EncodeException {
    PROYECTORCLOSING proyector = new PROYECTORCLOSING();
    proyector.setCommand(Command.PROYECTORCLOSING);
    proyector.setHeader(RESPONSE);
    proyector.setDataset(Protocol.DATASET);
    proyector.setUserid("louisxiv");

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"PROYECTORCLOSING\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\""
        + "}}";

    String result = encoder.encode(proyector);
    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encoderCHATONLINELIST_REQUEST() throws EncodeException {
    CHATONLINELIST chatonline = new CHATONLINELIST();
    chatonline.setCommand(Command.CHATONLINELIST);
    chatonline.setHeader(REQUEST);
    chatonline.setDataset(Protocol.DATASET);
    ChatOnline[] value = new ChatOnline[3];
    value[0] = new ChatOnline();
    value[0].setOnline(1);
    value[0].setRoom("Nexus");

    value[1] = new ChatOnline();
    value[1].setOnline(2);
    value[1].setRoom("Everyone");

    value[2] = new ChatOnline();
    value[2].setOnline(2);
    value[2].setRoom("Developer");

    chatonline.setChatOnline(value);

    String expected = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"CHATONLINELIST\","
        + "\"DATASET\":{\"CHATONLINELIST\":["
        + "{\"ROOM\":\"Nexus\",\"ONLINE\":1},"
        + "{\"ROOM\":\"Everyone\",\"ONLINE\":2},"
        + "{\"ROOM\":\"Developer\",\"ONLINE\":2}]}}";

    String result = encoder.encode(chatonline);

    assertEquals(expected, result, "enterroom,request");

  }



  @Test
  public void encoderDELETEUSER_CONFIRM() throws EncodeException {
    DELETEUSER deleteuser = new DELETEUSER();
    deleteuser.setCommand(Command.DELETE_USER);
    deleteuser.setHeader(CONFIRM);
    deleteuser.setDataset(Protocol.DATASET);
    deleteuser.setUserid("louisxiv");
    String result = encoder.encode(deleteuser);

    String bla = "{\"HEADER\":\"CONFIRM\"," + "\"COMMAND\":\"DELETE_USER\"" + "}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encoderDELETEUSER_ERROR() throws EncodeException {
    DELETEUSER deleteuser = new DELETEUSER();
    deleteuser.setCommand(Command.DELETE_USER);
    deleteuser.setHeader(ERROR);
    deleteuser.setDataset(Protocol.DATASET);
    deleteuser.setUserid("louisxiv");
    String result = encoder.encode(deleteuser);

    String bla = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"DELETE_USER\"" + "}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encoderDELETEUSER_REQUEST() throws EncodeException {
    DELETEUSER deleteuser = new DELETEUSER();
    deleteuser.setCommand(Command.DELETE_USER);
    deleteuser.setHeader(REQUEST);
    deleteuser.setDataset(Protocol.DATASET);
    deleteuser.setUserid("louisxiv");
    String result = encoder.encode(deleteuser);

    String bla = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"DELETE_USER\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\""
        + "}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeREADROOMS_REQUEST() throws EncodeException {
    READROOMS rooms = new READROOMS();
    rooms.setCommand(Command.READROOMS);
    rooms.setHeader(REQUEST);
    rooms.setDataset(Protocol.DATASET);
    rooms.setUserid("louisxiv");
    rooms.setRoomtype(Roomtype.GRUPPENRAUM);

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"READROOMS\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"ROOMTYPE\":4"
        + "}}";

    String result = encoder.encode(rooms);
    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeREADROOMS_REQUEST_ohne_ROOMTYPE() {
    READROOMS rooms = new READROOMS();
    rooms.setCommand(Command.READROOMS);
    rooms.setHeader(REQUEST);
    rooms.setDataset(Protocol.DATASET);
    rooms.setUserid("louisxiv");
//    rooms.setRoomtype(Roomtype.GRUPPENRAUM);

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"READROOMS\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\""
        + "}}";

    try {
      String result = encoder.encode(rooms);
      fail("Exception nicht geworfen");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void encodeREADROOMS_RESPONSE() throws EncodeException {
    READROOMS rooms = new READROOMS();
    rooms.setCommand(Command.READROOMS);
    rooms.setHeader(RESPONSE);
    rooms.setDataset(Protocol.DATASET);
    rooms.setUserid("louisxiv");
    rooms.setRoomtype(Roomtype.GRUPPENRAUM);
    rooms.setRooms(new String[] {"schach", "radsport"});

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"READROOMS\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"ROOMTYPE\":4,"
        + "\"ROOMS\":[\"schach\",\"radsport\"]"
        + "}}";

    String result = encoder.encode(rooms);
    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeREADROOMS_RESPONSE_EMPTY() throws EncodeException {
    READROOMS rooms = new READROOMS();
    rooms.setCommand(Command.READROOMS);
    rooms.setHeader(RESPONSE);
    rooms.setDataset(Protocol.DATASET);
    rooms.setUserid("louisxiv");
    rooms.setRoomtype(Roomtype.GRUPPENRAUM);
    rooms.setRooms(new String[] {});

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"READROOMS\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"ROOMTYPE\":4,"
        + "\"ROOMS\":[]"
        + "}}";

    String result = encoder.encode(rooms);
    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeREADTOPICROOMOWNER_REQUEST_OWNER_USERID() throws EncodeException {
    READTOPICROOMOWNER roomlist = new READTOPICROOMOWNER();
    roomlist.setCommand(Command.READTOPICROOMOWNER);
    roomlist.setHeader(REQUEST);
    roomlist.setDataset(Protocol.DATASET);
    roomlist.setUserid("12345678");
    roomlist.setOwner("viper");
    String result = encoder.encode(roomlist);
    String expected = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"READTOPICROOMOWNER\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"OWNER\":\"viper\""
        + "}}";

    assertEquals(expected, result, "roomlist,request");
  }



  @Test
  public void encodeREADTOPICROOMOWNER_RESPONSE() throws EncodeException {
    READTOPICROOMOWNER roomlist = new READTOPICROOMOWNER();
    roomlist.setCommand(Command.READTOPICROOMOWNER);
    roomlist.setHeader(RESPONSE);
    roomlist.setDataset(Protocol.DATASET);
    roomlist.setOwner("viper");
    roomlist.setUserid("12345678");
    String result = encoder.encode(roomlist);
    String expected = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"READTOPICROOMOWNER\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"OWNER\":\"viper\","
        + "\"ROOMLIST\":[]"
        + "}}";

    assertEquals(expected, result, "roomlist,request");
  }



  @Test
  public void encodeREADTOPICROOMOWNER_RESPONSE_ROOMLIST() throws EncodeException {
    READTOPICROOMOWNER roomlist = new READTOPICROOMOWNER();
    roomlist.setCommand(Command.READTOPICROOMOWNER);
    roomlist.setHeader(RESPONSE);
    roomlist.setDataset(Protocol.DATASET);
    roomlist.setOwner("viper");
    roomlist.setUserid("12345678");
    Room[] value = new Room[1];
    value[0] = new Room();
    value[0].setChatid("JFZI");
    value[0].setEntry(Entry.PROTECTED);
    value[0].setLifetime(Lifetime.PERMANENT);
    value[0].setOwner("viper");
    roomlist.setRoom(value);
    String result = encoder.encode(roomlist);
    String expected = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"READTOPICROOMOWNER\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"OWNER\":\"viper\","
        + "\"ROOMLIST\":"
        + "["
        + "{\"ROOM\":\"JFZI\",\"ENTRY\":\"PROTECTED\",\"LIFETIME\":\"PERMANENT\",\"OWNER\":\"viper\"}"
        + "]"
        + "}}";

    assertEquals(expected, result, "roomlist,request");
  }



  @Test
  public void encodeROOMLIST_REQUEST() throws EncodeException {
    ROOMLIST roomlist = new ROOMLIST();
    roomlist.setCommand(Command.ROOMLIST);
    roomlist.setHeader(REQUEST);
    String result = encoder.encode(roomlist);
    String expected = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"ROOMLIST\"}";

    assertEquals(expected, result, "roomlist,request");
  }



  @Test
  public void encoderROOMLIST_REQUEST() throws EncodeException {
    ROOMLIST roomlist = new ROOMLIST();
    roomlist.setCommand(Command.ROOMLIST);
    roomlist.setHeader(REQUEST);
    String result = encoder.encode(roomlist);
  }



  @Test
  public void encoderROOMLIST_RESPONSE() throws EncodeException {
    ROOMLIST roomlist = new ROOMLIST();
    roomlist.setCommand(Command.ROOMLIST);
    roomlist.setHeader(RESPONSE);
    roomlist.setDataset(Protocol.DATASET);

    Room[] room = new Room[2];
    room[0] = new Room();
    room[0].setChatid("Everyone");
    room[0].setEntry(Entry.PUBLIC);
    room[0].setLifetime(Lifetime.PERMANENT);
    room[0].setRoomtype(Roomtype.FORUM);

    room[1] = new Room();
    room[1].setChatid("Developer");
    room[1].setEntry(Entry.PROTECTED);
    room[1].setLifetime(Lifetime.PERMANENT);
    room[1].setRoomtype(Roomtype.BESPRECHUNGSRAUM);
    roomlist.setRoom(room);
    String result = encoder.encode(roomlist);

    log.info(roomlist.toString());

    String expected = "{\"HEADER\":\"RESPONSE\",\"COMMAND\":\"ROOMLIST\"," + "\"DATASET\":{\"ROOMLIST\":"
        + "[{\"ROOM\":\"Everyone\","
        + "\"ROOMTYPE\":\"FORUM\","
        + "\"ENTRY\":\"PUBLIC\","
        + "\"LIFETIME\":\"PERMANENT\"},"
        + "{\"ROOM\":\"Developer\","
        + "\"ROOMTYPE\":\"BESPRECHUNGSRAUM\","
        + "\"ENTRY\":\"PROTECTED\","
        + "\"LIFETIME\":\"PERMANENT\"}]}}";

    assertEquals(expected, result, "RESPONSE");
    assertEquals(roomlist.toString(), result, "RESPONSE");
  }



  @Test
  public void encoderROOMLIST_RESPONSE_CHATID_FEHLT() {
    ROOMLIST roomlist = new ROOMLIST();
    roomlist.setCommand(Command.ROOMLIST);
    roomlist.setHeader(RESPONSE);
    roomlist.setDataset(Protocol.DATASET);

    Room[] room = new Room[2];
    room[0] = new Room();
    room[0].setChatid("Everyone");
    room[0].setEntry(Entry.PUBLIC);
    room[0].setLifetime(Lifetime.PERMANENT);
    room[0].setRoomtype(Roomtype.FORUM);

    room[1] = new Room();
    room[1].setEntry(Entry.PROTECTED);
    room[1].setLifetime(Lifetime.PERMANENT);
    room[1].setRoomtype(Roomtype.BESPRECHUNGSRAUM);
    roomlist.setRoom(room);
    try {
      encoder.encode(roomlist);
      fail("Exception nicht geworfen, CHATID fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }

  }



  @Test
  public void encoderROOMLIST_RESPONSE_kein_ROOMS() throws EncodeException {
    ROOMLIST roomlist = new ROOMLIST();
    roomlist.setCommand(Command.ROOMLIST);
    roomlist.setHeader(RESPONSE);
    roomlist.setDataset(Protocol.DATASET);

    String result = encoder.encode(roomlist);

    String expected = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"ROOMLIST\","
        + "\"DATASET\":{\"ROOMLIST\":["
        + "]}}";

    assertEquals(expected, result, "RESPONSE");
  }



  @Test
  @DisplayName(value = "UPDATEUSER | CONFIRM")
  public void encoderUPDATEUSER_CONFIRM() throws EncodeException {
    UPDATEUSER updateuser = new UPDATEUSER();
    updateuser.setCommand(Command.UPDATEUSER);
    updateuser.setHeader(CONFIRM);
    updateuser.setDataset(Protocol.DATASET);
    updateuser.setBackgroundColor(-1);
    updateuser.setForegroundColor(-2);
    updateuser.setNickname("louisxiv");
    updateuser.setIdentity("deij");

    String result = encoder.encode(updateuser);
    String encoded = "{\"HEADER\":\"CONFIRM\"," + "\"COMMAND\":\"UPDATEUSER\","
        + "\"DATASET\":{"
        + "\"IDENTITY\":\"deij\","
        + "\"FOREGROUND_COLOR\":-2,"
        + "\"BACKGROUND_COLOR\":-1,"
        + "\"NICKNAME\":\"louisxiv\""
        + "}}";

    assertEquals(encoded, result, "JSON");
  }



  @Test
  @DisplayName(value = "UPDATEUSER | CONFIRM | BACKGROUND_COLOR | FEHLT")
  public void encoderUPDATEUSER_CONFIRM_BACKGROUND_COLOR_FEHLT() {
    UPDATEUSER updateuser = new UPDATEUSER();
    updateuser.setCommand(Command.UPDATEUSER);
    updateuser.setHeader(CONFIRM);
    updateuser.setDataset(Protocol.DATASET);
    updateuser.setForegroundColor(-2);
    updateuser.setNickname("louisxiv");
    updateuser.setIdentity("deij");

    try {
      encoder.encode(updateuser);
      fail("Exception nicht geworfen, BACKGROUND_COLOR fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  @DisplayName(value = "UPDATEUSER | CONFIRM | FOREGROUND_COLOR | FEHLT")
  public void encoderUPDATEUSER_CONFIRM_FOREGROUND_COLOR_FEHLT() {
    UPDATEUSER updateuser = new UPDATEUSER();
    updateuser.setCommand(Command.UPDATEUSER);
    updateuser.setHeader(CONFIRM);
    updateuser.setDataset(Protocol.DATASET);
    updateuser.setBackgroundColor(-1);
    updateuser.setNickname("louisxiv");
    updateuser.setIdentity("deij");

    try {
      encoder.encode(updateuser);
      fail("Exception nicht geworfen, FOREGROUND_COLOR fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  @DisplayName(value = "UPDATEUSER | CONFIRM | IDENTITY | FEHLT")
  public void encoderUPDATEUSER_CONFIRM_IDENTITY_FEHLT() {
    UPDATEUSER updateuser = new UPDATEUSER();
    updateuser.setCommand(Command.UPDATEUSER);
    updateuser.setHeader(CONFIRM);
    updateuser.setDataset(Protocol.DATASET);
    updateuser.setBackgroundColor(-1);
    updateuser.setForegroundColor(-2);
    updateuser.setNickname("louisxiv");

    try {
      encoder.encode(updateuser);
      fail("Exception nicht geworfen, IDENTITY fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  @DisplayName(value = "UPDATEUSER | CONFIRM | NICKNAME | FEHLT")
  public void encoderUPDATEUSER_CONFIRM_NICKNAME_FEHLT() {
    UPDATEUSER updateuser = new UPDATEUSER();
    updateuser.setCommand(Command.UPDATEUSER);
    updateuser.setHeader(CONFIRM);
    updateuser.setDataset(Protocol.DATASET);
    updateuser.setBackgroundColor(-1);
    updateuser.setForegroundColor(-2);
    updateuser.setIdentity("deij");

    try {
      encoder.encode(updateuser);
      fail("Exception nicht geworfen, NICKNAME fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void encoderUPDATEUSER_ERROR() throws EncodeException {
    UPDATEUSER updateuser = new UPDATEUSER();
    updateuser.setCommand(Command.UPDATEUSER);
    updateuser.setHeader(ERROR);
    updateuser.setDataset(Protocol.DATASET);
    updateuser.setText("Hallo Freunde");
    String result = encoder.encode(updateuser);

    String encoded = "{\"HEADER\":\"ERROR\",\"COMMAND\":\"UPDATEUSER\",\"DATASET\":{"
        + "\"TEXT\":\"Hallo Freunde\""
        + "}}";//
    assertEquals(encoded, result, "JSON");
  }



  @Test
  @DisplayName(value = "UPDATEUSER | ERROR | TEXT | FEHLT ")
  public void encoderUPDATEUSER_ERROR_TEXT_FEHLT() {
    UPDATEUSER updateuser = new UPDATEUSER();
    updateuser.setCommand(Command.UPDATEUSER);
    updateuser.setHeader(ERROR);
    updateuser.setDataset(Protocol.DATASET);
    try {
      encoder.encode(updateuser);
      fail("Exception nicht geworfen, TEXT fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }

  }



  @Test
  @DisplayName(value = "USRLOGIN | CONFIRM")
  public void encoderUSRLOGIN_CONFIRM() throws EncodeException, IOException {
    USRLOGIN login = new USRLOGIN();
    login.setCommand(Command.USRLOGIN);
    login.setHeader(CONFIRM);
    login.setDataset(Protocol.DATASET);

    login.setForegroundColor(-16777216);
    login.setBackgroundColor(-1);
    login.setNickname("anaconda");
    login.setSession("bc");
    login.setIdentity("jeidj");
    login.setOncall(true);
    login.setVolume(100);
    String result = encoder.encodeUSRLOGIN(login);

    String bla = "{\"HEADER\":\"CONFIRM\"," + "\"COMMAND\":\"USRLOGIN\","
        + "\"DATASET\":"
        + "{\"SESSION\":\"bc\","
        + "\"IDENTITY\":\"jeidj\","
        + "\"FOREGROUND_COLOR\":-16777216,"
        + "\"BACKGROUND_COLOR\":-1,"
        + "\"ONCALL\":true,"
        + "\"VOLUME\":100,"
        + "\"NICKNAME\":\"anaconda\"}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  @DisplayName(value = "CONFIRM | IDENTITY | FEHLT")
  public void encoderUSRLOGIN_CONFIRM_IDENTITY_FEHLT() {
    USRLOGIN login = new USRLOGIN();
    login.setCommand(Command.USRLOGIN);
    login.setHeader(CONFIRM);
    login.setDataset(Protocol.DATASET);

    login.setForegroundColor(-16777216);
    login.setBackgroundColor(-1);
    login.setNickname("anaconda");
    login.setSession("da");
    login.setOncall(false);

    try {
      encoder.encodeUSRLOGIN(login);
      fail("Exception nicht geworfen, IDENTITY fehlt");
    }
    catch (EncodeException | IOException e) {
      assertTrue(true);
    }

  }



  @Test
  @DisplayName(value = "USRLOGIN | CONFIRM | NICKNAME | FEHLT")
  public void encoderUSRLOGIN_CONFIRM_NICKNAME_FEHLT() {
    USRLOGIN login = new USRLOGIN();
    login.setCommand(Command.USRLOGIN);
    login.setHeader(CONFIRM);
    login.setDataset(Protocol.DATASET);

    login.setForegroundColor(-16777216);
    login.setBackgroundColor(-1);
    login.setSession("bc");
    login.setIdentity("jeidj");
    login.setOncall(true);
    login.setVolume(100);
    try {
      encoder.encodeUSRLOGIN(login);
      fail("Exception nicht geworfen, NICKNAME fehlt");
    }
    catch (EncodeException | IOException e) {
      assertTrue(true);
    }

  }



  @Test
  @DisplayName(value = "USRLOGIN | CONFRIM | ONCALL fehlt")
  public void encoderUSRLOGIN_CONFIRM_ONCALL_FEHLT() {
    USRLOGIN login = new USRLOGIN();
    login.setCommand(Command.USRLOGIN);
    login.setHeader(CONFIRM);
    login.setDataset(Protocol.DATASET);

    login.setForegroundColor(-16777216);
    login.setBackgroundColor(-1);
    login.setNickname("anaconda");
    login.setSession("bc");
    login.setIdentity("jeidj");
    login.setVolume(68);
    try {
      encoder.encodeUSRLOGIN(login);
      fail("Exception, ONCALL fehlt");
    }
    catch (EncodeException | IOException e) {
      assertTrue(true);
    }

  }



  @Test
  @DisplayName(value = "SCREENCAST REQUEST EncodeException")
  public void encodeSCREENCAST_EncodeException() throws EncodeException {
    SCREENCAST screencast = new SCREENCAST();
    screencast.setCommand(Command.SCREENCAST);
    screencast.setHeader(REQUEST);
    screencast.setDataset(Protocol.DATASET);

    try {
      encoder.encode(screencast);
      fail("EncodeException wurde nicht geworfen");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void encodeSCREENCAST_ERROR() throws EncodeException {
    SCREENCAST screencast = new SCREENCAST();
    screencast.setCommand(Command.SCREENCAST);
    screencast.setHeader(ERROR);
    screencast.setDataset(Protocol.DATASET);
    screencast.setReceiverNickname("freigeist");
    screencast.setErrorMessage("freigeist ist nicht mehr online.");

    String result = encoder.encode(screencast);
    String expected = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"SCREENCAST\","
        + "\"DATASET\":"
        + "{\"RECEIVER_NICKNAME\":\"freigeist\","
        + "\"ERRORMESSAGE\":\"freigeist ist nicht mehr online.\""
        + "}}";

    assertEquals(expected, result, "SCREENCAST_ERROR");
  }



  @Test
  public void encodeSCREENCAST_ERROR_RECEIVERUSERID() throws EncodeException {
    SCREENCAST screencast = new SCREENCAST();
    screencast.setCommand(Command.SCREENCAST);
    screencast.setHeader(ERROR);
    screencast.setDataset(Protocol.DATASET);
    screencast.setReceiverNickname("freigeist");
    screencast.setReceiverUserid("admin678");
    screencast.setErrorMessage("freigeist ist nicht mehr online.");

    String result = encoder.encode(screencast);
    String expected = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"SCREENCAST\","
        + "\"DATASET\":"
        + "{\"RECEIVER_NICKNAME\":\"freigeist\","
        + "\"RECEIVER_USERID\":\"admin678\","
        + "\"ERRORMESSAGE\":\"freigeist ist nicht mehr online.\""
        + "}}";

    assertEquals(expected, result, "SCREENCAST_ERROR");
  }



  @Test
  @DisplayName(value = "SCREENCAST REQUEST bis Server")
  public void encodeSCREENCAST_REQUEST() throws EncodeException {
    SCREENCAST screencast = new SCREENCAST();
    screencast.setCommand(Command.SCREENCAST);
    screencast.setHeader(REQUEST);
    screencast.setDataset(Protocol.DATASET);
    screencast.setCallerUserid("admin678");
    screencast.setCallerNickname("anaconda");
    screencast.setCallerIp("192.168.7.30");
    screencast.setCallerPort(29472);
    screencast.setReceiverNickname("freigeist");

    String result = encoder.encode(screencast);
    String expected = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"SCREENCAST\","
        + "\"DATASET\":"
        + "{\"CALLER_USERID\":\"admin678\","
        + "\"CALLER_NICKNAME\":\"anaconda\","
        + "\"CALLER_IP\":\"192.168.7.30\","
        + "\"CALLER_PORT\":29472,"
        + "\"RECEIVER_NICKNAME\":\"freigeist\""
        + "}}";

    assertEquals(expected, result, "SCREENCAST_REQUEST");
  }



  @Test
  @DisplayName(value = "SCREENCAST RESPONSE")
  public void encodeSCREENCAST_RESPONSE() throws EncodeException {
    SCREENCAST screencast = new SCREENCAST();
    screencast.setCommand(Command.SCREENCAST);
    screencast.setHeader(RESPONSE);
    screencast.setDataset(Protocol.DATASET);
    screencast.setReceiverUserid("admin678");
    screencast.setReceiverNickname("anaconda");
    screencast.setCallerUserid("caller78");

    String result = encoder.encode(screencast);
    String expected = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"SCREENCAST\","
        + "\"DATASET\":"
        + "{\"CALLER_USERID\":\"caller78\","
        + "\"RECEIVER_USERID\":\"admin678\","
        + "\"RECEIVER_NICKNAME\":\"anaconda\""
        + "}}";

    assertEquals(expected, result, "SCREENCAST_RESPONSE");
  }



  @Test
  public void encodeSEARCHFILES_REQUEST() throws EncodeException {
    SEARCHFILES searchfiles = new SEARCHFILES();
    searchfiles.setCommand(Command.SEARCHFILES);
    searchfiles.setHeader(REQUEST);
    searchfiles.setDataset(Protocol.DATASET);
    searchfiles.setUserid("louisxiv");
    searchfiles.setPatterns(new String[] {"abc.123", "hallo.456"});

    String result = encoder.encode(searchfiles);

    String bla = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"SEARCHFILES\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"SEARCHFILES\":["
        + "{\"FILE\":\"abc.123\"},"
        + "{\"FILE\":\"hallo.456\"}]}}";
    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeSEARCHFILES_REQUEST_2() throws EncodeException {
    SEARCHFILES searchfiles = new SEARCHFILES();
    searchfiles.setCommand(Command.SEARCHFILES);
    searchfiles.setHeader(REQUEST);
    searchfiles.setDataset(Protocol.DATASET);
    searchfiles.setUserid("louisxiv");
    searchfiles.setPatterns(new String[] {});

    String result = encoder.encode(searchfiles);

    String bla = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"SEARCHFILES\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"SEARCHFILES\":["
        + "]}}";
    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeSEARCHFILES_RESPONSE() throws EncodeException {
    SEARCHFILES searchfiles = new SEARCHFILES();
    searchfiles.setCommand(Command.SEARCHFILES);
    searchfiles.setHeader(RESPONSE);
    searchfiles.setDataset(Protocol.DATASET);
    searchfiles.setUserid("louisxiv");
    searchfiles.setStarted(true);

    Searchfile[] files = new Searchfile[2];
    files[0] = new Searchfile();
    files[0].setIndex(1);
    files[0].setFilename("abc.123");
    files[0].setFilesize(999);
    searchfiles.setSearchfiles(files);

    files[1] = new Searchfile();
    files[1].setIndex(2);
    files[1].setFilename("hallo.456");
    files[1].setFilesize(1234);
    searchfiles.setSearchfiles(files);

    String result = encoder.encode(searchfiles);

    String encode = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"SEARCHFILES\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"START\":true,"
        + "\"SEARCHFILE\":["
        + "{\"INDEX\":1,"
        + "\"FILENAME\":\"abc.123\","
        + "\"FILESIZE\":999},"
        + "{\"INDEX\":2,"
        + "\"FILENAME\":\"hallo.456\","
        + "\"FILESIZE\":1234}"
        + "]}}";

    assertEquals(encode, result, "JSON");
  }



  @Test
  public void encodeSEARCHFILES_RESPONSE_2() throws EncodeException {
    SEARCHFILES searchfiles = new SEARCHFILES();
    searchfiles.setCommand(Command.SEARCHFILES);
    searchfiles.setHeader(RESPONSE);
    searchfiles.setDataset(Protocol.DATASET);
    searchfiles.setUserid("louisxiv");
    searchfiles.setStarted(true);

    Searchfile[] files = new Searchfile[0];

    String result = encoder.encode(searchfiles);

    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"SEARCHFILES\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"START\":true,"
        + "\"SEARCHFILE\":["

        + "]}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeSEARCHFILES_RESPONSE_2_JERSEY() throws EncodeException {
    SEARCHFILES searchfiles = new SEARCHFILES();
    searchfiles.setCommand(Command.SEARCHFILES);
    searchfiles.setHeader(RESPONSE);
    searchfiles.setDataset(Protocol.DATASET);
    searchfiles.setUserid("louisxiv");
    searchfiles.setStarted(true);

    Searchfile[] files = new Searchfile[0];

    String result = searchfiles.toString();

    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"SEARCHFILES\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"START\":true,"
        + "\"SEARCHFILE\":["

        + "]}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeSEARCHFILES_RESPONSE_JERSEY() {
    SEARCHFILES searchfiles = new SEARCHFILES();
    searchfiles.setCommand(Command.SEARCHFILES);
    searchfiles.setHeader(RESPONSE);
    searchfiles.setDataset(Protocol.DATASET);
    searchfiles.setUserid("louisxiv");
    searchfiles.setStarted(true);

    Searchfile[] files = new Searchfile[2];
    files[0] = new Searchfile();
    files[0].setIndex(1);
    files[0].setFilename("abc.123");
    files[0].setFilesize(999);
    searchfiles.setSearchfiles(files);

    files[1] = new Searchfile();
    files[1].setIndex(2);
    files[1].setFilename("hallo.456");
    files[1].setFilesize(1234);
    searchfiles.setSearchfiles(files);

    String result = searchfiles.toString();

    String encode = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"SEARCHFILES\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"START\":true,"
        + "\"SEARCHFILE\":["
        + "{\"INDEX\":1,"
        + "\"FILENAME\":\"abc.123\","
        + "\"FILESIZE\":999},"
        + "{\"INDEX\":2,"
        + "\"FILENAME\":\"hallo.456\","
        + "\"FILESIZE\":1234}"
        + "]}}";

    assertEquals(encode, result, "JSON");
  }



  @Test
  public void encodeSEARCHFILES_RESPONSE_OHNE_USERID_JERSEY() throws EncodeException {
    SEARCHFILES searchfiles = new SEARCHFILES();
    searchfiles.setCommand(Command.SEARCHFILES);
    searchfiles.setHeader(RESPONSE);
    searchfiles.setDataset(Protocol.DATASET);
    searchfiles.setStarted(true);

    String result = searchfiles.toString();

    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"SEARCHFILES\","
        + "\"DATASET\":{"
        + "\"START\":true,"
        + "\"SEARCHFILE\":["

        + "]}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  @DisplayName("SIGNIN CONFIRM_PASSWORD JERSEY")
  public void encodeSIGNIN_CONFIRM_MULTILINGUAL_JERSEY() {
    SIGNIN login = new SIGNIN();
    login.setCommand(Command.SIGNIN);
    login.setHeader(CONFIRM);
    login.setDataset(Protocol.DATASET);

    login.setUserid("louisxiv");
    login.setEmail("llange@javacomm.net");
    login.setPassword("PASSWORD");
    login.setForegroundColor(-16777216);
    login.setBackgroundColor(-1);
    login.setNickname("anaconda");
    login.setMultilingualkey(KEY.SERVER_CONFIRM_PASSWORT);
    String result = login.toString();

    String encoded = "{\"HEADER\":\"CONFIRM\"," + "\"COMMAND\":\"SIGNIN\","
        + "\"DATASET\":"
        + "{\"EMAIL\":\"llange@javacomm.net\","
        + "\"USERID\":\"louisxiv\","
        + "\"NICKNAME\":\"anaconda\","
        + "\"PASSWORD\":\"PASSWORD\","
        + "\"FOREGROUND_COLOR\":-16777216,"
        + "\"BACKGROUND_COLOR\":-1,"
        + "\"MULTILINGUALKEY\":\"Server_confirm_Passwort\""
        + "}}";

    assertEquals(encoded, result, "JSON");
  }



  @Test
  @DisplayName(value = "SIGNIN | CONFIRM | MULTILINGUALKEY | FEHLT")
  public void encodeSIGNIN_CONFIRM_MULTILINGUALKEY_FEHLT() throws IOException {
    SIGNIN login = new SIGNIN();
    login.setCommand(Command.SIGNIN);
    login.setHeader(CONFIRM);
    login.setDataset(Protocol.DATASET);

    login.setUserid("louisxiv");
    login.setEmail("llange@javacomm.net");
    login.setPassword("PASSWORD");
    login.setForegroundColor(-16777216);
    login.setBackgroundColor(-1);
    login.setNickname("anaconda");
    try {
      encoder.encodeSIGNIN(login);
      fail("EncodeException nicht geworfen, MULTILINGUALKEY fehlt");
    }
    catch (EncodeException e) {}

  }



  @Test
  @DisplayName(value = "SIGNIN | CONFIRM | NICKNAME | FEHLT")
  public void encodeSIGNIN_CONFIRM_NICKNAME_FEHLT() {
    SIGNIN login = new SIGNIN();
    login.setCommand(Command.SIGNIN);
    login.setHeader(CONFIRM);
    login.setDataset(Protocol.DATASET);

    login.setUserid("louisxiv");
    login.setEmail("llange@javacomm.net");
    login.setPassword("PASSWORD");
    login.setForegroundColor(-16777216);
    login.setBackgroundColor(-1);
    login.setMultilingualkey(KEY.SERVER_CONFIRM_PASSWORT);
    login.setText(
        "An e-mail with a password is on the way. Check your inbox folder. If nothing has arrived, check the trash. If you click on 'Next' in the next step, you will be logged in automatically. Next, change your password via the menu 'Account/Edit'."
    );

    try {
      encoder.encodeSIGNIN(login);
      fail("Exception nicht geworfen, NICKNAME fehlt");
    }
    catch (EncodeException | IOException e) {
      assertTrue(true);
    }

  }



  @Test
  @DisplayName("SIGNIN CONFIRM | PASSWORD | FEHLT")
  public void encodeSIGNIN_CONFIRM_PASSWORD_FEHLT() {
    SIGNIN login = new SIGNIN();
    login.setCommand(Command.SIGNIN);
    login.setHeader(CONFIRM);
    login.setDataset(Protocol.DATASET);

    login.setUserid("louisxiv");
    login.setEmail("llange@javacomm.net");
    login.setForegroundColor(-16777216);
    login.setBackgroundColor(-1);
    login.setNickname("anaconda");
    login.setMultilingualkey(KEY.SERVER_WEBCONFIRM_PASSWORT);
    login.setText(
        "An e-mail with a password is on the way. Check your inbox folder. If nothing has arrived, check the trash. If you click on 'Next' in the next step, you will be logged in automatically. Next, change your password via the menu 'Account/Edit'."
    );

    try {
      encoder.encodeSIGNIN(login);
      fail("Exception geworfen, PASSWORD fehlt");
    }
    catch (EncodeException | IOException e) {
      assertTrue(true);
    }

  }



  @Test
  @DisplayName("SIGIN | CONFIRM")
  public void encodeSIGNIN_CONFIRM_TEXT() throws EncodeException, IOException {
    SIGNIN login = new SIGNIN();
    login.setCommand(Command.SIGNIN);
    login.setHeader(CONFIRM);
    login.setDataset(Protocol.DATASET);

    login.setUserid("louisxiv");
    login.setEmail("llange@javacomm.net");
    login.setPassword("PASSWORD");
    login.setForegroundColor(-16777216);
    login.setBackgroundColor(-1);
    login.setNickname("anaconda");
    login.setMultilingualkey(KEY.SERVER_WEBCONFIRM_PASSWORT);
    login.setText(
        "An e-mail with a password is on the way. Check your inbox folder. If nothing has arrived, check the trash. If you click on 'Next' in the next step, you will be logged in automatically. Next, change your password via the menu 'Account/Edit'."
    );
    String result = encoder.encodeSIGNIN(login);

    String bla = "{\"HEADER\":\"CONFIRM\"," + "\"COMMAND\":\"SIGNIN\","
        + "\"DATASET\":"
        + "{\"EMAIL\":\"llange@javacomm.net\","
        + "\"USERID\":\"louisxiv\","
        + "\"PASSWORD\":\"PASSWORD\","
        + "\"FOREGROUND_COLOR\":-16777216,"
        + "\"BACKGROUND_COLOR\":-1,"
        + "\"NICKNAME\":\"anaconda\","
        + "\"MULTILINGUALKEY\":\"Server_webconfirm_Passwort\","
        + "\"TEXT\":\"An e-mail with a password is on the way. Check your inbox folder. If nothing has arrived, check the trash. If you click on 'Next' in the next step, you will be logged in automatically. Next, change your password via the menu 'Account/Edit'.\""
        + "}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  @DisplayName(value = "SIGNIN | CONFIRM | USERID | FEHLT")
  public void encodeSIGNIN_CONFIRM_USERID_FEHLT() {
    SIGNIN login = new SIGNIN();
    login.setCommand(Command.SIGNIN);
    login.setHeader(CONFIRM);
    login.setDataset(Protocol.DATASET);

    login.setEmail("llange@javacomm.net");
    login.setPassword("PASSWORD");
    login.setForegroundColor(-16777216);
    login.setBackgroundColor(-1);
    login.setNickname("anaconda");
    login.setMultilingualkey(KEY.SERVER_WEBCONFIRM_PASSWORT);
    login.setText(
        "An e-mail with a password is on the way. Check your inbox folder. If nothing has arrived, check the trash. If you click on 'Next' in the next step, you will be logged in automatically. Next, change your password via the menu 'Account/Edit'."
    );
    try {
      encoder.encodeSIGNIN(login);
      fail("Exception nicht geworfen, USERID fehlt");
    }
    catch (EncodeException | IOException e) {
      assertTrue(true);
    }

  }



  @Test
  @DisplayName(value = "SIGNIN | ERROR")
  public void encodeSIGNIN_ERROR() throws EncodeException {
    SIGNIN signin = new SIGNIN();
    signin.setCommand(Command.SIGNIN);
    signin.setHeader(ERROR);
    signin.setDataset(Protocol.DATASET);
    signin.setEmail("llange@javacomm.net");
    signin.setErrorCode(33);
    signin.setErrorMessage("123");
    signin.setText("User could not be added.");
    signin.setMultilingualkey(KEY.SERVER_EMAIL_INVALID);
    String result = signin.toString();

    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"SIGNIN\","
        + "\"DATASET\":{"
        + "\"EMAIL\":\"llange@javacomm.net\","
        + "\"ERRORCODE\":33,"
        + "\"ERRORMESSAGE\":\"123\","
        + "\"TEXT\":\"User could not be added.\","
        + "\"MULTILINGUALKEY\":"
        + "\"Server_Email_invalid\""
        + "}}";

    encoder.encode(signin);

    assertEquals(encoded, result, "JSON");
  }



  @Test
  @DisplayName(value = "SIGNIN | ERROR | ERRORMESSAGE | FEHLT")
  public void encodeSIGNIN_ERROR_ERRORMESSAGE_FEHLT() throws EncodeException {
    SIGNIN signin = new SIGNIN();
    signin.setCommand(Command.SIGNIN);
    signin.setHeader(ERROR);
    signin.setDataset(Protocol.DATASET);
    signin.setEmail("llange@javacomm.net");
    signin.setErrorCode(33);
    signin.setText("User could not be added.");
    signin.setMultilingualkey(KEY.SERVER_EMAIL_INVALID);
    String result = signin.toString();

    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"SIGNIN\","
        + "\"DATASET\":{"
        + "\"EMAIL\":\"llange@javacomm.net\","
        + "\"ERRORCODE\":33,"
        + "\"TEXT\":\"User could not be added.\","
        + "\"MULTILINGUALKEY\":\"Server_Email_invalid\""
        + "}}";

    encoder.encode(signin);

    assertEquals(encoded, result, "JSON");
  }



  @Test
  @DisplayName("SIGNIN | ERROR | MULTILINGUALKEY | FEHLT")
  public void encodeSIGNIN_ERROR_MULTILINGUALKEY_FEHLT() {
    SIGNIN signin = new SIGNIN();
    signin.setCommand(Command.SIGNIN);
    signin.setHeader(ERROR);
    signin.setDataset(Protocol.DATASET);
    signin.setEmail("llange@javacomm.net");
    signin.setErrorCode(33);
    signin.setErrorMessage("123");
    signin.setText("User could not be added.");

    try {
      encoder.encode(signin);
      fail("Exception nicht geworfen, MULTILINGUALKEY fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }

  }



  @Test
  @DisplayName(value = "SIGNIN | TEXT | FEHLT")
  public void encodeSIGNIN_ERROR_TEXT_FEHLT() throws EncodeException {
    SIGNIN signin = new SIGNIN();
    signin.setCommand(Command.SIGNIN);
    signin.setHeader(ERROR);
    signin.setDataset(Protocol.DATASET);
    signin.setEmail("llange@javacomm.net");
    signin.setErrorCode(33);
    signin.setErrorMessage("123");
    signin.setMultilingualkey(KEY.SERVER_EMAIL_INVALID);
    String result = signin.toString();

    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"SIGNIN\","
        + "\"DATASET\":{"
        + "\"EMAIL\":\"llange@javacomm.net\","
        + "\"ERRORCODE\":33,"
        + "\"ERRORMESSAGE\":\"123\","
        + "\"MULTILINGUALKEY\":"
        + "\"Server_Email_invalid\""
        + "}}";
    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeSIGNIN_REQUEST() throws EncodeException {
    SIGNIN signin = new SIGNIN();
    signin.setCommand(Command.SIGNIN);
    signin.setHeader(REQUEST);
    signin.setDataset(Protocol.DATASET);
    signin.setEmail("llange@javacomm.net");
    String result = encoder.encode(signin);

    String bla = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"SIGNIN\","
        + "\"DATASET\":{"
        + "\"EMAIL\":\"llange@javacomm.net\""
        + "}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeSIGNIN_REQUEST_ohne_IP() throws EncodeException {
    SIGNIN signin = new SIGNIN();
    signin.setCommand(Command.SIGNIN);
    signin.setHeader(REQUEST);
    signin.setDataset(Protocol.DATASET);
    signin.setEmail("llange@javacomm.net");
    // signin.setIp("localhost");

    String result = encoder.encode(signin);

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"SIGNIN\","
        + "\"DATASET\":{"
        + "\"EMAIL\":\"llange@javacomm.net\""
        + "}}";

    assertEquals(encoded, result, "JSON");

  }



  @Test
  public void encodeSIGNIN_RESPONSE() throws EncodeException, IOException {
    SIGNIN signin = new SIGNIN();
    signin.setCommand(Command.SIGNIN);
    signin.setHeader(RESPONSE);
    signin.setDataset(Protocol.DATASET);

    signin.setUserid("louisxiv");
    signin.setNeuesPassword("PASSWORD");
    signin.setConfirmationKey("12345678");
    signin.setNickname("anaconda");
    signin.setBackgroundColor(-127);
    signin.setForegroundColor(-111);
    signin.setMultilingualkey(KEY.SERVER_NEUES_PASSWORT);
    signin.setText(KEY.SERVER_NEUES_PASSWORT.value());

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"SIGNIN\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"NEUES_PASSWORD\":\"PASSWORD\","
        + "\"CONFIRMATION_KEY\":\"12345678\","
        + "\"NICKNAME\":\"anaconda\","
        + "\"BACKGROUND_COLOR\":-127,"
        + "\"FOREGROUND_COLOR\":-111,"
        + "\"MULTILINGUALKEY\":\"Server_neues_Passwort\","
        + "\"TEXT\":\"Server_neues_Passwort\""
        + "}}";
    String result = encoder.encodeSIGNIN(signin);
    assertEquals(encoded, result, "JSON");

  }



  @Test
  @DisplayName(value = "SIGNIN | RESPONSE | CONFIRMATION_KEY | FEHLT")
  public void encodeSIGNIN_RESPONSE_CONFIRMATION_KEY_FEHLT() {
    SIGNIN signin = new SIGNIN();
    signin.setCommand(Command.SIGNIN);
    signin.setHeader(RESPONSE);
    signin.setDataset(Protocol.DATASET);

    signin.setUserid("louisxiv");
    signin.setNeuesPassword("PASSWORD");
    signin.setMultilingualkey(KEY.SERVER_NEUES_PASSWORT);
    signin.setText(KEY.SERVER_NEUES_PASSWORT.value());
    signin.setNickname("anaconda");
    signin.setBackgroundColor(-127);
    signin.setForegroundColor(-111);

    try {
      encoder.encodeSIGNIN(signin);
      fail("Exception nicht geworfen, CONFIRMATION_KEY fehlt");
    }
    catch (EncodeException | IOException e) {
      assertTrue(true);
    }

  }



  @Test
  @DisplayName(value = "SIGNIN | RESPONSE | MULTILINGUALKEY | FEHLT")
  public void encodeSIGNIN_RESPONSE_MULTILINGUALKEY_FEHLT() {
    SIGNIN login = new SIGNIN();
    login.setCommand(Command.SIGNIN);
    login.setHeader(RESPONSE);
    login.setDataset(Protocol.DATASET);

    login.setUserid("louisxiv");
    login.setEmail("llange@javacomm.net");
    login.setPassword("PASSWORD");
    login.setForegroundColor(-16777216);
    login.setBackgroundColor(-1);
    login.setNickname("anaconda");
    login.setConfirmationKey("12345678");

    try {
      encoder.encodeSIGNIN(login);
      fail("Exception wurde nicht geworfen, MULTILINGUALKEY fehlt");
    }
    catch (EncodeException | IOException e) {
      assertTrue(true);
    }

  }



  @Test
  public void encodeSIGNIN_RESPONSE_NEUES_PASSWORD_FEHLT() {
    SIGNIN signin = new SIGNIN();
    signin.setCommand(Command.SIGNIN);
    signin.setHeader(RESPONSE);
    signin.setDataset(Protocol.DATASET);

    signin.setUserid("louisxiv");
    signin.setConfirmationKey("12345678");
    signin.setMultilingualkey(KEY.SERVER_NEUES_PASSWORT);

    try {
      encoder.encodeSIGNIN(signin);
      fail("Exception nicht geworfen, NEUES_PASSORD fehlt");
    }
    catch (EncodeException | IOException e) {
      assertTrue(true);
    }

  }



  @Test
  @DisplayName(value = "STOPSENDINGVIDEO REQUEST")
  public void encodeSTOPSENDINGVIDEO_REQUEST() throws EncodeException {
    STOPSENDINGVIDEO stopSendingVideo = new STOPSENDINGVIDEO();
    stopSendingVideo.setCommand(Command.STOPSENDINGVIDEO);
    stopSendingVideo.setHeader(REQUEST);
    stopSendingVideo.setDataset(Protocol.DATASET);
    stopSendingVideo.setUserid("fritz678");

    String result = encoder.encode(stopSendingVideo);

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"STOPSENDINGVIDEO\","
        + "\"DATASET\":{"
        + "\"USERID\":\"fritz678\""
        + "}}";

    assertEquals(encoded, result);

  }

//  @Test
//  public void encodeTCPAUDIO_REQUEST() throws EncodeException {
//    final String polyfon = "Polyfon zwitschernd aßen Mäxchens Vögel Rüben, Joghurt und Quark";
//
//    TCPAUDIO audio = new TCPAUDIO();
//    audio.setCommand(Command.TCPAUDIO);
//    audio.setHeader(REQUEST);
//    audio.setDataset(Protocol.DATASET);
//    try {
//      audio.setContent(polyfon.getBytes("UTF-8"));
//    }
//    catch (UnsupportedEncodingException e) {} // Hallo meine Freunde
//    audio.setOrigin("7b");
//    audio.setTarget("8c");
//
//    String result = encoder.encode(audio);
//    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"TCPAUDIO\","
//        + "\"DATASET\":{"
//        + "\"TARGET\":\"8c\","
//        + "\"ORIGIN\":\"7b\","
//        + "\"CONTENT\":[80,111,108,121,102,111,110,32,122,119,105,116,115,99,104,101,114,110,100,32,97,-61,-97,101,110,32,77,-61,-92,120,99,104,101,110,115,32,86,-61,-74,103,101,108,32,82,-61,-68,98,101,110,44,32,74,111,103,104,117,114,116,32,117,110,100,32,81,117,97,114,107]"
//        + "}}";
//    assertEquals(encoded, result, "encodeINCOMINGCALL_REQUEST");
//
//  }



  @Test
  public void encodeTOPICMEMBER_EXCEPTION() {
    TOPICMEMBER member = new TOPICMEMBER();
    member.setCommand(Command.TOPICMEMBER);
    member.setHeader(RESPONSE);
    member.setDataset(Protocol.DATASET);
    member.setUserid("louisxiv");
    member.setRoom("fussball");
    member.setChatUser(new ChatUser[0]);

    String result;
    try {
      result = encoder.encode(member);
      fail("EncodeException nicht geworfen, ROOMTYPE fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }

//    String encoded = "{\"HEADER\":\"RESPONSE\","
//        + "\"COMMAND\":\"TOPICMEMBER\","
//        + "\"DATASET\":{"
//        + "\"USERID\":\"louisxiv\","
//        + "\"ROOM\":\"fussball\","
//        + "\"TOPICMEMBER\":["
//        + "{\"USERID\":\"jester\",\"NICKNAME\":\"anaconda\"},"
//        + "{\"USERID\":\"luis\",\"NICKNAME\":\"viper\""
//        + "}]"
//        + "}}";
//
//    assertEquals(encoded, result, "JSON");

  }



  @Test
  public void encodeTOPICMEMBER_REQUEST() throws EncodeException {
    TOPICMEMBER member = new TOPICMEMBER();
    member.setCommand(Command.TOPICMEMBER);
    member.setHeader(REQUEST);
    member.setDataset(Protocol.DATASET);
    member.setUserid("louisxiv");
    member.setRoom("fussball");

    String result = encoder.encode(member);

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"TOPICMEMBER\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"ROOM\":\"fussball\"}"
        + "}";
    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeTOPICMEMBER_REQUEST_EncodeException() {
    TOPICMEMBER member = new TOPICMEMBER();
    member.setCommand(Command.TOPICMEMBER);
    member.setHeader(REQUEST);
    member.setDataset(Protocol.DATASET);
    member.setRoom("fussball");

    try {
      String result = encoder.encode(member);
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void encodeTOPICMEMBER_RESPONSE() throws EncodeException {
    TOPICMEMBER member = new TOPICMEMBER();
    member.setCommand(Command.TOPICMEMBER);
    member.setHeader(RESPONSE);
    member.setDataset(Protocol.DATASET);
    member.setUserid("louisxiv");
    member.setRoom("fussball");
    member.setRoomtype(Roomtype.BESPRECHUNGSRAUM);
    member.setChatUser(new ChatUser[0]);

    String result = encoder.encode(member);

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"TOPICMEMBER\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"ROOM\":\"fussball\","
        + "\"ROOMTYPE\":\"BESPRECHUNGSRAUM\","
        + "\"TOPICMEMBER\":[]"
        + "}}";
    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeTOPICMEMBER_RESPONSE_mit_User() throws EncodeException {
    TOPICMEMBER member = new TOPICMEMBER();
    member.setCommand(Command.TOPICMEMBER);
    member.setHeader(RESPONSE);
    member.setDataset(Protocol.DATASET);
    member.setUserid("louisxiv");
    member.setRoom("fussball");
    member.setRoomtype(Roomtype.PAUSENRAUM);
    ChatUser[] chatuser = new ChatUser[1];
    chatuser[0] = new ChatUser();
    chatuser[0].setUserid("jester");
    chatuser[0].setNickname("python");
    member.setChatUser(chatuser);

    String result = encoder.encode(member);

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"TOPICMEMBER\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"ROOM\":\"fussball\","
        + "\"ROOMTYPE\":\"PAUSENRAUM\","
        + "\"TOPICMEMBER\":[{"
        + "\"USERID\":\"jester\","
        + "\"NICKNAME\":\"python\""
        + "}]"
        + "}}";
    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeTOPICMEMBER_RESPONSE_mit_User2() throws EncodeException {
    TOPICMEMBER member = new TOPICMEMBER();
    member.setCommand(Command.TOPICMEMBER);
    member.setHeader(RESPONSE);
    member.setDataset(Protocol.DATASET);
    member.setUserid("louisxiv");
    member.setRoom("fussball");
    member.setRoomtype(Roomtype.FORUM);
    ChatUser[] chatuser = new ChatUser[2];
    chatuser[0] = new ChatUser();
    chatuser[0].setUserid("jester");
    chatuser[0].setNickname("python");

    chatuser[1] = new ChatUser();
    chatuser[1].setUserid("luis");
    chatuser[1].setNickname("viper");
    chatuser[1].setBackgroundColor(33);

    member.setChatUser(chatuser);

    String result = encoder.encode(member);

    String encoded = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"TOPICMEMBER\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"ROOM\":\"fussball\","
        + "\"ROOMTYPE\":\"FORUM\","
        + "\"TOPICMEMBER\":["
        + "{\"USERID\":\"jester\",\"NICKNAME\":\"python\"},"
        + "{\"USERID\":\"luis\",\"NICKNAME\":\"viper\""
        + "}]"
        + "}}";
    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeTRANSFER_ERROR() throws EncodeException {
    TRANSFER transfer = new TRANSFER();
    transfer.setCommand(Command.TRANSFER);
    transfer.setHeader(ERROR);
    transfer.setDataset(Protocol.DATASET);
    transfer.setFilename("hallo.txt");
    transfer.setPathfile("/root/path/to/file");
    transfer.setSlot(1);
    transfer.setErrorMessage("Die Datei ist ungültig");

    String result = encoder.encode(transfer);

    String encoded = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"TRANSFER\","
        + "\"DATASET\":{\"SLOT\":1,"
        + "\"FILENAME\":\"hallo.txt\","
        + "\"PATHFILE\":\"/root/path/to/file\","
        + "\"ERRORMESSAGE\":\"Die Datei ist ungültig\"}}";

    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeTRANSFER_REQUEST_2() throws EncodeException {
    TRANSFER transfer = new TRANSFER();
    transfer.setCommand(Command.TRANSFER);
    transfer.setHeader(REQUEST);
    transfer.setDataset(Protocol.DATASET);
    transfer.setFilename("hallo.mp3");
    transfer.setPathfile("c:/");
    transfer.setSlot(1);
    transfer.setBlockindex(33);
    transfer.setEndflag(true);
    transfer.setPayload("273237");
    transfer.setMaxDatablocks(848);

    String result = encoder.encode(transfer);

    String bla = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"TRANSFER\","
        + "\"DATASET\":{"
        + "\"SLOT\":1,"
        + "\"BLOCKINDEX\":33,"
        + "\"FILENAME\":\"hallo.mp3\","
        + "\"PATHFILE\":\"c:/\","
        + "\"MAXDATABLOCKS\":848"
        + "}"
        + "}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeTRANSFER_RESPONSE() throws EncodeException {
    TRANSFER transfer = new TRANSFER();
    transfer.setCommand(Command.TRANSFER);
    transfer.setHeader(RESPONSE);
    transfer.setDataset(Protocol.DATASET);
    transfer.setFilename("hallo.mp3");
    transfer.setPathfile("c:/");
    transfer.setSlot(1);
    transfer.setBlockindex(33);
    transfer.setPayload("273237");
    transfer.setMaxDatablocks(848);
    transfer.setEndflag(true);

    String result = encoder.encode(transfer);

    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"TRANSFER\","
        + "\"DATASET\":{"
        + "\"SLOT\":1,"
        + "\"BLOCKINDEX\":33,"
        + "\"FILENAME\":\"hallo.mp3\","
        + "\"PATHFILE\":\"c:/\","
        + "\"ENDFLAG\":true,"
        + "\"PAYLOAD\":\"273237\","
        + "\"MAXDATABLOCKS\":848"
        + "}"
        + "}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeUPDATEFILETYPES_CONFIRM() throws EncodeException {
    UPDATEFILETYPES updatetypes = new UPDATEFILETYPES();
    updatetypes.setCommand(Command.UPDATEFILETYPES);
    updatetypes.setHeader(CONFIRM);
    updatetypes.setDataset(Protocol.DATASET);
    updatetypes.setUserid("userid123");

    String result = encoder.encode(updatetypes);

    String encoded = "{\"HEADER\":\"CONFIRM\"," + "\"COMMAND\":\"UPDATEFILETYPES\","
        + "\"DATASET\":{"
        + "\"USERID\":\"userid123\""
        + "}"
        + "}";

    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeUPDATEFILETYPES_REQUEST() throws EncodeException {
    UPDATEFILETYPES filetypes = new UPDATEFILETYPES();
    filetypes.setCommand(Command.UPDATEFILETYPES);
    filetypes.setHeader(REQUEST);
    filetypes.setDataset(Protocol.DATASET);
    filetypes.setUserid("userid123");
    BlacklistTypes[] list = new BlacklistTypes[2];
    list[0] = new BlacklistTypes();
    list[0].setDescription("Adobe");
    list[0].setSuffix("pdf");
    list[0].setChecked(true);
    list[1] = new BlacklistTypes();
    list[1].setDescription("Simple Text Format");
    list[1].setSuffix("txt");
    list[1].setChecked(false);
    filetypes.setFiletypes(list);

    String result = encoder.encode(filetypes);

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"UPDATEFILETYPES\","
        + "\"DATASET\":{"
        + "\"USERID\":\"userid123\","
        + "\"FILETYPES\":["
        + "{\"SUFFIX\":\"pdf\",\"DESCRIPTION\":\"Adobe\",\"CHECKED\":true},"
        + "{\"SUFFIX\":\"txt\",\"DESCRIPTION\":\"Simple Text Format\",\"CHECKED\":false}"
        + "]}"
        + "}";

    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeUPDATEPHONE_ERROR() throws EncodeException {
    UPDATEPHONE updatePhone = new UPDATEPHONE();
    updatePhone.setHeader(ERROR);
    updatePhone.setCommand(Command.UPDATEPHONE);
    updatePhone.setDataset(Protocol.DATASET);
    updatePhone.setIp("127.0.0.1");
    updatePhone.setPort(29477);
    String result = encoder.encode(updatePhone);
    String expected = "{\"HEADER\":\"ERROR\"," + "\"COMMAND\":\"UPDATEPHONE\"" + "}";
    assertEquals(expected, result, "encodeUPDATEPHONE_ERROR");
  }



  @Test
  public void encodeUPDATEPHONE_REQUEST() throws EncodeException {
    UPDATEPHONE updatePhone = new UPDATEPHONE();
    updatePhone.setHeader(REQUEST);
    updatePhone.setCommand(Command.UPDATEPHONE);
    updatePhone.setDataset(Protocol.DATASET);
    updatePhone.setUserid("12345678");
    updatePhone.setIp("127.0.0.1");
    updatePhone.setPort(29477);
    updatePhone.setOnBusy(false);
    String result = encoder.encode(updatePhone);
    String expected = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"UPDATEPHONE\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"IP\":\"127.0.0.1\","
        + "\"PORT\":29477,"
        + "\"ONBUSY\":false"
        + "}}";
    assertEquals(expected, result, "encodeUPDATEPHONE_REQUEST");
  }



  @Test
  public void encodeUPDATEPHONE_REQUEST_OHNE_IP() throws EncodeException {
    UPDATEPHONE updatePhone = new UPDATEPHONE();
    updatePhone.setHeader(REQUEST);
    updatePhone.setCommand(Command.UPDATEPHONE);
    updatePhone.setDataset(Protocol.DATASET);
    updatePhone.setUserid("12345678");
    updatePhone.setOnBusy(false);
    String result = encoder.encode(updatePhone);
    String expected = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"UPDATEPHONE\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"ONBUSY\":false"
        + "}}";
    assertEquals(expected, result, "encodeUPDATEPHONE_REQUEST");
  }



  @Test
  /**
   * Der Schlechtfall wird über den Decoder gefangen
   * 
   * @throws EncodeException
   */
  public void encodeUPDATEPHONE_REQUEST_OHNE_USERID() {
    UPDATEPHONE updatePhone = new UPDATEPHONE();
    updatePhone.setHeader(REQUEST);
    updatePhone.setCommand(Command.UPDATEPHONE);
    updatePhone.setDataset(Protocol.DATASET);
    updatePhone.setIp("127.0.0.1");
    updatePhone.setPort(29477);
    updatePhone.setOnBusy(false);
    try {
      encoder.encode(updatePhone);
      fail("EncoderException wurde nicht geworfen - USERID fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void encodeUPDATEPHONE_REQUEST_ONCALL() throws EncodeException {
    UPDATEPHONE updatePhone = new UPDATEPHONE();
    updatePhone.setHeader(REQUEST);
    updatePhone.setCommand(Command.UPDATEPHONE);
    updatePhone.setDataset(Protocol.DATASET);
    updatePhone.setUserid("12345678");
    updatePhone.setIp("127.0.0.1");
    updatePhone.setPort(29477);
    updatePhone.setOnBusy(true);
    String result = encoder.encode(updatePhone);
    String expected = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"UPDATEPHONE\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"IP\":\"127.0.0.1\","
        + "\"PORT\":29477,"
        + "\"ONBUSY\":true"
        + "}}";
    assertEquals(expected, result, "encodeUPDATEPHONE_REQUEST");
  }



  @Test
  public void encodeUPDATEPHONE_RESPONSE() throws EncodeException {
    UPDATEPHONE updatePhone = new UPDATEPHONE();
    updatePhone.setHeader(RESPONSE);
    updatePhone.setCommand(Command.UPDATEPHONE);
    updatePhone.setDataset(Protocol.DATASET);
    updatePhone.setUserid("12345678");
    updatePhone.setIp("127.0.0.1");
    updatePhone.setPort(29477);
    updatePhone.setOnBusy(true);
    String result = encoder.encode(updatePhone);
    String expected = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"UPDATEPHONE\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"IP\":\"127.0.0.1\","
        + "\"PORT\":29477,"
        + "\"ONBUSY\":true"
        + "}}";
    assertEquals(expected, result, "encodeUPDATEPHONE_RESPONSE");
  }



  @Test
  public void encodeUPDATEPHONE_RESPONSE_OHNE_ONBUSY() {
    UPDATEPHONE updatePhone = new UPDATEPHONE();
    updatePhone.setHeader(RESPONSE);
    updatePhone.setCommand(Command.UPDATEPHONE);
    updatePhone.setDataset(Protocol.DATASET);
    updatePhone.setUserid("12345678");
    updatePhone.setIp("127.0.0.1");
    updatePhone.setPort(29477);
    try {
      encoder.encode(updatePhone);
      fail("EncodeException wurde nicht geworfen - ONBUSY fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  public void encodeUPDATEPHONE_RESPONSE_OHNE_PORT() throws EncodeException {
    UPDATEPHONE updatePhone = new UPDATEPHONE();
    updatePhone.setHeader(RESPONSE);
    updatePhone.setCommand(Command.UPDATEPHONE);
    updatePhone.setDataset(Protocol.DATASET);
    updatePhone.setUserid("12345678");
    updatePhone.setOnBusy(true);
    String result = encoder.encode(updatePhone);
    String expected = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"UPDATEPHONE\","
        + "\"DATASET\":{"
        + "\"USERID\":\"12345678\","
        + "\"ONBUSY\":true"
        + "}}";
    assertEquals(expected, result, "encodeUPDATEPHONE_RESPONSE");
  }



  @Test
  public void encodeUPDATEPORT_REQUEST() throws EncodeException {
    UPDATEPORT updateport = new UPDATEPORT();
    updateport.setCommand(Command.UPDATEPORT);
    updateport.setHeader(REQUEST);
    updateport.setDataset(Protocol.DATASET);

    updateport.setUserid("louisxiv");
    updateport.setIp("localhost");
    updateport.setPort(8443);
    String result = encoder.encode(updateport);

    String bla = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"UPDATEPORT\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"IP\":\"localhost\","
        + "\"PORT\":8443"
        + "}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  public void encodeUPDATEPORT_RESPONSE() throws EncodeException {
    UPDATEPORT updateport = new UPDATEPORT();
    updateport.setCommand(Command.UPDATEPORT);
    updateport.setHeader(RESPONSE);
    updateport.setDataset(Protocol.DATASET);

    updateport.setUserid("louisxiv");
    updateport.setIp("localhost");
    updateport.setPort(8443);
    String result = encoder.encode(updateport);

    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"UPDATEPORT\","
        + "\"DATASET\":{"
        + "\"USERID\":\"louisxiv\","
        + "\"IP\":\"localhost\","
        + "\"PORT\":8443"
        + "}}";

    assertEquals(bla, result, "JSON");
  }



  @Test
  @DisplayName(value = "UPDATEUSER | ERROR")
  public void encodeUPDATEUSER_ERROR() throws EncodeException {
    UPDATEUSER login = new UPDATEUSER();
    login.setCommand(Command.UPDATEUSER);
    login.setHeader(ERROR);
    login.setDataset(Protocol.DATASET);
    login.setText("abc");
    String result = encoder.encode(login);
    String encoded = "{\"HEADER\":\"ERROR\",\"COMMAND\":\"UPDATEUSER\",\"DATASET\":{" + "\"TEXT\":\"abc\""
        + "}}";

    assertEquals(encoded, result, "JSON");
  }



  @Test
  @DisplayName(value = "UPDATEUSER | ERROR | TEXT | FEHLT")
  public void encodeUPDATEUSER_ERROR_TEXT_FEHLT() {
    UPDATEUSER login = new UPDATEUSER();
    login.setCommand(Command.UPDATEUSER);
    login.setHeader(ERROR);
    login.setDataset(Protocol.DATASET);
    try {
      encoder.encode(login);
      fail("Exception nicht geworfen, TEXT fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }
  }



  @Test
  @DisplayName(value = "UPDATEUSER | REQUEST")
  public void encodeUPDATEUSER_REQUEST() throws EncodeException {
    UPDATEUSER login = new UPDATEUSER();
    login.setCommand(Command.UPDATEUSER);
    login.setHeader(REQUEST);
    login.setDataset(Protocol.DATASET);

    login.setForegroundColor(-16777216);
    login.setBackgroundColor(-1);
    login.setNickname("anaconda");
    login.setIdentity("diejd");
    String result = encoder.encode(login);

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"UPDATEUSER\","
        + "\"DATASET\":{"
        + "\"IDENTITY\":\"diejd\","
        + "\"FOREGROUND_COLOR\":-16777216,"
        + "\"BACKGROUND_COLOR\":-1,"
        + "\"NICKNAME\":\"anaconda\"}}";

    assertEquals(encoded, result, "JSON");
  }



  @Test
  @DisplayName(value = "UPDATEUSER | IDENTITY | FEHLT")
  public void encodeUPDATEUSER_REQUEST_IDENTITY_FEHLT() {
    UPDATEUSER login = new UPDATEUSER();
    login.setCommand(Command.UPDATEUSER);
    login.setHeader(REQUEST);
    login.setDataset(Protocol.DATASET);

    login.setForegroundColor(-16777216);
    login.setBackgroundColor(-1);
    login.setNickname("anaconda");
    try {
      encoder.encode(login);
      fail("Exception nicht geworfen, IDENTITY fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }

  }



  @Test
  @DisplayName(value = "UPDATEUSER | REQUEST | NICKNAME | FEHLT")
  public void encodeUPDATEUSER_REQUEST_NICKNAME_FEHLT() {
    UPDATEUSER login = new UPDATEUSER();
    login.setCommand(Command.UPDATEUSER);
    login.setHeader(REQUEST);
    login.setDataset(Protocol.DATASET);

    login.setForegroundColor(-16777216);
    login.setBackgroundColor(-1);
    login.setIdentity("diejd");
    try {
      encoder.encode(login);
      fail("Exception nicht geworfen, NICKNAME fehlt");
    }
    catch (EncodeException e) {
      assertTrue(true);
    }

  }



  @Test
  public void encodeUPLOADFILES_REQUEST() {
    UPLOADFILES uploadfiles = new UPLOADFILES();
    uploadfiles.setCommand(Command.UPLOADFILES);
    uploadfiles.setHeader(REQUEST);
    uploadfiles.setDataset(Protocol.DATASET);
    uploadfiles.setUserid("louisxiv");
    uploadfiles.setIp("192.168.1.41");
//  	uploadfiles.setFiles(new String[]{"abc.123", "hallo.456"});

    Uploadfile uploadfile1 = new Uploadfile();
    uploadfile1.setFilename("abc.txt");
    uploadfile1.setFilesize(1000);
    Uploadfile uploadfile2 = new Uploadfile();
    uploadfile2.setFilename("zwei.txt");
    uploadfile2.setFilesize(1234);
    uploadfiles.setUploadfile(new Uploadfile[] {uploadfile1, uploadfile2});

    uploadfiles.setStarted(true);

    String result;
    try {
      result = encoder.encode(uploadfiles);
      fail("EncodeException nicht geworfen");
    }
    catch (EncodeException e) {}
  }



  @Test
  public void encodeUPLOADFILES_REQUEST_PORT() throws EncodeException {
    UPLOADFILES uploadfiles = new UPLOADFILES();
    uploadfiles.setCommand(Command.UPLOADFILES);
    uploadfiles.setHeader(REQUEST);
    uploadfiles.setDataset(Protocol.DATASET);
    uploadfiles.setUserid("louisxiv");
    uploadfiles.setIp("192.168.1.41");
//    uploadfiles.setFiles(new String[]{"abc.123", "hallo.456"});

    Uploadfile uploadfile1 = new Uploadfile();
    uploadfile1.setFilename("abc.txt");
    uploadfile1.setFilesize(1000);
    Uploadfile uploadfile2 = new Uploadfile();
    uploadfile2.setFilename("zwei.txt");
    uploadfile2.setFilesize(1234);
    uploadfiles.setUploadfile(new Uploadfile[] {uploadfile1, uploadfile2});

    uploadfiles.setStarted(true);
    uploadfiles.setPort(8443);

    String result = encoder.encode(uploadfiles);

    String encoded = "{\"HEADER\":\"REQUEST\"," + "\"COMMAND\":\"UPLOADFILES\","
        + "\"DATASET\":{"
        + "\"START\":true,"
        + "\"USERID\":\"louisxiv\","
        + "\"IP\":\"192.168.1.41\","
        + "\"PORT\":8443,"
        + "\"UPLOADFILES\":["
        + "{\"FILENAME\":\"abc.txt\","
        + "\"FILESIZE\":1000},"
        + "{\"FILENAME\":\"zwei.txt\","
        + "\"FILESIZE\":1234}"
        + "]}}";
    assertEquals(encoded, result, "JSON");
  }



  @Test
  public void encodeUPLOADFILES_RESPONSE() throws EncodeException {
    UPLOADFILES uploadfiles = new UPLOADFILES();
    uploadfiles.setCommand(Command.UPLOADFILES);
    uploadfiles.setHeader(RESPONSE);
    uploadfiles.setDataset(Protocol.DATASET);
    uploadfiles.setUserid("louisxiv");
    uploadfiles.setIp("192.168.1.41");
//  	uploadfiles.setFiles(new String[]{"abc.123", "hallo.456"});

    Uploadfile uploadfile1 = new Uploadfile();
    uploadfile1.setFilename("abc.txt");
    uploadfile1.setFilesize(1000);
    Uploadfile uploadfile2 = new Uploadfile();
    uploadfile2.setFilename("zwei.txt");
    uploadfile2.setFilesize(1234);
    uploadfiles.setUploadfile(new Uploadfile[] {uploadfile1, uploadfile2});

    uploadfiles.setStarted(true);

    String result = encoder.encode(uploadfiles);

    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"UPLOADFILES\","
        + "\"DATASET\":{"
        + "\"START\":true,"
        + "\"USERID\":\"louisxiv\","
        + "\"IP\":\"192.168.1.41\""
        + "}}";
    assertEquals(bla, result, "JSON");
  }



  @Test
  @DisplayName(value = "USERONLINELIST | RESPONSE")
  public void encodeUSERONLINELIST_RESPONSE() throws EncodeException {
    USERONLINELIST userlist = new USERONLINELIST();
    userlist.setCommand(Command.USERONLINELIST);
    userlist.setHeader(RESPONSE);
    userlist.setDataset(Protocol.DATASET);
    userlist.setUserid("python");
    UserOnline[] online = new UserOnline[1];
    online[0] = new UserOnline();
    online[0].setIp("localhost");
    online[0].setNickname("anaconda");
    online[0].setUserid("louisxiv");
    userlist.setUserOnline(online);

    String result = encoder.encode(userlist);

    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"USERONLINELIST\","
        + "\"DATASET\":{"
        + "\"USERID\":\"python\","
        + "\"USERONLINELIST\":["
        + "{\"IP\":\"localhost\","
        + "\"USERID\":\"louisxiv\","
        + "\"NICKNAME\":\"anaconda\"}]}}";
    assertEquals(bla, result, "JSON");
  }



  @Test
  @DisplayName(value = "USERONLINELIST | RESPONSE | IP | FEHLT")
  public void encodeUSERONLINELIST_RESPONSE_IP_FEHLT() throws EncodeException {
    USERONLINELIST userlist = new USERONLINELIST();
    userlist.setCommand(Command.USERONLINELIST);
    userlist.setHeader(RESPONSE);
    userlist.setDataset(Protocol.DATASET);
    userlist.setUserid("python");
    UserOnline[] online = new UserOnline[1];
    online[0] = new UserOnline();
    online[0].setNickname("anaconda");
    online[0].setUserid("louisxiv");
    userlist.setUserOnline(online);

    String result = encoder.encode(userlist);

    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"USERONLINELIST\","
        + "\"DATASET\":{"
        + "\"USERID\":\"python\","
        + "\"USERONLINELIST\":["
        + "{\"USERID\":\"louisxiv\","
        + "\"NICKNAME\":\"anaconda\"}]}}";
    assertEquals(bla, result, "JSON");
  }



  @Test
  @DisplayName(value = "USERONLINELIST | RESPONSE | USERID | FEHLT")
  public void encodeUSERONLINELIST_RESPONSE_USERID_FEHLT() throws EncodeException {
    USERONLINELIST userlist = new USERONLINELIST();
    userlist.setCommand(Command.USERONLINELIST);
    userlist.setHeader(RESPONSE);
    userlist.setDataset(Protocol.DATASET);
    UserOnline[] online = new UserOnline[1];
    online[0] = new UserOnline();
    online[0].setIp("localhost");
    online[0].setNickname("anaconda");
    online[0].setUserid("louisxiv");
    userlist.setUserOnline(online);

    String result = encoder.encode(userlist);

    String bla = "{\"HEADER\":\"RESPONSE\"," + "\"COMMAND\":\"USERONLINELIST\","
        + "\"DATASET\":{"
        + "\"USERONLINELIST\":["
        + "{\"IP\":\"localhost\","
        + "\"USERID\":\"louisxiv\","
        + "\"NICKNAME\":\"anaconda\"}]}}";
    assertEquals(bla, result, "JSON");
  }



  @Test
  @DisplayName(value = "USRLOGIN | CONFIRM | SESSION | FEHLT")
  public void encodeUSRLOGIN_CONFIRM_SESSION_FEHLT() {
    USRLOGIN login = new USRLOGIN();
    login.setCommand(Command.USRLOGIN);
    login.setHeader(CONFIRM);
    login.setDataset(Protocol.DATASET);

    login.setForegroundColor(-16777216);
    login.setBackgroundColor(-1);
    login.setNickname("anaconda");
    login.setIdentity("jeidj");
    login.setOncall(true);
    login.setVolume(100);
    try {
      encoder.encodeUSRLOGIN(login);
      fail("Exception nicht geworfen, SESSION fehlt");
    }
    catch (EncodeException | IOException e) {
      assertTrue(true);
    }

  }



  @Test
  @DisplayName(value = "USRLOGIN | ERROR")
  public void encodeUSRLOGIN_ERROR() throws EncodeException, IOException {
    USRLOGIN login = new USRLOGIN();
    login.setCommand(Command.USRLOGIN);
    login.setHeader(ERROR);
    login.setDataset(Protocol.DATASET);
    login.setText("String_Konto_Deaktiviert");
    String result = encoder.encodeUSRLOGIN(login);
    assertEquals(
        "{\"HEADER\":\"ERROR\",\"COMMAND\":\"USRLOGIN\",\"DATASET\":{"
            + "\"TEXT\":\"String_Konto_Deaktiviert\""
            + "}}",
        result, "JSON"
    );
  }



  @Test
  @DisplayName(value = "USRLOGIN | ERROR | TEXT | FEHLT")
  public void encodeUSRLOGIN_ERROR_TEXT_FEHLT() {
    USRLOGIN login = new USRLOGIN();
    login.setCommand(Command.USRLOGIN);
    login.setHeader(ERROR);
    login.setDataset(Protocol.DATASET);
    try {
      encoder.encodeUSRLOGIN(login);
      fail("Exception nicht geworfen, TEXT fehlt");
    }
    catch (EncodeException | IOException e) {
      assertTrue(true);
    }
  }



  @Test
  @DisplayName(value = "USRLOGIN | ERROR | MULTILIGUALKEY")
  public void encodeUSRLOGIN_ERROR_MULTILINGUALKEY() throws EncodeException, IOException {
    USRLOGIN login = new USRLOGIN();
    login.setCommand(Command.USRLOGIN);
    login.setHeader(ERROR);
    login.setDataset(Protocol.DATASET);
    login.setMultilingualkey(KEY.STRING_KONTO_DEAKTIVIERT);
    login.setText("String_Konto_Deaktiviert");
    String result = encoder.encodeUSRLOGIN(login);
    assertEquals(
        "{\"HEADER\":\"ERROR\",\"COMMAND\":\"USRLOGIN\",\"DATASET\":{"
            + "\"MULTILINGUALKEY\":\"String_Konto_Deaktiviert\","
            + "\"TEXT\":\"String_Konto_Deaktiviert\""
            + "}}",
        result, "JSON"
    );
  }



  @Test
  @DisplayName(value = "USRLOGIN | IDENTITY | FEHLT")
  public void encodeUSRLOGIN_REQUEST_IDENTITY_FEHLT() {

    USRLOGIN login = new USRLOGIN();
    login.setCommand(Command.USRLOGIN);
    login.setHeader(REQUEST);
    login.setDataset(Protocol.DATASET);

    try {
      encoder.encodeUSRLOGIN(login);
      fail("Exception nicht geworfen, IDENTITY fehlt");
    }
    catch (Exception e) {
      assertTrue(true);
    }

  }



  @Test
  @DisplayName(value = "USERLOGIN | REQUEST | IP | AGENT | IDENTITY | AES")
  public void encodeUSRLOGIN_REQUEST_IP_AGENT_IDENTITY() throws EncodeException, IOException {

    USRLOGIN login = new USRLOGIN();
    login.setCommand(Command.USRLOGIN);
    login.setHeader(REQUEST);
    login.setDataset(Protocol.DATASET);
    login.setAgent(Agent.SmartPhone);
    Token token = new Token();
    token.setUserid("Luis");
    token.setPassword("123");
    token.setEMail("bla@javacomm.net");
    token.setAES("iwjjd");

    // Key erstellen
    SecretKey secretkey = Crypto.createAES();
    String scretkey64 = Crypto.getBase64FromAES(secretkey);

    String geheim64 = Crypto.encryptAES(token.toString(), secretkey);

    login.setIdentity(geheim64);

    log.info(login.toString());

    String result = encoder.encodeUSRLOGIN(login);

    USRLOGIN back = USRLOGIN.toUSRLOGIN(result);

    String backIdentity = back.getIdentity();

    String orig = Crypto.decryptAES(backIdentity, secretkey);

    assertEquals(geheim64, back.getIdentity(), "IDENTITY");
    assertEquals(Agent.SmartPhone, back.getAgent(), "AGENT");
  }



  @Test
  @DisplayName(value = "USRLOGIN | REQUEST | IP | IDENTITY ohne AGENT")
  public void encodeUSRLOGIN_REQUEST_IP_IDENTITY_ohne_AGENT() {

    USRLOGIN login = new USRLOGIN();
    login.setCommand(Command.USRLOGIN);
    login.setHeader(REQUEST);
    login.setDataset(Protocol.DATASET);
    Token token = new Token();
    token.setEMail("max@javacomm.net");
    token.setUserid("admin678");
    token.setPassword("12345678");
    token.setAES("uwdh");
    login.setIdentity(token.toString());

    // Key erstellen
    SecretKey secretkey = Crypto.createAES();

    String geheim64 = Crypto.encryptAES(token.toString(), secretkey);

    login.setIdentity(geheim64);

    try {
      encoder.encodeUSRLOGIN(login);
      fail("Exception nicht geworfen, AGENT fehlt.");
    }
    catch (EncodeException | IOException e) {
      assertTrue(true);
    }

  }



  @BeforeEach
  public void setUp() {
    encoder = new MessageEncoder();
    encoder.init(null);
  }

}
