package net.javacomm.transfer;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNull;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;
import net.javacomm.protocol.Benutzerstatus;
import net.javacomm.protocol.Language;



public class TestTransferUser {

  TransferUser user;
  GsonBuilder builder;
  Gson gson;

  @BeforeEach
  public void setUp() {
    user = new TransferUser();
    builder = new GsonBuilder();
    gson = builder.create();
  }



  @Test
  public void serializeUser() {
    user.setBenutzerstatus(Benutzerstatus.MÖCHTE_VOLLMITGLIED_WERDEN.getStatus());
    user.setMail("team@javacomm.net");
    user.setUserid("louisxiv");
//    user.setLanguage(Language.ES);
    String result = gson.toJson(user);
    assertEquals("{\"userid\":\"louisxiv\"," + "\"mail\":\"team@javacomm.net\"," + "\"benutzerstatus\":2"
//            + "\"language\":\"es\""
        + "}", result
    );
  }



  @Test
  public void deserialize() {
    String encoded = "{\"userid\":\"louisxiv\",\"mail\":\"team@javacomm.net\",\"benutzerstatus\":2}";
    TransferUser result = gson.fromJson(encoded, TransferUser.class);
    assertEquals(2, result.getBenutzerstatus());
    assertEquals("team@javacomm.net", result.getMail());
    assertEquals("louisxiv", result.getUserid());
  }



  @Test
  public void toJson() {
    user.setBenutzerstatus(Benutzerstatus.VOLLMITGLIED.getStatus());
    user.setMail("javaanaconda@gmail.com");
    user.setUserid("anaconda");

    assertEquals(
        "{\"userid\":\"anaconda\",\"mail\":\"javaanaconda@gmail.com\",\"benutzerstatus\":1}", user.toString()
    );

  }



  @Test
  public void toJson_2() {
    user.setBenutzerstatus(Benutzerstatus.MÖCHTE_VOLLMITGLIED_WERDEN.getStatus());
    user.setMail("javaanaconda@gmail.com");
    user.setUserid("anaconda");

    assertEquals(
        "{\"userid\":\"anaconda\",\"mail\":\"javaanaconda@gmail.com\",\"benutzerstatus\":2}", user.toString()
    );

  }



  @Test
  public void serializeOnprojektor() {
    user.setUserid("12345678");
    user.setOnprojektor(true);
    assertEquals("{\"userid\":\"12345678\",\"onprojektor\":true}", user.toString());
  }



  @Test
  public void deserializeOnprojektor() {
    TransferUser transfer = gson
        .fromJson("{\"userid\":\"12345678\",\"onprojektor\":true}", TransferUser.class);
    assertEquals("12345678", transfer.getUserid());
    assertEquals(true, transfer.getOnprojektor());
  }



  @Test
  public void serializeOncall() {
    user.setUserid("12345678");
    user.setOncall(true);
    assertEquals("{\"userid\":\"12345678\",\"oncall\":true}", user.toString());
  }



  @Test
  public void deserializeOncall() {
    TransferUser transfer = gson.fromJson("{\"userid\":\"12345678\",\"oncall\":true}", TransferUser.class);
    assertEquals("12345678", transfer.getUserid());
    assertEquals(true, transfer.getOncall());
  }



  @Test
  public void serializeVolume() {
    user.setUserid("12345678");
    user.setVolume(100);
    assertEquals("{\"userid\":\"12345678\",\"volume\":100}", user.toString());
  }



  @Test
  @DisplayName(value = "Gutufall Mikrofon und Speaker")
  public void serializeSpeakerAndMikrofon() {
    user.setUserid("12345678");
    user.setMikrofon("Mikrofon");
    user.setSpeaker("Lautsprecher");
    assertEquals(
        "{\"userid\":\"12345678\",\"speaker\":\"Lautsprecher\",\"mikrofon\":\"Mikrofon\"}", user.toString()
    );
  }



  @Test
  @DisplayName(value = "Gutufall deserialize Mikrofon und Speaker")
  public void deserializeSpeakerAndMikrofon() {

    TransferUser transfer = gson.fromJson(
        "{\"userid\":\"12345678\",\"speaker\":\"Lautsprecher\",\"mikrofon\":\"Mikrofon\"}", TransferUser.class
    );

    assertEquals("12345678", transfer.getUserid());
    assertEquals("Lautsprecher", transfer.getSpeaker());
    assertEquals("Mikrofon", transfer.getMikrofon());

  }



  @Test
  public void serializeLanguage() {
    user.setUserid("12345678");
    user.setLanguage(Language.es);
    assertEquals("{\"userid\":\"12345678\",\"language\":\"es\"}", user.toString());
  }



  @Test
  public void deserializeLanguage() {
    TransferUser transfer = gson
        .fromJson("{\"userid\":\"12345678\",\"language\":\"de\"}", TransferUser.class);

    assertEquals("12345678", transfer.getUserid());
    assertEquals(Language.de, transfer.getLanguage());
  }



  @Test
  public void serializeHelado() {
    user.setUserid("12345678");
    user.setHelado(Helado.Fresa);
    assertEquals("{\"userid\":\"12345678\",\"helado\":\"Fresa\"}", user.toString());
  }



  @Test
  public void deserializeHelado() {
    TransferUser transfer = gson
        .fromJson("{\"userid\":\"12345678\",\"helado\":\"Fresa\"}", TransferUser.class);

    assertEquals("12345678", transfer.getUserid());
    assertEquals(Helado.Fresa, transfer.getHelado());
  }



  @Test
  @DisplayName(value = "serialisiere REKORD")
  public void serializeRekord() {
    user.setUserid("12345678");
    user.setOnrekord(true);
    user.setRekorderport(29478);
    assertEquals("{\"userid\":\"12345678\",\"rekorderport\":29478,\"onrekord\":true}", user.toString());
  }



  @Test
  @DisplayName(value = "deserialisiere REKORD")
  public void deserializeRekord() {
    TransferUser transfer = gson
        .fromJson("{\"userid\":\"12345678\",\"rekorderport\":29478,\"onrekord\":true}", TransferUser.class);
    assertEquals("12345678", transfer.getUserid());
    assertEquals(true, transfer.getOnrekord());
    assertEquals(29478, transfer.getRekorderport());

  }



  @Test
  public void serializePlayervolume() {
    user.setUserid("12345678");
    user.setPlayervolume(68);
    assertEquals("{\"userid\":\"12345678\",\"playervolume\":68}", user.toString());
  }



  @Test
  public void deserializePlayervolume() {
    TransferUser transfer = gson
        .fromJson("{\"userid\":\"12345678\",\"playervolume\":68}", TransferUser.class);
    assertEquals("12345678", transfer.getUserid());
    assertEquals(68, transfer.getPlayervolume());
  }



  @Test
  public void deserializeVolume() {
    TransferUser transfer = gson.fromJson("{\"userid\":\"12345678\",\"volume\":100}", TransferUser.class);
    assertEquals("12345678", transfer.getUserid());
    assertEquals(100, transfer.getVolume());
  }



  @Test
  public void deserializeTelefonport() {
    TransferUser transfer = gson
        .fromJson("{\"userid\":\"12345678\",\"telefonport\":29477}", TransferUser.class);
    assertEquals("12345678", transfer.getUserid());
  }



  @Test
  public void deserializeLogicallyDelete() {
    TransferUser transfer = gson.fromJson("{\"userid\":\"12345678\",\"ldelete\":true}", TransferUser.class);
    assertEquals(true, transfer.getLdelete());
  }



  @Test
  public void serializeLogicallyDelete() {
    user.setUserid("12345678");
    user.setLdelete(true);
    assertEquals("{\"userid\":\"12345678\",\"ldelete\":true}", user.toString());
  }



  @Test
  public void serializeTelefon() {
    user.setUserid("12345678");
    user.setPlayervolume(68);
    user.setTelefonpuffer(Sprachguete.MEDIUM);
    assertEquals(
        "{\"userid\":\"12345678\",\"playervolume\":68,\"telefonpuffer\":\"MEDIUM\"}", user.toString()
    );
  }



  @Test
  @DisplayName("serialisiere User mit CHATDENIED")
  public void serializeChatdenied() {
    user.setUserid("12345678");
    user.setPlayervolume(68);
    user.setTelefonpuffer(Sprachguete.MEDIUM);
    user.setChatdenied(true);
    assertEquals(
        "{\"userid\":\"12345678\",\"playervolume\":68,\"telefonpuffer\":\"MEDIUM\",\"chatdenied\":true}",
        user.toString()
    );
  }



  @Test
  @DisplayName("serialisiere REKORDERPORT")
  public void serializeRekorderport() {
    user.setUserid("12345678");
    user.setRekorderport(29478);
    assertEquals("{\"userid\":\"12345678\",\"rekorderport\":29478}", user.toString());
  }



  @Test
  @DisplayName("serialisiere ChatuploadDir")
  public void serializeChatuploaddir() {
    user.setUserid("12345678");
    user.setRekorderport(29478);
    user.setChatuploaddir("upload.mp4");
    assertEquals(
        "{\"userid\":\"12345678\",\"chatuploaddir\":\"upload.mp4\",\"rekorderport\":29478}", user.toString()
    );
  }



  @Test
  @DisplayName("serialisiere ChatDownloadDir")
  public void serializeChatdownloaddir() {
    user.setUserid("12345678");
    user.setRekorderport(29478);
    user.setChatuploaddir("upload.mp4");
    user.setChatdownloaddir("bla.mp4");
    assertEquals(
        "{\"userid\":\"12345678\",\"chatdownloaddir\":\"bla.mp4\",\"chatuploaddir\":\"upload.mp4\",\"rekorderport\":29478}",
        user.toString()
    );
  }



  @Test
  @DisplayName("deserialisiere REKORDERPORT")
  public void deserializeRekorderport() {
    TransferUser transfer = gson
        .fromJson("{\"userid\":\"12345678\",\"rekorderport\":29478}", TransferUser.class);

    assertEquals("12345678", transfer.getUserid(), "userid");
    assertEquals(29478, transfer.getRekorderport(), "userid");

  }



  @Test
  @DisplayName("deserialisiere User mit CHATDENIED")
  public void deserializeChatdenied() {
    TransferUser transfer = gson.fromJson(
        "{\"userid\":\"12345678\",\"telefonport\":29477,\"telefonpuffer\":\"MEDIUM\",\"chatdenied\":true}",
        TransferUser.class
    );
    assertEquals("12345678", transfer.getUserid());
    assertEquals(Sprachguete.MEDIUM, transfer.getTelefonpuffer());
    assertEquals(true, transfer.getChatdenied());

  }



  @Test
  @DisplayName("deserialisiere User mit CHATTOOLTIP")
  public void deserializeChattooltip() {
    TransferUser transfer = gson.fromJson(
        "{\"userid\":\"12345678\",\"telefonport\":29477,\"telefonpuffer\":\"MEDIUM\",\"chatdenied\":true,\"chattooltip\":true}",
        TransferUser.class
    );
    assertEquals("12345678", transfer.getUserid());
    assertEquals(Sprachguete.MEDIUM, transfer.getTelefonpuffer());
    assertEquals(true, transfer.getChatdenied());
    assertEquals(true, transfer.getChattooltip());
  }



  @Test
  public void deserializeTelefon() {
    TransferUser transfer = gson.fromJson(
        "{\"userid\":\"12345678\",\"telefonport\":29477,\"telefonpuffer\":\"MEDIUM\"}", TransferUser.class
    );
    assertEquals("12345678", transfer.getUserid());
    assertEquals(Sprachguete.MEDIUM, transfer.getTelefonpuffer());

  }



  @Test
  public void deserializeUser() {
    TransferUser transfer = gson.fromJson(
        "{\"userid\":\"admin678\"," + "\"mail\":\"admin@javacomm.net\","
            + "\"benutzerstatus\":1,"
            + "\"volume\":68,"
            + "\"playervolume\":68,"
            + "\"iptvvolume\":68,"
            + "\"oncall\":false,"
            + "\"telefonport\":29477,"
            + "\"telefonpuffer\":\"MEDIUM\","
            + "\"language\":\"de\""
            + "}",
        TransferUser.class
    );

    assertEquals("admin678", transfer.getUserid());
    assertEquals("admin@javacomm.net", transfer.getMail());
    assertEquals(Language.de, transfer.getLanguage());
    assertEquals(Sprachguete.MEDIUM, transfer.getTelefonpuffer());
    assertNull(transfer.getChatdownloaddir());
    assertNull(transfer.getChatuploaddir());
  }



  @Test
  public void deserializeChatuploaddir() {
    TransferUser transfer = gson.fromJson(
        "{\"userid\":\"admin678\"," + "\"mail\":\"admin@javacomm.net\","
            + "\"benutzerstatus\":1,"
            + "\"volume\":68,"
            + "\"playervolume\":68,"
            + "\"iptvvolume\":68,"
            + "\"oncall\":false,"
            + "\"telefonport\":29477,"
            + "\"chatuploaddir\":\"hallo.mp4\","
            + "\"telefonpuffer\":\"MEDIUM\","
            + "\"language\":\"de\""
            + "}",
        TransferUser.class
    );

    assertEquals("admin678", transfer.getUserid());
    assertEquals("admin@javacomm.net", transfer.getMail());
    assertEquals(Language.de, transfer.getLanguage());
    assertEquals(Sprachguete.MEDIUM, transfer.getTelefonpuffer());
    assertEquals("hallo.mp4", transfer.getChatuploaddir());
    assertNull(transfer.getChatdownloaddir());
  }



  @Test
  public void deserializeChatdownloaddir() {
    TransferUser transfer = gson.fromJson(
        "{\"userid\":\"admin678\"," + "\"mail\":\"admin@javacomm.net\","
            + "\"benutzerstatus\":1,"
            + "\"volume\":68,"
            + "\"playervolume\":68,"
            + "\"iptvvolume\":68,"
            + "\"oncall\":false,"
            + "\"telefonport\":29477,"
            + "\"chatuploaddir\":\"hallo.mp4\","
            + "\"chatdownloaddir\":\"test.mp4\","
            + "\"telefonpuffer\":\"MEDIUM\","
            + "\"language\":\"de\""
            + "}",
        TransferUser.class
    );

    assertEquals("admin678", transfer.getUserid());
    assertEquals("admin@javacomm.net", transfer.getMail());
    assertEquals(Language.de, transfer.getLanguage());
    assertEquals(Sprachguete.MEDIUM, transfer.getTelefonpuffer());
    assertEquals("hallo.mp4", transfer.getChatuploaddir());
    assertEquals("test.mp4", transfer.getChatdownloaddir());
  }

}
