/**
 * This software is released as part of the Pumpernickel project.
 * 
 * All com.pump resources in the Pumpernickel project are distributed under the
 * MIT License:
 * https://github.com/mickleness/pumpernickel/raw/master/License.txt
 * 
 * More information about the Pumpernickel project is available here:
 * https://mickleness.github.io/pumpernickel/
 */
package com.pump.showcase.demo;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;

import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JEditorPane;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JScrollPane;
import javax.swing.JSpinner;
import javax.swing.SpinnerNumberModel;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.text.JTextComponent;

import com.pump.swing.TextSearchBar;
import com.pump.swing.TextSearchDialog;

/**
 * A demo app/applet for the {@link TextSearchBar} classes.
 *
 * 
 * <!-- ======== START OF AUTOGENERATED SAMPLES ======== -->
 * <p>
 * <img src=
 * "https://github.com/mickleness/pumpernickel/raw/master/resources/samples/TextSearchDemo/sample.png"
 * alt="new&#160;com.pump.swing.TextSearchDemo()"> <!-- ======== END OF
 * AUTOGENERATED SAMPLES ======== -->
 */
public class TextSearchDemo extends JPanel {
	private static final long serialVersionUID = 1L;

	public static void main(String[] args) {
		SwingUtilities.invokeLater(new Runnable() {
			public void run() {
				JFrame frame = new JFrame("TextSearchDemo");
				frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
				TextSearchDemo demo = new TextSearchDemo();
				frame.add(demo);
				frame.pack();
				frame.setVisible(true);
			}
		});
	}

	JTextComponent text;
	JPanel header = new JPanel(new GridBagLayout());
	TextSearchBar firefoxBar;
	TextSearchBar safariBar;
	JCheckBox firefoxButton = new JCheckBox("Firefox Bar", true);
	JCheckBox safariButton = new JCheckBox("Safari Bar", true);
	JButton findButton = new JButton("Find Dialog");
	JSpinner duration = new JSpinner(
			new SpinnerNumberModel(500, 100, 5000, 50));
	TextSearchDialog dialog;
	JRadioButton aquaEffect = new JRadioButton("Aqua");
	JRadioButton firefoxEffect = new JRadioButton("Firefox");
	JRadioButton nudgeEffect = new JRadioButton("Nudge");

	ActionListener actionListener = new ActionListener() {
		public void actionPerformed(ActionEvent e) {
			if (e.getSource() == firefoxButton) {
				firefoxBar.setVisible(firefoxButton.isSelected());
			} else if (e.getSource() == safariButton) {
				safariBar.setVisible(safariButton.isSelected());
			} else if (e.getSource() == findButton) {
				if (dialog == null) {
					dialog = new TextSearchDialog(text);
					dialog.setLocationRelativeTo(null);
				}
				dialog.setVisible(true);
			} else if (e.getSource() == aquaEffect) {
				UIManager.put("textSearchHighlightEffect",
						"com.pump.plaf.AquaSearchHighlight");
			} else if (e.getSource() == firefoxEffect) {
				UIManager.put("textSearchHighlightEffect",
						"com.pump.plaf.FirefoxSearchHighlight");
			} else if (e.getSource() == nudgeEffect) {
				UIManager.put("textSearchHighlightEffect",
						"com.pump.plaf.NudgeSearchHighlight");
			}
		}
	};

	ChangeListener changeListener = new ChangeListener() {
		public void stateChanged(ChangeEvent e) {
			Number number = (Number) duration.getValue();
			UIManager.put("SearchHighlight.duration", number);
		}
	};

	ComponentListener componentListener = new ComponentAdapter() {
		@Override
		public void componentShown(ComponentEvent e) {
			updateButtons(e);
		}

		@Override
		public void componentHidden(ComponentEvent e) {
			updateButtons(e);
		}

		private void updateButtons(ComponentEvent e) {
			if (e.getSource() == safariBar) {
				safariButton.setSelected(safariBar.isVisible());
			} else if (e.getSource() == firefoxBar) {
				firefoxButton.setSelected(firefoxBar.isVisible());
			}
		}
	};

	public TextSearchDemo() {
		try {
			text = new JEditorPane(
					TextSearchDemo.class.getResource("book.htm"));
		} catch (Exception e) {
			e.printStackTrace();
			text = new JEditorPane();
			text.setText(
					"I tried to load a webpage, but failed.  Do you have an active internet connection?  Oh well, no matter.  You can still search this text component, but it's filled with a lot of rambling words...");
		}
		text.setPreferredSize(new Dimension(500, 500));

		firefoxBar = TextSearchBar.createFirefoxStyleBar(text);
		safariBar = TextSearchBar.createSafariStyleBar(text);

		setLayout(new GridBagLayout());
		GridBagConstraints c = new GridBagConstraints();
		c.gridx = 0;
		c.gridy = 0;
		c.weightx = 1;
		c.weighty = 0;
		c.fill = GridBagConstraints.NONE;
		c.anchor = GridBagConstraints.WEST;
		add(header, c);
		c.gridy++;
		c.fill = GridBagConstraints.HORIZONTAL;
		add(safariBar, c);
		c.gridy++;
		c.weighty = 1;
		c.fill = GridBagConstraints.BOTH;
		add(new JScrollPane(text), c);
		c.gridy++;
		c.weighty = 0;
		add(firefoxBar, c);

		JPanel effects = new JPanel(new GridBagLayout());

		c = new GridBagConstraints();
		c.insets = new Insets(4, 4, 4, 4);
		c.gridx = 0;
		c.gridy = 0;
		c.weightx = 0;
		c.anchor = GridBagConstraints.EAST;
		header.add(new JLabel("Show:"), c);
		c.gridx++;
		c.anchor = GridBagConstraints.WEST;
		header.add(safariButton, c);
		c.gridx++;
		header.add(firefoxButton, c);
		c.gridx++;
		header.add(findButton, c);
		c.gridy++;
		c.gridx = 0;
		c.anchor = GridBagConstraints.EAST;
		header.add(new JLabel("Effect Duration:"), c);
		c.anchor = GridBagConstraints.WEST;
		c.gridx++;
		c.gridwidth = GridBagConstraints.REMAINDER;
		header.add(duration, c);
		c.gridy++;
		c.gridx = 0;
		c.anchor = GridBagConstraints.EAST;
		c.gridwidth = 1;
		header.add(new JLabel("Effect Type:"), c);
		c.anchor = GridBagConstraints.WEST;
		c.gridx++;
		c.gridwidth = GridBagConstraints.REMAINDER;
		header.add(effects, c);

		c = new GridBagConstraints();
		c.gridx = 0;
		c.gridy = 0;
		c.weightx = 0;
		c.anchor = GridBagConstraints.WEST;
		c.insets = new Insets(0, 0, 0, 6);
		effects.add(aquaEffect, c);
		c.gridx++;
		effects.add(firefoxEffect, c);
		c.gridx++;
		effects.add(nudgeEffect, c);

		ButtonGroup group = new ButtonGroup();
		group.add(firefoxEffect);
		group.add(aquaEffect);
		group.add(nudgeEffect);

		aquaEffect.doClick();

		duration.addChangeListener(changeListener);
		firefoxEffect.addActionListener(actionListener);
		aquaEffect.addActionListener(actionListener);
		nudgeEffect.addActionListener(actionListener);
		safariButton.addActionListener(actionListener);
		firefoxButton.addActionListener(actionListener);
		findButton.addActionListener(actionListener);

		firefoxBar.addComponentListener(componentListener);
		safariBar.addComponentListener(componentListener);

		header.setOpaque(false);
		safariBar.setOpaque(false);
		firefoxBar.setOpaque(false);
		effects.setOpaque(false);

		safariButton.setOpaque(false);
		firefoxButton.setOpaque(false);
		aquaEffect.setOpaque(false);
		firefoxEffect.setOpaque(false);
		nudgeEffect.setOpaque(false);

		// added mostly for Applets
		setBackground(Color.white);
	}
}