/**
 *  Copyright © 2023-2025, Luis Andrés Lange <https://javacomm.net>
 *
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  ----------------------------------------------------------------------------
 *
 *  Exhibit B - "Incompatible With Secondary Licenses" Notice
 *
 *  This Source Code Form is "Incompatible With Secondary Licenses",
 *  as defined by the Mozilla Public License, v. 2.0.
 *
 */
package net.javacomm.window.manager;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;



/**
 * Ein Fehlerbericht wird geöffnet und die Baumansicht in allen Fenstern wird
 * aktualisiert.
 *
 * @author lromero
 *
 */
public class WindowReport {

  private List<WindowReportListener> windowReportList = Collections
      .synchronizedList(new LinkedList<WindowReportListener>());

  private String filename;

  public WindowReport() {}



  public void addWindowReportListener(WindowReportListener listener) {
    windowReportList.add(listener);
  }



  public void removeWindowReportListener(WindowReportListener listener) {
    windowReportList.remove(listener);
  }



  /**
   * Die Quelle für das Ereignis wird nicht vorgegeben.
   *
   *
   * @param event
   *              diese Quelle löste das Event aus
   */
  public void trigger(WindowReportEvent event) {
    for (WindowReportListener listener : windowReportList) {
      listener.open(event);
    }
  }



  /**
   * Diese Methode immer aufrufen, wenn der Fehlerbericht geöffnet wird. Erst
   * danach das Ereignis mit {@link #trigger(WindowReportEvent)} auslösen. Der
   * Filename erscheint im Fensterrahmen und in der Baumansicht.
   *
   * @param filename
   */
  public void setTriggerFilename(String filename) {
    this.filename = filename;
  }



  /**
   * Der Dateiname erscheint im Fenstertitel und auf der rechten Seite in der
   * Baumansicht.
   *
   * @return Dateiname
   */
  public String getTriggerFilename() {
    return filename;
  }

}
